import * as React from 'react';
export function useHistoryState(nodes, edges, setNodes, setEdges) {
    const [history, setHistory] = React.useState([]);
    const [historyIndex, setHistoryIndex] = React.useState(0);
    const [canRedo, setCanRedo] = React.useState(false);
    const takeHistorySnapshot = React.useCallback(() => {
        const newHistory = history.slice(0, historyIndex);
        setHistory([...newHistory, { nodes, edges }]);
        setHistoryIndex(newHistory.length + 1);
    }, [edges, history, historyIndex, nodes]);
    const onUndo = React.useCallback(() => {
        // If the history index is at the end, we need to take a snapshot of the current state before going back
        if (historyIndex === history.length) {
            takeHistorySnapshot();
        }
        const historyItem = history[historyIndex - 1];
        if (!historyItem)
            return;
        setNodes(historyItem.nodes);
        setEdges(historyItem.edges);
        setHistoryIndex(historyIndex - 1);
        setCanRedo(true);
    }, [history, historyIndex, setEdges, setNodes, takeHistorySnapshot]);
    const onRedo = React.useCallback(() => {
        const historyItem = history[historyIndex + 1];
        if (!historyItem)
            return;
        setNodes(historyItem.nodes);
        setEdges(historyItem.edges);
        setHistoryIndex(historyIndex + 1);
        if (historyIndex + 1 === history.length - 1) {
            setCanRedo(false);
        }
    }, [history, historyIndex, setEdges, setNodes]);
    const resetHistory = React.useCallback(() => {
        setHistory([]);
        setHistoryIndex(0);
        setCanRedo(false);
    }, []);
    // Computed states
    const isWorkflowDirty = React.useMemo(() => {
        return historyIndex !== 0 || (historyIndex === 0 && history.length > 0);
    }, [history, historyIndex]);
    const isUndoDisabled = React.useMemo(() => {
        return historyIndex === 0;
    }, [historyIndex]);
    const isRedoDisabled = React.useMemo(() => {
        return !canRedo;
    }, [canRedo]);
    return React.useMemo(() => ({
        history,
        historyIndex,
        canRedo,
        isWorkflowDirty,
        isUndoDisabled,
        isRedoDisabled,
        takeHistorySnapshot,
        onUndo,
        onRedo,
        resetHistory,
    }), [
        history,
        historyIndex,
        canRedo,
        isWorkflowDirty,
        isUndoDisabled,
        isRedoDisabled,
        takeHistorySnapshot,
        onUndo,
        onRedo,
        resetHistory,
    ]);
}
//# sourceMappingURL=use-history-state.js.map