import * as React from 'react';
import { autoResizeGroup, LayoutDefaults } from './workflow-layout-utils';
export function useGroupAutoResize(nodes, setNodes, isDragging, getStyleWidth, getStyleHeight) {
    const shouldAutoResizeGroups = React.useRef(false);
    const groupsToResize = React.useRef(new Set());
    const scheduleGroupResize = React.useCallback((groupId) => {
        if (!groupId)
            return;
        groupsToResize.current.add(groupId);
        shouldAutoResizeGroups.current = true;
    }, []);
    // Dynamically resize group nodes to fit their children
    React.useEffect(() => {
        if (!shouldAutoResizeGroups.current)
            return;
        // Only resize groups that have been specifically scheduled for resize
        // Don't auto-resize all groups to avoid conflicts with manual resizing
        const groupIds = groupsToResize.current.size
            ? new Set(Array.from(groupsToResize.current))
            : new Set();
        if (groupIds.size === 0) {
            // Reset flags if no groups to resize
            shouldAutoResizeGroups.current = false;
            groupsToResize.current.clear();
            return;
        }
        const paddingX = LayoutDefaults.group.paddingX;
        const paddingY = LayoutDefaults.group.paddingY;
        let changed = false;
        const updated = nodes.map(n => ({ ...n }));
        for (const groupId of groupIds) {
            const groupIndex = updated.findIndex(n => n.id === groupId);
            if (groupIndex < 0) {
                // Skip invalid group reference
                break;
            }
            const children = updated.filter((n) => n.parentId === groupId);
            if (children.length === 0) {
                // Nothing to resize
                break;
            }
            const currentWidthNum = getStyleWidth(updated[groupIndex], 300);
            const currentHeightNum = getStyleHeight(updated[groupIndex], 200);
            const { width: newWidth, height: newHeight, shiftX, shiftY, } = autoResizeGroup({
                children: children.map(c => ({
                    position: { x: c.position?.x ?? 0, y: c.position?.y ?? 0 },
                    width: typeof c.width === 'number' ? c.width : undefined,
                    height: typeof c.height === 'number' ? c.height : undefined,
                })),
                currentStyle: { width: currentWidthNum, height: currentHeightNum },
                paddingX,
                paddingY,
                minWidth: 300,
                minHeight: 200,
            });
            const currentWidth = currentWidthNum;
            const currentHeight = currentHeightNum;
            if (newWidth !== currentWidth || newHeight !== currentHeight || shiftX > 0 || shiftY > 0) {
                if ((shiftX > 0 || shiftY > 0) && !isDragging.current) {
                    for (let i = 0; i < updated.length; i += 1) {
                        const n = updated[i];
                        if (n.parentId === groupId) {
                            updated[i] = {
                                ...n,
                                position: {
                                    x: (n.position?.x ?? 0) + shiftX,
                                    y: (n.position?.y ?? 0) + shiftY,
                                },
                            };
                        }
                    }
                }
                updated[groupIndex] = {
                    ...updated[groupIndex],
                    style: { ...updated[groupIndex].style, width: newWidth, height: newHeight },
                };
                // Do not reposition anchored repeat; keep its absolute position unchanged
                changed = true;
            }
        }
        if (changed) {
            setNodes(updated);
        }
        // Reset flags
        shouldAutoResizeGroups.current = false;
        groupsToResize.current.clear();
    }, [nodes, setNodes, getStyleWidth, getStyleHeight, isDragging]);
    return React.useMemo(() => ({
        scheduleGroupResize,
    }), [scheduleGroupResize]);
}
//# sourceMappingURL=use-group-auto-resize.js.map