import * as React from 'react';
import { computeGroupAndAnchors } from './workflow-layout-utils';
/**
 * Hook that handles post-layout alignment of groups and anchored repeats.
 * Centralizes the logic for positioning and centering related elements.
 */
export function useAnchoredRepeatLayout(nodes, setNodes) {
    // Post-layout alignment: ensure anchored repeat and group are centered and half-overlapped on initial render
    React.useEffect(() => {
        const groupsWithAnchors = nodes.filter(n => n.type === 'group' && n.data?.anchoredRepeatId);
        if (groupsWithAnchors.length === 0)
            return;
        let changed = false;
        const updated = nodes.map(n => ({ ...n }));
        groupsWithAnchors.forEach(g => {
            const repeatId = g.data.anchoredRepeatId;
            const rIndex = updated.findIndex(n => n.id === repeatId);
            if (rIndex < 0) {
                return;
            }
            const r = updated[rIndex];
            const endRepeatId = g.data.anchoredEndRepeatId;
            const endRepeatIndex = endRepeatId ? updated.findIndex(n => n.id === endRepeatId) : -1;
            const groupWidth = g.style?.width ?? g.width ?? 410;
            const groupHeight = g.style?.height ?? g.height ?? 170;
            const repeatWidth = r.width ?? 260;
            const repeatHeight = r.height ?? 120;
            const { groupPos: desiredGroup, anchoredRepeatPos: desiredRepeat, anchoredEndPos, } = computeGroupAndAnchors({
                repeatPos: { x: r.position?.x ?? 0, y: r.position?.y ?? 0 },
                repeatSize: { width: repeatWidth, height: repeatHeight },
                groupSize: { width: groupWidth, height: groupHeight },
                endSize: {
                    width: updated.find(n => n.id === g.data.anchoredEndRepeatId)?.width ?? 120,
                    height: updated.find(n => n.id === g.data.anchoredEndRepeatId)?.height ?? 32,
                },
            });
            if ((g.position?.x ?? 0) !== desiredGroup.x || (g.position?.y ?? 0) !== desiredGroup.y) {
                const gIndex = updated.findIndex(n => n.id === g.id);
                if (gIndex >= 0) {
                    updated[gIndex] = {
                        ...updated[gIndex],
                        position: { x: desiredGroup.x, y: desiredGroup.y },
                    };
                    changed = true;
                }
            }
            if ((r.position?.x ?? 0) !== desiredRepeat.x || (r.position?.y ?? 0) !== desiredRepeat.y) {
                updated[rIndex] = { ...r, position: { x: desiredRepeat.x, y: desiredRepeat.y } };
                changed = true;
            }
            // Keep End repeat centered at group boundary (50% inside, 50% outside)
            if (endRepeatIndex >= 0) {
                const er = updated[endRepeatIndex];
                if ((er.position?.x ?? 0) !== anchoredEndPos.x || (er.position?.y ?? 0) !== anchoredEndPos.y) {
                    updated[endRepeatIndex] = { ...er, position: anchoredEndPos };
                    changed = true;
                }
            }
        });
        if (changed) {
            setNodes(updated);
        }
    }, [nodes, setNodes]);
}
//# sourceMappingURL=use-anchored-repeat-layout.js.map