import { Handle, Position } from 'reactflow';
import * as React from 'react';
import { getTextForLocale } from '@sage/xtrem-shared';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import { openPageDialog } from '../../../../service/dialog-service';
import { removeTransientNodeDataProperties, useSubsequentWorkflowNodes, useWorkflowNodeVariables, } from '../workflow-component-utils';
import { resolveDetailedIcon } from '../../../../utils/detailed-icons-utils';
import { WorkflowContext } from '../workflow-context-provider';
import Button from 'carbon-react/esm/components/button';
import SplitButton from 'carbon-react/esm/components/split-button';
import { localize } from '../../../../service/i18n-service';
import { getStore } from '../../../../redux';
import * as tokens from '@sage/design-tokens/js/base/common';
export function StandardNode(props) {
    const { data, isConnectable, id, nodeType, configurationPage } = props;
    const context = React.useContext(WorkflowContext);
    const subsequentNodes = useSubsequentWorkflowNodes(id, context?.screenId, context?.elementId);
    const locale = getStore().getState().applicationContext?.locale || 'base';
    const { inputVariables, oldRootPaths } = useWorkflowNodeVariables(id, context?.screenId, context?.elementId);
    const eventLog = React.useMemo(() => {
        return data.eventLog;
    }, [data.eventLog]);
    const errorMessage = React.useMemo(() => {
        if (data.message) {
            return data.message;
        }
        if (eventLog?.event === 'error') {
            return eventLog?.message;
        }
        return null;
    }, [data.message, eventLog?.event, eventLog?.message]);
    const headerLabel = React.useMemo(() => {
        if (nodeType?.type === 'event') {
            return localize('@sage/xtrem-ui/workflow-component-header-label-start', 'Starts when');
        }
        return localize('@sage/xtrem-ui/workflow-component-header-label-action', 'Do');
    }, [nodeType?.type]);
    const onConfigurationOpen = React.useCallback(async (isReadOnly) => {
        if (!configurationPage || !context?.onNodeDataChange) {
            return;
        }
        try {
            const result = (await openPageDialog(configurationPage, {
                inputVariables: inputVariables,
                oldRootPaths: oldRootPaths,
                isReadOnly: Boolean(isReadOnly),
                errorMessage: errorMessage || '',
            }, {
                values: removeTransientNodeDataProperties(data),
                rightAligned: true,
                size: 'extra-large',
            }));
            // Preserve internal grouping metadata if present (e.g., groupId for repeat)
            const preserved = {};
            if (data?.groupId)
                preserved.groupId = data.groupId;
            if (result) {
                context.onNodeDataChange(id, {
                    ...result,
                    ...preserved,
                    type: data.type,
                });
            }
        }
        catch (e) {
            // intentionally left empty
        }
    }, [configurationPage, inputVariables, oldRootPaths, data, context, id, errorMessage]);
    const onInsertBelowButtonClick = React.useCallback(() => {
        context?.onNodeBelowNode(id);
    }, [context, id]);
    const onInsertBelowConditionButtonClick = React.useCallback(() => {
        context?.onNodeBelowNode(id, 'condition');
    }, [context, id]);
    const onInsertBelowActionsButtonClick = React.useCallback(() => {
        context?.onNodeBelowNode(id, 'action');
    }, [context, id]);
    // Open Add dialog filtered to Repeat
    const onInsertBelowRepeatOpenDialog = React.useCallback(() => {
        context?.onNodeBelowNode(id, 'repeat-open-dialog');
    }, [context, id]);
    const showAddButton = !context?.isReadOnly && subsequentNodes.length === 0;
    return (React.createElement("div", { className: "e-workflow-node-wrapper" },
        React.createElement(Handle, { type: "target", position: Position.Top, id: "in", isConnectable: isConnectable },
            React.createElement("div", { className: "e-workflow-node-header" },
                React.createElement("div", { className: "e-workflow-node-header-content", style: {
                        backgroundColor: 'var(--colorsSemanticNeutral500)',
                        color: 'var(--colorsYang100)',
                    } }, headerLabel))),
        React.createElement("div", { className: `e-workflow-node e-workflow-node-${nodeType.type} e-workflow-node-${nodeType.key} ${eventLog?.event ? `e-workflow-node-event-${eventLog.event}` : ''}`, "data-testid": `e-workflow-node e-workflow-node-${nodeType.type} e-workflow-node-${nodeType.key}`, "data-nodeid": id },
            React.createElement("div", { className: "e-workflow-node-icon-wrapper", style: { backgroundColor: nodeType.color } },
                React.createElement("div", { className: "e-workflow-node-icon", style: {
                        mask: nodeType.icon
                            ? `url(${resolveDetailedIcon(nodeType.icon)}) no-repeat left center`
                            : undefined,
                        maskSize: '24px 24px',
                        backgroundColor: 'var(--colorsUtilityYang100)',
                    } })),
            React.createElement("div", { className: `e-workflow-node-body e-workflow-node-${nodeType.type}-body` },
                React.createElement("div", { className: "e-workflow-node-title" },
                    React.createElement("div", { className: "e-workflow-node-title-label" }, getTextForLocale(data.localizedTitle, locale) || data.title || nodeType.title),
                    errorMessage && (React.createElement("div", { className: "e-workflow-node-error-message" },
                        React.createElement(Icon, { type: "error", tooltipMessage: errorMessage, ariaLabel: errorMessage, color: tokens.colorsSemanticNegative500, tooltipBgColor: tokens.colorsSemanticNegative500, tooltipFontColor: tokens.colorsYang100 }))),
                    configurationPage && !context?.isReadOnly && (React.createElement(IconButton, { "data-pendoid": "workflow-node-open-configuration", onClick: () => onConfigurationOpen(false) },
                        React.createElement(Icon, { ml: "8px", type: "edit", color: "--colorsUtilityYin090" })))),
                data.subtitle && React.createElement("div", { className: "e-workflow-node-subtitle" }, data.subtitle),
                data.details && React.createElement("div", { className: "e-workflow-node-details" }, data.details)),
            (eventLog || context?.isReadOnly) && (React.createElement("div", { className: "e-workflow-node-event-view-details", onClick: () => onConfigurationOpen(true) },
                React.createElement(Icon, { type: "view" }))),
            React.createElement(Handle, { type: "source", position: Position.Bottom, id: "out", isConnectable: isConnectable })),
        showAddButton && (React.createElement("div", { className: "e-workflow-node-add-button" },
            React.createElement(SplitButton, { "data-pendoid": "workflow-node-add-step", text: localize('@sage/xtrem-ui/workflow-component-add-step', 'Add step'), size: "small", iconType: "plus", onClick: onInsertBelowButtonClick },
                React.createElement(Button, { "data-pendoid": "workflow-node-add-action", iconType: "tick", onClick: onInsertBelowActionsButtonClick }, localize('@sage/xtrem-ui/workflow-component-add-action', 'Add action')),
                React.createElement(Button, { "data-pendoid": "workflow-node-add-condition", iconType: "split", onClick: onInsertBelowConditionButtonClick }, localize('@sage/xtrem-ui/workflow-component-add-condition', 'Add condition')),
                React.createElement(Button, { "data-pendoid": "workflow-node-add-repeat", iconType: "add", onClick: onInsertBelowRepeatOpenDialog }, localize('@sage/xtrem-ui/workflow-component-add-repeat', 'Add repeat')))))));
}
//# sourceMappingURL=standard-node.js.map