import { Handle, Position } from 'reactflow';
import * as React from 'react';
import { getTextForLocale } from '@sage/xtrem-shared';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import { openPageDialog } from '../../../../service/dialog-service';
import { hasEdgeConnectedToNode, removeTransientNodeDataProperties, useWorkflowNodeVariables, } from '../workflow-component-utils';
import { WorkflowContext } from '../workflow-context-provider';
import { localize } from '../../../../service/i18n-service';
import { getStore } from '../../../../redux';
import { ConditionNodeLabel } from './condition-node-label';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import { useFieldValue } from '../../../../utils/hooks/effects/use-set-field-value';
import * as tokens from '@sage/design-tokens/js/base/common';
export function ConditionNode(props) {
    const { data, isConnectable, id, nodeType, configurationPage } = props;
    const context = React.useContext(WorkflowContext);
    const locale = getStore().getState().applicationContext?.locale || 'base';
    const { inputVariables, oldRootPaths } = useWorkflowNodeVariables(id, context?.screenId, context?.elementId);
    const fieldValue = useFieldValue(context?.screenId, context?.elementId);
    const [hasOutgoingTrueEdge, hasOutgoingFalseEdge] = React.useMemo(() => {
        const trueBranch = hasEdgeConnectedToNode(id, fieldValue?.edges || [], 'out-true');
        const falseBranch = hasEdgeConnectedToNode(id, fieldValue?.edges || [], 'out-false');
        return [trueBranch, falseBranch];
    }, [id, fieldValue?.edges]);
    const eventLog = React.useMemo(() => {
        return data.eventLog;
    }, [data.eventLog]);
    const errorMessage = React.useMemo(() => {
        if (data.message) {
            return data.message;
        }
        if (eventLog?.event === 'error') {
            return eventLog?.message;
        }
        return null;
    }, [data.message, eventLog?.event, eventLog?.message]);
    const headerLabel = localize('@sage/xtrem-ui/workflow-component-header-label-condition', 'Condition');
    const onConfigurationOpen = React.useCallback(async (isReadOnly) => {
        if (!configurationPage || !context?.onNodeDataChange) {
            return;
        }
        try {
            const result = (await openPageDialog(configurationPage, {
                inputVariables: inputVariables,
                oldRootPaths: oldRootPaths,
                isReadOnly: Boolean(isReadOnly),
                errorMessage: errorMessage || '',
            }, {
                values: removeTransientNodeDataProperties(data),
                rightAligned: true,
                size: 'extra-large',
            }));
            // Preserve internal grouping metadata if present (e.g., groupId for repeat)
            const preserved = {};
            if (data?.groupId)
                preserved.groupId = data.groupId;
            if (result) {
                context.onNodeDataChange(id, {
                    ...result,
                    ...preserved,
                    type: data.type,
                });
            }
        }
        catch (e) {
            // intentionally left empty
        }
    }, [configurationPage, inputVariables, oldRootPaths, data, context, id, errorMessage]);
    const onInsertTrueButtonClick = React.useCallback(() => {
        context?.onNodeBelowNode(id, undefined, 'out-true');
    }, [context, id]);
    const onInsertFalseButtonClick = React.useCallback(() => {
        context?.onNodeBelowNode(id, undefined, 'out-false');
    }, [context, id]);
    return (React.createElement("div", { className: "e-workflow-node-wrapper" },
        React.createElement("div", { className: "e-workflow-node-floating-buttons" },
            data.ifTrueBranch && !hasOutgoingTrueEdge && (React.createElement(ButtonMinor, { "data-pendoid": "workflow-node-add-true-branch", iconType: "plus", buttonType: "secondary", size: "small", onClick: onInsertTrueButtonClick, iconTooltipMessage: localize('@sage/xtrem-ui/workflow-component-add-step', 'Add step'), "aria-label": localize('@sage/xtrem-ui/workflow-component-add-step', 'Add step'), className: "e-workflow-node-floating-button-left e-workflow-add-node-on-edge-button" })),
            data.ifFalseBranch && !hasOutgoingFalseEdge && (React.createElement(ButtonMinor, { "data-pendoid": "workflow-node-add-false-branch", iconType: "plus", buttonType: "secondary", size: "small", onClick: onInsertFalseButtonClick, iconTooltipMessage: localize('@sage/xtrem-ui/workflow-component-add-step', 'Add step'), "aria-label": localize('@sage/xtrem-ui/workflow-component-add-step', 'Add step'), className: "e-workflow-node-floating-button-right e-workflow-add-node-on-edge-button" }))),
        React.createElement(Handle, { type: "target", position: Position.Top, id: "in", isConnectable: isConnectable },
            React.createElement("div", { className: "e-workflow-node-header" },
                React.createElement("div", { className: "e-workflow-node-header-content", style: {
                        backgroundColor: 'var(--colorsSemanticNeutral500)',
                        color: 'var(--colorsYang100)',
                    } }, headerLabel))),
        React.createElement("div", { className: `e-workflow-node e-workflow-node-${nodeType.type} e-workflow-node-${nodeType.key} ${eventLog?.event ? `e-workflow-node-event-${eventLog.event}` : ''}`, "data-testid": `e-workflow-node e-workflow-node-${nodeType.type} e-workflow-node-${nodeType.key}`, "data-nodeid": id },
            React.createElement("div", { className: "e-workflow-node-body e-workflow-node-condition-body" },
                React.createElement("div", { className: "e-workflow-node-title" },
                    React.createElement("div", { className: "e-workflow-node-title-label" }, getTextForLocale(data.localizedTitle, locale) || data.title || nodeType.title),
                    errorMessage && (React.createElement("div", { className: "e-workflow-node-error-message" },
                        React.createElement(Icon, { type: "error", tooltipMessage: errorMessage, ariaLabel: errorMessage, color: tokens.colorsSemanticNegative500, tooltipBgColor: tokens.colorsSemanticNegative500, tooltipFontColor: tokens.colorsYang100 }))),
                    configurationPage && !context?.isReadOnly && (React.createElement(IconButton, { "data-pendoid": "workflow-node-open-configuration", onClick: () => onConfigurationOpen(false) },
                        React.createElement(Icon, { type: "edit", color: "--colorsUtilityYin090" })))),
                data.subtitle && React.createElement("div", { className: "e-workflow-node-subtitle" }, data.subtitle),
                data.details && React.createElement("div", { className: "e-workflow-node-details" }, data.details)),
            (eventLog || context?.isReadOnly) && (React.createElement("div", { className: "e-workflow-node-event-view-details", onClick: () => onConfigurationOpen(true) },
                React.createElement(Icon, { type: "view" }))),
            (data.ifTrueBranch || hasOutgoingTrueEdge) && (React.createElement(ConditionNodeLabel, { sourceHandleId: "out-true", sourceX: -55, sourceY: -40 })),
            (data.ifFalseBranch || hasOutgoingFalseEdge) && (React.createElement(ConditionNodeLabel, { sourceHandleId: "out-false", sourceX: 250, sourceY: -40 })),
            React.createElement(Handle, { type: "source", position: Position.Right, id: "out-false", isConnectable: isConnectable }),
            React.createElement(Handle, { type: "source", position: Position.Left, id: "out-true", isConnectable: isConnectable }))));
}
//# sourceMappingURL=condition-node.js.map