import * as React from 'react';
import Button from 'carbon-react/esm/components/button';
import Form from 'carbon-react/esm/components/form';
import { WorkflowNodeSelectorComponent } from './workflow-node-selector-component';
import { StepSequence, StepSequenceItem } from 'carbon-react/esm/components/step-sequence';
import Typography from 'carbon-react/esm/components/typography';
import { ScreenComponent } from '../../ui/screen-component';
import { useWorkflowNodeVariables } from './workflow-component-utils';
import BusinessActions from '../../container/footer/business-actions';
import { ContextType } from '../../../types';
import { getArtifactDescription } from '../../../utils/transformers';
import { getPageTitlesFromPageDefinition } from '../../../utils/page-utils';
import { localize } from '../../../service/i18n-service';
import Sidebar from 'carbon-react/esm/components/sidebar';
import { useDeepEqualSelector } from '../../../utils/hooks/use-deep-equal-selector';
export function AddWorkflowNodeDialog({ isOpen, onClose, onConfirm, previousNodeId, screenId, elementId, filterType, }) {
    const workflowNodes = useDeepEqualSelector(s => s.workflowNodes);
    const { inputVariables, oldRootPaths } = useWorkflowNodeVariables(previousNodeId, screenId, elementId, true);
    const [selectedNodeType, setSelectedNodeType] = React.useState(null);
    const [step, setStep] = React.useState(1);
    const selectedWorkflowType = workflowNodes?.find(n => n.key === selectedNodeType);
    const defaultConfig = JSON.parse(selectedWorkflowType?.defaultConfig || '{}');
    const configPageArtifactDescription = selectedWorkflowType
        ? getArtifactDescription(selectedWorkflowType.configurationPage)
        : null;
    const configPageDefinition = useDeepEqualSelector(s => {
        if (!configPageArtifactDescription) {
            return null;
        }
        return s.screenDefinitions[configPageArtifactDescription.name] || null;
    });
    const availableWorkflowNodes = React.useMemo(() => {
        const all = workflowNodes || [];
        if (!filterType || filterType.length === 0)
            return all;
        return all.filter(n => {
            // Standard filter by node.type (e.g., 'event', 'action', 'condition')
            if (filterType.indexOf(n.type) !== -1)
                return true;
            // Extra: allow filtering by key base (e.g., 'repeat')
            const baseKey = (n.key || '').split('/')?.[0];
            if (filterType.includes(baseKey))
                return true;
            return false;
        });
    }, [filterType, workflowNodes]);
    const onWorkflowConfigPageFinish = React.useCallback((values) => {
        if (selectedNodeType) {
            onConfirm({ selectedNodeType, values });
        }
    }, [onConfirm, selectedNodeType]);
    const onPositiveButtonClick = React.useCallback(() => {
        if (step === 1) {
            setStep(2);
        }
    }, [step]);
    const onNegativeButtonClick = React.useCallback(() => {
        if (step === 1) {
            onClose();
        }
        if (step === 2) {
            setStep(1);
        }
    }, [onClose, step]);
    React.useEffect(() => {
        setStep(1);
        setSelectedNodeType(null);
    }, [isOpen]);
    React.useEffect(() => {
        if (availableWorkflowNodes?.length === 1 && step === 1) {
            setStep(2);
            setSelectedNodeType(availableWorkflowNodes[0].key);
        }
    }, [availableWorkflowNodes, step]);
    const dialogTitle = React.useMemo(() => {
        if (!configPageDefinition || step === 1) {
            if (filterType?.length === 1 && filterType[0] === 'event') {
                return localize('@sage/xtrem-ui/workflow-component-wizard-title-trigger', 'Trigger gallery');
            }
            return localize('@sage/xtrem-ui/workflow-component-wizard-title-action', 'Action gallery');
        }
        const { title } = getPageTitlesFromPageDefinition(configPageDefinition, 'en-US');
        return title;
    }, [configPageDefinition, filterType, step]);
    const hiddenCompleteLabel = localize('@sage/xtrem-ui/step-sequence-item-aria-complete', 'Complete');
    const hiddenCurrentLabel = localize('@sage/xtrem-ui/step-sequence-item-aria-current', 'Current');
    return (React.createElement(Sidebar, { open: isOpen, onCancel: onClose, enableBackgroundUI: true, size: "extra-large", "data-element": "workflow-dialog", header: React.createElement("div", null,
            React.createElement(Typography, { variant: "h1" }, dialogTitle),
            availableWorkflowNodes?.length !== 1 && (React.createElement(StepSequence, { mb: "0", mt: "16px" },
                React.createElement(StepSequenceItem, { "aria-label": localize('@sage/xtrem-ui/step-sequence-item-aria-count', 'Step {{0}} of {{1}}', [1, 2]), hiddenCompleteLabel: hiddenCompleteLabel, hiddenCurrentLabel: hiddenCurrentLabel, indicator: "1", status: step === 1 ? 'current' : 'complete' }, filterType?.length === 1 && filterType[0] === 'event'
                    ? localize('@sage/xtrem-ui/workflow-component-wizard-trigger-selection', 'Trigger selection')
                    : localize('@sage/xtrem-ui/workflow-component-wizard-event-selection', 'Event selection')),
                React.createElement(StepSequenceItem, { "aria-label": localize('@sage/xtrem-ui/step-sequence-item-aria-count', 'Step {{0}} of {{1}}', [2, 2]), hiddenCompleteLabel: hiddenCompleteLabel, hiddenCurrentLabel: hiddenCurrentLabel, indicator: "2", status: step === 1 ? 'incomplete' : 'current' }, localize('@sage/xtrem-ui/workflow-component-wizard-step-configuration', 'Configuration'))))) },
        React.createElement(Form, { "data-testid": "e-workflow-node-dialog-form", stickyFooter: true, height: "500px", leftSideButtons: availableWorkflowNodes?.length > 1 && (React.createElement(Button, { onClick: onNegativeButtonClick }, step === 1
                ? localize('@sage/xtrem-ui/cancel', 'Cancel')
                : localize('@sage/xtrem-ui/wizard-previous', 'Previous'))), saveButton: configPageDefinition && step === 2 ? (React.createElement(BusinessActions, { contextType: ContextType.dialog, defaultButtonType: "primary", screenId: configPageDefinition.metadata.screenId, businessActions: configPageDefinition.page.$.businessActions })) : (React.createElement(Button, { buttonType: "primary", type: "button", onClick: onPositiveButtonClick, disabled: !selectedNodeType || step === 2 }, localize('@sage/xtrem-ui/wizard-next', 'Next'))) },
            step === 1 && (React.createElement("div", { className: "e-workflow-add-dialog" },
                React.createElement(WorkflowNodeSelectorComponent, { onChange: setSelectedNodeType, value: selectedNodeType, workflowNodes: availableWorkflowNodes }))),
            step === 2 && selectedWorkflowType && (React.createElement(ScreenComponent, { screenPath: selectedWorkflowType.configurationPage, queryParameters: { inputVariables, oldRootPaths }, values: { ...defaultConfig }, onFinish: onWorkflowConfigPageFinish })))));
}
//# sourceMappingURL=add-workflow-node-dialog.js.map