/**
 * @packageDocumentation
 * @module root
 * */
import { partition } from 'lodash';
import { standardDecoratorImplementation, standardExtensionDecoratorImplementation, } from '../../../utils/decorator-utils';
import { AbstractFieldDecorator } from '../../abstract-field-decorator';
import { AbstractFieldLayoutBuilder } from '../../abstract-field-layout-builder';
import { TableControlObject } from '../../control-objects';
import { allowedMainFieldTypes } from '../../nested-fields';
import { FieldKey } from '../../types';
import { addColumnsToProperties, addMobileCardDefinitionToProperties, addNodeToProperties, } from '../../../utils/data-type-utils';
class TableDecorator extends AbstractFieldDecorator {
    constructor() {
        super(...arguments);
        this._layout = AbstractFieldLayoutBuilder;
        this._controlObjectConstructor = TableControlObject;
    }
    getComponentPropertiesFromDataType(dataType, propertyDetails) {
        const properties = {};
        addNodeToProperties({ dataType, propertyDetails, properties });
        addColumnsToProperties({
            dataType,
            propertyDetails,
            properties,
            dataTypes: this.dataTypes,
            nodeTypes: this.nodeTypes,
        });
        addMobileCardDefinitionToProperties({
            dataType,
            propertyDetails,
            properties,
            dataTypes: this.dataTypes,
            nodeTypes: this.nodeTypes,
        });
        return properties;
    }
}
/**
 * Initializes the decorated member as a [Table]{@link TableControlObject} field with the provided properties
 *
 * @param properties The properties that the [Table]{@link TableControlObject} field will be initialized with
 */
export function tableField(properties) {
    // Backwards support of the renamed rowAction property. In case of X3, they may deploy older platform with new app code. We should remove this around August 2023.
    if (properties.rowActions) {
        // eslint-disable-next-line no-param-reassign
        properties = {
            ...properties,
            dropdownActions: properties.rowActions,
        };
    }
    if (Object.prototype.hasOwnProperty.call(properties, 'mainField')) {
        const mainField = properties.mainField;
        const [allowedColumns, disallowedColumns] = partition(properties.columns, c => allowedMainFieldTypes.includes(c.type));
        const allowedColumnBinds = allowedColumns.map(c => `"${c.properties.bind}"`);
        if (disallowedColumns.find(c => c.properties.bind === mainField)) {
            throw new Error(`"${mainField}" cannot be used as "mainField". Please use one of the following values: ${allowedColumnBinds.join(', ')}.`);
        }
        if (!allowedColumns.find(c => c.properties.bind === mainField)) {
            throw new Error(`"${mainField}" cannot be used as "mainField". Please use one of the following values: ${allowedColumnBinds.join(', ')} or add a new column bound to "${mainField}".`);
        }
    }
    return standardDecoratorImplementation(properties, TableDecorator, FieldKey.Table, true);
}
export function tableFieldOverride(properties) {
    return standardExtensionDecoratorImplementation(properties);
}
//# sourceMappingURL=table-decorator.js.map