import { CsvExportModule } from '@ag-grid-community/csv-export';
import { AgGridReact } from '@ag-grid-community/react';
import { EnterpriseCoreModule } from '@ag-grid-enterprise/core';
import { ExcelExportModule } from '@ag-grid-enterprise/excel-export';
import { MenuModule } from '@ag-grid-enterprise/menu';
import { RichSelectModule } from '@ag-grid-enterprise/rich-select';
import { RowGroupingModule } from '@ag-grid-enterprise/row-grouping';
import { ServerSideRowModelModule } from '@ag-grid-enterprise/server-side-row-model';
import { SetFilterModule } from '@ag-grid-enterprise/set-filter';
import { isValidDatePropertyValue } from '@sage/xtrem-date-time';
import { flat, objectKeys } from '@sage/xtrem-shared';
import { cloneDeep, difference, get, isEmpty, isEqual, isNil, set, sortBy } from 'lodash';
import * as React from 'react';
import { renderToStaticMarkup } from 'react-dom/server';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { ActionType } from '../../../redux';
import { subscribeToActions } from '../../../redux/middleware/action-subscription-middleware';
import { setAgGridLicence } from '../../../service/ag-grid-licence-service';
import { RecordActionType } from '../../../service/collection-data-types';
import { getGroupKey } from '../../../service/collection-data-utils';
import { localize } from '../../../service/i18n-service';
import { getScreenElement } from '../../../service/screen-base-definition';
import { exportTableData } from '../../../service/table-export-service';
import { showToast } from '../../../service/toast-service';
import { ContextType } from '../../../types';
import { findColumnDefinitionByBind, getNestedFieldElementId } from '../../../utils/abstract-fields-utils';
import { cellExportFormatter, getColumnsToExport, getRowClassRules, } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import { COLUMN_ID_AUTO_COLUMN, COLUMN_ID_LINE_NUMBER, COLUMN_ID_ROW_ACTIONS, COLUMN_ID_ROW_SELECTION, COLUMN_ID_VALIDATIONS, frameworkComponents, getSelectionColumnDef, } from '../../../utils/ag-grid/ag-grid-column-config';
import { onRowClick } from '../../../utils/ag-grid/ag-grid-event-handlers';
import { getColumns } from '../../../utils/ag-grid/ag-grid-service';
import localeText from '../../../utils/ag-grid/ag-grid-strings';
import { GROUP_ROW_HEIGHT, ROW_HEIGHT, callGridMethod, copySelectedCellValue, getColumnStatesForColumnPanel, getFilterModel, getFirstEditableColumn, getSafeGridApiContext, getSelectionFilter, getTotalRecordCount, mapAgGridFilterToXtremFilters, pasteToSelectedCell, tryToCommitPhantomRow, } from '../../../utils/ag-grid/ag-grid-table-utils';
import { xtremConsole } from '../../../utils/console';
import { isRowValueInDateGroup } from '../../../utils/date-utils';
import { triggerFieldEvent } from '../../../utils/events';
import { convertDeepBindToPath, convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition } from '../../../utils/state-utils';
import { AUTO_COLUMN_ID, getActiveOptionsMenu, getCardDefinitionFromColumns, getOrderByFromSortModel, getTableContext, getTableViewColumnHidden, getTableViewColumnOrder, getTableViewFilter, getTableViewGrouping, getTableViewSortOrder, isColumnBindVariant, setTableContext, } from '../../../utils/table-component-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { hasAgGridLogging } from '../../../utils/window';
import { navigationPanelId } from '../../container/navigation-panel/navigation-panel-types';
import { FieldKey, SET } from '../../types';
import { AsyncCalendarBodyComponent } from '../../ui/calendar-body/async-calendar-body-component';
import { Icon } from '../../ui/icon/icon-component';
import { TableLoadingCellRenderer } from '../../ui/table-shared/cell/table-loading-cell-renderer';
import { DesktopTableBulkActionBar } from '../../ui/table-shared/desktop-table-bulk-action-bar';
import { DesktopTableHeaderComponent } from '../../ui/table-shared/desktop-table-header';
import { TableConfigurationDialog } from '../../ui/table-shared/table-configuration-dialog/table-configuration-dialog';
import { generateFieldId, getFieldTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { getReferenceValueField } from '../reference/reference-utils';
const activatedAgGridModules = [
    ServerSideRowModelModule,
    EnterpriseCoreModule,
    MenuModule,
    RichSelectModule,
    SetFilterModule,
    RowGroupingModule,
    ExcelExportModule,
    CsvExportModule,
];
var DataOperationMode;
(function (DataOperationMode) {
    DataOperationMode[DataOperationMode["NONE"] = 0] = "NONE";
    DataOperationMode[DataOperationMode["FETCH_PAGE"] = 1] = "FETCH_PAGE";
    DataOperationMode[DataOperationMode["REMOVE_RECORD"] = 2] = "REMOVE_RECORD";
    DataOperationMode[DataOperationMode["RESET_TABLE"] = 3] = "RESET_TABLE";
    DataOperationMode[DataOperationMode["FILTER_ERRORS"] = 4] = "FILTER_ERRORS";
})(DataOperationMode || (DataOperationMode = {}));
export class DesktopTableComponent extends React.Component {
    constructor(props) {
        super(props);
        this.gridApi = null;
        this.gridId = generateFieldId({
            screenId: this.props.screenId,
            elementId: this.props.elementId,
            fieldProperties: this.props.fieldProperties,
            isNested: false,
        });
        this.dataOperationMode = DataOperationMode.NONE;
        this.autoSizeStrategy = {
            type: 'fitCellContents',
        };
        this.isFetching = false;
        this.isFilteringErrors = false;
        this.containerRef = React.createRef();
        this.defaultColDef = { flex: 1, resizable: false, singleClickEdit: true };
        this.loadingCellRendererParams = {
            elementId: this.props.elementId,
        };
        this.loadingOverlayComponentParams = { size: 'large' };
        this.getRowNode = ({ id, gridApi = this.gridApi, }) => {
            let node = callGridMethod(gridApi, 'getRowNode', id);
            if (node) {
                return node;
            }
            node = gridApi?.getPinnedTopRow(0);
            if (node?.data?._id === id) {
                return node;
            }
            return undefined;
        };
        this.focusPhantomRowAndStartEditing = () => {
            setTimeout(async () => {
                if (!this.gridApi) {
                    return;
                }
                const phantomRecords = await this.props.value.getPhantomRecords();
                const colKey = getFirstEditableColumn(this.gridApi, phantomRecords?.[0]);
                if (colKey === undefined) {
                    return;
                }
                callGridMethod(this.gridApi, 'setFocusedCell', 0, colKey, 'top');
                callGridMethod(this.gridApi, 'ensureColumnVisible', colKey);
                callGridMethod(this.gridApi, 'startEditingCell', {
                    colKey,
                    rowIndex: 0,
                    key: 'Enter',
                    rowPinned: 'top',
                });
            }, 50);
        };
        this.showHideNoRowsOverlay = (show) => {
            setTimeout(() => {
                if (show) {
                    callGridMethod(this.gridApi, 'showNoRowsOverlay');
                }
                else {
                    callGridMethod(this.gridApi, 'hideOverlay');
                }
            }, 100);
        };
        this.onCollectionUpdated = (type, rowValue) => {
            if (this.gridApi) {
                if (type === RecordActionType.MODIFIED) {
                    if (rowValue?.__phantom && rowValue?.__forceRowUpdate) {
                        delete rowValue.__forceRowUpdate;
                        this.setState({ phantomRows: [rowValue] });
                    }
                    else {
                        // In this case we target one specific row in order to reduce the amplitude of rendering
                        callGridMethod(this.gridApi, 'applyServerSideTransaction', {
                            update: [rowValue],
                        });
                    }
                }
                else if (rowValue?.__phantom !== undefined && type === RecordActionType.ADDED) {
                    this.setState({ phantomRows: [rowValue] }, () => {
                        if (this.props.isTableInFocus) {
                            this.focusPhantomRowAndStartEditing();
                        }
                    });
                }
                else if (type === RecordActionType.ADDED) {
                    this.dataOperationMode = DataOperationMode.FETCH_PAGE;
                    const nodes = this.props.value.getData({ noLimit: true });
                    const addIndex = nodes.findIndex(n => n._id === rowValue._id);
                    callGridMethod(this.gridApi, 'applyServerSideTransaction', {
                        add: [rowValue],
                        addIndex: addIndex === -1 ? 0 : addIndex,
                    });
                    this.redrawLineNumbers();
                }
                else if (type === RecordActionType.REMOVED) {
                    this.dataOperationMode = DataOperationMode.REMOVE_RECORD;
                    callGridMethod(this.gridApi, 'applyServerSideTransaction', {
                        remove: [rowValue],
                    });
                    this.redrawLineNumbers();
                }
                const tableHasData = this.props.value &&
                    this.props.value.getData &&
                    this.props.value.getData({ temporaryRecords: this.props.additionalLookupRecords?.() }).length > 0;
                this.setState({ tableHasData });
                this.showHideNoRowsOverlay(!tableHasData);
            }
        };
        this.applyTableView = () => {
            callGridMethod(this.gridApi, 'setFilterModel', getTableViewFilter(this.props.tableUserSettings));
            const currentColumnState = callGridMethod(this.gridApi, 'getColumnState');
            if (currentColumnState) {
                const cleanCurrentColumnState = cloneDeep(currentColumnState).map(c => c.colId === AUTO_COLUMN_ID
                    ? c
                    : {
                        ...c,
                        rowGroup: false,
                        sort: null,
                        sortIndex: null,
                        rowGroupIndex: null,
                    });
                const columnVisibility = getTableViewColumnHidden(this.props.tableUserSettings);
                const newTableViewGrouping = getTableViewGrouping(this.props.tableUserSettings);
                const newTableSortOrder = getTableViewSortOrder(this.props.tableUserSettings);
                const newColumnState = objectKeys(columnVisibility ?? {}).reduce((acc, curr) => {
                    const col = acc.find(c => c.colId === curr);
                    const s = (newTableSortOrder ?? []).find(c => c.colId === curr);
                    if (col && col.colId !== COLUMN_ID_VALIDATIONS) {
                        const isGroupColumn = curr === newTableViewGrouping?.key;
                        col.sort = s?.sort ?? null;
                        col.sortIndex = s?.sortIndex ?? null;
                        col.hide = isGroupColumn ? true : (columnVisibility?.[col.colId] ?? false);
                        col.rowGroup = isGroupColumn;
                        col.rowGroupIndex = isGroupColumn ? 0 : null;
                        col.aggFunc = isGroupColumn ? (newTableViewGrouping?.aggFunc ?? null) : null;
                    }
                    return acc;
                }, cleanCurrentColumnState);
                if (newColumnState && !isEqual(currentColumnState, newColumnState) && this.gridApi) {
                    // Only apply new group setting if it has changed
                    if (this.state.groupByColumn?.aggFunc !== newTableViewGrouping?.aggFunc ||
                        this.state.groupByColumn?.field !== newTableViewGrouping?.key) {
                        if (newTableViewGrouping) {
                            const colDef = callGridMethod(this.gridApi, 'getColumnDef', newTableViewGrouping.key);
                            if (colDef) {
                                this.groupByColumn({
                                    api: this.gridApi,
                                    colDef,
                                    aggFunc: newTableViewGrouping.aggFunc,
                                    sort: newTableViewGrouping.sort,
                                }, () => {
                                    callGridMethod(this.gridApi, 'applyColumnState', { state: newColumnState });
                                });
                            }
                        }
                        else {
                            this.ungroupByColumn({ api: this.gridApi }, () => {
                                callGridMethod(this.gridApi, 'applyColumnState', {
                                    state: newColumnState.filter(c => c.colId !== AUTO_COLUMN_ID),
                                });
                            });
                        }
                    }
                    else {
                        callGridMethod(this.gridApi, 'applyColumnState', {
                            state: newColumnState.filter(c => c.colId !== AUTO_COLUMN_ID),
                        });
                    }
                }
                const currentColumnOrder = this.getCurrentColumnOrder();
                const columnOrder = getTableViewColumnOrder(this.props.tableUserSettings);
                if (columnOrder && !isEqual(currentColumnState, currentColumnOrder)) {
                    callGridMethod(this.gridApi, 'moveColumns', columnOrder, this.moveStartIndex);
                }
            }
        };
        this.resetOptionsMenu = () => {
            const optionsMenu = resolveByValue({
                propertyValue: this.props.fieldProperties.optionsMenu,
                rowValue: null,
                screenId: this.props.screenId,
                fieldValue: null,
                skipHexFormat: true,
            });
            this.props.setTableViewOptionsMenuItemAndViewFilter?.(0, optionsMenu?.[0], {});
        };
        this.insertDataSourceIntoTable = (getRowsParams, data, lastRow) => {
            getRowsParams.success({ rowData: data, rowCount: lastRow });
            callGridMethod(getRowsParams.api, 'hideOverlay');
            const isEmptyTable = getRowsParams.request.groupKeys?.length === 0 && lastRow === 0;
            this.showHideNoRowsOverlay(isEmptyTable);
            if (lastRow && lastRow !== -1) {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onAllDataLoaded');
            }
            else {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onDataLoaded');
            }
        };
        this.updateColumnValidation = (globalValidationState) => {
            const ctx = this.gridApi ? getTableContext(this.gridApi) : undefined;
            if (this.gridApi && ctx) {
                // Refresh column's header
                const invalidColumns = objectKeys(globalValidationState?.[0] || {});
                const headerClasses = invalidColumns.reduce((acc, column) => {
                    acc[column] = false;
                    return acc;
                }, {});
                setTableContext(this.gridApi, c => {
                    c.headerClasses = headerClasses;
                });
                this.gridApi
                    .getAllGridColumns()
                    .map(c => c.getColDef().field)
                    .forEach(k => {
                    if (!k) {
                        return;
                    }
                    if (invalidColumns.includes(k)) {
                        document
                            .querySelectorAll(`[grid-id="${this.gridId}"] .ag-header-cell[col-id="${k}"]`)
                            .forEach(element => {
                            element.classList.add('e-table-field-header-column-error');
                        });
                    }
                    else {
                        document
                            .querySelectorAll(`[grid-id="${this.gridId}"] .ag-header-cell[col-id="${k}"]`)
                            .forEach(element => {
                            element.classList.remove('e-table-field-header-column-error');
                        });
                    }
                });
                // Display/hide validation column
                getSafeGridApiContext(column => {
                    const isValidationColumnVisible = Boolean(column?.isVisible());
                    const shouldValidationColumnBeVisible = invalidColumns.length > 0;
                    if (isValidationColumnVisible !== shouldValidationColumnBeVisible) {
                        callGridMethod(this.gridApi, 'setColumnVisible', COLUMN_ID_VALIDATIONS, shouldValidationColumnBeVisible);
                        this.autoSizeAllColumns();
                    }
                }, this.gridApi, 'getColumn', COLUMN_ID_VALIDATIONS);
            }
        };
        this.getUnknownLastRow = () => {
            return this.isInfiniteScroll() ? -1 : undefined;
        };
        this.getRowCount = ({ data, getRowsParams, }) => {
            return data.length < (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0)
                ? (getRowsParams.request.startRow || 0) + data.length
                : this.getUnknownLastRow();
        };
        this.isNavigationPanel = () => this.props.elementId === navigationPanelId;
        this.belongsToGroup = ({ groupKey, groupValue, column, aggFunc, }) => aggFunc == null
            ? (n) => get(n.data, getGroupKey({ groupKey, type: column.type }), null) ===
                (groupValue === '' || isNil(groupValue) ? null : groupValue)
            : (n) => {
                const rowValue = get(n.data, groupKey);
                if (groupValue === '' || isNil(groupValue)) {
                    return isNil(rowValue);
                }
                if (!isValidDatePropertyValue(rowValue)) {
                    return false;
                }
                return isRowValueInDateGroup(rowValue, groupValue, aggFunc);
            };
        this.selectTableItems = (api) => {
            (xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata.uiComponentProperties[this.props.elementId]?.selectedRecords ?? []).forEach(id => {
                const rowNode = this.getRowNode({ id, gridApi: api });
                if (rowNode && !rowNode.isSelected()) {
                    // IMPORTANT: DO NOT handle in 'onSelectionChanged' event
                    rowNode.setSelected(true, false, 'rowDataChanged');
                }
            });
        };
        /**
         * Ag-grid callback which is triggered whenever the table component needs new data from our framework
         * @param getRowsParams
         */
        this.getRows = async (getRowsParams) => {
            const hasGroups = getRowsParams.request.rowGroupCols.length > 0;
            /**
             * Set flag to disable row field selection so events will not be triggered when rows selected
             * programmatically.
             */
            const pageSize = this.props.fieldProperties.pageSize || 20;
            const pageNumber = Math.max(0, Math.ceil((getRowsParams.request.endRow || 0) / pageSize) - 1);
            callGridMethod(getRowsParams.api, 'hideOverlay');
            if (hasGroups) {
                callGridMethod(getRowsParams.api, 'showLoadingOverlay');
                const colId = getRowsParams.request.rowGroupCols[0].id;
                const column = findColumnDefinitionByBind(this.props.fieldProperties.columns || [], colId);
                if (!column) {
                    throw new Error(`Cannot find column with the following id: ${colId}`);
                }
                const groupKey = column.type === FieldKey.Reference
                    ? `${convertDeepBindToPathNotNull(column?.properties.bind)}.${getReferenceValueField(column?.properties)}`
                    : convertDeepBindToPathNotNull(column?.properties.bind);
                const groupValue = column.type === FieldKey.Reference
                    ? (get(getRowsParams.parentNode.data, getGroupKey({ groupKey, type: column.type })) ??
                        getRowsParams.request.groupKeys?.[0])
                    : getRowsParams.request.groupKeys?.[0];
                let cursor;
                const nodes = [];
                if (!isNil(groupValue)) {
                    callGridMethod(getRowsParams.api, 'forEachNode', n => {
                        if (!n.group &&
                            this.belongsToGroup({
                                groupKey,
                                groupValue,
                                column,
                                aggFunc: getRowsParams.request.rowGroupCols[0]?.aggFunc,
                            })(n)) {
                            nodes.push(n);
                        }
                    });
                    cursor = nodes[(getRowsParams.request.startRow ?? 1) - 1]?.data?.__cursor;
                    if (cursor === undefined && getRowsParams.request.startRow !== 0) {
                        getRowsParams.success({
                            rowData: [],
                            rowCount: nodes.length,
                        });
                        callGridMethod(getRowsParams.api, 'showNoRowsOverlay');
                        return;
                    }
                }
                else {
                    callGridMethod(getRowsParams.api, 'forEachNode', n => {
                        if (n.group && n.data) {
                            nodes.push(n);
                        }
                    });
                    cursor = nodes[(getRowsParams.request.startRow ?? 1) - 1]?.data?.__cursor;
                    if (cursor === undefined && getRowsParams.request.startRow !== 0) {
                        getRowsParams.success({
                            rowData: nodes,
                            rowCount: nodes.length,
                        });
                        return;
                    }
                }
                let sortOrder = getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef);
                if (isEmpty(sortOrder)) {
                    sortOrder = { _id: -1 };
                }
                const orderBy = groupValue !== undefined
                    ? sortOrder
                    : objectKeys(flat(sortOrder)).reduce((acc, key) => {
                        if (key === groupKey) {
                            set(acc, key, get(sortOrder, key));
                        }
                        return acc;
                    }, {});
                // Map Ag-grid filter objects into GraphQL/MongoDB friendly filter data structure
                const userFilters = this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field));
                const gridPageSize = (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0);
                const groupedData = await this.props.value.getPage({
                    tableFieldProperties: this.props.fieldProperties,
                    filters: userFilters,
                    orderBy,
                    pageSize: gridPageSize,
                    group: {
                        key: groupKey,
                        value: groupValue,
                        aggFunc: getRowsParams.request.rowGroupCols[0]?.aggFunc,
                        type: column.type,
                    },
                    cursor,
                    cleanMetadata: false,
                    pageNumber: cursor ? pageNumber : undefined,
                    selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                });
                const rowCount = this.getRowCount({ data: groupedData, getRowsParams });
                getRowsParams.success({
                    rowData: groupedData,
                    rowCount,
                });
                callGridMethod(getRowsParams.api, 'hideOverlay');
                return;
            }
            if (!this.props.value) {
                this.insertDataSourceIntoTable(getRowsParams, [], 0);
                return;
            }
            if (!this.isInfiniteScroll()) {
                callGridMethod(getRowsParams.api, 'showLoadingOverlay');
            }
            try {
                switch (this.dataOperationMode) {
                    case DataOperationMode.NONE:
                        const data = this.props.value.getData({
                            cleanMetadata: false,
                            temporaryRecords: this.props.additionalLookupRecords?.(),
                            limit: this.props.fieldProperties.pageSize || 20,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, data, this.getRowCount({ data, getRowsParams }));
                        this.collectionValueChangeSubscription = this.props.value.subscribeForValueChanges(this.onCollectionUpdated);
                        this.collectionValidityChangeSubscription = this.props.value.subscribeForValidityChanges(({ globalValidationState, recordValidationState, recordId }) => {
                            if (this.gridApi) {
                                this.updateColumnValidation(globalValidationState);
                                const rowNode = this.getRowNode({ id: recordId });
                                if (rowNode) {
                                    const previousValidationStateKeys = objectKeys(rowNode.data.__validationState || {});
                                    rowNode.setData({
                                        ...rowNode.data,
                                        __validationState: recordValidationState,
                                    });
                                    // Refresh cells
                                    const columns = difference(objectKeys(recordValidationState), previousValidationStateKeys);
                                    this.gridApi?.refreshCells({
                                        rowNodes: [rowNode],
                                        columns,
                                        force: true,
                                        suppressFlash: true,
                                    });
                                }
                            }
                        });
                        this.selectTableItems(getRowsParams.api);
                        break;
                    case DataOperationMode.FETCH_PAGE:
                        this.isFetching = true;
                        const newData = await this.props.value.getPageWithCurrentQueryArguments({
                            tableFieldProperties: this.props.fieldProperties,
                            pageSize,
                            pageNumber,
                            group: undefined,
                            cleanMetadata: false,
                            cursor: this.getCursor(getRowsParams.api),
                        });
                        this.updateColumnValidation(this.props.value.getValidationStateByColumn());
                        this.insertDataSourceIntoTable(getRowsParams, newData, this.getRowCount({ data: newData, getRowsParams }));
                        this.selectTableItems(getRowsParams.api);
                        this.isFetching = false;
                        break;
                    case DataOperationMode.REMOVE_RECORD:
                        this.onUnselectAll();
                        // Populating values
                        const mergedData = await this.props.value.getRecordWithCurrentQueryArguments({
                            tableFieldProperties: this.props.fieldProperties,
                            pageSize,
                            pageNumber,
                            group: undefined,
                            cursor: this.getCursor(getRowsParams.api),
                            cleanMetadata: false,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, mergedData, this.getRowCount({ data: mergedData, getRowsParams }));
                        this.selectTableItems(getRowsParams.api);
                        this.updateColumnValidation(this.props.value.getValidationStateByColumn());
                        break;
                    case DataOperationMode.RESET_TABLE:
                        this.onUnselectAll();
                        if (this.isNavigationPanel()) {
                            this.props.value.cleanCollectionData();
                            callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
                        }
                        // Map Ag-grid sorting properties
                        let orderBy = getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef);
                        if (isEmpty(orderBy)) {
                            orderBy = { _id: -1 };
                        }
                        // Map Ag-grid filter objects into GraphQL/MongoDB friendly filter data structure
                        const userFilters = this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field));
                        // Map Ag-grid page sizing properties
                        const gridPageSize = (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0);
                        // Populating values
                        const pageData = await this.props.value.getPage({
                            tableFieldProperties: this.props.fieldProperties,
                            filters: userFilters,
                            orderBy,
                            pageSize: gridPageSize,
                            group: undefined,
                            cursor: this.getCursor(getRowsParams.api),
                            cleanMetadata: false,
                            pageNumber: !this.getCursor(getRowsParams.api) ? pageNumber : undefined,
                            selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, pageData, this.getRowCount({ data: pageData, getRowsParams }));
                        this.selectTableItems(getRowsParams.api);
                        callGridMethod(this.gridApi, 'paginationGoToFirstPage');
                        break;
                    case DataOperationMode.FILTER_ERRORS:
                        if (this.props.value.getAllInvalidRecords().length === 0) {
                            this.insertDataSourceIntoTable(getRowsParams, [], 0);
                        }
                        else if (this.props.fieldProperties.isTransient) {
                            this.insertDataSourceIntoTable(getRowsParams, this.props.value.getAllInvalidRecords(), this.getRowCount({ data: this.props.value.getAllInvalidRecords(), getRowsParams }));
                        }
                        else {
                            await this.props.value.fetchInvalidUnloadedRecords();
                            const errorPageData = await this.props.value.getPage({
                                tableFieldProperties: this.props.fieldProperties,
                                filters: this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field)),
                                orderBy: getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef),
                                pageSize: (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0),
                                group: undefined,
                                cursor: this.getCursor(getRowsParams.api),
                                cleanMetadata: false,
                                fetchPageSize: 500,
                            });
                            this.insertDataSourceIntoTable(getRowsParams, errorPageData, this.getRowCount({ data: errorPageData, getRowsParams }));
                        }
                        break;
                    default:
                        break;
                }
                if (this.props.value.getData({
                    cleanMetadata: false,
                    temporaryRecords: this.props.additionalLookupRecords?.(),
                }).length > 0) {
                    this.setState({ tableHasData: true });
                }
                else {
                    this.setState({ tableHasData: false });
                }
                this.dataOperationMode = DataOperationMode.FETCH_PAGE;
            }
            catch (error) {
                xtremConsole.log(error);
                getRowsParams.fail();
            }
            finally {
                await this.setPhantomRow();
            }
        };
        this.serverSideDataSource = {
            getRows: this.getRows,
        };
        this.isDisabled = () => {
            const x = isFieldDisabled(this.props.screenId, this.props.fieldProperties, this.props.recordContext?._id || null, this.props.recordContext || null);
            return this.props.isParentDisabled || x;
        };
        this.isReadOnly = () => resolveByValue({
            fieldValue: null,
            propertyValue: this.props.isReadOnly,
            skipHexFormat: true,
            rowValue: null,
            screenId: this.props.screenId,
        }) || isFieldReadOnly(this.props.screenId, this.props.fieldProperties, null, undefined);
        this.resizeListener = () => {
            this.autoSizeAllColumns();
        };
        this.onGridEvent = async (eventType, event) => {
            /**
             * ag-grid correctly selects all children when a group is expanded or data is added
             * but it doesn't automatically trigger the 'onSelectionChanged' event
             */
            if (eventType === 'displayedRowsChanged' &&
                Object.values(event.api.getCacheBlockState() ?? {}).some((b) => (b?.loadedRowCount ?? 0) > 0)) {
                this.selectTableItems(event.api);
                this.onSelectionChanged(event);
            }
        };
        /**
         * Ag-grid lifecycle event listener. It is triggered when the table is prepared with the initial rendering of the
         * controller components and is ready to receive data.
         **/
        this.onGridReady = async (params) => {
            this.gridApi = params.api;
            params.api.addGlobalListener(this.onGridEvent);
            // Timeout needed in order to avoid ag-grid redraw error
            setTimeout(() => {
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            });
            if (this.props.contextType === ContextType.dialog) {
                (callGridMethod(this.gridApi, 'getRenderedNodes') ?? [])?.[0]?.setSelected?.(true);
            }
            if (this.gridApi) {
                const grouping = getTableViewGrouping(this.props.tableUserSettings);
                const filter = getTableViewFilter(this.props.tableUserSettings);
                if (grouping?.key !== undefined && grouping?.key !== this.state.groupByColumn?.colId) {
                    // Restore group
                    const colDef = callGridMethod(this.gridApi, 'getColumnDefs')?.find(c => c.colId === grouping.key);
                    if (colDef != null) {
                        setTimeout(api => {
                            this.groupByColumn({
                                api,
                                colDef: {
                                    ...colDef,
                                    sort: grouping.sort,
                                },
                                aggFunc: grouping.aggFunc,
                            }, () => {
                                callGridMethod(this.gridApi, 'setFilterModel', filter);
                            });
                        }, 0, this.gridApi);
                    }
                }
                else {
                    callGridMethod(this.gridApi, 'setFilterModel', filter);
                }
            }
            window.addEventListener('resize', this.resizeListener);
        };
        this.setPhantomRow = async () => {
            const fp = this.props.fieldProperties;
            const canAdd = !!fp?.canAddNewLine && !fp?.isPhantomRowDisabled;
            const phantomRows = await (canAdd ? this.props.value?.getPhantomRecords?.() : []);
            // Points to the first element to avoid race conditions that could return more than 1 row:
            this.setState(({ phantomRows: currentPhantomRows }) => {
                const newPhantomRows = phantomRows[0] ? [phantomRows[0]] : [];
                return {
                    phantomRows: isEqual(currentPhantomRows, newPhantomRows) ? currentPhantomRows : newPhantomRows,
                };
            });
        };
        this.onSwitchView = (view) => {
            this.props.setFieldProperties?.(this.props.elementId, {
                ...xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata.uiComponentProperties[this.props.elementId],
                tableViewMode: view,
            });
        };
        this.onSwitchCalendarView = (selectedCalendarView) => {
            this.props.onTelemetryEvent?.(`tableCalendarViewSwitched-${this.props.elementId}`, {
                view: selectedCalendarView,
                elementId: this.props.elementId,
                screenId: this.props.screenId,
            });
            this.setState({ selectedCalendarView });
        };
        this.handleFilterChanged = (params) => {
            const filterModel = getFilterModel(params?.api || this.gridApi, this.state.groupByColumn?.field);
            if (this.props.setTableViewFilter) {
                this.props.setTableViewFilter(0, filterModel);
            }
            this.setShouldResetTable();
        };
        this.handleColumnRowGroupChanged = (params) => {
            // IMPORTANT to check if rowGroup is set in user provided column def
            if (this.props.setTableViewGrouping && params.column?.getUserProvidedColDef()?.rowGroup) {
                const columnDefinition = params.column?.getColDef();
                this.props.setTableViewGrouping(0, {
                    key: params.column?.getColId() || '',
                    sort: columnDefinition?.sort || 'asc',
                    aggFunc: columnDefinition?.aggFunc || undefined,
                });
            }
            this.setShouldResetTable();
        };
        this.setShouldResetTable = () => {
            this.dataOperationMode = DataOperationMode.RESET_TABLE;
        };
        this.onOptionsMenuItemChange = (selectedOptionsMenuItem) => {
            if (this.gridApi) {
                this.setShouldResetTable();
                this.props.setTableViewOptionsMenuItemAndViewFilter?.(0, selectedOptionsMenuItem, {});
                this.getColumnDefinitions({
                    keepVisibility: true,
                    refreshDataSource: false,
                    keepGrouping: true,
                });
            }
        };
        this.getRowId = ({ data }) => data._id;
        this.getRowClass = params => {
            if (!params.data || this.props.fieldProperties.isChangeIndicatorDisabled)
                return undefined;
            if (params.data.__isGroup) {
                return 'ag-user-row-group';
            }
            if (params.data?.__phantom === undefined &&
                (params.data?.__action === RecordActionType.ADDED || params.data?.__action === RecordActionType.MODIFIED)) {
                return 'ag-row-edited';
            }
            return undefined;
        };
        this.selectionColumnDef = getSelectionColumnDef({
            contextType: this.props.contextType,
            fieldProperties: this.props.fieldProperties,
        });
        this.gridOptions = {
            getRowClass: this.getRowClass,
            selectionColumnDef: this.selectionColumnDef,
        };
        this.getColumnDefinitions = ({ refreshDataSource = true, resetSelection = true, keepVisibility = false, resetTableFilters = false, keepGrouping = false, } = {}) => {
            const columnsData = (this.props.fieldProperties.columns || []).map(columnDefinition => ({
                columnDefinition,
                bind: convertDeepBindToPath(columnDefinition.properties.bind),
                elementId: getNestedFieldElementId(columnDefinition),
            }));
            let columns = getColumns({
                accessBindings: this.props.accessBindings,
                columnsData,
                currentTableView: this.props.tableUserSettings?.$current?.content?.[0],
                customizedOrderBy: getTableViewSortOrder(this.props.tableUserSettings),
                elementId: this.props.elementId,
                enumTypes: this.props.enumTypes,
                fieldProperties: () => this.props.fieldProperties,
                isDisabled: this.isDisabled(),
                // eslint-disable-next-line react/no-access-state-in-setstate
                groupBy: this.state.groupByColumn?.colId,
                // eslint-disable-next-line react/no-access-state-in-setstate
                hasFloatingFilters: this.state.hasFloatingFilters,
                isParentDisabled: () => !!this.props.isParentDisabled,
                isReadOnly: this.isReadOnly(),
                level: 0,
                locale: this.props.locale,
                lookupSelectionMode: this.props.selectionMode,
                nodeTypes: this.props.nodeTypes,
                dataTypes: this.props.dataTypes,
                pageNode: this.props.pageNode,
                screenId: this.props.screenId,
                value: () => this.props.value,
            });
            if (keepVisibility) {
                const columnHidden = getTableViewColumnHidden(this.props.tableUserSettings);
                columns = columns.map(k => {
                    const colState = this.state.columnStates?.find(cs => cs.colId === k.field);
                    if (colState) {
                        return { ...k, hide: !!colState.hide };
                    }
                    if (columnHidden && k.field) {
                        return { ...k, hide: !!columnHidden[k.field] };
                    }
                    return k;
                });
            }
            this.setState({ columns }, () => {
                if (this.gridApi) {
                    callGridMethod(this.gridApi, 'stopEditing');
                    if (resetSelection) {
                        this.onUnselectAll();
                    }
                    /**
                     * Column definitions have to be reset in order the new properties based on the new collection value
                     * object and event listeners to take effect
                     */
                    callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', this.state.columns);
                    // AG-grid will not update the order of columns unless we explicitly tell it so
                    this.sortColumns();
                    if (refreshDataSource) {
                        /**
                         * Datasource has to be reset so the grid detects the new collection value and rerenders itself.
                         */
                        callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
                        if (resetTableFilters) {
                            callGridMethod(this.gridApi, 'setFilterModel', getTableViewFilter(this.props.tableUserSettings));
                        }
                    }
                    if (keepGrouping && this.state.groupByColumn) {
                        this.groupByColumn({
                            api: this.gridApi,
                            colDef: this.state.groupByColumn,
                            aggFunc: this.state.groupByColumn.aggFunc,
                        });
                    }
                }
            });
        };
        this.hasFilterableColumns = () => this.state.columns.filter(c => c.filter).length > 0;
        this.toggleFloatingFilters = () => {
            this.setState(state => ({
                hasFloatingFilters: !state.hasFloatingFilters,
                autoGroupColumnDef: { ...state.groupByColumn, floatingFilter: !state.hasFloatingFilters },
                groupByColumn: state.groupByColumn
                    ? {
                        ...state.groupByColumn,
                        floatingFilter: !state.hasFloatingFilters,
                    }
                    : undefined,
            }), () => {
                this.props.onTelemetryEvent?.(`tableFloatingFilterRowToggled-${this.props.elementId}`, {
                    hasFloatingFilters: this.state.hasFloatingFilters,
                    elementId: this.props.elementId,
                    screenId: this.props.screenId,
                });
                this.getColumnDefinitions({
                    refreshDataSource: false,
                    resetSelection: false,
                    keepVisibility: true,
                    keepGrouping: true,
                });
            });
        };
        this.exportToExcelWithInfiniteScroll = async (target, options) => {
            const gapi = this.gridApi;
            this.props.onTelemetryEvent?.(`tableDataExported-${target}-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
                target,
            });
            if (gapi) {
                if (!this.props.fieldProperties.isTransient) {
                    this.props.value.cleanCollectionData();
                }
                const pageData = await this.props.value.getPagesIteratively({
                    filters: this.getFilters(getFilterModel(gapi, this.state.groupByColumn?.field)),
                    tableFieldProperties: this.props.fieldProperties,
                    pageSize: 1000,
                    cleanMetadata: false,
                    selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                });
                gapi.applyServerSideRowData({
                    successParams: {
                        rowData: pageData,
                        rowCount: pageData.length,
                    },
                });
                if (target === 'excel') {
                    callGridMethod(gapi, 'exportDataAsExcel', options);
                }
                else {
                    callGridMethod(gapi, 'exportDataAsCsv', options);
                }
            }
        };
        this.exportToExcelWithPagination = (target, options) => {
            this.props.onTelemetryEvent?.(`tableDataExported-${target}-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
                target,
            });
            getSafeGridApiContext(currentPage => {
                if (typeof this.props.setGlobalLoading === 'function') {
                    this.props.setGlobalLoading(true);
                }
                const gettingPages = setInterval(() => {
                    getSafeGridApiContext(isLastPageFound => {
                        if (isLastPageFound) {
                            clearInterval(gettingPages);
                            if (target === 'excel') {
                                callGridMethod(this.gridApi, 'exportDataAsExcel', options);
                            }
                            else {
                                callGridMethod(this.gridApi, 'exportDataAsCsv', options);
                            }
                            callGridMethod(this.gridApi, 'paginationGoToPage', currentPage || 0);
                            if (typeof this.props.setGlobalLoading === 'function') {
                                this.props.setGlobalLoading(false);
                            }
                            this.setState({ isExportingExcel: false });
                        }
                        else if (!this.isFetching) {
                            callGridMethod(this.gridApi, 'paginationGoToNextPage');
                            this.setState({ isExportingExcel: true });
                        }
                    }, this.gridApi, 'paginationIsLastPageFound');
                }, 500);
            }, this.gridApi, 'paginationGetCurrentPage');
        };
        this.exportToExcel = async (target) => {
            if (this.props.elementId === navigationPanelId && this.props.fieldProperties.node) {
                try {
                    await exportTableData({
                        node: String(this.props.fieldProperties.node),
                        columns: this.state.columns,
                        nodeTypes: this.props.nodeTypes,
                        filter: this.props.value.getFilter({ level: 0 }),
                        orderBy: this.props.value.getOrderBy({ level: 0 }),
                        outputFormat: target === 'excel' ? 'xlsx' : target,
                    });
                    showToast(localize('@sage/xtrem-ui/table-export-started', 'Export started.'), { type: 'info' });
                }
                catch (error) {
                    showToast(localize('@sage/xtrem-ui/table-export-failed', 'Failed to export main list content'), {
                        type: 'error',
                    });
                }
                return;
            }
            getSafeGridApiContext(columns => {
                const getBindIndexFromColumnId = (column) => {
                    const colId = column.getColDef().field;
                    const tempIndex = Number(column.getColId().replace(`${colId}_`, ''));
                    const index = Number.isNaN(Number(tempIndex)) ? 0 : tempIndex;
                    return { bind: colId, index };
                };
                const options = {
                    columnKeys: getColumnsToExport(columns ?? []),
                    processCellCallback: (params) => {
                        const { bind, index } = getBindIndexFromColumnId(params.column);
                        const filteredColumns = (this.props.fieldProperties.columns || []).filter(c => getNestedFieldElementId(c) === bind ||
                            getNestedFieldElementId(c).startsWith(`${bind}__`));
                        const column = filteredColumns[index];
                        return column ? cellExportFormatter(column, params.value) : '';
                    },
                    fileName: getFieldTitle(this.props.screenId, this.props.fieldProperties, null) || this.props.elementId,
                };
                if (this.isInfiniteScroll()) {
                    this.exportToExcelWithInfiniteScroll(target, options);
                }
                else {
                    this.exportToExcelWithPagination(target, options);
                }
            }, this.gridApi, 'getColumns');
        };
        this.getCacheBlockSize = () => this.props.fieldProperties.pageSize || 20;
        this.isServerSideGroup = (dataItem) => Boolean(dataItem.__isGroup);
        this.excludeInternalColumnsFilter = (column) => {
            const colId = column.getColId();
            return (colId !== undefined &&
                colId !== COLUMN_ID_ROW_SELECTION &&
                colId !== COLUMN_ID_ROW_ACTIONS &&
                colId !== COLUMN_ID_VALIDATIONS);
        };
        this.getAllColumnIds = (gridApi = this.gridApi) => {
            return (callGridMethod(gridApi, 'getColumns') ?? [])
                .filter(this.excludeInternalColumnsFilter)
                .map(column => column.getColId());
        };
        this.getAllDisplayedColumnIds = (gridApi = this.gridApi) => {
            return (callGridMethod(gridApi, 'getAllDisplayedColumns') ?? []).map(column => {
                const colId = column.getColId();
                return colId;
            });
        };
        this.autoSizeAllColumns = (gridApi = this.gridApi) => {
            if (!gridApi) {
                return;
            }
            callGridMethod(gridApi, 'autoSizeColumns', this.getAllColumnIds(gridApi));
            const availableWidth = this.containerRef.current?.clientWidth ?? 0;
            const usedWidth = (gridApi.getAllDisplayedColumns() ?? []).reduce((acc, c) => acc + c.getActualWidth(), 0);
            if (availableWidth - usedWidth > 20) {
                setTimeout(() => {
                    callGridMethod(gridApi, 'sizeColumnsToFit');
                }, 0);
            }
            if (this.state.resizedColumns.length > 0) {
                callGridMethod(this.gridApi, 'setColumnWidths', this.state.resizedColumns);
            }
        };
        this.onFirstDataRendered = (event) => {
            setTimeout(() => {
                this.autoSizeAllColumns(event.api);
            }, 100);
        };
        this.canBeGroupedBy = (column) => {
            const supportedFieldTypes = [
                FieldKey.Reference,
                FieldKey.Label,
                FieldKey.Date,
                FieldKey.RelativeDate,
            ];
            const colDef = column.getColDef();
            // Allow grouping by Text when the whole table is transient (UI-only data)
            // This must take precedence over column-level transient checks
            if (this.props.fieldProperties?.isTransient &&
                (colDef.type === FieldKey.Text || colDef.type === FieldKey.Label)) {
                return true;
            }
            if (colDef.cellRendererParams?.fieldProperties?.isTransient) {
                return false;
            }
            return ((supportedFieldTypes.includes(colDef.type) && this.isNavigationPanel()) ||
                column.getId() === AUTO_COLUMN_ID);
        };
        this.ungroupByColumn = ({ colDef, api, }, cb) => {
            let newColumns;
            this.setState(prevState => {
                newColumns = [
                    ...(prevState.columns ?? []).map(c => {
                        const isPrevGroupByColumn = prevState.groupByColumn && c.field === prevState.groupByColumn.field;
                        return {
                            ...c,
                            ...(isPrevGroupByColumn && { filter: prevState.groupByColumn.filter }),
                            aggFunc: null,
                            rowGroup: false,
                            hide: c.field === (colDef?.field ?? prevState.groupByColumn?.field) ? false : c.hide,
                        };
                    }),
                ];
                return {
                    groupByColumn: undefined,
                    columns: newColumns,
                    autoGroupColumnDef: undefined,
                };
            }, () => {
                this.onUnselectAll();
                setTimeout(() => {
                    const filter = getTableViewFilter(this.props.tableUserSettings);
                    // TODO: revisit this fix. It fixes XT-39123.
                    callGridMethod(api, 'setFilterModel', filter);
                    callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', newColumns);
                    if (this.props.setTableViewGrouping) {
                        this.props.setTableViewGrouping(0, undefined);
                    }
                    cb?.();
                    this.autoSizeAllColumns();
                }, 10);
            });
        };
        this.groupByColumn = ({ colDef, api, aggFunc, sort, sortIndex, }, cb) => {
            let newColumns;
            this.setState(currState => {
                newColumns = [
                    ...(currState.columns ?? []).map(c => {
                        const isGroupByColumn = c.field === colDef.field;
                        const isPrevGroupByColumn = currState.groupByColumn && c.field === currState.groupByColumn.field;
                        let hide = c.hide;
                        let filter = c.filter;
                        if (isGroupByColumn) {
                            hide = true;
                            filter = false;
                            c.aggFunc = aggFunc ?? null;
                        }
                        else if (isPrevGroupByColumn) {
                            hide = false;
                            filter = currState.groupByColumn.filter;
                            c.aggFunc = null;
                        }
                        return {
                            ...c,
                            rowGroup: isGroupByColumn,
                            hide,
                            filter,
                        };
                    }),
                ];
                const { cellRenderer, cellEditor, editable, resizable, ...colDefProps } = colDef;
                const menuTabs = ['filterMenuTab', 'generalMenuTab'];
                return {
                    groupByColumn: { ...colDef, aggFunc },
                    autoGroupColumnDef: {
                        ...colDefProps,
                        colId: AUTO_COLUMN_ID,
                        lockPosition: 'left',
                        rowGroup: true,
                        menuTabs,
                        flex: undefined,
                        width: 250,
                        minWidth: 250,
                        resizable: true,
                        editable: false,
                        ...(this.props.groupTitle && { headerName: this.props.groupTitle }),
                        // by default sort by grouping column in ascending order
                        sort: sort !== undefined ? sort : (colDef.sort ?? 'asc'),
                        sortIndex,
                        sortable: true,
                        headerName: this.props.groupTitle ? this.props.groupTitle : colDef.headerName,
                        cellRendererParams: {
                            ...colDefProps.cellRendererParams,
                            innerRenderer: cellRenderer,
                        },
                    },
                    columns: newColumns,
                };
            }, () => {
                setTimeout(() => {
                    callGridMethod(api, 'setGridOption', 'columnDefs', newColumns);
                    cb?.();
                    this.autoSizeAllColumns();
                }, 0);
            });
        };
        this.updateColDefsStates = () => {
            return (getSafeGridApiContext((newColumns) => {
                return new Promise(resolve => {
                    this.setState({ columns: newColumns ?? [] }, () => {
                        callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', this.state.columns);
                        resolve();
                    });
                });
            }, this.gridApi, 'getColumnDefs') ?? Promise.resolve());
        };
        this.sortColumns = () => {
            (callGridMethod(this.gridApi, 'getColumnDefs') ?? []).forEach((c) => {
                if (this.gridApi && c.colId && c.context.xtremSortIndex) {
                    callGridMethod(this.gridApi, 'moveColumns', [c.colId], c.context.xtremSortIndex + this.moveStartIndex);
                }
            });
        };
        this.getGroupByMenuItem = ({ isGroupedByThisColumn, colDef, api, }) => {
            if ([FieldKey.Date, FieldKey.RelativeDate].includes(colDef.type)) {
                return {
                    cssClasses: isGroupedByThisColumn ? ['e-table-field-ungroup'] : ['e-table-field-group-by'],
                    name: isGroupedByThisColumn
                        ? localize('@sage/xtrem-ui/ungroup', 'Ungroup')
                        : localize('@sage/xtrem-ui/group-by', 'Group by'),
                    icon: isGroupedByThisColumn
                        ? renderToStaticMarkup(
                        // eslint-disable-next-line react/jsx-indent
                        React.createElement("div", { style: { position: 'relative' } },
                            React.createElement(Icon, { type: "bullet_list" }),
                            React.createElement("div", { style: {
                                    position: 'absolute',
                                    width: '24px',
                                    bottom: '2px',
                                    left: '2px',
                                    borderTop: '1px solid black',
                                    transform: 'rotate(-45deg)',
                                    transformOrigin: '0% 0%',
                                } })))
                        : renderToStaticMarkup(React.createElement(Icon, { type: "bullet_list" })),
                    subMenu: isGroupedByThisColumn
                        ? undefined
                        : [
                            {
                                cssClasses: ['e-table-field-group-by-year'],
                                name: localize('@sage/xtrem-ui/group-by-year', 'Year'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'year' });
                                    }, 10);
                                },
                            },
                            {
                                cssClasses: ['e-table-field-group-by-month'],
                                name: localize('@sage/xtrem-ui/group-by-month', 'Month'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'month' });
                                    }, 10);
                                },
                            },
                            {
                                cssClasses: ['e-table-field-group-by-day'],
                                name: localize('@sage/xtrem-ui/group-by-day', 'Day'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'day' });
                                    }, 10);
                                },
                            },
                        ],
                    action: () => {
                        if (isGroupedByThisColumn) {
                            this.ungroupByColumn({ colDef, api });
                        }
                    },
                };
            }
            return {
                cssClasses: isGroupedByThisColumn ? ['e-table-field-ungroup'] : ['e-table-field-group-by'],
                name: isGroupedByThisColumn
                    ? localize('@sage/xtrem-ui/ungroup', 'Ungroup')
                    : localize('@sage/xtrem-ui/group-by-this-column', 'Group by this column'),
                icon: isGroupedByThisColumn
                    ? renderToStaticMarkup(
                    // eslint-disable-next-line react/jsx-indent
                    React.createElement("div", { style: { position: 'relative' } },
                        React.createElement(Icon, { type: "bullet_list" }),
                        React.createElement("div", { style: {
                                position: 'absolute',
                                width: '24px',
                                bottom: '2px',
                                left: '2px',
                                borderTop: '1px solid black',
                                transform: 'rotate(-45deg)',
                                transformOrigin: '0% 0%',
                            } })))
                    : renderToStaticMarkup(React.createElement(Icon, { type: "bullet_list" })),
                action: () => {
                    if (isGroupedByThisColumn) {
                        this.ungroupByColumn({ colDef, api });
                    }
                    else {
                        setTimeout(() => {
                            this.groupByColumn({ api, colDef });
                        }, 10);
                    }
                },
            };
        };
        this.getMainMenuItems = ({ column, api }) => {
            /**
             * params.defaultItems are
            [
                "pinSubMenu",
                "separator",
                "autoSizeThis",
                "autoSizeAll",
                "separator",
                "separator",
                "resetColumns"
            ]
             */
            const defaultMenuItems = ['pinSubMenu', 'separator', 'autoSizeThis', 'autoSizeAll'];
            const colId = column.getColId();
            const isAutoColumn = colId === AUTO_COLUMN_ID;
            const colDef = isAutoColumn ? this.state.groupByColumn : this.state.columns.find(c => c.field === colId);
            if (!colDef) {
                throw new Error('Unable to find column definition');
            }
            const isGroupedByThisColumn = this.state.groupByColumn?.colId === colId || isAutoColumn;
            return this.canBeGroupedBy(column)
                ? [...defaultMenuItems, 'separator', this.getGroupByMenuItem({ api, isGroupedByThisColumn, colDef })]
                : defaultMenuItems;
        };
        this.isElementInViewport = (e) => {
            const rect = e.getBoundingClientRect();
            const header = document.querySelector('[data-testid="e-header"]');
            const headerHeight = header ? header.getBoundingClientRect().height : 0;
            const footer = document.querySelector('.e-page-footer-container');
            const footerHeight = footer ? footer.getBoundingClientRect().height : 0;
            return (rect.top >= headerHeight &&
                rect.left >= 0 &&
                rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) - footerHeight &&
                rect.right <= (window.innerWidth || document.documentElement.clientWidth));
        };
        this.onCellFocused = (params) => {
            if (this.props.onFocus) {
                this.props.onFocus(this.props.screenId, this.props.elementId, String(params.rowIndex || ''), typeof params.column === 'object' && params.column ? params.column.getColId() : '');
            }
            if (this.props.selectionMode === 'single') {
                callGridMethod(this.gridApi, 'forEachNode', node => {
                    if (node.rowIndex === params.rowIndex) {
                        node.setSelected(true);
                    }
                });
            }
            const focusedCell = params.api.getFocusedCell();
            if (!focusedCell) {
                return;
            }
            const focusedElement = document
                .querySelector(`[grid-id="${this.gridId}"]`)
                ?.querySelector(`[row-index="${focusedCell.rowIndex}"]`);
            if (focusedElement && !this.isElementInViewport(focusedElement)) {
                focusedElement.scrollIntoView({ block: 'center', inline: 'nearest' });
            }
        };
        this.onCellKeyDown = (event) => {
            const keyboardEvent = event.event;
            if (keyboardEvent?.key === 'Enter') {
                this.props.onRowClick?.(event.data._id, keyboardEvent)();
            }
            if (event.data?.__phantom && keyboardEvent.key === 'Enter') {
                if (keyboardEvent.getModifierState('Meta') || keyboardEvent.getModifierState('Control')) {
                    this.props.onTelemetryEvent?.(`tablePhantomRowCommittedByKeyboard-${this.props.elementId}`, {
                        screenId: this.props.screenId,
                        elementId: this.props.elementId,
                    });
                    tryToCommitPhantomRow({
                        api: event.api,
                        screenId: this.props.screenId,
                        elementId: this.props.elementId,
                        value: this.props.value,
                    });
                }
                return;
            }
            const isCtrlOrCmd = keyboardEvent.ctrlKey || keyboardEvent.metaKey;
            if (isCtrlOrCmd && keyboardEvent.key === 'c') {
                keyboardEvent.preventDefault();
                copySelectedCellValue(String(event.value ?? ''));
            }
            if (isCtrlOrCmd && keyboardEvent.key === 'v') {
                keyboardEvent.preventDefault();
                pasteToSelectedCell(event.api);
            }
        };
        this.getAllNotVisibleNotHiddenColumns = (tableProperties) => {
            return (tableProperties.columns || [])
                .filter(c => {
                return c.properties.isHiddenOnMainField;
            })
                .map(c => {
                return convertDeepBindToPathNotNull(c.properties.bind);
            });
        };
        this.hideUnhideColumns = (event) => {
            const gridApi = event?.api ?? this.gridApi;
            if (!gridApi ||
                gridApi.isDestroyed() ||
                !this.props.fieldProperties.canUserHideColumns ||
                (event && ['toolPanelUi', 'api', 'gridOptionsUpdated'].includes(event.source))) {
                return;
            }
            const hiddenColumns = this.state?.columnStates?.filter(c => c.colId && c.hide).map(c => c.colId);
            const allColumns = this.getAllColumnIds(gridApi);
            const visibleColumns = this.getAllDisplayedColumnIds();
            const notVisibleNotHiddenColumns = this.getAllNotVisibleNotHiddenColumns(this.props.fieldProperties);
            const currentlyHiddenColumns = difference(allColumns, visibleColumns);
            const columnsToHide = visibleColumns
                .filter(c => c !== COLUMN_ID_VALIDATIONS)
                .filter(colId => hiddenColumns && hiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)));
            const columnsToUnhide = currentlyHiddenColumns
                .filter(c => c !== COLUMN_ID_VALIDATIONS && c !== this.state.groupByColumn?.field)
                .filter(colId => hiddenColumns && !hiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)))
                .filter(colId => !notVisibleNotHiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)));
            callGridMethod(gridApi, 'setColumnsVisible', columnsToHide, false);
            callGridMethod(gridApi, 'setColumnsVisible', columnsToUnhide, true);
            this.autoSizeAllColumns();
        };
        this.onFilterByErrors = async () => {
            this.props.onTelemetryEvent?.(`tableFilteredByErrors-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
            });
            this.isFilteringErrors = true;
            this.dataOperationMode = DataOperationMode.FILTER_ERRORS;
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            }
        };
        this.onUnsetFilterByErrors = async () => {
            this.isFilteringErrors = false;
            this.props.value.resetFilter();
            this.dataOperationMode = DataOperationMode.FETCH_PAGE;
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setFilterModel', null);
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            }
        };
        this.onColumnPanelHiddenStateChange = async (colId, isHidden) => {
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setColumnVisible', colId, isHidden);
                await this.updateColumnState();
                await this.updateColDefsStates();
            }
            if (this.state.columnStates && this.props.setTableViewColumnHidden) {
                const visible = this.getAllDisplayedColumnIds();
                const hidden = difference(this.getAllColumnIds(), visible);
                const hiddenColumns = {};
                visible.forEach(column => {
                    hiddenColumns[column] = false;
                });
                hidden.forEach(column => {
                    hiddenColumns[column] = true;
                });
                this.props.setTableViewColumnHidden(0, hiddenColumns);
            }
        };
        this.getCurrentColumnOrder = () => {
            if (!this.state.columnStates) {
                return [];
            }
            return this.state.columnStates
                .filter(columnState => columnState.colId !== undefined &&
                columnState.colId !== COLUMN_ID_ROW_SELECTION &&
                columnState.colId !== COLUMN_ID_ROW_ACTIONS &&
                columnState.colId !== COLUMN_ID_VALIDATIONS &&
                columnState.colId !== COLUMN_ID_AUTO_COLUMN)
                .map(columnState => columnState.colId);
        };
        this.moveStartIndex = this.props.fieldProperties.canSelect ? 1 : 0;
        this.onColumnPanelOrderChangeChange = async (colIds) => {
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'moveColumns', colIds, this.moveStartIndex);
                await this.updateColumnState();
                await this.updateColDefsStates();
            }
            if (this.state.columnStates && this.props.setTableViewColumnOrder) {
                this.props.setTableViewColumnOrder(0, this.getCurrentColumnOrder());
            }
        };
        this.onSortChanged = (event) => {
            if (event.source === 'gridOptionsUpdated') {
                return;
            }
            getSafeGridApiContext(columnState => {
                const groupBySort = columnState.find(c => c.colId === AUTO_COLUMN_ID)?.sort;
                const grouping = getTableViewGrouping(this.props.tableUserSettings);
                if (groupBySort &&
                    this.state.groupByColumn?.field &&
                    groupBySort !== grouping?.key &&
                    this.props.setTableViewGrouping) {
                    this.props.setTableViewGrouping(0, {
                        key: this.state.groupByColumn.field,
                        sort: groupBySort,
                        aggFunc: this.state.groupByColumn.aggFunc || undefined,
                    });
                }
            }, event.api, 'getColumnState');
            const sortOrder = event.api
                .getColumnState()
                .filter(columnState => columnState.colId !== undefined &&
                columnState.colId !== COLUMN_ID_ROW_SELECTION &&
                columnState.colId !== COLUMN_ID_ROW_ACTIONS &&
                columnState.colId !== COLUMN_ID_VALIDATIONS &&
                columnState.colId !== COLUMN_ID_AUTO_COLUMN)
                .map(columnState => {
                return {
                    colId: columnState.colId,
                    sort: columnState.sort ?? null,
                    sortIndex: columnState.sortIndex ?? null,
                };
            });
            if (this.props.setTableViewSortOrder) {
                this.props.setTableViewSortOrder(0, sortOrder);
            }
            if (this.state.groupByColumn) {
                // Force scroll to top when sorting with active grouping.
                // Otherwise, ag-Grid may call getRows() with a high startRow,
                // but cursors for those rows no exist after resetting the data source.
                event.api.ensureIndexVisible(0, 'top');
            }
            this.setShouldResetTable();
        };
        this.onCellClicked = async (event) => {
            // If we are in a phantom row and the user clicked on the minus we reset the row
            if (event.data?.__phantom &&
                event.column.getColId() === COLUMN_ID_ROW_ACTIONS &&
                event.event?.target?.getAttribute('data-element') === 'minus') {
                await this.props.value.resetRowToDefaults({
                    id: event.data._id,
                    level: event.data.__level,
                    parentId: event.data.__parentId,
                    isOrganicChange: true,
                    resetDirtiness: true,
                });
            }
        };
        this.onCellEditingStopped = async (event) => {
            const isPhantom = event.data?.__phantom !== undefined;
            if (!isPhantom) {
                return;
            }
            if (event.value !== event.oldValue) {
                const columnId = event.colDef.field;
                const newValue = event.data[event.colDef.field];
                let mergedValue = this.props.value.getMergedValue({
                    recordId: event.data._id,
                    columnId,
                    value: newValue,
                    level: event.data.__level,
                });
                const dirtyColumns = new Set([...mergedValue.__dirtyColumns, columnId]);
                const validationResult = await this.props.value.runValidationOnRecord({
                    recordData: {
                        ...mergedValue,
                        __dirtyColumns: dirtyColumns,
                    },
                    changedColumnId: event.colDef.field,
                });
                mergedValue = this.props.value.getMergedValue({
                    recordId: event.data._id,
                    columnId,
                    value: newValue,
                    level: event.data.__level,
                });
                const updatedData = {
                    ...mergedValue,
                    __validationState: validationResult,
                };
                this.props.value.setRecordValue({
                    recordData: updatedData,
                    level: event.data.__level,
                    toBeMarkedAsDirty: Array.from(dirtyColumns),
                    isOrganicChange: true,
                    changedColumnId: columnId,
                });
            }
        };
        this.onColumnResized = async (event) => {
            if (event.finished && event.column) {
                const columnKey = event.column.getColId();
                const columnWidth = event.column.getActualWidth();
                const updatedColumn = { key: columnKey, newWidth: columnWidth };
                this.setState(prevState => {
                    const currentColumns = [...prevState.resizedColumns];
                    const columnIndex = currentColumns.findIndex(col => col.key === columnKey);
                    if (columnIndex !== -1) {
                        currentColumns[columnIndex] = updatedColumn;
                    }
                    else {
                        currentColumns.push(updatedColumn);
                    }
                    return { resizedColumns: currentColumns };
                });
            }
        };
        this.onOpenColumnPanel = () => {
            this.setState({ isConfigurationDialogOpen: true });
        };
        this.onCloseColumnPanel = () => {
            this.setState({ isConfigurationDialogOpen: false });
        };
        this.noRowsOverlayComponentParams = {
            ...this.props,
            isEditable: !this.isReadOnly() && !this.isDisabled(),
        };
        this.onRowClicked = e => {
            return onRowClick(this.props.onRowClick)(e);
        };
        this.rowClassRules = getRowClassRules({
            canSelect: this.props.fieldProperties.canSelect || false,
            isChangeIndicatorDisabled: this.props.fieldProperties.isChangeIndicatorDisabled,
            openedRecordId: this.props.openedRecordId,
        });
        this.isInfiniteScroll = () => {
            return Boolean(this.props.isUsingInfiniteScroll);
        };
        this.getRowHeight = (params) => {
            return params?.data?.__isGroup ? GROUP_ROW_HEIGHT : ROW_HEIGHT;
        };
        this.onUnselectAll = async () => {
            if (!this.gridApi) {
                return;
            }
            callGridMethod(this.gridApi, 'deselectAll');
            this.setState({
                selectedRowCount: 0,
                isSelectAllChecked: false,
            }, () => {
                const currentFieldProps = xtremRedux.getStore().getState().screenDefinitions[this.props.screenId]
                    .metadata.uiComponentProperties[this.props.elementId];
                this.props.setFieldProperties?.(this.props.elementId, {
                    ...currentFieldProps,
                    selectedRecords: [],
                });
            });
        };
        this.onSelectionChanged = async (event) => {
            if (event.source === 'rowDataChanged') {
                // IMPORTANT: IGNORE when called from selectTableItems
                return;
            }
            const allNodes = [];
            event.api.forEachNode(node => {
                if (node.data) {
                    allNodes.push(node);
                }
            });
            const selectedNodes = event.api.getSelectedNodes() ?? [];
            const selectedGridIds = selectedNodes.filter(s => !s.group).map(s => s.data._id);
            const currentFieldProps = xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata
                .uiComponentProperties[this.props.elementId];
            const currentlySelectedIds = currentFieldProps?.selectedRecords ?? [];
            const shouldResetSelection = selectedNodes.length === 0 &&
                Object.values(event.api.getCacheBlockState() ?? {}).filter((b) => (b?.loadedRowCount ?? 0) > 0)
                    .length > 1;
            const selectedIds = shouldResetSelection
                ? []
                : selectedGridIds.concat(difference(currentlySelectedIds, allNodes.map(n => n.data._id)));
            const newlySelected = difference(selectedIds, currentlySelectedIds);
            const newlyUnselected = shouldResetSelection
                ? currentlySelectedIds
                : difference(currentlySelectedIds.filter(id => this.getRowNode({ id })), selectedIds);
            let selectedRecords = currentlySelectedIds;
            if (newlySelected.length > 0 || newlyUnselected.length > 0) {
                selectedRecords = difference(currentlySelectedIds, newlyUnselected).concat(newlySelected);
                this.props.setFieldProperties?.(this.props.elementId, {
                    ...currentFieldProps,
                    selectedRecords,
                });
                newlySelected.forEach(selectedId => {
                    triggerFieldEvent(this.props.screenId, this.props.elementId, 'onRowSelected', selectedId, splitValueToMergedValue(this.props.value.getRawRecord({ id: selectedId })));
                });
                newlyUnselected.forEach(selectedId => {
                    triggerFieldEvent(this.props.screenId, this.props.elementId, 'onRowUnselected', selectedId, splitValueToMergedValue(this.props.value.getRawRecord({ id: selectedId })));
                });
            }
            const selectionState = event.api.getServerSideSelectionState();
            if (selectionState === null) {
                this.setState({
                    selectedRowCount: 0,
                    isSelectAllChecked: false,
                });
                return;
            }
            let isSelectAllChecked = false;
            if ('selectAllChildren' in selectionState) {
                const nodeCount = allNodes.length;
                isSelectAllChecked =
                    Boolean(selectionState.selectAllChildren) ||
                        (selectedNodes.length > 0 &&
                            ((nodeCount > 0 && nodeCount === event.api.getSelectedNodes().filter(n => n.data).length) ||
                                selectedRecords.length > allNodes.length));
            }
            if (this.state.groupByColumn) {
                let selectedCount = 0;
                if (isSelectAllChecked) {
                    const total = await getTotalRecordCount({
                        api: event.api,
                        fieldProperties: this.props.fieldProperties,
                        screenId: this.props.screenId,
                        node: this.props.fieldProperties.node,
                        activeOptionsMenuItem: this.props.selectedOptionsMenuItem,
                        groupByColumnField: event.api.getRowGroupColumns()[0]?.getColId(),
                    });
                    selectedCount = (selectionState.toggledNodes ?? []).reduce((acc, node) => {
                        if (typeof node === 'string' || !node.nodeId) {
                            return acc;
                        }
                        const n = this.getRowNode({ id: node.nodeId });
                        if (!n) {
                            return acc;
                        }
                        if (node.selectAllChildren) {
                            return acc - (node.toggledNodes ?? []).length;
                        }
                        return acc - ((n.data.__groupCount ?? 0) - (node.toggledNodes ?? []).length);
                    }, total);
                }
                else {
                    selectedCount = (selectionState.toggledNodes ?? []).reduce((acc, node) => {
                        if (typeof node === 'string' || !node.nodeId) {
                            return acc;
                        }
                        const n = this.getRowNode({ id: node.nodeId });
                        if (!n) {
                            return acc;
                        }
                        if (node.selectAllChildren) {
                            return acc + ((n.data.__groupCount ?? 0) - (node.toggledNodes ?? []).length);
                        }
                        return acc + (node.toggledNodes ?? []).length;
                    }, 0);
                }
                this.setState({
                    selectedRowCount: selectedCount,
                    isSelectAllChecked,
                });
            }
            else if (isSelectAllChecked) {
                const total = await getTotalRecordCount({
                    api: event.api,
                    fieldProperties: this.props.fieldProperties,
                    screenId: this.props.screenId,
                    node: this.props.fieldProperties.node,
                    activeOptionsMenuItem: this.props.selectedOptionsMenuItem,
                    groupByColumnField: event.api.getRowGroupColumns()[0]?.getColId(),
                });
                const allRowsSelected = selectionState?.toggledNodes?.length === total;
                this.setState({
                    selectedRowCount: allRowsSelected ? total : total - (selectionState.toggledNodes ?? []).length,
                    isSelectAllChecked,
                });
            }
            else {
                this.setState({
                    selectedRowCount: (selectionState.toggledNodes ?? []).length,
                    isSelectAllChecked,
                });
            }
            if (isSelectAllChecked && !this.state.isSelectAllChecked) {
                this.props.onTelemetryEvent?.(`tableSelectAll-${this.props.elementId}`, {
                    elementId: this.props.elementId,
                    screenId: this.props.screenId,
                    contextType: this.props.contextType,
                    fieldType: this.props.fieldProperties._controlObjectType,
                });
            }
            else if (!isSelectAllChecked && this.state.isSelectAllChecked) {
                this.props.onTelemetryEvent?.(`tableUnselectAll-${this.props.elementId}`, {
                    elementId: this.props.elementId,
                    screenId: this.props.screenId,
                    contextType: this.props.contextType,
                    fieldType: this.props.fieldProperties._controlObjectType,
                });
            }
        };
        this.getRowSelection = () => {
            if (!this.props.fieldProperties.canSelect) {
                return undefined;
            }
            if (this.props.selectionMode === 'single') {
                return 'single';
            }
            return {
                mode: 'multiRow',
                groupSelects: 'filteredDescendants',
                selectAll: 'filtered',
            };
        };
        this.rowSelection = this.getRowSelection();
        this.getTableHeight = () => {
            if (!this.isInfiniteScroll() ||
                this.props.contextType === ContextType.dialog ||
                this.props.contextType === ContextType.navigationPanel) {
                return '100%';
            }
            if (!this.state.tableHasData) {
                return `${10 * ROW_HEIGHT}px`;
            }
            const numberOfRows = this.props.numberOfVisibleRows || this.props.fieldProperties.pageSize || 20;
            // Add one extra to account for table header
            const numberOfVisibleRows = this.props.fieldProperties.canAddNewLine && !this.props.fieldProperties.isPhantomRowDisabled
                ? numberOfRows + 2
                : numberOfRows + 1;
            // Add 15px to account for horizontal scrollbar
            return `${numberOfVisibleRows * ROW_HEIGHT + 15}px`;
        };
        setAgGridLicence();
        this.state = {
            autoGroupColumnDef: {
                colId: AUTO_COLUMN_ID,
                rowGroup: true,
                menuTabs: ['filterMenuTab', 'generalMenuTab'],
                flex: undefined,
                width: 250,
                minWidth: 250,
                resizable: true,
                editable: false,
                ...(this.props.groupTitle && { headerName: this.props.groupTitle }),
                sortable: true,
            },
            isConfigurationDialogOpen: false,
            columns: [],
            // Enable floating filters for lookups
            hasFloatingFilters: this.props.selectionMode !== undefined,
            isExportingExcel: false,
            isSelectAllChecked: false,
            phantomRows: [],
            selectedRowCount: 0,
            selectedCalendarView: 'dayGridMonth',
            tableHasData: props.value &&
                props.value.getData &&
                props.value.getData({ temporaryRecords: this.props.additionalLookupRecords?.() }).length > 0,
            resizedColumns: [],
        };
    }
    componentDidMount() {
        this.getColumnDefinitions({
            keepVisibility: true,
        });
        this.actionSubscription = subscribeToActions(action => {
            if (action.type === ActionType.RedrawComponent &&
                this.gridApi &&
                this.props.screenId === action.value.screenId &&
                this.props.elementId === action.value.elementId) {
                const columns = action.value.columnBind
                    ? (callGridMethod(this.gridApi, 'getColumnDefs') ?? [])
                        ?.filter((c) => c.field?.indexOf(action.value.columnBind) === 0)
                        .map((c) => c.colId)
                    : undefined;
                if (columns && columns.length > 0) {
                    this.gridApi.refreshCells({ force: true, columns });
                }
                else {
                    this.getColumnDefinitions({ keepVisibility: true });
                }
            }
        });
    }
    async componentDidUpdate(prevProps) {
        if (!this.isNavigationPanel() && this.props.selectedRecordId !== prevProps.selectedRecordId) {
            this.resetOptionsMenu();
        }
        if (!this.gridApi) {
            return;
        }
        if (!this.props.isInFocus && prevProps.isInFocus) {
            callGridMethod(this.gridApi, 'stopEditing', true);
        }
        // When the value changes (e.g the user switches between records), we should unset filters that were applied to the table
        if (!this.isNavigationPanel() && prevProps.value !== this.props.value) {
            callGridMethod(this.gridApi, 'setFilterModel', {});
        }
        else if (this.isNavigationPanel() &&
            this.props.value &&
            !isEqual(prevProps.tableUserSettings, this.props.tableUserSettings)) {
            await this.initColumnState();
            this.setShouldResetTable();
            this.applyTableView();
            setTimeout(() => {
                this.autoSizeAllColumns();
            }, 0);
        }
        if (!this.state.columnStates) {
            await this.initColumnState();
        }
        if (!isEqual(prevProps.selectedOptionsMenuItem, this.props.selectedOptionsMenuItem)) {
            callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
        }
        const fp = this.props.fieldProperties;
        if (prevProps.fieldProperties.isPhantomRowDisabled !== fp.isPhantomRowDisabled ||
            prevProps.fieldProperties.canAddNewLine !== fp.canAddNewLine) {
            await this.setPhantomRow();
            setTimeout(() => callGridMethod(this.gridApi, 'refreshServerSide', { purge: true }), 0);
        }
        if (prevProps.value !== this.props.value ||
            prevProps.fieldProperties.isDisabled !== this.props.fieldProperties.isDisabled ||
            prevProps.fieldProperties.tableViewMode !== this.props.fieldProperties.tableViewMode ||
            prevProps.isParentDisabled !== this.props.isParentDisabled ||
            prevProps.isReadOnly !== this.props.isReadOnly) {
            this.dataOperationMode =
                prevProps.fieldProperties.tableViewMode !== this.props.fieldProperties.tableViewMode
                    ? DataOperationMode.RESET_TABLE
                    : DataOperationMode.NONE;
            const resetTableFilters = this.isNavigationPanel();
            this.getColumnDefinitions({
                keepVisibility: true,
                keepGrouping: true,
                resetTableFilters,
                resetSelection: false,
            });
            // Revalidate field with current data
            this.props.value?.validateField();
            const ctx = getTableContext(this.gridApi);
            if (ctx) {
                // Reset headers
                setTableContext(this.gridApi, c => {
                    c.headerClasses = {};
                });
                if (!this.isNavigationPanel()) {
                    this.gridApi.refreshHeader();
                }
            }
        }
        if (!this.isNavigationPanel() && this.props.openedRecordId !== prevProps.openedRecordId) {
            const rowNodes = [];
            if (typeof prevProps.openedRecordId === 'string') {
                const node = this.getRowNode({ id: prevProps.openedRecordId, gridApi: this.gridApi });
                if (node) {
                    rowNodes.push(node);
                }
            }
            if (typeof this.props.openedRecordId === 'string') {
                const node = this.getRowNode({ id: this.props.openedRecordId, gridApi: this.gridApi });
                if (node) {
                    rowNodes.push(node);
                }
            }
            if (rowNodes.length) {
                setTimeout(() => callGridMethod(this.gridApi, 'redrawRows', { rowNodes }));
            }
        }
        if (!isEqual(sortBy(prevProps.fieldProperties.hiddenColumns), sortBy(this.props.fieldProperties.hiddenColumns))) {
            this.hideUnhideColumns();
            this.autoSizeAllColumns();
        }
        if (!isEqual(sortBy(prevProps.fieldProperties.selectedRecords), sortBy(this.props.fieldProperties.selectedRecords))) {
            const currentSelectedNodes = callGridMethod(this.gridApi, 'getSelectedNodes');
            const currentSelectedIds = (currentSelectedNodes ?? []).filter(n => n.data?._id).map(n => n.data._id);
            const newlySelected = difference(this.props.fieldProperties.selectedRecords ?? [], currentSelectedIds);
            const newlyUnselected = difference(currentSelectedIds, this.props.fieldProperties.selectedRecords ?? []);
            newlyUnselected.forEach(id => {
                const n = this.getRowNode({ id });
                if (n && n.isSelected()) {
                    n.setSelected(false);
                }
            });
            newlySelected.forEach(id => {
                const n = this.getRowNode({ id });
                if (n && !n.isSelected()) {
                    n.setSelected(true);
                }
            });
        }
    }
    componentWillUnmount() {
        if (this.actionSubscription) {
            this.actionSubscription();
        }
        if (this.collectionValueChangeSubscription) {
            this.collectionValueChangeSubscription();
        }
        if (this.collectionValidityChangeSubscription) {
            this.collectionValidityChangeSubscription();
        }
        window.removeEventListener('resize', this.resizeListener);
        this.gridApi?.removeGlobalListener(this.onGridEvent);
    }
    // eslint-disable-next-line react/no-unused-class-component-methods
    getSelectionFilter(mode = 'server') {
        if (!this.gridApi) {
            return {};
        }
        return getSelectionFilter({
            gridApi: this.gridApi,
            screenId: this.props.screenId,
            isSelectAllChecked: this.state.isSelectAllChecked,
            tableFieldProperties: this.props.fieldProperties,
            activeOptionsMenuItem: this.props.selectedOptionsMenuItem,
            mode,
        });
    }
    // eslint-disable-next-line react/no-unused-class-component-methods
    isSelectAllChecked() {
        return this.state.isSelectAllChecked;
    }
    redrawLineNumbers() {
        if (this.props.fieldProperties.hasLineNumbers) {
            callGridMethod(this.gridApi, 'refreshCells', {
                force: true,
                columns: [COLUMN_ID_LINE_NUMBER],
            });
        }
    }
    findLastLoadedRowData(api) {
        const lastKnownRowIndex = callGridMethod(api, 'getLastDisplayedRow');
        if (!lastKnownRowIndex) {
            return undefined;
        }
        for (let i = lastKnownRowIndex; i >= 0; i -= 1) {
            const rowNode = callGridMethod(api, 'getDisplayedRowAtIndex', i);
            if (rowNode?.data?._id) {
                return rowNode?.data;
            }
        }
        return undefined;
    }
    getCursor(api) {
        return this.findLastLoadedRowData(api)?.__cursor;
    }
    getFilters(filterModel) {
        return objectKeys(filterModel)
            .map(mapAgGridFilterToXtremFilters(filterModel))
            .concat(this.isFilteringErrors
            ? [
                {
                    id: '_id',
                    value: [
                        {
                            filterValue: this.props.value.getAllInvalidRecords().map(r => r._id),
                            filterType: {
                                text: '_id in list',
                                value: SET,
                            },
                        },
                    ],
                },
            ]
            : []);
    }
    async updateColumnState() {
        getSafeGridApiContext(columnStates => {
            return new Promise(resolve => {
                this.setState({ columnStates }, resolve);
            });
        }, this.gridApi, 'getColumnState');
    }
    renderCalendarView() {
        const eventCard = getCardDefinitionFromColumns({
            columns: this.props.fieldProperties.columns,
            mobileCard: this.props.fieldProperties.mobileCard,
            screenId: this.props.screenId,
            isGreaterThanSmall: true,
            hiddenColumns: this.props.fieldProperties.hiddenColumns,
            sortColumns: this.props.fieldProperties.sortColumns,
        });
        if (!this.props.fieldProperties.startDateField) {
            // This should never be the case because the calendar view is only available if the startDateField property is defined
            return null;
        }
        return (React.createElement("div", { className: "e-navigation-panel-calendar-wrapper" },
            React.createElement(AsyncCalendarBodyComponent, { cardColor: this.props.fieldProperties.cardColor, elementId: this.props.elementId, endDateField: this.props.fieldProperties.endDateField, eventCard: eventCard, selectedView: this.state.selectedCalendarView, fieldProperties: this.props.fieldProperties, isGreaterThanSmall: true, onEventClick: (_id, event, isModifierKeyPushed) => {
                    this.props.onRowClick?.(_id, event, isModifierKeyPushed)();
                }, screenId: this.props.screenId, selectedOptionsMenuItem: this.props.selectedOptionsMenuItem, startDateField: this.props.fieldProperties.startDateField, value: this.props.value, isEventMovable: this.props.fieldProperties.isEventMovable })));
    }
    async initColumnState() {
        getSafeGridApiContext(columnStates => {
            return new Promise(resolve => {
                const columnHidden = getTableViewColumnHidden(this.props.tableUserSettings) || {};
                columnStates.forEach((column) => {
                    const isTableViewColumnHidden = columnHidden[column.colId];
                    if (isTableViewColumnHidden != null &&
                        Boolean(column.hide) !== Boolean(isTableViewColumnHidden)) {
                        column.hide = Boolean(columnHidden[column.colId]);
                    }
                });
                this.setState({ columnStates }, resolve);
            });
        }, this.gridApi, 'getColumnState');
    }
    render() {
        const domLayout = this.isInfiniteScroll() ? 'normal' : 'autoHeight';
        const hasBulkActions = (this.props.bulkActions ?? []).length > 0;
        const shouldRenderBulkActionBar = hasBulkActions && this.state.selectedRowCount > 0 && this.gridApi;
        const hasAggregatedGroupColumns = !!this.props.fieldProperties?.columns?.find(c => !!c.properties.groupAggregationMethod);
        const columnPanelColumnStates = getColumnStatesForColumnPanel(this.props.screenId, (callGridMethod(this.gridApi, 'getColumnDefs') ?? []), this.state.columnStates);
        const addItemActions = resolveByValue({
            propertyValue: this.props.fieldProperties.addItemActions,
            rowValue: null,
            screenId: this.props.screenId,
            fieldValue: null,
            skipHexFormat: true,
        }) ?? [];
        const hasAddItemsButton = Boolean((this.props.fieldProperties.canAddNewLine && !this.props.fieldProperties.isPhantomRowDisabled) ||
            addItemActions.filter(action => !action.isHidden).length > 0);
        return (React.createElement(React.Fragment, null,
            process.env.NODE_ENV === 'test' && this.gridApi && React.createElement("p", { "data-testid": "api", style: { display: 'none' } }),
            shouldRenderBulkActionBar && this.gridApi && (React.createElement(DesktopTableBulkActionBar, { screenId: this.props.screenId, isSelectAllChecked: Boolean(this.state.isSelectAllChecked), bulkActions: this.props.bulkActions, gridApi: this.gridApi, selectedRowCount: this.state.selectedRowCount, groupByColumn: this.state.groupByColumn, onTelemetryEvent: this.props.onTelemetryEvent, onClearSelection: this.onUnselectAll })),
            (!this.props.isLookupDialog || this.props.elementId === navigationPanelId) &&
                !shouldRenderBulkActionBar && (React.createElement(DesktopTableHeaderComponent, { canAddNewLine: Boolean(this.props.fieldProperties.canAddNewLine), columnPanelDisabled: !!this.state.groupByColumn || !this.props.fieldProperties.canUserHideColumns, elementId: this.props.elementId, fieldProperties: this.props.fieldProperties, hasAddItemsButton: hasAddItemsButton, hasCalendarView: !!this.props.fieldProperties.startDateField, hasData: this.state.tableHasData, hasFloatingFilters: this.state.hasFloatingFilters, hasSidebar: Boolean(this.props.fieldProperties.sidebar), isDisabled: this.isDisabled(), isExportDisabled: !!this.state.groupByColumn, isReadOnly: this.isReadOnly(), onExport: this.exportToExcel, onFilterByErrors: this.onFilterByErrors, onFocusPhantomRow: this.focusPhantomRowAndStartEditing, onOpenColumnPanel: this.onOpenColumnPanel, onOptionsMenuItemChange: this.onOptionsMenuItemChange, onSwitchCalendarView: this.onSwitchCalendarView, onSwitchView: this.onSwitchView, onTelemetryEvent: this.props.onTelemetryEvent, onToggleFilters: this.hasFilterableColumns() ? this.toggleFloatingFilters : undefined, onUnsetFilterByErrors: this.onUnsetFilterByErrors, screenId: this.props.screenId, selectedCalendarView: this.state.selectedCalendarView, selectedOptionsMenuItem: this.props.selectedOptionsMenuItem, tableViewMode: this.props.fieldProperties.tableViewMode, validationErrors: this.props.validationErrors })),
            React.createElement("div", { "data-testid": `e-table-field-${this.props.elementId}`, style: {
                    height: this.getTableHeight(),
                    overflow: 'hidden',
                }, className: "e-table-field-desktop-wrapper" },
                this.props.fieldProperties.tableViewMode === 'calendar' && this.renderCalendarView(),
                (this.props.fieldProperties.tableViewMode === 'table' ||
                    !this.props.fieldProperties.tableViewMode) && (React.createElement("div", { ref: this.containerRef, style: {
                        height: '100%',
                        width: '100%',
                        visibility: this.state.isExportingExcel ? 'hidden' : 'visible',
                    }, className: "ag-theme-balham" },
                    React.createElement(AgGridReact, { stopEditingWhenCellsLoseFocus: true, gridId: this.gridId, autoSizeStrategy: this.autoSizeStrategy, autoGroupColumnDef: this.state.autoGroupColumnDef, cacheBlockSize: this.getCacheBlockSize(), columnDefs: this.state.columns, components: frameworkComponents, debug: hasAgGridLogging(), defaultColDef: this.defaultColDef, domLayout: domLayout, floatingFiltersHeight: ROW_HEIGHT, getMainMenuItems: this.getMainMenuItems, getRowId: this.getRowId, gridOptions: this.gridOptions, groupDisplayType: "singleColumn", headerHeight: ROW_HEIGHT, isServerSideGroup: this.isServerSideGroup, loadingCellRenderer: TableLoadingCellRenderer, loadingCellRendererParams: this.loadingCellRendererParams, loadingOverlayComponent: "Loader", loadingOverlayComponentParams: this.loadingOverlayComponentParams, localeText: localeText(), maxConcurrentDatasourceRequests: 1, modules: activatedAgGridModules, noRowsOverlayComponent: "DesktopTableEmptyComponent", noRowsOverlayComponentParams: this.noRowsOverlayComponentParams, onCellClicked: this.onCellClicked, onCellEditingStopped: this.onCellEditingStopped, onCellFocused: this.onCellFocused, onCellKeyDown: this.onCellKeyDown, onColumnEverythingChanged: this.hideUnhideColumns, onColumnResized: this.onColumnResized, onColumnRowGroupChanged: this.handleColumnRowGroupChanged, onFilterChanged: this.handleFilterChanged, onFirstDataRendered: this.onFirstDataRendered, onGridReady: this.onGridReady, onRowClicked: this.onRowClicked, onSelectionChanged: this.onSelectionChanged, onSortChanged: this.onSortChanged, pagination: !this.isInfiniteScroll(), paginationPageSize: this.props.fieldProperties.pageSize || 20, paginationPageSizeSelector: false, pinnedTopRowData: this.state.phantomRows ?? [], rowClassRules: this.rowClassRules, getRowHeight: this.getRowHeight, rowModelType: "serverSide", rowSelection: this.rowSelection, serverSideSortAllLevels: true, suppressAggFuncInHeader: true, suppressCellFocus: this.isDisabled(), suppressColumnVirtualisation: process.env.NODE_ENV === 'test', suppressContextMenu: true, suppressDragLeaveHidesColumns: true, suppressExcelExport: false, suppressPaginationPanel: this.isDisabled(), suppressRowClickSelection: true, undoRedoCellEditing: true, valueCacheNeverExpires: true, groupTotalRow: hasAggregatedGroupColumns ? 'bottom' : undefined })))),
            React.createElement(TableConfigurationDialog, { isDialogOpen: this.state.isConfigurationDialogOpen, onDialogClose: this.onCloseColumnPanel, columnPanelColumnStates: columnPanelColumnStates, onChangeColumnHiddenState: this.onColumnPanelHiddenStateChange, onOrderChange: this.onColumnPanelOrderChangeChange })));
    }
}
const mapStateToProps = (state, props) => {
    const screenDefinition = getPageDefinitionFromState(props.screenId, state);
    const screenElement = getScreenElement(screenDefinition);
    const selectedRecordId = screenDefinition.queryParameters?._id;
    const openSidebarDialogContent = Object.values(state.activeDialogs).find(dialog => dialog.screenId === props.screenId &&
        dialog.type === 'table-sidebar' &&
        dialog.content.elementId === props.elementId)?.content;
    const pageNode = getPagePropertiesFromPageDefinition(screenDefinition).node;
    const tableUserSettings = screenDefinition.userSettings[props.elementId];
    const selectedOptionsMenuItem = getActiveOptionsMenu(props.screenId, tableUserSettings, props.fieldProperties.optionsMenu);
    const isTableInFocus = !!state.focusPosition &&
        state.focusPosition.elementId === props.elementId &&
        state.focusPosition.screenId === props.screenId;
    return {
        ...props,
        isTableInFocus,
        isInFocus: Boolean(isTableInFocus && !state.focusPosition?.nestedField),
        graphApi: screenElement.$.graph,
        selectedOptionsMenuItem,
        selectedRecordId,
        openedRecordId: openSidebarDialogContent?.recordId,
        activeLookupDialog: state.activeLookupDialog,
        tableUserSettings,
        pageNode,
        accessBindings: screenDefinition.accessBindings,
        dataTypes: state.dataTypes,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
    };
};
const mapDispatchToProps = (dispatch, props) => ({
    setTableViewColumnHidden: (level, columnHidden) => {
        dispatch(xtremRedux.actions.setTableViewColumnHidden(props.screenId, props.elementId, level, columnHidden));
    },
    setTableViewColumnOrder: (level, columnOrder) => {
        dispatch(xtremRedux.actions.setTableViewColumnOrder(props.screenId, props.elementId, level, columnOrder));
    },
    setTableViewFilter: (level, filter) => {
        dispatch(xtremRedux.actions.setTableViewFilter(props.screenId, props.elementId, level, filter));
    },
    setTableViewGrouping: (level, grouping) => {
        dispatch(xtremRedux.actions.setTableViewGrouping(props.screenId, props.elementId, level, grouping));
    },
    setTableViewOptionsMenuItem: (level, optionsMenuItem) => {
        dispatch(xtremRedux.actions.setTableViewOptionMenuItem(props.screenId, props.elementId, level, optionsMenuItem));
    },
    setTableViewOptionsMenuItemAndViewFilter: (level, optionsMenuItem, filter) => {
        dispatch(xtremRedux.actions.setTableViewOptionsMenuItemAndViewFilter(props.screenId, props.elementId, level, optionsMenuItem, filter));
    },
    setTableViewSortOrder: (level, sortOrder) => {
        dispatch(xtremRedux.actions.setTableViewSortOrder(props.screenId, props.elementId, level, sortOrder));
    },
});
export const ConnectedDesktopTableComponent = connect(mapStateToProps, mapDispatchToProps, null, { forwardRef: true })(DesktopTableComponent);
export default ConnectedDesktopTableComponent;
//# sourceMappingURL=desktop-table-component.js.map