import React from 'react';
import { connect } from 'react-redux';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { getDataTestIdAttribute } from '../../../utils/dom';
import { triggerFieldEvent } from '../../../utils/events';
import { useFocus } from '../../../utils/hooks/effects/use-focus';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { addOptionsAndLocalizationToProps } from '../../../utils/transformers';
import { getFieldIndicatorStatus, getFieldTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { HelperText, FieldLabel } from '../carbon-utility-components';
import { CarbonWrapper } from '../carbon-wrapper';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { resolveDetailedIcon } from '../../../utils/detailed-icons-utils';
import { SelectionCard } from '@sage/xtrem-ui-components';
import Search from 'carbon-react/esm/components/search';
import { localize } from '../../../service/i18n-service';
export function SelectionCardComponent(props) {
    const [filterValue, setFilterValue] = React.useState('');
    const componentRef = React.useRef(null);
    useFocus(componentRef, props.isInFocus, 'button');
    const getDescription = React.useCallback((option) => resolveByValue({
        propertyValue: props.fieldProperties.description,
        skipHexFormat: true,
        screenId: props.screenId,
        fieldValue: option,
        rowValue: null, // Nested selection-cards are not supported
    }), [props.fieldProperties.description, props.screenId]);
    const getTitle = React.useCallback((option) => props.localizedOptions && props.localizedOptions[option]
        ? props.localizedOptions[option]
        : props.fieldProperties.map?.apply({}, [option]) || option, [props.fieldProperties.map, props.localizedOptions]);
    const optionsToUse = React.useMemo(() => {
        const options = props.enumOptions ||
            resolveByValue({
                propertyValue: props.fieldProperties.options,
                skipHexFormat: true,
                screenId: props.screenId,
                fieldValue: props.value,
                rowValue: null, // Nested selection-cards are not supported
            }) ||
            [];
        const sortedOptions = props.fieldProperties.isSortedAlphabetically
            ? [...options].sort((a, b) => {
                const aLabel = props.localizedOptions && props.localizedOptions[a]
                    ? props.localizedOptions[a]
                    : props.fieldProperties.map?.apply({}, [a]) || a;
                const bLabel = props.localizedOptions && props.localizedOptions[b]
                    ? props.localizedOptions[b]
                    : props.fieldProperties.map?.apply({}, [b]) || b;
                return aLabel.localeCompare(bLabel);
            })
            : options;
        return sortedOptions.filter(option => {
            if (!props.fieldProperties.hasFilter || !filterValue) {
                return true;
            }
            if (getDescription(option).toLowerCase().includes(filterValue.toLowerCase())) {
                return true;
            }
            if (getTitle(option).toLowerCase().includes(filterValue.toLowerCase())) {
                return true;
            }
            return false;
        });
    }, [
        filterValue,
        getDescription,
        getTitle,
        props.enumOptions,
        props.fieldProperties.hasFilter,
        props.fieldProperties.isSortedAlphabetically,
        props.fieldProperties.map,
        props.fieldProperties.options,
        props.localizedOptions,
        props.screenId,
        props.value,
    ]);
    const isMandatory = React.useMemo(() => resolveByValue({
        propertyValue: props.fieldProperties.isMandatory,
        skipHexFormat: true,
        screenId: props.screenId,
        fieldValue: props.value,
        rowValue: null, // Nested selection-cards are not supported
    }), [props.fieldProperties.isMandatory, props.screenId, props.value]);
    const isReadOnly = React.useMemo(() => isFieldDisabled(props.screenId, props.fieldProperties, props.value, null) ||
        isFieldReadOnly(props.screenId, props.fieldProperties, props.value, null), [props.fieldProperties, props.screenId, props.value]);
    const onClick = React.useCallback((option) => () => {
        handleChange(props.elementId, option, props.setFieldValue, props.validate, triggerChangeListener(props.screenId, props.elementId));
    }, [props.elementId, props.screenId, props.setFieldValue, props.validate]);
    const title = React.useMemo(() => {
        const resolvedTitle = getFieldTitle(props.screenId, props.fieldProperties, null);
        return `${resolvedTitle || ''}${isMandatory ? ' *' : ''}`;
    }, [props.fieldProperties, props.screenId, isMandatory]);
    const hasTitle = React.useMemo(() => !props.fieldProperties.isTitleHidden && title !== '' && title !== undefined, [props.fieldProperties.isTitleHidden, title]);
    const testId = React.useMemo(() => getDataTestIdAttribute('selection-card-component', 'selection-card-button', props.elementId), [props.elementId]);
    const getIcon = React.useCallback((option) => resolveDetailedIcon(resolveByValue({
        propertyValue: props.fieldProperties.icon,
        skipHexFormat: true,
        screenId: props.screenId,
        fieldValue: option,
        rowValue: null, // Nested selection-cards are not supported
    })), [props.fieldProperties.icon, props.screenId]);
    const { error, warning, info } = getFieldIndicatorStatus(props);
    return (React.createElement(CarbonWrapper, { noReadOnlySupport: true, ...props, className: "e-selection-card-field", componentName: "selection-card", value: props.value, componentRef: componentRef },
        hasTitle && React.createElement(FieldLabel, { label: title, errorMessage: error, warningMessage: warning, infoMessage: info }),
        props.fieldProperties.hasFilter && (React.createElement(Search, { placeholder: localize('@sage/xtrem-ui/selection-card-filter-placeholder', 'Filter...'), value: filterValue, onChange: e => setFilterValue(e.target.value), mb: "8px", mt: "8px", "data-testid": "e-selection-card-filter" })),
        React.createElement("div", { className: "selection-card-container" }, optionsToUse.map(option => {
            return (React.createElement(SelectionCard, { key: option, ref: componentRef, _id: testId, isSelected: props.value === option, isReadOnly: isReadOnly, title: getTitle(option), icon: getIcon(option), description: getDescription(option), onClick: onClick(option) }));
        })),
        React.createElement(HelperText, { helperText: props.fieldProperties.helperText })));
}
const triggerChangeListener = (screenId, elementId) => () => {
    triggerFieldEvent(screenId, elementId, 'onChange');
};
export const ConnectedSelectionCardComponent = connect((state, externalProps) => addOptionsAndLocalizationToProps(state, mapStateToProps()(state, externalProps)), mapDispatchToProps())(SelectionCardComponent);
export default ConnectedSelectionCardComponent;
//# sourceMappingURL=selection-card-component.js.map