import { Datetime, DateValue, formatDateToCurrentLocale, datePropertyValueToDateString } from '@sage/xtrem-date-time';
import { intervalToDuration } from 'date-fns';
import { getStore } from '../../../redux';
import { isDateOrDateTime } from '../../../utils/formatters';
const getFormattedValue = (date) => {
    const valueType = isDateOrDateTime(date);
    if (valueType === 'date') {
        return DateValue.parse(datePropertyValueToDateString(date));
    }
    return Datetime.parse(datePropertyValueToDateString(date, true));
};
export function toRelative_(type, lang, end, start = new Date(Date.now())) {
    const startDate = getFormattedValue(start);
    const endDate = getFormattedValue(end);
    /* eslint-enable */
    const format = (value, unit) => {
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/RelativeTimeFormat/RelativeTimeFormat#parameters
        const intlRelativeTimeFormat = new Intl.RelativeTimeFormat(lang, {
            numeric: 'auto',
            localeMatcher: 'best fit',
            style: 'long',
        });
        return intlRelativeTimeFormat.format(value, unit);
    };
    let weeks = 0;
    let days = 0;
    const { years, months, days: originalDays, hours, minutes, seconds, } = intervalToDuration({ start: startDate.toJsDate().getTime(), end: endDate.toJsDate().getTime() });
    if (originalDays && Math.abs(originalDays) >= 7) {
        weeks = Math.trunc(originalDays / 7);
        days = originalDays % 7;
    }
    if (years && Math.abs(years) > 0) {
        return format(years, 'year');
    }
    if (months && Math.abs(months) > 0) {
        return format(months, 'month');
    }
    if (weeks && Math.abs(weeks) > 0) {
        return format(weeks, 'week');
    }
    if (type === 'date') {
        const startDateValue = DateValue.parse(startDate.toString().substring(0, 10));
        const endDateValue = DateValue.parse(endDate.toString().substring(0, 10));
        const day = endDateValue.daysDiff(startDateValue);
        return format(day, 'day');
    }
    if (days && Math.abs(days) > 0) {
        return format(days, 'day');
    }
    if (hours && Math.abs(hours) > 0) {
        return format(hours, 'hour');
    }
    if (minutes && Math.abs(minutes) > 0) {
        return format(minutes, 'minute');
    }
    if ((seconds && Math.abs(seconds) >= 0) || seconds === 0) {
        if (Math.abs(seconds) < 20) {
            return format(0, 'second');
        }
        return format(seconds, 'second');
    }
    if (endDate.toJsDate().getTime() - startDate.toJsDate().getTime() < 1000) {
        return format(0, 'second');
    }
    return false;
}
export function toRelative(type, end) {
    const state = getStore().getState();
    return toRelative_(type, state.applicationContext?.locale || 'en-US', end);
}
export const getRelativeDateDisplayValue = (value, scope) => {
    let correctedScope = scope || null;
    if (!correctedScope) {
        correctedScope = isDateOrDateTime(value);
    }
    return toRelative(correctedScope ?? 'datetime', value) || formatDateToCurrentLocale(value);
};
//# sourceMappingURL=relative-date-utils.js.map