import { RadioButton, RadioButtonGroup } from 'carbon-react/esm/components/radio-button';
import React, { useRef } from 'react';
import { connect } from 'react-redux';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { getDataTestIdAttribute } from '../../../utils/dom';
import { triggerFieldEvent } from '../../../utils/events';
import { useFocus } from '../../../utils/hooks/effects/use-focus';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { addOptionsAndLocalizationToProps } from '../../../utils/transformers';
import { getFieldIndicatorStatus, getLabelTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { HelperText } from '../carbon-utility-components';
import { CarbonWrapper } from '../carbon-wrapper';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import styled from 'styled-components';
import RadioButtonGroupStyle from 'carbon-react/esm/components/radio-button/radio-button-group/radio-button-group.style';
import CheckBoxStyle from 'carbon-react/esm/components/checkbox/checkbox.style';
const StyledCarbonWrapper = styled(CarbonWrapper) `
    ${CheckBoxStyle} {
        &:not(:last-child) {
            margin-right: 72px;
        }
        display: inline-block;
    }
    ${RadioButtonGroupStyle} {
        display: block;
    }
`;
export function RadioComponent(props) {
    const componentRef = useRef(null);
    useFocus(componentRef, props.isInFocus, 'input');
    const title = getLabelTitle(props.screenId, props.fieldProperties, null) || '';
    const { error, warning, info } = getFieldIndicatorStatus(props);
    const radioButtonGroupProps = {
        groupName: props.elementId,
        label: title,
        name: props.elementId,
        legend: title,
        value: props.value,
        error,
        warning,
        info,
        onChange: evt => handleChange(props.elementId, evt.currentTarget.value, props.setFieldValue, props.validate, triggerChangeListener(props.screenId, props.elementId)),
    };
    let optionsToUse = props.enumOptions ||
        resolveByValue({
            propertyValue: props.fieldProperties.options,
            skipHexFormat: true,
            screenId: props.screenId,
            fieldValue: props.value,
            rowValue: null, // Nested radio buttons are not supported
        }) ||
        [];
    const isReadOnly = isFieldDisabled(props.screenId, props.fieldProperties, props.value, null) ||
        isFieldReadOnly(props.screenId, props.fieldProperties, props.value, null);
    if (props.fieldProperties.isSortedAlphabetically) {
        optionsToUse = [...optionsToUse].sort((a, b) => {
            const aLabel = props.localizedOptions && props.localizedOptions[a]
                ? props.localizedOptions[a]
                : props.fieldProperties.map?.apply({}, [a]) || a;
            const bLabel = props.localizedOptions && props.localizedOptions[b]
                ? props.localizedOptions[b]
                : props.fieldProperties.map?.apply({}, [b]) || b;
            return aLabel.localeCompare(bLabel);
        });
    }
    return (React.createElement(StyledCarbonWrapper, { noReadOnlySupport: true, ...props, className: "e-radio-field", componentName: "radio", value: props.value, componentRef: componentRef },
        React.createElement(RadioButtonGroup, { "data-testid": getDataTestIdAttribute('radio-component', 'radio-button-group', props.elementId), ...radioButtonGroupProps }, optionsToUse.map(option => renderRadioButton(option, props.elementId, isReadOnly, props.onFocus, componentRef, props.localizedOptions, props.fieldProperties.map?.apply({}, [option])))),
        React.createElement(HelperText, { helperText: props.fieldProperties.helperText })));
}
const triggerChangeListener = (screenId, elementId) => () => {
    triggerFieldEvent(screenId, elementId, 'onChange');
};
const renderRadioButton = (value, elementId, disabled = false, onFocus, ref, localizedOptions, mappedText) => {
    const label = localizedOptions && localizedOptions[value] ? localizedOptions[value] : mappedText || value;
    const carbonRadioProps = {
        value,
        label,
        size: 'small',
        key: value,
        disabled,
    };
    return (React.createElement(RadioButton, { ref: ref, onFocus: onFocus, className: "e-radio-button-item", "data-testid": getDataTestIdAttribute('radio-component', 'radio-button', elementId), ...carbonRadioProps }));
};
export const ConnectedRadioComponent = connect((state, externalProps) => addOptionsAndLocalizationToProps(state, mapStateToProps()(state, externalProps)), mapDispatchToProps())(RadioComponent);
export default ConnectedRadioComponent;
//# sourceMappingURL=radio-component.js.map