var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
import { objectKeys } from '@sage/xtrem-shared';
import { get, set, sortBy, uniq } from 'lodash';
import { ActionType, getStore } from '../../../redux';
import { openTableSidebar } from '../../../redux/actions';
import { CollectionValue } from '../../../service/collection-data-service';
import { CollectionFieldTypes } from '../../../service/collection-data-types';
import { fetchNestedDefaultValues } from '../../../service/graphql-service';
import { PromiseTracker } from '../../../service/promise-tracker';
import { showToast } from '../../../service/toast-service';
import { xtremConsole } from '../../../utils/console';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { ControlObjectProperty } from '../../property-decorators/control-object-property-decorator';
import { ReadonlyFieldControlObject } from '../../readonly-field-control-object';
import { TableDisplayMode } from '../../types';
import { getLevelMap } from './nested-grid-utils';
/**
 * [Field]{@link ReadonlyFieldControlObject} that holds a set of values of any type. It can contain nested fields
 */
export class NestedGridControlObject extends ReadonlyFieldControlObject {
    static { this.defaultUiProperties = {
        ...ReadonlyFieldControlObject.defaultUiProperties,
        canFilter: true,
        canSelect: true,
        canUserHideColumns: true,
        displayMode: TableDisplayMode.comfortable,
        pageSize: 20,
    }; }
    ensureNestedGridHasValue(initialValues = []) {
        const currentValue = this._getValue();
        return !currentValue ? this.setFieldValue(initialValues) : currentValue;
    }
    setFieldValue(initialValues = []) {
        const levels = this.getUiComponentProperty('levels');
        const nodes = levels.map(l => String(l.node));
        const nestedGridFilters = levels.map(l => resolveByValue({
            screenId: this.screenId,
            skipHexFormat: true,
            propertyValue: l.filter,
            rowValue: null,
            fieldValue: null,
        }));
        const value = new CollectionValue({
            bind: this.uiComponentProperties.bind,
            columnDefinitions: levels.map(level => level.columns),
            elementId: this.elementId,
            fieldType: CollectionFieldTypes.NESTED_GRID,
            filter: nestedGridFilters,
            hasNextPage: false,
            initialValues,
            levelMap: getLevelMap(levels),
            mapServerRecordFunctions: levels.map(l => l.mapServerRecord),
            isTransient: !!this.getUiComponentProperty('isTransient'),
            nodes,
            orderBy: levels.map(l => l.orderBy),
            screenId: this.screenId,
        });
        this._setValue(value);
        return value;
    }
    defaultSelectedRecords() {
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        return this.getUiComponentProperty('levels').map((_) => []);
    }
    /** Whether user can hide columns or not */
    get canUserHideColumns() {
        const propertyValue = this.getUiComponentProperty('canUserHideColumns');
        return propertyValue === undefined ? true : propertyValue;
    }
    /** Whether user can hide columns or not */
    set canUserHideColumns(newValue) {
        this.setUiComponentProperties('canUserHideColumns', newValue);
    }
    /**
     * Indicate additional warning message, rendered as tooltip and blue border.
     */
    get infoMessage() {
        return resolveByValue({
            fieldValue: this.value,
            propertyValue: this.getUiComponentProperty('infoMessage'),
            rowValue: undefined,
            screenId: this.screenId,
            skipHexFormat: true,
        });
    }
    /**
     * Indicate additional warning message, rendered as tooltip and blue border.
     */
    set infoMessage(infoMessage) {
        this.setUiComponentProperties('infoMessage', infoMessage);
    }
    /**
     * Indicate additional information, rendered as tooltip and orange border
     */
    get warningMessage() {
        return resolveByValue({
            fieldValue: this.value,
            propertyValue: this.getUiComponentProperty('warningMessage'),
            rowValue: undefined,
            screenId: this.screenId,
            skipHexFormat: true,
        });
    }
    /**
     * Indicate additional information, rendered as tooltip and orange border
     */
    set warningMessage(warningMessage) {
        this.setUiComponentProperties('warningMessage', warningMessage);
    }
    /** Selected rows identifiers */
    get selectedRecords() {
        this.ensureNestedGridHasValue();
        const selectedRecords = this.getUiComponentProperty('selectedRecords') || this.defaultSelectedRecords();
        return selectedRecords.map(l => uniq(l));
    }
    /** Selected rows identifiers */
    set selectedRecords(rowIds) {
        this.ensureNestedGridHasValue();
        this.setUiComponentProperties('selectedRecords', rowIds.map(l => sortBy(uniq(l))));
    }
    /** Selects a given row by its identifier*/
    selectRecord(recordId, level) {
        this.ensureNestedGridHasValue();
        const rowIds = get(this.getUiComponentProperty('selectedRecords'), level, []);
        if (rowIds.indexOf(String(recordId)) === -1) {
            rowIds.push(String(recordId));
            const newSelectedRecords = set(this.getUiComponentProperty('selectedRecords') || this.defaultSelectedRecords(), level, uniq(sortBy(rowIds)));
            this.setUiComponentProperties('selectedRecords', newSelectedRecords);
        }
    }
    async refresh() {
        await PromiseTracker.withTracker(() => this._refresh({ keepPageInfo: true }).catch(e => {
            showToast(e.message || e, { type: 'warning' });
        }));
    }
    /** Unselects a given row by its identifier */
    unselectRecord(recordId, level) {
        this.ensureNestedGridHasValue();
        const rowIds = get(this.getUiComponentProperty('selectedRecords'), level, []);
        const rowIdIndex = rowIds.indexOf(String(recordId));
        if (rowIdIndex !== -1) {
            rowIds.splice(rowIdIndex, 1);
            const newSelectedRecords = set(this.getUiComponentProperty('selectedRecords') || this.defaultSelectedRecords(), level, rowIds);
            this.setUiComponentProperties('selectedRecords', newSelectedRecords);
        }
    }
    /** Unselects all items */
    unselectAllRecords() {
        this.ensureNestedGridHasValue();
        this.setUiComponentProperties('selectedRecords', this.defaultSelectedRecords());
    }
    /**
     * Return all data known to the client (it does not include rows which were not previously fetched by some
     * user interaction such as pagination) in normalized collection form.
     * */
    get normalizedValue() {
        const value = this._getValue();
        if (!value) {
            return [];
        }
        return value.getRawRecords();
    }
    /**
     * Return all data known to the client (it does not include rows which were not previously fetched by some
     * user interaction such as pagination) in a tree form.
     * */
    get value() {
        const value = this._getValue();
        if (!value) {
            return [];
        }
        return value.getAllDataAsTree({
            cleanInputTypes: false,
        });
    }
    /**
     * Resets the value of the nested grid, all pending changes will be overridden and all cached rows will be discarded
     * */
    set value(newValue) {
        if (!this.getUiComponentProperty('isTransient')) {
            xtremConsole.warn('Reassigning nestedGrid values is discouraged. Please update individual rows using the setRecordValue(), addRecord(), removeRecord() functions.');
        }
        this.setUiComponentProperties('selectedRecords', objectKeys(this.getUiComponentProperty('levels')).map(() => []));
        this.setFieldValue(newValue);
    }
    validateLevel(level) {
        const maxLevel = this.getUiComponentProperty('levels').length - 1;
        if (level < 0 || level > maxLevel) {
            throw new Error(`Invalid level '${level}' received. Please specify a value between 0 and ${maxLevel}.`);
        }
    }
    openSidebar(recordId, level = 0) {
        const sidebarDefinition = this.properties.levels[level].sidebar;
        const columns = this.properties.levels[level].columns;
        const cardDefinition = level === 0 ? this.properties.mobileCard : undefined;
        const dispatch = getStore().dispatch;
        dispatch(openTableSidebar({
            screenId: this.screenId,
            elementId: this.elementId,
            recordId,
            level,
            sidebarDefinition,
            cardDefinition,
            columns,
        }));
    }
    /** Update the value of a single row in the collection */
    setRecordValue(recordValue, level) {
        const value = this.ensureNestedGridHasValue();
        value.setRecordValue({ recordData: recordValue, level });
    }
    /** Return a single record that is already known to the client */
    getRecordValue(recordId, level) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        if (!value) {
            return null;
        }
        return value.getRecordWithChildren({
            id: recordId,
            level,
        });
    }
    /** Add a single row to the nested grid and in its dataset with default values coming from the server */
    async addRecordWithDefaults(level, parentId) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        const recordValue = await fetchNestedDefaultValues({
            screenId: this.screenId,
            elementId: this.elementId,
            level,
        });
        return splitValueToMergedValue(value.addRecord({
            recordData: recordValue?.nestedDefaults ?? {},
            level,
            parentId,
        }));
    }
    /** Add a single row to the nested grid and in its dataset */
    addRecord(recordValue, level, parentId) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        return splitValueToMergedValue(value.addRecord({
            recordData: recordValue,
            level,
            parentId,
        }));
    }
    /** Remove a single row from the nested grid and its dataset */
    removeRecord(recordId, level) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        const rowLevel = level === 0 ? undefined : level;
        value.removeRecord({ recordId, ...(rowLevel && { level: rowLevel }) });
    }
    /** Add or update row in the nested grid depending of the existence of the ID field */
    addOrUpdateRecordValue(recordData, level, parentId) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        return splitValueToMergedValue(value.addOrUpdateRecordValue({
            recordData,
            level,
            parentId,
        }));
    }
    /** Add or update row in the nested grid (depending of the existence of its "_id" field and a given "level") */
    getRecordByFieldValue(fieldName, fieldValue, level) {
        const value = this.ensureNestedGridHasValue();
        this.validateLevel(level);
        return value.getRawRecordByFieldValue({ fieldName: String(fieldName), fieldValue, level });
    }
    /** Gets all records that have been added/updated/removed. */
    getChangedRecords() {
        const value = this.ensureNestedGridHasValue();
        return value.getChangedRecordsAsTree();
    }
    async refreshRecord(recordId, level, skipUpdate = false) {
        const value = this.ensureNestedGridHasValue();
        return value.refreshRecord({ recordId, level, skipUpdate });
    }
    /**
     * Fetches data for children record based on parent record id and child level. Default limit is 100. If the
     * openLevel attribute is set to true, the parent record is unfolded on the user interface and the children are
     * displayed.
     */
    async loadChildRecords({ childLevel, parentRecordId, limit = 100, filter, openLevel = false, }) {
        const value = this.ensureNestedGridHasValue();
        const maxLevel = this.getUiComponentProperty('levels').length - 1;
        const parentLevel = childLevel - 1;
        if (childLevel < 1 || childLevel > maxLevel) {
            throw new Error(`Invalid level '${childLevel}' received. Please specify a value between 1 and ${maxLevel}.`);
        }
        const parentRecord = value.getRawRecord({ id: parentRecordId, level: parentLevel });
        if (!parentRecord) {
            throw new Error(`Invalid record: could not find a record with ID ${parentRecordId} on level ${parentLevel}.`);
        }
        const parentLevelProps = this.getUiComponentProperty('levels')[parentLevel];
        const levelProps = this.getUiComponentProperty('levels')[childLevel];
        const first = limit > 0 && limit <= 500 ? limit : 500;
        const queryArguments = { first };
        if (filter) {
            queryArguments.filter = JSON.stringify(filter);
        }
        const data = (await value.fetchNestedGrid({
            level: childLevel,
            queryArguments,
            rootNode: String(parentLevelProps.node),
            selectedRecordId: parentRecordId,
            levelProps: levelProps,
            childProperty: String(parentLevelProps.childProperty),
        }));
        if (openLevel) {
            const recordPath = value.getIdPathToNestedRecord(parentRecordId, parentLevel);
            getStore().dispatch({
                type: ActionType.OpenNestedGridLevel,
                value: {
                    elementId: this.elementId,
                    screenId: this.screenId,
                    recordPath,
                },
            });
        }
        return data;
    }
    redraw() {
        return this._redraw();
    }
}
__decorate([
    ControlObjectProperty()
    /** Number of lines displayed by default in the nested grid (defaults to 20) */
], NestedGridControlObject.prototype, "pageSize", void 0);
__decorate([
    ControlObjectProperty()
    /** Whether the rows of this nested grid can be selected or not */
], NestedGridControlObject.prototype, "canSelect", void 0);
__decorate([
    ControlObjectProperty()
    /** Whether the rows of this nested grid can be filtered or not */
], NestedGridControlObject.prototype, "canFilter", void 0);
__decorate([
    ControlObjectProperty()
    /** Whether the nested grid should check for child records*/
], NestedGridControlObject.prototype, "checkForChildRecords", void 0);
//# sourceMappingURL=nested-grid-control-object.js.map