import { deepMerge } from '@sage/xtrem-shared';
import { get, isEqual, isNil, set } from 'lodash';
import * as React from 'react';
import * as xtremRedux from '../../redux';
import { runAndDispatchFieldValidation } from '../../service/dispatch-service';
import { getScreenElement } from '../../service/screen-base-definition';
import { ContextType } from '../../types';
import { handleChange } from '../../utils/abstract-fields-utils';
import { getComponentClass, getDataTestIdAttribute } from '../../utils/dom';
import { triggerFieldEvent, triggerNestedFieldEvent } from '../../utils/events';
import { convertDeepBindToPathNotNull } from '../../utils/nested-field-utils';
import { resolveByValue } from '../../utils/resolve-value-utils';
import { splitValueToMergedValue } from '../../utils/transformers';
import { getFieldTitle, isFieldDisabled, isFieldReadOnly, onFocusHandler } from './carbon-helpers';
export class FieldBaseComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.componentRef = React.createRef();
        /**
         * 'data-label' is used for alignment purposes only
         *
         * @private
         * @memberof FieldBaseComponent
         */
        this.getDataLabel = ({ contextType, title, }) => {
            if (contextType === ContextType.navigationPanel || (!this.props.fieldProperties.isTitleHidden && title)) {
                return { 'data-label': title ?? '' };
            }
            return undefined;
        };
        this.getComponentClass = (specificClassNames, rowValue) => {
            return getComponentClass(
            // TS 5.2 replaced FieldValue by any
            // this.props as unknown as BaseEditableComponentProperties<FieldUiProperties, FieldValue>,
            this.props, specificClassNames, rowValue);
        };
        this.getBaseAttributesDivWrapper = (componentName, specificClassNames, contextType, rowValue, isNested = false) => {
            const title = this.getTitle();
            return {
                'data-testid': getDataTestIdAttribute(componentName, title, this.props.elementId),
                className: this.getComponentClass(specificClassNames, rowValue),
                ...this.getDataLabel({ contextType, title }),
                ...(isNested && { 'data-nested': 'true' }),
            };
        };
        this.getValue = () => {
            const map = this.props.fieldProperties.map;
            if (map) {
                return resolveByValue({
                    screenId: this.props.screenId,
                    fieldValue: this.props.value,
                    rowValue: this.props.handlersArguments?.rowValue
                        ? splitValueToMergedValue(this.props.handlersArguments?.rowValue)
                        : this.props.handlersArguments?.rowValue,
                    skipHexFormat: true,
                    propertyValue: map,
                });
            }
            return this.props.value;
        };
        this.getClickHandler = () => () => {
            const nestedFieldsAdditionalProperties = this.props;
            if (nestedFieldsAdditionalProperties.isNested) {
                triggerNestedFieldEvent(this.props.screenId, nestedFieldsAdditionalProperties.parentElementId || this.props.elementId, this.props.fieldProperties, 'onClick', ...nestedFieldsAdditionalProperties.handlersArguments.onClick);
            }
            else {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onClick');
            }
        };
        this.triggerChangeListener = (newValue) => {
            const nestedFieldsAdditionalProperties = this.props;
            if (nestedFieldsAdditionalProperties.isNested) {
                // Merge rowValue with new value from change event to ensure latest changes are sent to listeners
                const rowValue = newValue !== undefined
                    ? deepMerge(nestedFieldsAdditionalProperties.handlersArguments.rowValue, set({}, convertDeepBindToPathNotNull(nestedFieldsAdditionalProperties.columnDefinition?.properties?.bind ||
                        this.props.fieldProperties?.bind ||
                        this.props.elementId), newValue))
                    : nestedFieldsAdditionalProperties.handlersArguments?.rowValue;
                triggerNestedFieldEvent(this.props.screenId, nestedFieldsAdditionalProperties.parentElementId || this.props.elementId, this.props.fieldProperties, 'onChange', rowValue?._id, rowValue);
            }
            else {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onChange');
            }
        };
    }
    shouldComponentUpdate(nextProps, nextState) {
        if (this.constructor.name === 'CarbonWrapper') {
            return true;
        }
        const isMandatory = this.props.fieldProperties.isMandatory ?? false;
        const validationErrors = this.props
            .validationErrors ?? false;
        if (!isEqual(this.state, nextState)) {
            return true;
        }
        if (!isEqual(this.props.value, nextProps.value)) {
            return true;
        }
        if (!isEqual(this.props.fieldProperties, nextProps.fieldProperties)) {
            return true;
        }
        if (this.props.isInFocus !== nextProps.isInFocus) {
            return true;
        }
        if (this.props.isParentReadOnly !== nextProps.isParentReadOnly) {
            return true;
        }
        if (!isEqual(this.props.recordContext, nextProps.recordContext)) {
            return true;
        }
        // Trigger updates on detail-panel-component when validation is updated:
        if (isMandatory && !this.props.value) {
            return true;
        }
        if (!isEqual(validationErrors, nextProps
            .validationErrors)) {
            return true;
        }
        return false;
    }
    componentDidUpdate(prevProps) {
        // TODO: This can be probably replaced by Carbon 12's autofocus
        const previousProps = prevProps;
        const currentProps = this.props;
        // TODO: Sometimes the 'CarbonWrapper' triggers the update instead of the actual component (i.e. reference)
        const componentRef = this.constructor.name === 'CarbonWrapper'
            ? this.props?.componentRef?.current
            : this.componentRef.current;
        if (!previousProps.isInFocus && currentProps.isInFocus && componentRef) {
            const elementInFocus = document.activeElement;
            const input = this.getFocusableElement(componentRef);
            if (input && elementInFocus !== input) {
                input.focus();
                onFocusHandler(currentProps);
            }
        }
    }
    /**
     * @deprecated Use handleChange from abstract-fields-utils instead of this.
     */
    handleChange(elementId, value, // TODO Type this properly
    setFieldValue, validate, onChange) {
        handleChange(elementId, value, setFieldValue, validate, onChange);
    }
    getResolvedProperty(key, skipHexFormat = true) {
        return resolveByValue({
            screenId: this.props.screenId,
            propertyValue: this.props.fieldProperties[key],
            skipHexFormat,
            rowValue: this.props.handlersArguments?.rowValue
                ? splitValueToMergedValue(this.props.handlersArguments?.rowValue)
                : null,
            fieldValue: null,
        });
    }
    getFocusableElement(fieldContainer) {
        return fieldContainer.querySelector('input');
    }
    getTitle() {
        return getFieldTitle(this.props.screenId, this.props.fieldProperties, this.props.handlersArguments?.rowValue);
    }
    isDisabled() {
        return (Boolean(this.props.isParentDisabled) ||
            isFieldDisabled(this.props.screenId, this.props.fieldProperties, this.props.value, this.props.handlersArguments?.rowValue));
    }
}
const shouldFieldBaseUpdate = (props, nextProps, parentShouldComponentUpdate) => {
    if (parentShouldComponentUpdate) {
        return true;
    }
    if (!isEqual(props.validationErrors, nextProps.validationErrors)) {
        return true;
    }
    if (props.isInFocus !== nextProps.isInFocus) {
        return true;
    }
    // INFO: Rerender all fields nested in sidebar's to ensure callback function values are
    //       updated when they depend on another field's value (XT-50323).
    if (props.contextType === 'sidebar') {
        return true;
    }
    return false;
};
export class ReadonlyFieldBaseComponent extends FieldBaseComponent {
    shouldComponentUpdate(nextProps, nextState) {
        const parentShouldComponentUpdate = super.shouldComponentUpdate?.(nextProps, nextState);
        if (shouldFieldBaseUpdate(this.props, nextProps, parentShouldComponentUpdate)) {
            return true;
        }
        return false;
    }
}
export class EditableFieldBaseComponent extends FieldBaseComponent {
    shouldComponentUpdate(nextProps, nextState) {
        const parentShouldComponentUpdate = super.shouldComponentUpdate?.(nextProps, nextState);
        if (shouldFieldBaseUpdate(this.props, nextProps, parentShouldComponentUpdate)) {
            return true;
        }
        return false;
    }
    isReadOnly() {
        return (Boolean(this.props.isParentReadOnly) ||
            isFieldReadOnly(this.props.screenId, this.props.fieldProperties, this.props.value, this.props.handlersArguments?.rowValue, this.props.contextType));
    }
}
export class ErrorableFieldBaseComponent extends FieldBaseComponent {
    shouldComponentUpdate(nextProps, nextState) {
        const parentShouldComponentUpdate = super.shouldComponentUpdate?.(nextProps, nextState);
        if (shouldFieldBaseUpdate(this.props, nextProps, parentShouldComponentUpdate)) {
            return true;
        }
        return false;
    }
    isReadOnly() {
        return isFieldReadOnly(this.props.screenId, this.props.fieldProperties, this.props.value, this.props.handlersArguments?.rowValue);
    }
}
export const mapReadonlyStateToProps = () => 
// eslint-disable-next-line func-names
function (state, props) {
    const fieldProperties = state.screenDefinitions[props.screenId].metadata.uiComponentProperties[props.elementId];
    const screenElement = getScreenElement(state.screenDefinitions[props.screenId]);
    const fieldValue = state.screenDefinitions[props.screenId].values[props.elementId];
    const pageNode = state.screenDefinitions[props.screenId].metadata.uiComponentProperties[props.screenId].node;
    const componentProperties = {
        // We always normalize undefined to null here
        value: isNil(fieldValue) ? null : fieldValue,
        browser: state.browser,
        nodeTypes: state.nodeTypes,
        pageNode,
        fieldProperties,
        locale: state.applicationContext?.locale || 'base',
        isInFocus: !!state.focusPosition &&
            state.focusPosition.elementId === props.elementId &&
            state.focusPosition.screenId === props.screenId,
        onFocus: xtremRedux.actions.actionStub,
    };
    // TODO Define a Trait for headerActions
    /* eslint-disable */
    const headerActions = get(fieldProperties, 'headerActions');
    if (fieldProperties && typeof headerActions === 'function') {
        set(componentProperties.fieldProperties, 'headerActionsMap', headerActions.apply(screenElement));
    }
    /* eslint-enable */
    return componentProperties;
};
export const mapStateToProps = () => 
// eslint-disable-next-line func-names
function (state, props) {
    return {
        ...mapReadonlyStateToProps()(state, props),
        validationErrors: state.screenDefinitions[props.screenId].errors[props.elementId],
        setFieldValue: xtremRedux.actions.actionStub,
        removeNonNestedErrors: xtremRedux.actions.actionStub,
        validate: xtremRedux.actions.actionStub,
        setFieldProperties: xtremRedux.actions.actionStub,
    };
};
export const mapDispatchToProps = (callback) => 
// eslint-disable-next-line func-names
function (dispatch, props) {
    let dispatchToProps = {
        setFieldValue: (elementId, value) => dispatch(xtremRedux.actions.setFieldValue(props.screenId, elementId, value, true)),
        onFocus: (row, nestedField) => {
            dispatch(xtremRedux.actions.setFocusPosition(props.screenId, props.elementId, row, nestedField));
        },
        validate: (elementId, value) => runAndDispatchFieldValidation(props.screenId, elementId, value),
        removeNonNestedErrors: (elementId) => {
            dispatch(xtremRedux.actions.removeNonNestedErrors(props.screenId, elementId));
        },
        setFieldProperties: (elementId, value) => {
            dispatch(xtremRedux.actions.setFieldProperties(props.screenId, elementId, value));
        },
    };
    if (callback) {
        dispatchToProps = {
            ...dispatchToProps,
            ...callback(dispatch, props),
        };
    }
    return dispatchToProps;
};
//# sourceMappingURL=field-base-component.js.map