import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { getGraphQLFilter } from '../../../service/filter-service';
import { localize } from '../../../service/i18n-service';
import { showToast } from '../../../service/toast-service';
import { refreshField } from '../../../service/field-state-service';
import { getPropertyScalarType } from '../../../utils/abstract-fields-utils';
import { schemaTypeNameFromNodeName } from '../../../utils/transformers';
import { FieldKey } from '../../types';
import { FiltersComponent, FiltersLabels } from '../../ui/filter/filters-component';
import { HelperText } from '../carbon-utility-components';
import { mapReadonlyStateToProps, ReadonlyFieldBaseComponent } from '../field-base-component';
import { FieldHeader } from '../field-header';
import { findDeepPropertyType } from '../../../utils/node-utils';
import { convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { ChartTypes } from '../../chart-types';
import { LineChartComponent } from './line/line-chart-component';
import { BarChartComponent } from './bar/bar-chart-component';
import { PieChartComponent } from './pie/pie-chart-component';
import { getChartOptions } from './chart-utils';
export class ChartComponent extends ReadonlyFieldBaseComponent {
    constructor(props) {
        super(props);
        this.onFilterChange = () => {
            const columns = this.getNestedFields();
            const filter = getGraphQLFilter(this.state.filters, columns);
            if (this.props.setFieldProperties) {
                this.props.setFieldProperties(this.props.elementId, {
                    ...this.props.fieldProperties,
                    activeUserFilter: filter,
                });
            }
            return refreshField({
                screenId: this.props.screenId,
                elementId: this.props.elementId,
                keepPageInfo: true,
            }).catch((e) => {
                showToast(e.message || e, { type: 'warning' });
            });
        };
        this.saveFilters = (filters) => {
            this.setState({ filters }, () => this.onFilterChange());
        };
        this.getFilterableFields = (node, nestedFields) => {
            return nestedFields.map(nestedField => {
                const bind = convertDeepBindToPathNotNull(nestedField.properties.bind);
                const fieldType = findDeepPropertyType(schemaTypeNameFromNodeName(node), nestedField.properties.bind, this.props.nodeTypes, true)?.type || FieldKey.Text;
                return {
                    type: nestedField.type,
                    properties: {
                        ...nestedField.properties,
                        bind,
                        title: this.getTitle(),
                        type: getPropertyScalarType(this.props.nodeTypes, fieldType, nestedField.type),
                    },
                };
            });
        };
        this.getNestedFields = () => {
            const nestedFields = this.props.fieldProperties.chart.series;
            const node = this.props.fieldProperties.node;
            return this.getFilterableFields(node, nestedFields);
        };
        this.state = {
            filters: [],
        };
    }
    renderChart() {
        const options = getChartOptions({
            properties: this.props.fieldProperties,
            collection: this.props.value,
            screenId: this.props.screenId,
            locale: this.props.locale,
        });
        switch (this.props.fieldProperties.chart.type) {
            case ChartTypes.Line:
                return React.createElement(LineChartComponent, { ...options });
            case ChartTypes.Bar:
                return React.createElement(BarChartComponent, { ...options });
            case ChartTypes.Pie:
                return React.createElement(PieChartComponent, { ...options });
            default:
                throw new Error(`Unsupported chart type: ${this.props.fieldProperties.chart.type}`);
        }
    }
    render() {
        return (React.createElement("div", { ...this.getBaseAttributesDivWrapper('chart', 'e-chart-field', this.props.contextType, this.props.handlersArguments?.rowValue, this.props.isNested), onClick: this.getClickHandler() },
            React.createElement(FieldHeader, { elementId: this.props.elementId, screenId: this.props.screenId, title: this.getTitle(), isDisabled: this.isDisabled(), isTitleHidden: this.props.fieldProperties.isTitleHidden }, this.props.fieldProperties.canFilter && this.props.fieldProperties.node && (React.createElement(FiltersComponent, { screenId: this.props.screenId, fields: this.getNestedFields(), handleSave: this.saveFilters, filters: this.state.filters, isDisabled: this.isDisabled() }))),
            this.props.fieldProperties.canFilter && (React.createElement(FiltersLabels, { screenId: this.props.screenId, fields: this.getNestedFields(), handleSave: this.saveFilters, filters: this.state.filters, isDisabled: this.isDisabled() })),
            this.props.value ? (this.renderChart()) : (React.createElement("p", null, localize('@sage/xtrem-ui/chart-component-no-data', 'No data found'))),
            this.props.fieldProperties.helperText && (React.createElement(HelperText, { helperText: this.props.fieldProperties.helperText }))));
    }
}
const mapDispatchToProps = (dispatch, props) => {
    return {
        setFieldProperties: (elementId, value) => {
            dispatch(xtremRedux.actions.setFieldProperties(props.screenId, elementId, value));
        },
    };
};
const mapStateToProps = (state, props) => {
    const componentProperties = mapReadonlyStateToProps()(state, props);
    return {
        ...props,
        ...componentProperties,
        nodeTypes: state.nodeTypes,
        setFieldProperties: xtremRedux.actions.actionStub,
    };
};
export const ConnectedChartComponent = connect(mapStateToProps, mapDispatchToProps)(ChartComponent);
export default ConnectedChartComponent;
//# sourceMappingURL=chart-component.js.map