import { objectKeys } from '@sage/xtrem-shared';
import { isNil } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { RenderingRouter } from '../../../render/rendering-router';
import { subscribe, unsubscribe } from '../../../service/shortcut-service';
import { getSectionValidationMessage } from '../../../service/validation-service';
import { ContextType } from '../../../types';
import { scrollWithAnimationTo } from '../../../utils/dom';
import { getHeaderSection, getVisibleSections } from '../../../utils/layout-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getNavigationPanelState, getPageDefinitionFromState, getPagePropertiesFromState, } from '../../../utils/state-utils';
import { getFieldTitle } from '../../field/carbon-helpers';
import { XtremTabs } from '../../ui/tabs/xtrem-tabs';
import { ConnectedPageHeaderCardComponent } from './page-header-card';
import IconButton from 'carbon-react/esm/components/icon-button';
import Icon from 'carbon-react/esm/components/icon';
import { getRouter } from '../../../service/router';
import { localize } from '../../../service/i18n-service';
import * as tokens from '@sage/design-tokens/js/base/common';
import { getIdFieldValue } from '../../../utils/id-field-utils';
import { PageTitle } from '../../ui/page-title';
import FieldWrapper from '../../../render/field-wrapper';
import { getScreenElement } from '../../../service/screen-base-definition';
import HeaderNavigationArrows from './header-navigation-arrows';
import { getHeaderTitles } from '../../../utils/page-utils';
import { Page360Switch } from '../page-360/page-360-switch';
import { HEADER_IMAGE, HEADER_TITLE, NEW_PAGE } from '../../../utils/constants';
import { Portrait } from '../../ui/portrait-component';
import { getImageUrlFromValue } from '../../field/image/image-utils';
import { navigationPanelId } from '../navigation-panel/navigation-panel-types';
import { WizardSteps } from './wizard-steps';
import BackArrowButton from './header-back-arrow';
import { PageHeaderInsightsButton } from '../../ui/insights-button';
import { HeaderLineActionIconButtonGroup } from './header-line-action-icon-button-group';
import { HeaderSectionToggleButton } from './header-section-toggle-button';
import { GridRow } from '@sage/xtrem-ui-components';
export class XtremHeader extends React.Component {
    constructor() {
        super(...arguments);
        this.tabDictionary = {};
        this.hotkeySubscriptions = [];
        this.parentRef = React.createRef();
        this.getActiveSectionIndex = () => this.props.sections.findIndex(s => this.props.activeSection === s.$containerId);
        this.setActiveSectionByIndex = (index) => {
            if (index < 0 || index >= this.props.sections.length) {
                return;
            }
            this.setActiveSection(this.props.sections[index].$containerId);
        };
        this.setActiveSection = (containerId) => {
            this.props.setActiveSection(containerId);
            if (this.props.pageBodyRef.current && this.props.pageMode !== 'tabs' && this.props.pageMode !== 'wizard') {
                scrollWithAnimationTo(this.props.pageBodyRef.current.querySelector(`#${containerId}`), this.props.pageBodyRef.current, 80);
            }
        };
        this.hasTabs = () => this.getVisibleTabs().length > 1;
        this.getClasses = () => {
            const classes = ['e-header'];
            if (this.props.contextType) {
                classes.push(`e-header-context-${this.props.contextType}`);
            }
            if (!this.hasTabs()) {
                classes.push('e-no-tabs');
            }
            return classes.join(' ');
        };
        this.getVisibleTabs = (skipHeaderSection = false) => {
            if (this.props.areNavigationTabsHidden) {
                return [];
            }
            return this.props.sections
                .filter(item => getFieldTitle(this.props.screenId, this.props.sectionsProperties[item.$containerId], null))
                .filter(item => !resolveByValue({
                screenId: this.props.screenId,
                skipHexFormat: true,
                propertyValue: this.props.sectionsProperties[item.$containerId].isHidden,
                rowValue: null,
            }))
                .filter(item => skipHeaderSection && this.props.headerSection?.$containerId
                ? item.$containerId !== this.props.headerSection.$containerId
                : true)
                .map(c => c.$containerId);
        };
        this.onRecordClose = () => {
            this.props.navigationPanelValue?.restoreFilterSnapshot();
            getRouter(this.props.screenId).closeRecord();
        };
        this.isDialogContext = () => {
            return this.props.contextType === ContextType.dialog;
        };
        this.hasHeaderSection = () => {
            return !isNil(this.props.headerSection);
        };
        this.filterByHeaderSection = (sections) => {
            return this.hasHeaderSection()
                ? sections.filter(({ id }) => {
                    return id !== this.props.headerSection?.$containerId;
                })
                : sections;
        };
        this.renderTabs = () => {
            if (!this.hasTabs()) {
                return null;
            }
            const sections = this.getVisibleTabs().map((id) => ({
                id,
                title: getFieldTitle(this.props.screenId, this.props.sectionsProperties[id], null),
                validationMessage: this.props.sectionValidationErrors[id],
                indicatorContent: this.props.sectionsProperties[id].indicatorContent,
            }));
            if (this.props.pageMode === 'wizard') {
                return (React.createElement("div", { className: "e-header-wizard-steps-container" },
                    React.createElement(WizardSteps, { screenId: this.props.screenId, sections: sections, selectedSection: this.props.activeSection })));
            }
            const skipHeaderSection = this.isDialogContext() || this.props.isMobileScreen;
            return (React.createElement("div", { className: "e-header-nav", "data-testid": "e-header-nav-tabs" },
                React.createElement(XtremTabs, { selectedTabId: this.props.activeSection || this.getVisibleTabs(skipHeaderSection)[0], onTabChange: this.setActiveSection, tabs: skipHeaderSection ? this.filterByHeaderSection(sections) : sections })));
        };
        this.renderCloseIcon = () => {
            if (this.props.hasNavigationPanel) {
                return (React.createElement("div", { className: "e-header-close-icon-col" },
                    React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-ui/close-record', 'Close record'), "data-testid": "e-page-close-button", "data-pendoid": "closeRecord", onClick: this.onRecordClose },
                        React.createElement(Icon, { type: "close", color: tokens.colorsUtilityMajor400, tooltipMessage: localize('@sage/xtrem-ui/close-record', 'Close record') }))));
            }
            return null;
        };
        this.renderTitleImage = (title) => (React.createElement("div", { className: "e-header-image-container" },
            this.props.headerImage && (React.createElement("div", { className: "e-header-image", style: { backgroundImage: `url('${getImageUrlFromValue(this.props.headerImage)}')` }, "aria-label": title, role: "img" })),
            !this.props.headerImage && this.props.headerTitle && (React.createElement(Portrait, { size: "M", placeholderMode: "Initials", placeholderValue: this.props.headerTitle, icon: "image" })),
            !this.props.headerImage && !this.props.headerTitle && this.props.title && (React.createElement(Portrait, { size: "M", placeholderMode: "Initials", placeholderValue: this.props.idFieldValue || this.props.title, icon: "image" })),
            !this.props.headerImage && !this.props.headerTitle && !this.props.title && (React.createElement("img", { className: "e-header-image", src: "/images/no-available-image.svg", alt: localize('@sage/xtrem-ui/no-available-image', 'No available image') }))));
    }
    componentDidMount() {
        this.setIndicatorPosition();
        this.hotkeySubscriptions.push(subscribe(['escape', 'arrowup'], () => {
            const index = this.getActiveSectionIndex();
            this.setActiveSectionByIndex(index - 1);
        }));
        this.hotkeySubscriptions.push(subscribe(['escape', 'arrowdown'], () => {
            const index = this.getActiveSectionIndex();
            this.setActiveSectionByIndex(index + 1);
        }));
        // We subscribe to each button in order to prevent issues from overlapping screens
        for (let i = 1; i < 10; i += 1) {
            this.hotkeySubscriptions.push(subscribe(['escape', i.toString()], () => {
                this.setActiveSectionByIndex(i - 1);
            }));
        }
    }
    componentDidUpdate(prevProps) {
        this.setIndicatorPosition();
        if (this.props.isNavigationPanelOpened !== prevProps.isNavigationPanelOpened) {
            this.parentRef.current?.focus();
        }
        const visibleSections = this.getVisibleTabs();
        if (this.props.activeSection &&
            this.props.pageMode !== 'wizard' &&
            !this.props.areNavigationTabsHidden &&
            this.getVisibleTabs().indexOf(this.props.activeSection) === -1) {
            this.props.setActiveSection(visibleSections[0]);
        }
        if (prevProps.isHeaderSectionClosed !== this.props.isHeaderSectionClosed && this.props.pageBodyRef.current) {
            /**
             * Even if the parent element gets resized by hiding the top section,
             * the resize event is only triggered automatically if the window is resized
             *  */
            const event = new UIEvent('resize', { view: window, bubbles: true, cancelable: false, detail: 0 });
            this.props.pageBodyRef.current.dispatchEvent(event);
        }
    }
    componentWillUnmount() {
        this.hotkeySubscriptions.forEach(unsubscribe);
    }
    setIndicatorPosition() {
        if (this.activeIndicator &&
            this.tabContainer &&
            this.props.activeSection &&
            this.tabDictionary[this.props.activeSection]) {
            const activeTabRect = this.tabDictionary[this.props.activeSection].getBoundingClientRect();
            const containerX = this.tabContainer.getBoundingClientRect().x || 0;
            this.activeIndicator.style.left = `${(activeTabRect.x || 0) - containerX}px`;
            this.activeIndicator.style.width = `${activeTabRect.width - 16}px`;
        }
    }
    render() {
        const { subtitle, title } = getHeaderTitles({
            title: this.props.title,
            subtitle: this.props.subtitle,
            idFieldValue: this.props.idFieldValue,
            isRecordCreationPage: this.props.isRecordCreationPage,
            objectTypePlural: this.props.objectTypePlural,
            objectTypeSingular: this.props.objectTypeSingular,
        });
        const headerTitleColumnClasses = ['e-header-title-col'];
        if (this.props.insightCount && this.props.insightCount > 0) {
            headerTitleColumnClasses.push('e-header-title-col-insight');
        }
        return (React.createElement(React.Fragment, null,
            React.createElement("header", { className: `e-header-title-row
                        ${this.props.hasToggleButton ? 'e-with-toggle-button' : ''}
                        ${this.props.isDetailPanelOpen ? 'e-detail-panel-open' : ''}`, "data-testid": "e-header" },
                this.props.contextType === ContextType.page && !this.props.isTitleHidden && (React.createElement("div", { className: headerTitleColumnClasses.join(' ') },
                    React.createElement("div", { className: "e-header-title-container" },
                        this.props.shouldDisplayBackArrow && (React.createElement(BackArrowButton, { screenId: this.props.screenId })),
                        this.props.shouldDisplayNavigationArrows && (React.createElement(HeaderNavigationArrows, { screenId: this.props.screenId })),
                        this.props.isHeaderImageNeeded && this.renderTitleImage(title),
                        React.createElement("div", null,
                            React.createElement(PageTitle, { title: title }),
                            !!subtitle && React.createElement("p", { className: "e-header-subtitle" }, subtitle)),
                        this.props.headerField && (React.createElement("div", { className: "e-header-field" },
                            React.createElement(GridRow, { columns: 2, margin: 0, gutter: 0 },
                                React.createElement(FieldWrapper, { item: { $bind: this.props.headerField.id }, screenId: this.props.screenId, contextType: ContextType.pageHeader })))),
                        this.props.headerLabelField && (React.createElement("div", { className: "e-header-title-label" },
                            React.createElement(FieldWrapper, { item: { $bind: this.props.headerLabelField.id }, screenId: this.props.screenId, contextType: ContextType.pageHeader })))))),
                this.props.contextType !== ContextType.dialog && (React.createElement(PageHeaderInsightsButton, { screenId: this.props.screenId })),
                this.props.contextType !== ContextType.dialog && (React.createElement(HeaderLineActionIconButtonGroup, { screenId: this.props.screenId, contextType: this.props.contextType })),
                this.props.contextType !== ContextType.dialog && React.createElement(Page360Switch, { screenId: this.props.screenId }),
                (this.props.hasNavigationPanel || this.props.headerSection) && (React.createElement("span", { className: "e-header-line-icon-group e-header-line-icon-group-close" },
                    this.props.contextType !== ContextType.dialog && (React.createElement(HeaderSectionToggleButton, { screenId: this.props.screenId })),
                    this.renderCloseIcon())),
                this.props.headerLineBlock && (React.createElement("div", { className: "e-header-line-block" },
                    React.createElement(RenderingRouter, { screenId: this.props.screenId, item: this.props.headerLineBlock, contextType: ContextType.header, availableColumns: 4 })))),
            React.createElement("div", { className: this.getClasses(), "data-testid": "e-header" },
                React.createElement(ConnectedPageHeaderCardComponent, { screenId: this.props.screenId }),
                this.props.headerSection && !this.props.is360ViewOn && !this.props.isHeaderSectionClosed && (React.createElement("div", { className: "e-header-section", "data-testid": "e-header-section" },
                    React.createElement(RenderingRouter, { screenId: this.props.screenId, item: this.props.headerSection, availableColumns: this.props.availableColumns }))),
                !this.props.areNavigationTabsHidden && !this.props.is360ViewOn && this.renderTabs())));
    }
}
const mapStateToProps = (state, props) => {
    const pageDefinition = getPageDefinitionFromState(props.screenId, state);
    const sectionIds = pageDefinition.metadata.layout.$items
        .filter(section => !!section.$containerId)
        .map(section => section.$containerId);
    const sectionsProperties = objectKeys(pageDefinition.metadata.uiComponentProperties)
        .filter(property => sectionIds.indexOf(property) > -1)
        .reduce((properties, nextKey) => ({
        ...properties,
        [nextKey]: pageDefinition.metadata.uiComponentProperties[nextKey],
    }), {});
    const navigationPanelState = getNavigationPanelState(props.screenId, state);
    const pageProperties = getPagePropertiesFromState(props.screenId, state);
    const navPanelDefinition = pageDefinition.metadata.uiComponentProperties[navigationPanelId];
    const isHeaderImageNeeded = !!navPanelDefinition?.mobileCard?.image;
    const sectionValidationErrors = sectionIds.reduce((prevValue, sectionId) => {
        return { ...prevValue, [sectionId]: getSectionValidationMessage(props.screenId, sectionId, state) };
    }, {});
    const locale = state.applicationContext?.locale || 'en-US';
    const idFieldValue = getIdFieldValue(pageDefinition, locale);
    const headerLabelField = pageProperties.headerLabel?.apply(getScreenElement(pageDefinition));
    const headerField = pageProperties.headerField?.apply(getScreenElement(pageDefinition));
    const isHeaderFieldHidden = !headerField || headerField.isHidden;
    const headerSection = getHeaderSection(pageDefinition, false);
    const has360View = resolveByValue({
        skipHexFormat: true,
        screenId: props.screenId,
        propertyValue: pageProperties.has360View,
        rowValue: null,
        fieldValue: null,
    }) || false;
    const headerImage = state.screenDefinitions[props.screenId]?.values[HEADER_IMAGE] ||
        state.screenDefinitions[props.screenId]?.values.image;
    const headerTitle = state.screenDefinitions[props.screenId]?.values[HEADER_TITLE] ||
        state.screenDefinitions[props.screenId]?.values.name;
    const getLastValue = (obj) => {
        if (typeof obj !== 'object' || obj === null)
            return obj;
        const values = Object.values(obj);
        if (values.length === 0)
            return '';
        return getLastValue(values[values.length - 1]);
    };
    const finalHeaderTitleValue = typeof headerTitle === 'object' && headerTitle !== null ? getLastValue(headerTitle) : headerTitle || '';
    const finalHeaderImageValue = typeof headerImage === 'object' && headerImage !== null
        ? getLastValue(headerImage)
        : headerImage?.value || undefined;
    const history = state.navigation.history;
    return {
        ...props,
        has360View,
        headerField: isHeaderFieldHidden ? undefined : headerField,
        headerImage: finalHeaderImageValue,
        headerLabelField,
        headerSection: headerSection?.layout || null,
        headerTitle: finalHeaderTitleValue,
        history,
        idFieldValue,
        insightCount: pageDefinition.insightCount,
        is360ViewOn: !!pageDefinition.is360ViewOn,
        isHeaderImageNeeded,
        isHeaderSectionClosed: pageDefinition.isHeaderSectionClosed || false,
        isMobileScreen: state.browser.lessThan.m,
        isNavigationPanelHeaderHidden: !!navigationPanelState?.isHeaderHidden,
        isNavigationPanelOpened: !!navigationPanelState?.isOpened,
        isNewPage: pageDefinition.queryParameters?._id === NEW_PAGE,
        isTitleHidden: pageProperties.isTitleHidden,
        navigationPanelValue: pageDefinition.navigationPanel?.value,
        objectTypePlural: pageProperties.objectTypePlural,
        objectTypeSingular: pageProperties.objectTypeSingular,
        openSectionInDialog: xtremRedux.actions.actionStub,
        pageMode: pageProperties?.mode || 'default',
        sections: getVisibleSections(pageDefinition, state.activeDialogs, state.browser.lessThan.m),
        sectionsProperties,
        sectionValidationErrors,
        shouldDisplayBackArrow: history.length > 0,
        shouldDisplayNavigationArrows: props.hasNavigationPanel && !state.browser.is.xs && !navigationPanelState?.isOpened,
        subtitle: pageProperties.subtitle,
        title: (pageProperties && getFieldTitle(props.screenId, pageProperties, null)) || '',
    };
};
export const ConnectedXtremHeader = connect(mapStateToProps)(XtremHeader);
//# sourceMappingURL=xtrem-header.js.map