import * as React from 'react';
import { connect } from 'react-redux';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import { getPageDefinitionFromState, getPagePropertiesFromState } from '../../../utils/state-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { camelCase } from 'lodash';
import { triggerFieldEvent } from '../../../utils/events';
import * as tokens from '@sage/design-tokens/js/base/common';
import { getElementAccessStatus } from '../../../utils/access-utils';
function HeaderQuickAction({ icon, isDisabled, title, isHidden, screenId, id, onTelemetryEvent, }) {
    const onClick = () => {
        onTelemetryEvent?.(`headerQuickActionItemTriggered-${id}`, { screenId, elementId: id });
        triggerFieldEvent(screenId, id, 'onClick');
    };
    if (isHidden) {
        return null;
    }
    return (React.createElement("span", { className: "e-header-quick-action" },
        React.createElement(IconButton, { onClick: onClick, disabled: isDisabled, "aria-label": title, "data-testid": `e-header-quick-action-label-${camelCase(title)}`, "data-pendoid": `e-header-quick-action-${id}` },
            React.createElement(Icon, { type: icon, tooltipMessage: title, color: tokens.colorsUtilityMajor400 }))));
}
const ConnectedHeaderQuickAction = connect((state, { screenId, id }) => {
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const pageProperties = getPagePropertiesFromState(screenId, state);
    const actionProperties = pageDefinition.metadata.uiComponentProperties[id];
    const accessRule = getElementAccessStatus({
        accessBindings: pageDefinition.accessBindings,
        bind: id,
        elementProperties: actionProperties,
        contextNode: pageProperties?.node,
        nodeTypes: state.nodeTypes,
        dataTypes: state.dataTypes,
    });
    const isDisabled = resolveByValue({
        screenId,
        propertyValue: actionProperties.isDisabled,
        skipHexFormat: true,
        rowValue: null,
        fieldValue: null,
    });
    const isHidden = resolveByValue({
        screenId,
        propertyValue: actionProperties.isHidden,
        skipHexFormat: true,
        rowValue: null,
        fieldValue: null,
    }) ||
        (actionProperties.access && accessRule !== 'authorized');
    const title = resolveByValue({
        screenId,
        propertyValue: actionProperties.title,
        skipHexFormat: true,
        rowValue: null,
        fieldValue: null,
    });
    const icon = actionProperties.icon;
    if (!icon) {
        throw new Error(`Quick actions must have a valid icon value. Screen Id: ${screenId}, Action Id: ${id}`);
    }
    return {
        id,
        icon,
        isDisabled,
        isHidden,
        screenId,
        title,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
    };
})(HeaderQuickAction);
export function HeaderQuickActions({ screenId, headerQuickActions, }) {
    if (!headerQuickActions) {
        return null;
    }
    return (React.createElement("div", { className: "e-header-quick-action-container" }, headerQuickActions.map(a => (React.createElement(ConnectedHeaderQuickAction, { key: a.id, screenId: screenId, id: a.id })))));
}
export const ConnectedHeaderQuickActions = connect((state, props) => {
    const pageProperties = getPagePropertiesFromState(props.screenId, state);
    const rawQuickActions = props.actions ??
        resolveByValue({
            skipHexFormat: true,
            screenId: props.screenId,
            propertyValue: pageProperties.headerQuickActions,
            rowValue: null,
            fieldValue: null,
        });
    return {
        ...props,
        headerQuickActions: rawQuickActions,
    };
})(HeaderQuickActions);
//# sourceMappingURL=header-quick-actions.js.map