import * as React from 'react';
import { RecordActionType } from '../service/collection-data-types';
import { objectKeys } from '@sage/xtrem-shared';
import { cleanMetadataFromRecord } from '../utils/transformers';
import { isEqual } from 'lodash';
import { useDeepEqualSelector } from '../utils/hooks/use-deep-equal-selector';
class BaseCollectionValueSubscriptionHOC extends React.Component {
    componentDidMount() {
        this.subscribeToChanges();
    }
    componentDidUpdate(prevProps) {
        if (prevProps.value !== this.props.value) {
            this.subscribeToChanges();
        }
    }
    componentWillUnmount() {
        if (this.validityChangeSubscription) {
            this.validityChangeSubscription();
        }
        if (this.valueChangeSubscription) {
            this.valueChangeSubscription();
        }
    }
    subscribeToChanges() {
        this.valueChangeSubscription = this.props.value?.subscribeForValueChanges((type, rowValue) => {
            if (this.shouldUpdateOnChange(type, rowValue)) {
                this.forceUpdate();
            }
        }, this.props.isUncommitted);
        this.validityChangeSubscription = this.props.value?.subscribeForValidityChanges(({ globalValidationState, recordValidationState, recordId, recordLevel }) => {
            if (this.shouldUpdateOnValidityChange(globalValidationState, recordValidationState, recordId, recordLevel)) {
                this.forceUpdate();
            }
        }, this.props.isUncommitted);
    }
}
class CollectionValueSubscription extends BaseCollectionValueSubscriptionHOC {
    shouldUpdateOnChange(type) {
        return type === RecordActionType.ADDED || type === RecordActionType.REMOVED;
    }
    shouldUpdateOnValidityChange() {
        return false;
    }
}
/**
 * If a CollectionValue `value` property is provided, the HOC subscribes to the add and remove events of collection value
 * items and re-renders the child component on change
 * @param WrappedComponent
 */
export const withCollectionValueSubscription = (WrappedComponent) => class extends CollectionValueSubscription {
    render() {
        return React.createElement(WrappedComponent, { ...this.props });
    }
};
class CollectionValueAndOrderSubscription extends CollectionValueSubscription {
    constructor(props) {
        super(props);
        this.order = props.value?.getData().map(d => d._id) || [];
    }
    shouldUpdateOnChange(type) {
        let shouldUpdate = super.shouldUpdateOnChange(type);
        if (!shouldUpdate && type === RecordActionType.MODIFIED) {
            const newOrder = this.props.value?.getData().map(d => d._id) || [];
            shouldUpdate = !isEqual(this.order, newOrder);
            this.order = newOrder;
            return shouldUpdate;
        }
        return shouldUpdate;
    }
    shouldUpdateOnValidityChange() {
        return false;
    }
}
/**
 * If a CollectionValue `value` property is provided, the HOC subscribes to the add and remove events of collection value
 * items and re-renders the child component on change
 * @param WrappedComponent
 */
export const withCollectionValueAndOrderSubscription = (WrappedComponent) => class extends CollectionValueAndOrderSubscription {
    render() {
        return React.createElement(WrappedComponent, { ...this.props });
    }
};
/**
 * If a CollectionValue `value` property and a valid `recordId` property are provided, the HOC subscribes to changes to
 * the item tracked by the record ID and re-renders the child component if the item changes.
 * @param WrappedComponent
 */
export const withCollectionValueItemSubscription = (WrappedComponent) => class CollectionValueItemSubscription extends BaseCollectionValueSubscriptionHOC {
    constructor() {
        super(...arguments);
        this.onChange = async (bind, value) => {
            if (this.props.value && this.props.recordId) {
                this.props.value.setCellValue({
                    recordId: this.props.recordId,
                    level: this.props.level,
                    isOrganicChange: true,
                    columnId: bind,
                    value,
                    isUncommitted: this.props.isUncommitted,
                    shouldFetchDefault: this.props.shouldFetchDefault,
                });
            }
            if (this.props.onChange) {
                await this.props.onChange(bind, value, this.props.recordId);
            }
        };
    }
    shouldUpdateOnValidityChange(globalValidationState, recordValidationState, id, level) {
        return id === this.props.recordId && (level ?? 0) === (this.props.level ?? 0);
    }
    shouldUpdateOnChange(type, rowValue) {
        return (!!rowValue._id &&
            rowValue._id === this.props.recordId &&
            (rowValue.__level ?? 0) === (this.props.level ?? 0));
    }
    getRecordValue() {
        if (this.props.value && this.props.recordId) {
            return this.props.value?.getRawRecord({
                id: this.props.recordId,
                level: this.props.level,
                cleanMetadata: false,
                isUncommitted: this.props.isUncommitted,
            });
        }
        return undefined;
    }
    render() {
        const value = this.getRecordValue();
        // Remap the errors into a validation array from the object structure of collection value
        const validationErrors = objectKeys(value?.__validationState || {}).reduce((prevValue, key) => {
            prevValue.push(value.__validationState[key]);
            return prevValue;
        }, []);
        return (React.createElement(WrappedComponent, { ...this.props, onChange: this.onChange, recordValue: cleanMetadataFromRecord(value), validationErrors: validationErrors, validate: () => Promise.resolve(undefined) }));
    }
};
export function useCollectionItemRecord({ elementId, screenId, recordId, level = 0, isUncommitted = false, }) {
    const [recordValue, setRecordValue] = React.useState(null);
    const collectionValue = useDeepEqualSelector(state => {
        return state.screenDefinitions[screenId].values[elementId];
    });
    const onRecordChange = React.useCallback((type, rowValue) => {
        const recordLevel = rowValue.__level ?? 0;
        if (rowValue._id !== recordId || recordLevel === level) {
            setRecordValue(null);
        }
        if (type === RecordActionType.ADDED || type === RecordActionType.MODIFIED) {
            setRecordValue(rowValue);
        }
    }, [level, recordId]);
    React.useEffect(() => {
        setRecordValue(collectionValue.getRawRecord({
            id: recordId,
            level,
            isUncommitted,
            cleanMetadata: false,
        }));
        return collectionValue.subscribeForValueChanges(onRecordChange, isUncommitted);
    }, [collectionValue, recordId, level, isUncommitted, onRecordChange]);
    return recordValue;
}
//# sourceMappingURL=connected-collection.js.map