import type { DirectInput, DirectInputLine, GraphApi, ReceiptMovementAddressed } from '@sage/wh-input-api';
import type { SerialGroup } from '@sage/wh-master-data-api';
import { getSelectedSiteDepositor } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import type { Product } from '@sage/wh-product-data-api';
import { getProductConfiguration } from '@sage/wh-product-data/lib/client-functions/get-product-configuration';
import { ProductConfiguration } from '@sage/wh-product-data/lib/interfaces/environnement';
import type { SerialNumber, SerialNumberBinding } from '@sage/wh-stock-data-api';
import { generateAggregateSerialNumberRequest } from '@sage/wh-stock-data/lib/client-functions/aggregators';
import { AggregateGroupSelector, Edges, ExtractEdges, extractEdges, Filter } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';

/**
 *  Defines the error messages that can be returned by the serial number management functions
 *  Except for adding a new serial number, all errors are blocking
 */
export enum SerialNumberManagementError {
    quantityIsMandatory = 'quantityIsMandatory',
    serialNumberAlreadySelected = 'serialNumberAlreadySelected',
    tooManySerialNumbers = 'tooManySerialNumbers',
    noSerialNumberFound = 'noSerialNumberFound',
    noSerialNumberForReturn = 'noSerialNumberForReturn',
    tooManySerialNumbersToReturn = 'tooManySerialNumbersToReturn',
    serialGroupRequired = 'serialGroupRequired',
    addNewSerialNumber = 'addNewSerialNumber',
}

/**
 *  Defines the mode of serial number movement
 */
export enum SerialNumberMovementMode {
    none = 'none',
    line = 'line',
    movement = 'movement',
}

interface DoubleOccurrence {
    double: number;
    occurrence: number;
}

interface DoubleOccurrenceResult extends DoubleOccurrence {
    isValid: boolean;
}

const maximumNumberOfSerialNumbers = 1000;

export class SerialNumberManagement {
    constructor(
        private _serialGroup: ui.fields.Reference,
        private _serialNumber: ui.fields.FilterSelect,
        private _serialNumbers: ui.fields.Table,
    ) {
        this.reinitialize();
    }

    /* ***************************************************************************************** */
    /*                                                                                           */
    /*                                 Internal objects                                          */
    /*                                                                                           */
    /* ***************************************************************************************** */

    private _siteCode?: string;

    private _depositorCode?: string;

    private _maxRecords = maximumNumberOfSerialNumbers;

    private _productConfiguration?: ProductConfiguration;

    private _receiptMovementAddressed?: ReceiptMovementAddressed;

    private _serialNumbersAvailable: ExtractEdges<SerialNumber>[] = [];

    private _numberSerialNumbersInLine = 0;

    private _numberSerialNumbersInMovement = 0;

    private _homogeneousQuantityInConsumptionUnit = 0;

    /* ***************************************************************************************** */
    /*                                                                                           */
    /* 🔵                               Getters and Setters                                     */
    /*                                                                                           */
    /* ***************************************************************************************** */

    /**
     *  returns the available quantity in consumption unit to create new serial numbers.
     *  The maximum authorized quantity available for entry depends on the movement or line mode.
     */
    get availableQuantityInConsumptionUnit(): number {
        const _quantityInConsumptionUnit = this.homogeneousQuantityInConsumptionUnit;

        // The quantity must have been initialized, there must be remaining quantity to cover
        if (!_quantityInConsumptionUnit) {
            return 0;
        }

        // The maximum authorized quantity available for entry depends on the movement or line mode.
        const _quantityInConsumptionUnitToCheck =
            this.movementMode === SerialNumberMovementMode.movement
                ? _quantityInConsumptionUnit
                : this.directInputLineNumberOfConsumptionUnitRemaining;

        // There must be available quantity to create new serial numbers
        return Math.max(_quantityInConsumptionUnitToCheck - this._serialNumbersAvailable.length, 0);
    }

    /**
     * returns the direct input linked to the receipt movement addressed
     */
    get directInput(): DirectInput | undefined {
        return this._receiptMovementAddressed?.directInput;
    }

    /**
     *  returns the direct input code linked to the receipt movement addressed
     */
    get directInputCode(): string | undefined {
        return this.directInput?.code;
    }

    /**
     * returns the direct input line linked to the receipt movement addressed
     */
    get directInputLine(): DirectInputLine | undefined {
        return this._receiptMovementAddressed?.directInputLine;
    }

    /**
     *  returns the number of consumption unit remaining in the direct input line
     *  The total number of Serial Numbers still available cannot exceed this value.
     */
    get directInputLineNumberOfConsumptionUnitRemaining(): number {
        return Number(this.directInputLine?.numberOfConsumptionUnitRemaining ?? 0);
    }

    /**
     *  returns the direct input line pointer number linked to the receipt movement addressed
     * */
    get directInputLinePointer(): number | undefined {
        return this.directInputLine?.linePointerNumber;
    }

    /**
     *  returns the homogeneous quantity in consumption unit
     */
    get homogeneousQuantityInConsumptionUnit(): number {
        return this._homogeneousQuantityInConsumptionUnit;
    }

    /**
     *  sets the homogeneous quantity in consumption unit which will be limited to the maximum usable
     *  @param value
     */
    set homogeneousQuantityInConsumptionUnit(value: number) {
        if (value > 0) {
            this._homogeneousQuantityInConsumptionUnit = this._getQuantityInConsumptionUnit(value);
        }
        this.resetSerialNumberNewEnabledState();
    }

    /**
     * returns the initial quantity in consumption unit
     */
    get initialQuantityInConsumptionUnit(): number {
        return Number(this.movement?.quantityInConsumptionUnit ?? 0);
    }

    /**
     * returns whether the receipt movement addressed is a customer return
     */
    get isCustomerReturn(): boolean | undefined {
        return this._receiptMovementAddressed?.directInputLine?.isCustomerReturn;
    }

    /**
     *  returns whether the product is localized serial number managed
     */
    get isLocalizedSerialNumberAllowed(): boolean {
        return this._productConfiguration?.isLocalizedSerialNumberAllowed ?? false;
    }

    /**
     *  returns whether a new serial number can be created depending on the active line or movement mode
     */
    get isNewSerialNumberAllowed(): boolean {
        const _quantityInConsumptionUnit = this.homogeneousQuantityInConsumptionUnit;

        // The quantity must have been initialized, there must be remaining quantity to cover
        if (!_quantityInConsumptionUnit) {
            return false;
        }

        const _remainingQuantity = Math.max(_quantityInConsumptionUnit - this._serialNumbers.value.length, 0);

        // All serial numbers are stored, no need to create new ones
        if (!_remainingQuantity) {
            return false;
        }

        // The maximum authorized quantity available for entry depends on the movement or line mode.
        return this.availableQuantityInConsumptionUnit > 0;
    }

    /**
     *  returns the maximum number of records to fetch
     */
    get maxRecords(): number {
        return this._maxRecords > 10 ? this._maxRecords : maximumNumberOfSerialNumbers;
    }

    /**
     * returns the receipt movement addressed
     */
    get movement(): ReceiptMovementAddressed | undefined {
        return this._receiptMovementAddressed;
    }

    /**
     * returns the movement code linked to the receipt movement addressed
     */
    get movementCode(): string | undefined {
        return this._receiptMovementAddressed?.code;
    }

    /**
     *  returns the movement mode depending on the serial numbers available and the movement code
     */
    get movementMode(): SerialNumberMovementMode {
        if (this._serialNumbersAvailable.length === 0) {
            return SerialNumberMovementMode.movement;
        }

        // If the number of serial numbers in line is equal to the number of serial numbers in movement,
        // we are in movement mode
        if (this.numberSerialNumbersInLine === this.numberSerialNumbersInMovement) {
            return SerialNumberMovementMode.movement;
        }

        return this._serialNumbersAvailable.some(sn => sn.inputMovement?.code !== undefined)
            ? SerialNumberMovementMode.movement
            : SerialNumberMovementMode.line;
    }

    /**
     *  returns the number of serial numbers already linked to the line
     */
    get numberSerialNumbersInLine(): number {
        return this._numberSerialNumbersInLine;
    }

    /**
     * returns the number of serial numbers already linked to the movement
     */
    get numberSerialNumbersInMovement(): number {
        return this._numberSerialNumbersInMovement;
    }

    /**
     * returns the product linked to the receipt movement addressed
     */
    get product(): Product | undefined {
        return this._receiptMovementAddressed?.product;
    }

    /**
     * returns the product code linked to the receipt movement addressed
     */
    get productCode(): string | undefined {
        return this._receiptMovementAddressed?.product?.code;
    }

    /**
     * returns the product configuration linked to the receipt movement addressed
     */
    get productConfiguration(): ProductConfiguration | undefined {
        return this._productConfiguration;
    }

    /**
     * returns the serial group filter or undefined if no serial group available
     */
    get serialGroupFilter(): Filter<SerialGroup> {
        const _serialNumbersExcluded = this.serialNumbersExcluded;
        const _serialGroupsAvailable = Array.from(
            new Set(
                this._serialNumbersAvailable
                    .filter(
                        sn =>
                            sn.serialGroup?.code !== undefined &&
                            !_serialNumbersExcluded.includes(sn.keyCodeForDuplicates),
                    )
                    .map(sn => sn.serialGroup.code)
                    .sort(),
            ),
        );
        const _filter: Filter<SerialGroup> = {
            site: { code: this._siteCode },
            ...(_serialGroupsAvailable.length > 0 && { code: { _in: _serialGroupsAvailable } }),
            ...(!_serialGroupsAvailable.length && { code: '***No_Match***' }),
        };
        return _filter;
    }

    /**
     * returns the serial numbers key excluded
     */
    get serialNumbersExcluded(): string[] {
        return this._serialNumbers.value.map(sn => sn.keyCodeForDuplicates).sort();
    }

    /**
     *  returns the serial number filter
     */
    get serialNumberFilter(): Filter<SerialNumber> {
        const _serialNumbersExcluded = this.serialNumbersExcluded;
        const _serialNumbersAvailable = Array.from(
            new Set(
                this.serialNumbersAvailable
                    .filter(sn => !_serialNumbersExcluded.includes(sn.keyCodeForDuplicates))
                    .map(sn => sn.keyCodeForDuplicates)
                    .sort(),
            ),
        );
        const _filter: Filter<SerialNumber> = {
            ...this._getDefaultSerialNumberFilter(),
            ...(_serialNumbersAvailable.length > 0 && { keyCodeForDuplicates: { _in: _serialNumbersAvailable } }),
            ...(_serialNumbersExcluded.length > 0 && { keyCodeForDuplicates: { _nin: _serialNumbersExcluded } }),
        };
        return _filter;
    }

    /**
     * returns the serial numbers available
     */
    get serialNumbersAvailable(): ExtractEdges<SerialNumber>[] {
        return this._serialNumbersAvailable;
    }

    /* ***************************************************************************************** */
    /*                                                                                           */
    /* 🔷                               Private methods                                          */
    /*                                                                                           */
    /* ***************************************************************************************** */

    /**
     *
     * @returns the base filter for serial numbers
     */
    private _getBaseSerialNumberFilter(additionalFilters?: Filter<SerialNumber>): Filter<SerialNumber> {
        return <Filter<SerialNumber>>{
            site: { code: this._siteCode },
            depositor: { code: this._depositorCode },
            product: { code: this.productCode },
            ...additionalFilters,
        };
    }

    /**
     *
     * @returns the default filter for serial numbers
     */
    private _getDefaultSerialNumberFilter(additionalFilters?: Filter<SerialNumber>): Filter<SerialNumber> {
        return <Filter<SerialNumber>>{
            ...this._getBaseSerialNumberFilter(),
            directInput: { code: this.directInputCode },
            linePointerNumber: this.directInputLinePointer,
            inputMovement: { code: { _in: [this.movementCode, undefined] } },
            ...additionalFilters,
        };
    }

    /**
     * Get the quantity in consumption unit
     * @param quantityInConsumptionUnit
     * @returns
     */
    private _getQuantityInConsumptionUnit(quantityInConsumptionUnit: number): number {
        return Math.min(this.initialQuantityInConsumptionUnit, Math.max(0, quantityInConsumptionUnit));
    }

    /**
     * 🔶 Get the receipt movement addressed
     * @param pageInstance
     * @param movementCode
     * @returns
     */
    private async _getReceiptMovementAddressed(
        pageInstance: any,
        movementCode: string,
    ): Promise<ReceiptMovementAddressed | undefined> {
        if (!this._siteCode || !this._depositorCode || !movementCode) {
            return undefined;
        }
        try {
            const _response = extractEdges<any>(
                await pageInstance.$.graph
                    .node('@sage/wh-input/ReceiptMovementAddressed')
                    .query(
                        ui.queryUtils.edgesSelector<ReceiptMovementAddressed>(
                            {
                                _id: true,
                                storingList: { code: true, _id: true },
                                code: true,
                                directInput: {
                                    _id: true,
                                    code: true,
                                },
                                directInputLine: {
                                    _id: true,
                                    lineNumber: true,
                                    linePointerNumber: true,
                                    receiptLineNumber: true,
                                    isCustomerReturn: true,
                                    numberOfConsumptionUnit: true,
                                    numberOfConsumptionUnitStored: true,
                                    numberOfConsumptionUnitRemaining: true,
                                },
                                product: {
                                    _id: true,
                                    code: true,
                                },

                                quantityInConsumptionUnit: true,
                                numberOfContainers: true,
                                container: { container: { code: true, _id: true }, _id: true },
                                homogeneousQuantity: true,
                                homogeneousContainer: { container: { code: true, _id: true }, _id: true },
                            },
                            {
                                filter: {
                                    site: { code: this._siteCode },
                                    depositor: { code: this._depositorCode },
                                    code: movementCode,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            return <ReceiptMovementAddressed>_response[0] ?? undefined;
        } catch (error) {
            ui.console.error('Error reading receipt movements addressed:\n', JSON.stringify(error));
            return undefined;
        }
    }

    /**
     * 🔶 Get serial groups
     * @param pageInstance
     * @param serialGroupCode
     * @returns serial group selected or undefined
     */
    private async _getSerialGroups(
        pageInstance: any,
        serialGroupCode: string,
    ): Promise<ExtractEdges<SerialGroup> | undefined> {
        if (!this._siteCode) {
            return undefined;
        }
        try {
            const _response = extractEdges<SerialGroup>(
                await pageInstance.$.graph
                    .node('@sage/wh-input/SerialGroup')
                    .query(
                        ui.queryUtils.edgesSelector<SerialGroup>(
                            {
                                _id: true,
                                code: true,
                                parentGroup: { code: true, _id: true },
                            },
                            {
                                filter: {
                                    site: { code: this._siteCode },
                                    code: serialGroupCode,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            return _response.length > 0 ? _response[0] : undefined;
        } catch (error) {
            ui.console.error('Error reading serial groups:\n', JSON.stringify(error));
            return undefined;
        }
    }

    /**
     * 🔶 Get highest serial numbers (minimum data)
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @param maxRecords The maximum number of records to return.
     * @returns
     */
    // eslint-disable-next-line require-await
    private async _getHighestSerialNumbers(
        pageInstance: any,
        additionalFilters?: Filter<SerialNumber>,
        maxRecords?: number,
    ): Promise<ExtractEdges<SerialNumber>[]> {
        return this._getSerialNumbers(pageInstance, additionalFilters, true, (maxRecords ?? 0) > 0 ? maxRecords : 1);
    }

    /**
     * 🔶 Get serial numbers (minimum data)
     * @param pageInstance  The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @param reverseSort Whether to sort in reverse order.
     * @param maxRecords The maximum number of records to return.
     * @returns
     */
    private async _getSerialNumbers(
        pageInstance: any,
        additionalFilters?: Filter<SerialNumber>,
        reverseSort?: boolean,
        maxRecords?: number,
    ): Promise<ExtractEdges<SerialNumber>[]> {
        if (!this._siteCode || !this._depositorCode) {
            return [];
        }
        try {
            const _response = extractEdges<SerialNumber>(
                await pageInstance.$.graph
                    .node('@sage/wh-stock-data/SerialNumber')
                    .query(
                        ui.queryUtils.edgesSelector<SerialNumber>(
                            {
                                _id: true,
                                code: true,
                                double: true,
                                occurrence: true,
                                directInput: { code: true, _id: true },
                                linePointerNumber: true,
                                inputMovement: { code: true, _id: true },
                                receiptDate: true,
                                outputDate: true,
                                transferDate: true,
                                stockObject: { code: true, _id: true },
                                serialGroup: { code: true, _id: true },
                                displayCode: true,
                                keyCodeForDuplicates: true,
                            },
                            {
                                filter: {
                                    ...{
                                        site: { code: this._siteCode },
                                        depositor: { code: this._depositorCode },
                                        product: { code: this.productCode },
                                    },
                                    ...additionalFilters,
                                },
                                first: (maxRecords ?? 0) > 0 ? maxRecords : this.maxRecords,
                                orderBy: reverseSort
                                    ? { code: -1, double: -1, occurrence: -1 }
                                    : { code: 1, double: 1, occurrence: 1 },
                            },
                        ),
                    )
                    .execute(),
            );

            return _response ?? [];
        } catch (error) {
            ui.console.error('Error reading serial numbers:\n', JSON.stringify(error));
            return [];
        }
    }

    /**
     * 🔶 Get the count of serial numbers by direct input.
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns The count of serial numbers.
     */
    private async _getSerialNumberCountByDirectInput(
        pageInstance: ui.Page<GraphApi>,
        additionalFilters?: Filter<SerialNumber>,
    ): Promise<number> {
        const _numberOfSerialNumbers = await this._getSerialNumbersCountByProduct(pageInstance, <Filter<SerialNumber>>{
            directInput: { code: this.directInputCode },
            linePointerNumber: this.directInputLinePointer,
            ...additionalFilters,
        });
        return _numberOfSerialNumbers;
    }

    /**
     * 🔶 Get the count of serial numbers by input movement.
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns The count of serial numbers.
     */
    private async _getSerialNumberCountByInputMovement(
        pageInstance: ui.Page<GraphApi>,
        additionalFilters?: Filter<SerialNumber>,
    ): Promise<number> {
        const _numberOfSerialNumbers = await this._getSerialNumberCountByDirectInput(pageInstance, <
            Filter<SerialNumber>
        >{
            inputMovement: { code: this.movementCode },
            ...additionalFilters,
        });
        return _numberOfSerialNumbers;
    }

    /**
     * 🔶 Count serial numbers by product (minimum data)
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns number of serial numbers found
     */
    private async _getSerialNumbersCountByProduct(
        pageInstance: ui.Page<GraphApi>,
        additionalFilters?: Filter<SerialNumber>,
    ): Promise<number> {
        const _requests = {
            aggregateSerialNumber: generateAggregateSerialNumberRequest<SerialNumber>(
                pageInstance,
                {
                    group: <AggregateGroupSelector<SerialNumber>>{
                        product: { code: { _by: 'value' }, _id: { _by: 'value' } },
                    },
                    values: {
                        keyCodeForDuplicates: { distinctCount: true },
                    },
                },
                {
                    filter: <Filter<SerialNumber>>{
                        site: this._siteCode,
                        depositor: this._depositorCode,
                        product: this.productCode,
                        ...additionalFilters,
                    },
                },
            ),
        };

        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();

        interface AggregateBySerialNumber {
            group: { product: { code: string; _id: string } };
            values: { keyCodeForDuplicates: { distinctCount: number } };
        }

        const _aggregateSerialNumber = extractEdges<AggregateBySerialNumber>(
            _response.aggregateSerialNumber as Edges<any>,
        );

        return _aggregateSerialNumber[0]?.values?.keyCodeForDuplicates?.distinctCount ?? 0;
    }

    /**
     * 🔶 Get serial numbers (full data)
     * @param pageInstance
     * @param additionalFilters
     * @returns
     */
    private async _getSerialNumbersDetailed(
        pageInstance: any,
        additionalFilters?: Filter<SerialNumber>,
    ): Promise<ExtractEdges<SerialNumber>[]> {
        if (!this._siteCode || !this._depositorCode) {
            return [];
        }
        try {
            const _response = extractEdges<SerialNumber>(
                await pageInstance.$.graph
                    .node('@sage/wh-stock-data/SerialNumber')
                    .query(
                        ui.queryUtils.edgesSelector<any>(
                            {
                                _id: true,
                                code: true,
                                double: true,
                                occurrence: true,
                                directInput: { code: true, _id: true },
                                linePointerNumber: true,
                                inputMovement: { code: true, _id: true },
                                receiptDate: true,
                                deliveryOrder: { code: true, _id: true },
                                deliveryOrderLineNumber: true,
                                outputMovement: { code: true, _id: true },
                                outputDate: true,
                                inputAdjustment: { code: true, _id: true },
                                outputAdjustment: { code: true, _id: true },
                                transfer: { code: true, _id: true },
                                transferMovement: { code: true, _id: true },
                                transferDate: true,
                                stockObject: { code: true, _id: true },
                                serialGroup: { code: true, _id: true },
                                serialParentGroup: { code: true, _id: true },
                                free1: true,
                                free2: true,
                                free3: true,
                                free4: true,
                                free5: true,
                                validatedStoringList: true,
                                validatedPreparationOrder: true,
                                postponed: true,
                                displayCode: true,
                                keyCodeForDuplicates: true,
                            },
                            {
                                filter: {
                                    site: { code: this._siteCode },
                                    depositor: { code: this._depositorCode },
                                    product: { code: this.productCode },
                                    ...additionalFilters,
                                },
                                first: this.maxRecords,
                                orderBy: { code: 1, double: 1, occurrence: 1 },
                            },
                        ),
                    )
                    .execute(),
            );

            return _response ?? [];
        } catch (error) {
            ui.console.error('Error reading serial numbers:\n', JSON.stringify(error));
            return [];
        }
    }

    /**
     *🔹Initialize the serial number management
     * @param pageInstance
     * @param movementCode code of the movement to address
     * @param homogeneousQuantityInConsumptionUnit current quantity > 0
     * @param maxRecords maximum number of records to fetch
     * @returns true if initialization is successful
     */
    public async initialize(
        pageInstance: ui.Page<GraphApi>,
        movementCode: string | undefined | null,
        homogeneousQuantityInConsumptionUnit: number,
        _maxRecords: number = 100,
    ): Promise<boolean> {
        this._serialNumbers.value = [];
        if (movementCode && this.initializeSiteDepositor(pageInstance)) {
            const _movement = await this._getReceiptMovementAddressed(pageInstance, movementCode);
            const _productConfiguration = _movement
                ? await getProductConfiguration(pageInstance, _movement.product.code)
                : undefined;
            if (_productConfiguration && _productConfiguration.isLocalizedSerialNumberAllowed) {
                // We ignore the serial numbers that have receipt date
                const _serialFilter = <Filter<SerialNumber>>{
                    receiptDate: { _eq: undefined },
                };
                this._maxRecords = _maxRecords;
                this._productConfiguration = _productConfiguration;
                this._receiptMovementAddressed = _movement;
                // We are looking for all serial numbers associated with the line, assigned to the movement or available
                this._serialNumbersAvailable = await this._getSerialNumbers(
                    pageInstance,
                    this._getDefaultSerialNumberFilter(_serialFilter),
                );

                // Compute the number of serial numbers already linked to the line and movement
                this._numberSerialNumbersInLine = await this._getSerialNumberCountByDirectInput(
                    pageInstance,
                    _serialFilter,
                );
                this._numberSerialNumbersInMovement = this._numberSerialNumbersInLine
                    ? await this._getSerialNumberCountByInputMovement(pageInstance, _serialFilter)
                    : 0;

                // Set the homogeneous quantity in consumption unit (movement is loaded now)
                this.homogeneousQuantityInConsumptionUnit = homogeneousQuantityInConsumptionUnit;

                return true;
            }
        }
        this.reinitialize();
        return false;
    }

    /**
     *🔹Initialize the site and depositor codes
     * @param pageInstance
     * @returns
     */
    public initializeSiteDepositor(pageInstance: ui.Page<GraphApi>): boolean {
        const _selectedSiteDepositor = getSelectedSiteDepositor(pageInstance);
        const _siteCode = _selectedSiteDepositor?.site;
        const _depositorCode = _selectedSiteDepositor?.depositor;
        if (_siteCode && _depositorCode) {
            this._siteCode = _siteCode;
            this._depositorCode = _depositorCode;
            return true;
        }
        return false;
    }

    /**
     *🔹Reinitialize all values to undefined or empty array
     */
    public reinitialize(): void {
        this._productConfiguration = undefined;
        this._receiptMovementAddressed = undefined;
        this._maxRecords = 0;
        this._serialNumbersAvailable = [];
        this._numberSerialNumbersInLine = 0;
        this._numberSerialNumbersInMovement = 0;
        this._serialNumbers.value = [];
    }

    /**
     * 🔵 Handle changes to the serial group field
     * @param pageInstance
     * @returns error message type or undefined if no error
     */
    // eslint-disable-next-line require-await
    public async onChange_serialGroup(
        pageInstance: ui.Page<GraphApi>,
    ): Promise<SerialNumberManagementError | undefined> {
        if (this._serialGroup.value) {
            // We are looking for all serial numbers associated with the line, assigned to the movement or available
            const _serialGroupCode = this._serialGroup.value.code?.trim();
            // const _serialGroup = await this.getSerialGroups(pageInstance, _serialGroupCode);
            const _serialNumbers = this._serialNumbersAvailable.filter(
                sn =>
                    sn.serialGroup?.code === _serialGroupCode &&
                    !this.serialNumbersExcluded.includes(sn.keyCodeForDuplicates),
            );
            const _result = this._validateSerialNumbers(pageInstance, _serialNumbers);
            if (!_result || _result !== SerialNumberManagementError.addNewSerialNumber) {
                this._serialGroup.value = null;
                this.resetSerialNumberNewEnabledState();
                return _result;
            }

            // If the group does not exist in what is available, ignore it.
            this._serialGroup.value = null;
            return SerialNumberManagementError.serialGroupRequired;
        }
        return undefined;
    }

    /**
     * 🔵 Handle changes to the serial number field
     * @param pageInstance
     * @returns error message type or undefined if no error
     */
    public async onChange_serialNumber(
        pageInstance: ui.Page<GraphApi>,
    ): Promise<SerialNumberManagementError | undefined> {
        const _serialNumberCode = this._serialNumber.value?.trim()?.replace(/\s+/g, '')?.toUpperCase();
        if (_serialNumberCode) {
            // We check if the serial number exists in what was initially entered.
            // If it exists, we add it to the table if possible.
            // If it does not exist, we check if we can create it.
            const _serialNumber = (
                await this._getSerialNumbers(pageInstance, <Filter<SerialNumber>>{
                    ...this._getDefaultSerialNumberFilter(),
                    code: _serialNumberCode,
                })
            )[0];

            const _keyCodeForDuplicates = _serialNumber?.keyCodeForDuplicates;
            let _serialNumbers: ExtractEdges<SerialNumber>[] = [];

            if (!_serialNumber) {
                // If the number is being created, a generic value must be given.
                _serialNumbers = [
                    <ExtractEdges<SerialNumber>>{
                        code: _serialNumberCode,
                        double: 1,
                        occurrence: 1,
                        free1: '',
                        free2: '',
                        free3: '',
                        free4: '',
                        free5: '',
                        displayCode: `${_serialNumberCode}-1-1`,
                        keyCodeForDuplicates: `${_serialNumberCode}|1|1`,
                    },
                ];
            } else {
                _serialNumbers = this._serialNumbersAvailable.filter(
                    sn => sn.keyCodeForDuplicates === _keyCodeForDuplicates,
                );
            }

            const _result = this._validateSerialNumbers(pageInstance, _serialNumbers);

            if (!_result || _result !== SerialNumberManagementError.addNewSerialNumber) {
                this._serialNumber.value = null;
                this.resetSerialNumberNewEnabledState();
                return _result;
            }

            // 🔷 If the serial number does not exist in what is available, we create it if possible.

            // We check if the serial number already exists in the system
            // to determine the double and occurrence to assign to the new serial number.
            // In the case of a customer return, we check for serial numbers that are not linked to
            // the current direct input line or movement to avoid conflicts with serial numbers
            // already selected for the current line.

            let _createResult: DoubleOccurrence | SerialNumberManagementError | undefined;

            if (this.isCustomerReturn) {
                // In the case of a customer return, we check that it is possible, and we recover the duplicate number.
                _createResult = await this._findNextAvailableDoubleOccurrence(pageInstance, _serialNumberCode);

                // If an error is detected, it is returned.
                if (SerialNumberManagement._isSerialNumberError(_createResult)) {
                    this._serialNumber.value = null;
                    return _createResult;
                }

                // Then we take the highest available line number.
                const _newSerialNumber = (
                    await this._getHighestSerialNumbers(pageInstance, <Filter<SerialNumber>>{
                        ...this._getBaseSerialNumberFilter(),
                        code: _serialNumberCode,
                        double:
                            _createResult && SerialNumberManagement._isDoubleOccurrence(_createResult)
                                ? _createResult.double
                                : 1,
                    })
                )[0];

                // If not found, we cannot create the serial number (should never happen).
                if (!_newSerialNumber) {
                    this._serialNumber.value = null;
                    return SerialNumberManagementError.noSerialNumberFound;
                }

                _createResult = <DoubleOccurrence>{
                    double: _newSerialNumber.double,
                    occurrence: _newSerialNumber.occurrence + 1,
                };
            } else {
                // In the standard case, we just take the duplicate number plus 1
                const _newSerialNumber = (
                    await this._getHighestSerialNumbers(pageInstance, <Filter<SerialNumber>>{
                        ...this._getBaseSerialNumberFilter(),
                        code: _serialNumberCode,
                    })
                )[0];

                // If it is not found, we move on to creation.
                _createResult = <DoubleOccurrence>{
                    double: _newSerialNumber ? _newSerialNumber.double + 1 : 1,
                    occurrence: 1,
                };
            }

            if (SerialNumberManagement._isDoubleOccurrence(_createResult)) {
                const _newSerialNumber = <ExtractEdges<SerialNumberBinding>>{
                    site: { code: this._siteCode },
                    depositor: { code: this._depositorCode },
                    product: { code: this.productCode },
                    directInput: { code: this.directInputCode },
                    linePointerNumber: this.directInputLinePointer,
                    ...(this.movementMode !== SerialNumberMovementMode.line && {
                        inputMovement: { code: this.movementCode },
                    }),
                    code: _serialNumberCode,
                    double: _createResult.double,
                    occurrence: _createResult.occurrence,
                    free1: '',
                    free2: '',
                    free3: '',
                    free4: '',
                    free5: '',
                    displayCode: `${_serialNumberCode}-${_createResult.double}-${_createResult.occurrence}`,
                    keyCodeForDuplicates: `${_serialNumberCode}|${_createResult.double}|${_createResult.occurrence}`,
                    _id: `${this._siteCode}|${this._depositorCode}|${this.productCode}|${_serialNumberCode}|${_createResult.double}|${_createResult.occurrence}`,
                    actionImport: 'C',
                };

                this._serialNumbers.addRecord(_newSerialNumber);
            }
            this.resetSerialNumberNewEnabledState();
        }
        this._serialNumber.value = null;
        return undefined;
    }

    /**
     * Updates the enabled state for creating a new serial number.
     * This is a placeholder for UI logic; implement as needed.
     */
    public resetSerialNumberNewEnabledState(): boolean {
        const _setSerialNumberNewEnabled = this.isNewSerialNumberAllowed;
        // Update the "isNewEnabled" property of the serial number field
        this._serialNumber.isNewEnabled = _setSerialNumberNewEnabled;
        return _setSerialNumberNewEnabled;
    }

    /**
     * 🔶 Find next available double/occurrence for customer returns
     * Analyzes each double group to determine the last valid occurrence
     * @param pageInstance
     * @param serialNumberCode
     * @returns Promise<{double: number, occurrence: number}>
     */
    private async _findNextAvailableDoubleOccurrence(
        pageInstance: ui.Page<GraphApi>,
        serialNumberCode: string,
    ): Promise<DoubleOccurrence | SerialNumberManagementError | undefined> {
        const _serialNumbersHistory = await this._getSerialNumbers(pageInstance, {
            ...this._getBaseSerialNumberFilter(),
            code: serialNumberCode,
            _and: [
                {
                    _or: [
                        { directInput: { code: { _ne: this.directInputCode } } },
                        { inputMovement: { code: { _ne: this.movementCode } } },
                    ],
                },
            ],
        });

        if (!_serialNumbersHistory.length) {
            // Creation of a new serial number is not allowed for a customer return
            return SerialNumberManagementError.noSerialNumberFound;
        }

        // Group by double
        const _doubleGroups = SerialNumberManagement._groupSerialNumbersByDouble(_serialNumbersHistory);

        // Analyze each couple of double/occurrence
        const _validGroups = Array.from(_doubleGroups.entries())
            .map(([double, serialNumbers]) => SerialNumberManagement._analyzeDoubleGroup(double, serialNumbers))
            .filter(result => result.isValid)
            .map(result => <DoubleOccurrence>{ double: result.double, occurrence: result.occurrence });

        // Application rules for validation

        if (_validGroups.length === 0) {
            // The serial number is not out of stock.
            return SerialNumberManagementError.noSerialNumberForReturn;
        }

        if (_validGroups.length > 1) {
            // There are too many duplicate outputs on this serial number
            return SerialNumberManagementError.tooManySerialNumbersToReturn;
        }

        // This pair can be used for customer returns.
        return _validGroups[0];
    }

    /**
     * 🔶 Analyze a double group to check if return is possible
     * @param double
     * @param serialNumbers
     * @returns analysis result
     */
    private static _analyzeDoubleGroup(
        double: number,
        serialNumbers: ExtractEdges<SerialNumber>[],
    ): DoubleOccurrenceResult {
        // Check if all serial numbers in the group have been outputted
        const _allOutputted = serialNumbers.every(sn => sn.outputDate !== undefined && sn.outputDate !== null);

        if (!_allOutputted) {
            return { isValid: false, double, occurrence: 0 };
        }

        // Get the last occurrence
        const _lastOccurrence = Math.max(...serialNumbers.map(sn => sn.occurrence));

        return <DoubleOccurrenceResult>{
            isValid: true,
            double,
            occurrence: _lastOccurrence,
        };
    }

    /**
     * 🔶 Group serial numbers by double value, sorted by double
     * @param serialNumbers
     * @returns Map<number, ExtractEdges<SerialNumber>[]> sorted by double
     */
    private static _groupSerialNumbersByDouble(
        serialNumbers: ExtractEdges<SerialNumber>[],
    ): Map<number, ExtractEdges<SerialNumber>[]> {
        const _groupedByDouble = serialNumbers.reduce((groups, sn) => {
            const _double = sn.double;
            if (!groups.has(_double)) {
                groups.set(_double, []);
            }
            groups.get(_double)!.push(sn);
            return groups;
        }, new Map<number, ExtractEdges<SerialNumber>[]>());

        // Sort each group by occurrence
        const _sortedGroups = new Map(
            Array.from(_groupedByDouble.entries())
                .sort(([a], [b]) => a - b)
                .map(([double, group]) => [double, [...group].sort((a, b) => a.occurrence - b.occurrence)]),
        );

        return _sortedGroups;
    }

    /**
     * 🔶 Type guard to check if result is DoubleOccurrence
     * @param result
     * @returns boolean
     */
    private static _isDoubleOccurrence(
        result: DoubleOccurrence | SerialNumberManagementError | undefined,
    ): result is DoubleOccurrence {
        return result !== undefined && typeof result === 'object' && 'double' in result && 'occurrence' in result;
    }

    /**
     * 🔶 Type guard to check if result is a SerialNumberManagementError
     * @param result
     * @returns boolean
     */
    private static _isSerialNumberError(
        result: DoubleOccurrence | SerialNumberManagementError | undefined,
    ): result is SerialNumberManagementError {
        return typeof result === 'string' && Object.values(SerialNumberManagementError).includes(result);
    }

    /**
     * 🔸🔹Validate the serial number against the available stock and selected items.
     *      The available quantity represents the number of serial numbers that can still be entered,
     *      the maximum depending on the movement or line mode, without exceeding the quantity already
     *      existing on the movement.
     *
     * @param pageInstance The current page instance.
     * @param _serialNumber The serial number to validate.
     * @returns An error if validation fails, otherwise undefined.
     */

    private _validateSerialNumbers(
        _pageInstance: ui.Page<GraphApi, any>,
        _serialNumbers: ExtractEdges<SerialNumber>[],
    ): SerialNumberManagementError | undefined {
        const _quantityInConsumptionUnit = this.homogeneousQuantityInConsumptionUnit;
        const _remainingQuantity = Math.max(_quantityInConsumptionUnit - this._serialNumbers.value.length, 0);
        const _availableQuantity = this.availableQuantityInConsumptionUnit;

        // We are looking for all serial numbers associated with the line, assigned to the movement or available
        if (!_quantityInConsumptionUnit) {
            return SerialNumberManagementError.quantityIsMandatory;
        }

        if (!_remainingQuantity) {
            return SerialNumberManagementError.tooManySerialNumbers;
        }

        const _requestedKeys = _serialNumbers.map(sn => sn.keyCodeForDuplicates);
        const _requestedCodes = _serialNumbers.map(sn => sn.code);
        const _existingKeys = this._serialNumbers.value.map(sn => sn.keyCodeForDuplicates);
        const _existingCodes = this._serialNumbers.value.map(sn => sn.code);

        // Check duplicate with optimized Array methods
        const _validation = SerialNumberManagement._validateSerialNumberDuplicates(
            _requestedKeys,
            _requestedCodes,
            _existingKeys,
            _existingCodes,
        );
        if (_validation) {
            return _validation;
        }

        // Check availability
        return this._validateSerialNumberAvailability(_requestedKeys, _remainingQuantity, _availableQuantity);
    }

    /**
     * 🔶 Validate serial number duplicates
     * @param requestedKeys
     * @param requestedCodes
     * @param existingKeys
     * @param existingCodes
     * @returns error or undefined
     */
    private static _validateSerialNumberDuplicates(
        requestedKeys: string[],
        requestedCodes: string[],
        existingKeys: string[],
        existingCodes: string[],
    ): SerialNumberManagementError | undefined {
        if (existingKeys.length === 0) {
            return undefined;
        }

        // Use Set for O(1) checks instead of includes() O(n)
        const _existingKeySet = new Set(existingKeys);
        const _existingCodeSet = new Set(existingCodes);

        const _hasKeyDuplicate = requestedKeys.some(key => _existingKeySet.has(key));
        const _hasCodeDuplicate = requestedCodes.some(code => _existingCodeSet.has(code));

        if (_hasKeyDuplicate || _hasCodeDuplicate) {
            return SerialNumberManagementError.serialNumberAlreadySelected;
        }

        return undefined;
    }

    /**
     * 🔶 Validate serial number availability
     * @param requestedKeys
     * @param remainingQuantity
     * @param availableQuantity
     * @returns error or undefined
     */
    private _validateSerialNumberAvailability(
        requestedKeys: string[],
        remainingQuantity: number,
        availableQuantity: number,
    ): SerialNumberManagementError | undefined {
        if (!this._serialNumbersAvailable.length) {
            return availableQuantity < 1
                ? SerialNumberManagementError.tooManySerialNumbers
                : SerialNumberManagementError.addNewSerialNumber;
        }

        // Use Set for O(1) lookups instead of includes() O(n)
        const _excludedSet = new Set(this.serialNumbersExcluded);
        const _requestedSet = new Set(requestedKeys);

        const _availableSerialNumbers = this._serialNumbersAvailable.filter(
            available =>
                _requestedSet.has(available.keyCodeForDuplicates) && !_excludedSet.has(available.keyCodeForDuplicates),
        );

        if (_availableSerialNumbers.length > remainingQuantity) {
            return SerialNumberManagementError.tooManySerialNumbers;
        }

        if (_availableSerialNumbers.length > 0) {
            _availableSerialNumbers.forEach(availableSerial => {
                this._serialNumbers.addRecord(availableSerial);
            });
            return undefined;
        }

        return availableQuantity < 1
            ? SerialNumberManagementError.tooManySerialNumbers
            : SerialNumberManagementError.addNewSerialNumber;
    }
}
