import type { DirectInput, DirectInputLine, GraphApi, ReceiptMovementAddressed } from '@sage/wh-input-api';
import type { SerialGroup } from '@sage/wh-master-data-api';
import type { Product } from '@sage/wh-product-data-api';
import { ProductConfiguration } from '@sage/wh-product-data/lib/interfaces/environnement';
import type { SerialNumber } from '@sage/wh-stock-data-api';
import { ExtractEdges, Filter } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
/**
 *  Defines the error messages that can be returned by the serial number management functions
 *  Except for adding a new serial number, all errors are blocking
 */
export declare enum SerialNumberManagementError {
    quantityIsMandatory = "quantityIsMandatory",
    serialNumberAlreadySelected = "serialNumberAlreadySelected",
    tooManySerialNumbers = "tooManySerialNumbers",
    noSerialNumberFound = "noSerialNumberFound",
    noSerialNumberForReturn = "noSerialNumberForReturn",
    tooManySerialNumbersToReturn = "tooManySerialNumbersToReturn",
    serialGroupRequired = "serialGroupRequired",
    addNewSerialNumber = "addNewSerialNumber"
}
/**
 *  Defines the mode of serial number movement
 */
export declare enum SerialNumberMovementMode {
    none = "none",
    line = "line",
    movement = "movement"
}
export declare class SerialNumberManagement {
    private _serialGroup;
    private _serialNumber;
    private _serialNumbers;
    constructor(_serialGroup: ui.fields.Reference, _serialNumber: ui.fields.FilterSelect, _serialNumbers: ui.fields.Table);
    private _siteCode?;
    private _depositorCode?;
    private _maxRecords;
    private _productConfiguration?;
    private _receiptMovementAddressed?;
    private _serialNumbersAvailable;
    private _numberSerialNumbersInLine;
    private _numberSerialNumbersInMovement;
    private _homogeneousQuantityInConsumptionUnit;
    /**
     *  returns the available quantity in consumption unit to create new serial numbers.
     *  The maximum authorized quantity available for entry depends on the movement or line mode.
     */
    get availableQuantityInConsumptionUnit(): number;
    /**
     * returns the direct input linked to the receipt movement addressed
     */
    get directInput(): DirectInput | undefined;
    /**
     *  returns the direct input code linked to the receipt movement addressed
     */
    get directInputCode(): string | undefined;
    /**
     * returns the direct input line linked to the receipt movement addressed
     */
    get directInputLine(): DirectInputLine | undefined;
    /**
     *  returns the number of consumption unit remaining in the direct input line
     *  The total number of Serial Numbers still available cannot exceed this value.
     */
    get directInputLineNumberOfConsumptionUnitRemaining(): number;
    /**
     *  returns the direct input line pointer number linked to the receipt movement addressed
     * */
    get directInputLinePointer(): number | undefined;
    /**
     *  returns the homogeneous quantity in consumption unit
     */
    get homogeneousQuantityInConsumptionUnit(): number;
    /**
     *  sets the homogeneous quantity in consumption unit which will be limited to the maximum usable
     *  @param value
     */
    set homogeneousQuantityInConsumptionUnit(value: number);
    /**
     * returns the initial quantity in consumption unit
     */
    get initialQuantityInConsumptionUnit(): number;
    /**
     * returns whether the receipt movement addressed is a customer return
     */
    get isCustomerReturn(): boolean | undefined;
    /**
     *  returns whether the product is localized serial number managed
     */
    get isLocalizedSerialNumberAllowed(): boolean;
    /**
     *  returns whether a new serial number can be created depending on the active line or movement mode
     */
    get isNewSerialNumberAllowed(): boolean;
    /**
     *  returns the maximum number of records to fetch
     */
    get maxRecords(): number;
    /**
     * returns the receipt movement addressed
     */
    get movement(): ReceiptMovementAddressed | undefined;
    /**
     * returns the movement code linked to the receipt movement addressed
     */
    get movementCode(): string | undefined;
    /**
     *  returns the movement mode depending on the serial numbers available and the movement code
     */
    get movementMode(): SerialNumberMovementMode;
    /**
     *  returns the number of serial numbers already linked to the line
     */
    get numberSerialNumbersInLine(): number;
    /**
     * returns the number of serial numbers already linked to the movement
     */
    get numberSerialNumbersInMovement(): number;
    /**
     * returns the product linked to the receipt movement addressed
     */
    get product(): Product | undefined;
    /**
     * returns the product code linked to the receipt movement addressed
     */
    get productCode(): string | undefined;
    /**
     * returns the product configuration linked to the receipt movement addressed
     */
    get productConfiguration(): ProductConfiguration | undefined;
    /**
     * returns the serial group filter or undefined if no serial group available
     */
    get serialGroupFilter(): Filter<SerialGroup>;
    /**
     * returns the serial numbers key excluded
     */
    get serialNumbersExcluded(): string[];
    /**
     *  returns the serial number filter
     */
    get serialNumberFilter(): Filter<SerialNumber>;
    /**
     * returns the serial numbers available
     */
    get serialNumbersAvailable(): ExtractEdges<SerialNumber>[];
    /**
     *
     * @returns the base filter for serial numbers
     */
    private _getBaseSerialNumberFilter;
    /**
     *
     * @returns the default filter for serial numbers
     */
    private _getDefaultSerialNumberFilter;
    /**
     * Get the quantity in consumption unit
     * @param quantityInConsumptionUnit
     * @returns
     */
    private _getQuantityInConsumptionUnit;
    /**
     * 🔶 Get the receipt movement addressed
     * @param pageInstance
     * @param movementCode
     * @returns
     */
    private _getReceiptMovementAddressed;
    /**
     * 🔶 Get serial groups
     * @param pageInstance
     * @param serialGroupCode
     * @returns serial group selected or undefined
     */
    private _getSerialGroups;
    /**
     * 🔶 Get highest serial numbers (minimum data)
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @param maxRecords The maximum number of records to return.
     * @returns
     */
    private _getHighestSerialNumbers;
    /**
     * 🔶 Get serial numbers (minimum data)
     * @param pageInstance  The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @param reverseSort Whether to sort in reverse order.
     * @param maxRecords The maximum number of records to return.
     * @returns
     */
    private _getSerialNumbers;
    /**
     * 🔶 Get the count of serial numbers by direct input.
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns The count of serial numbers.
     */
    private _getSerialNumberCountByDirectInput;
    /**
     * 🔶 Get the count of serial numbers by input movement.
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns The count of serial numbers.
     */
    private _getSerialNumberCountByInputMovement;
    /**
     * 🔶 Count serial numbers by product (minimum data)
     * @param pageInstance The current page instance.
     * @param additionalFilters Additional filters to apply.
     * @returns number of serial numbers found
     */
    private _getSerialNumbersCountByProduct;
    /**
     * 🔶 Get serial numbers (full data)
     * @param pageInstance
     * @param additionalFilters
     * @returns
     */
    private _getSerialNumbersDetailed;
    /**
     *🔹Initialize the serial number management
     * @param pageInstance
     * @param movementCode code of the movement to address
     * @param homogeneousQuantityInConsumptionUnit current quantity > 0
     * @param maxRecords maximum number of records to fetch
     * @returns true if initialization is successful
     */
    initialize(pageInstance: ui.Page<GraphApi>, movementCode: string | undefined | null, homogeneousQuantityInConsumptionUnit: number, _maxRecords?: number): Promise<boolean>;
    /**
     *🔹Initialize the site and depositor codes
     * @param pageInstance
     * @returns
     */
    initializeSiteDepositor(pageInstance: ui.Page<GraphApi>): boolean;
    /**
     *🔹Reinitialize all values to undefined or empty array
     */
    reinitialize(): void;
    /**
     * 🔵 Handle changes to the serial group field
     * @param pageInstance
     * @returns error message type or undefined if no error
     */
    onChange_serialGroup(pageInstance: ui.Page<GraphApi>): Promise<SerialNumberManagementError | undefined>;
    /**
     * 🔵 Handle changes to the serial number field
     * @param pageInstance
     * @returns error message type or undefined if no error
     */
    onChange_serialNumber(pageInstance: ui.Page<GraphApi>): Promise<SerialNumberManagementError | undefined>;
    /**
     * Updates the enabled state for creating a new serial number.
     * This is a placeholder for UI logic; implement as needed.
     */
    resetSerialNumberNewEnabledState(): boolean;
    /**
     * 🔶 Find next available double/occurrence for customer returns
     * Analyzes each double group to determine the last valid occurrence
     * @param pageInstance
     * @param serialNumberCode
     * @returns Promise<{double: number, occurrence: number}>
     */
    private _findNextAvailableDoubleOccurrence;
    /**
     * 🔶 Analyze a double group to check if return is possible
     * @param double
     * @param serialNumbers
     * @returns analysis result
     */
    private static _analyzeDoubleGroup;
    /**
     * 🔶 Group serial numbers by double value, sorted by double
     * @param serialNumbers
     * @returns Map<number, ExtractEdges<SerialNumber>[]> sorted by double
     */
    private static _groupSerialNumbersByDouble;
    /**
     * 🔶 Type guard to check if result is DoubleOccurrence
     * @param result
     * @returns boolean
     */
    private static _isDoubleOccurrence;
    /**
     * 🔶 Type guard to check if result is a SerialNumberManagementError
     * @param result
     * @returns boolean
     */
    private static _isSerialNumberError;
    /**
     * 🔸🔹Validate the serial number against the available stock and selected items.
     *      The available quantity represents the number of serial numbers that can still be entered,
     *      the maximum depending on the movement or line mode, without exceeding the quantity already
     *      existing on the movement.
     *
     * @param pageInstance The current page instance.
     * @param _serialNumber The serial number to validate.
     * @returns An error if validation fails, otherwise undefined.
     */
    private _validateSerialNumbers;
    /**
     * 🔶 Validate serial number duplicates
     * @param requestedKeys
     * @param requestedCodes
     * @param existingKeys
     * @param existingCodes
     * @returns error or undefined
     */
    private static _validateSerialNumberDuplicates;
    /**
     * 🔶 Validate serial number availability
     * @param requestedKeys
     * @param remainingQuantity
     * @param availableQuantity
     * @returns error or undefined
     */
    private _validateSerialNumberAvailability;
}
//# sourceMappingURL=serial-number-management.d.ts.map