import { Point } from '../model';
import { ShapeChanges } from './shape';
declare type Vector = Point;
/**
 * Adds two given vectors.
 *
 * @export
 * @param {Vector} v1 first vector
 * @param {Vector} v2 second vector
 * @returns {Vector} resulting vector
 */
export declare function add(v1: Vector, v2: Vector): Vector;
/**
 * Subtracts one vector from another.
 *
 * @export
 * @param {Vector} v1 first vector
 * @param {Vector} v2 second vector
 * @returns {Vector} resulting vector
 */
export declare function subtract(v1: Vector, v2: Vector): Vector;
/**
 * Returns the magnitude of a vector.
 *
 * @export
 * @param {Vector} v vector
 * @returns {number} the vector's magnitude
 */
export declare function magnitude(v: Vector): number;
/**
 * Returns the unit/direction vector of a given vector.
 *
 * @export
 * @param {Vector} v vector
 * @returns {Vector} unit vector
 */
export declare function unit(v: Vector): Vector;
/**
 * Multiplies a vector by a scalar.
 *
 * @export
 * @param {Vector} v vector
 * @param {number} scalar scalar
 * @returns {Vector} multiplied vector
 */
export declare function multiply(v: Vector, scalar: number): Vector;
export declare function getAngleBetweenTwoVectors(v1: Vector, v2: Vector): number;
/**
 * Gets the angle between the line intersecting 'v1' and 'v2'
 * with respect to the vertical axis.
 *
 * @export
 * @param {Vector} v1 first point
 * @param {Vector} v2 second point
 * @returns {number} angle in radians
 */
export declare function getVerticalAngle(v1: Vector, v2: Vector): number;
/**
 * Gets the angle between the line intersecting 'v1' and 'v2'
 * with respect to the horizontal axis.
 *
 * @export
 * @param {Vector} v1 first point
 * @param {Vector} v2 second point
 * @returns {number} angle in radians
 */
export declare function getHorizontalAngle(v1: Vector, v2: Vector): number;
interface PointsAlongLineInput {
    start: Vector;
    end: Vector;
    distance: number;
}
interface PointsAlongLineOutput {
    start: Vector;
    end: Vector;
    outsideLineEnd: Vector;
    outsideLineStart: Vector;
}
/**
 * Returns the coordinates of a point along the line intersecting 'start' and 'end' at a given distance.
 *
 * @export
 * @param {PointsAlongLineInput} { start: 'starting point', end: 'ending point', distance, direction }
 * @returns {Vector} point at given distance
 */
export declare function getPointsAlongLineAtDistance({ start: v1, end: v2, distance, }: PointsAlongLineInput): PointsAlongLineOutput;
export declare function getDistanceBetweenTwoPoints(v1: Vector, v2: Vector): number;
export declare function getBoundingRectangle(...vectors: Vector[]): {
    top: number;
    left: number;
    right: number;
    bottom: number;
};
/**
 * Rotates a vector by a given angle.
 *
 * @export
 * @param {Vector} v vector
 * @param {number} angle angle
 * @returns {Vector} rotated vector
 */
export declare function rotate(v: Vector, angle: number): Vector;
/**
 * Calculates the centroid of N given vectors.
 *
 * @export
 * @param {...Vector[]} vectors
 * @returns the centroid
 */
export declare function centroid(...vectors: Vector[]): {
    x: number;
    y: number;
};
/**
 * Scales a vector horizontally and vertically.
 *
 * @export
 * @param {Vector} v vector
 * @param {number} scaleX scale factor for the X axis
 * @param {number} scaleY scale factor for the Y axis
 * @returns scaled vector
 */
export declare function scale(v: Vector, scaleX?: number, scaleY?: number): {
    x: number;
    y: number;
};
/**
 * Returns the center of gravity of a shape defined by N vectors.
 * This is different from the 'centroid' in case the shape doesn't have a uniform density.
 * In the case of a polygon the given shape must be closed, i.e. the last vertex/vector must be the same as the first.
 *
 * @export
 * @param {...Vector[]} vectors
 * @returns {Vector}
 */
export declare function center(...vectors: Vector[]): Vector;
/**
 * Rounds a number according to the given precision.
 *
 * @export
 * @param {number} num input number
 * @param {number} [precision=4] precision
 * @returns {number} rounded number
 */
export declare function round(num: number, precision?: number): number;
/**
 * Transforms a vector, i.e. scale it, rotate it and translate it.
 * scaleX = scaling factor along the X axis.
 * scaleY = scaling factor along the Y axis.
 * rotation = rotation in degrees.
 * {x, y} vector used for translation.
 *
 * @export
 * @param {ShapeChanges} { x, y, scaleX = 1, scaleY = 1, rotation = 0 }
 * @param {...Vector[]} vectors
 * @returns
 */
export declare function transform({ x, y, scaleX, scaleY, rotation, centerX, centerY }: ShapeChanges, ...vectors: Vector[]): Point[];
/**
 * Sorts a series of points clockwise around their centroid.
 *
 * @export
 * @param {...Vector[]} points
 * @returns {Vector[]}
 */
export declare function getClockwisePointsAroundCenter(...points: Vector[]): Vector[];
export {};
//# sourceMappingURL=vector.d.ts.map