# Visual Process Editor

This library provides an editor and a renderer for visual processes defined in a proprietary JSON format. The library is implemented as a React component, but a vanilla JavaScript adapter is also provided.

### Importing

Once the library is imported to a project, it declares the `vp` object on the global scope which contains all objects and functions which are needed.

### TypeScript

The library is implemented in TypeScript. When the project is compiled, a `.d.ts` file is generated to support integration and type security with other TypeScript project. The library is still compiled into JavaScript, so it can be integrated to any standard JavaScript project.

Given that the library is imported as a global object, the following declaration needs to be added manually:

```javascript
import * as VP from '@sage/etna-visual-process-editor';

declare global {
    interface Window {
        vp: typeof VP;
    }
}
```

### Arguments / Properties

Given that the visual process editor lives inside other projects, a number of arguments has to be provided in order to control the libraries behavior and ensure that it integrates correctly to the containing application.

-   **name**: `string` — the name of the visual process document. It will be used in the file name when the document is exported from the editor.
-   **value**: `object` — the visual process chart value. It is described as a JSON document.
-   **onChange**: `function(updatedValue: object) => void` — This callback notifies the consumer whenever the user makes changes to the visual process document.
-   **canEdit**: boolean — Whether the library allows user edits or is only used as a rendering engine (read-only mode).
-   **icons**: `object<string[]>` — Sets of icons which can be added by the user to the visual process document. It must be an objects with string keys. The object keys represent icon collection IDs. The object property values are string arrays. An item in the array is the ID of an icon in the collection.
-   **resolveImageUrl**: `function(collectionId: string, resourceId: string) => string` — This callback is used to resolve icons or images to actual URLs. It is called when an image or an icon is rendered on the canvas, the arguments are the resource's collection and resource ID. The expected return value is a URL in string format; this value is used to load the images.
-   **onLinkClick**: `function(link: object) => void` — Executed when the user clicks a link in read-only mode.
-   **getData**: `function(action: string, args?: { code?: string; transaction?: string }) => Promise<{value: string; label: string | null}>` — This callback retrieves data from the consumer application. It is used for link form validation and reference lookups. For example, when the user clicks on the ... button next to an ERP link, the `FUNCTION_LINK_LOOKUP` is triggered. The expected return value is a promise.

    -   In case of validations, if the process is resolved, the form is considered to be valid. When the returned promise is rejected with a string error message, an error dialog is displayed.
    -   In the case of a lookup action, it should resolve to an object which contains two properties: a technical `value` and a user-friendly `label`. If the promise is resolved to `null` the library considers that the user just cancelled the lookup action in the consumer.

    The following actions can occur:

```javascript
    /** No arguments needed */
    | 'FUNCTION_LINK_LOOKUP'
    /** Args:
     * - code: the selected function code.
     */
    | 'FUNCTION_TRANSACTION_LINK_LOOKUP'
    /** Args:
     * - code: the selected function code.
     * - transaction: the selected transaction code
     */
    | 'FUNCTION_LINK_VALIDATION'
    /** No arguments needed */
    | 'REPORT_LINK_LOOKUP'
    /** Args:
     * - code: the selected report code.
     */
    | 'REPORT_LINK_VALIDATION'
    /** No arguments needed */
    | 'PROCESS_LINK_LOOKUP'
    /** Args:
     * - code: the selected report code.
     */
    | 'PROCESS_LINK_VALIDATION'
    /** No arguments needed */
    | 'BO_REPORT_LINK_LOOKUP'
    /** Args:
     * - code: the selected business objects report code.
     */
    | 'BO_REPORT_LINK_VALIDATION'
    /** No arguments needed */
    | 'QUERY_LINK_LOOKUP'
    /** Args:
     * - code: the selected query report code.
     */
    | 'QUERY_LINK_VALIDATION'
    /** No arguments needed */
    | 'STATISTICS_LINK_LOOKUP'
    /** Args:
     * - code: the selected statistics report code.
     */
    | 'STATISTICS_LINK_VALIDATION'
    /** No arguments needed */
    | 'INQUERY_LINK_LOOKUP'
    /** Args:
     * - code: the selected inquery code.
     */
    | 'INQUERY_SCREEN_LINK_LOOKUP'
    /** Args:
     * - code: the selected statistics report code.
     * - transaction: the selected inquery screen code.
     */
    | 'INQUERY_LINK_VALIDATION'
    /** No arguments needed */
    | 'HELP_FIELD_LINK_LOOKUP'
    /** Args:
     * - code: the selected field code.
     */
    | 'HELP_FIELD_LINK_VALIDATION'
    /** No arguments needed */
    | 'HELP_FUNCTION_LINK_LOOKUP'
    /** Args:
     * - code: the selected function code.
     */
    | 'HELP_FUNCTION_LINK_VALIDATION';
```

### Using the library as a React component

The visual process component follows the [controlled component](https://reactjs.org/docs/forms.html#controlled-components) approach, which means that the component does not keep track of its own value. Whenever the process is changed, the consuming application is notified using the `onChange` callback. It is the consumer's responsibility to keep track of the state and set the updated value in the `value` property.

```javascript
import '@sage/etna-visual-process-editor';

...

render() {
    return (
        <window.vp.VisualProcessEditorComponent
            name={this.state.name}
            canEdit={!this.state.readonly}
            onChange={this.onChange}
            resolveImageUrl={this.resolveImageUrl}
            value={this.state.value}
            icons={this.props.iconSet}
            resolveImageUrl={this.resolveImageUrl}
            getData={this.getData}
            onLinkClick={this.props.dispatchLink}
        />
    )
}
```

### Using in a vanilla JS project (or any other framework than React)

The library provides a vanilla JS wrapper that can be constructed as a pure JavaScript object. Unlike the React component, the vanilla JS wrapper keeps track of the document's state, so the consumer doesn't need to reset the value after every change. When external changes set using the `setValue` function, the visual process editor will update the document on the screen.

```javascript
require('@sage/etna-visual-process-editor');

const vp = new VisualProcessEditor(
    this.name,
    this.domItem, // Target DOM node that the library is rendered into
    this.canEdit(),
    this.onChange.bind(this),
    icons.default,
    this.resolveImageUrl.bind(this),
    this.getData.bind(this),
    this.onLinkClick.bind(this),
);

vp.render(); // The visual process editor is added to the DOM

vp.setValue(newValue); // Updates the value of the editor / viewer

vp.dispose(); // Removes all event listeners and the visual process editor from the DOM
```
