import { ASYNC_CONTEXT_CONST, AsyncContext } from '@sage/async-context-provider';
import { Context, Logger } from '@sage/xtrem-core';
import { FileTimeToLive, S3BodyGet, S3BodyPut, TenantDataType } from '@sage/xtrem-file-storage';
import { TenantAppHelper } from '@sage/xtrem-infra';
export type BaseContextValueType = {
    replyTopic: string;
};
export interface FileList {
    IsTruncated: boolean;
    Contents: {
        Key: string;
        LastModified: string;
        ETag: string;
    }[];
    Prefix: string;
    Name: string;
    KeyCount: number;
}
export declare class InfrastructureHelper {
    private static logger;
    static getLogger(): Logger;
    /**
     * The test async context object has a Map to cache the contexts, so we need to store it in a static for unit tests
     */
    private static testAsyncContextXtremRepository;
    /**
     * @returns
     */
    private static getAsyncContextXtremRepository;
    private static getTenantId;
    private static getClusterId;
    /**
     * Create an async context
     * @param context
     * @param contextKind
     * @param contextValue
     * @param options
     * @returns nanoId - contextId
     */
    static createAsyncContext<ContextValueT extends BaseContextValueType>(context: Context, contextKind: keyof typeof ASYNC_CONTEXT_CONST, contextValue: ContextValueT, options: {
        contextId?: string;
        expireDatetimeIso?: string;
        responseQueueName?: string;
        mimeType?: string | string[];
        s3FileTTL: FileTimeToLive;
    }): Promise<string>;
    /**
     * Get an async context for a specified tenant and contextId
     * @param context
     * @param contextId
     * @returns
     */
    static getAsyncContext(contextId: string): Promise<AsyncContext | undefined>;
    /**
     * delete an async context
     * @param context
     * @param contextId
     * @returns
     */
    static deleteAsyncContext(contextId: string): Promise<void>;
    /**
     * Get a file storage handler to perform actions on tenant specific S3 buckets
     * @returns
     */
    private static getFileStorageHandler;
    private static isPureDevelopmentConfig;
    private static getDevUrl;
    /**
     * Usually before you want to generate a nanoid and use it will async context, see InfrastructureHelper.createAsyncContext
     * returns a presigned url that can be used to upload the attachment.
     * location would be : s3://bucket-of-cluster/tenant-data/{app}/{tenant-id}/attachments/{async-context-id}
     * @param context
     * @param objectKey
     * @param ttlSeconds
     * @returns
     */
    static generateAttachmentDownloadPresignedUrl(context: Context, objectKey: string, urlTimeToLiveInSec?: number, filename?: string): Promise<string>;
    /**
     * Copy a file from a place to another one
     * @param context The context
     * @param sourceObjectKey the objectKey of the file to copy
     * @param targetObjectKey the target objectKey of the new file
     * @param targetTimeToLive the time to live of the new file
     */
    static copyFileToFolder(context: Context, sourceObjectKey: string, targetObjectKey: string, targetTimeToLive: FileTimeToLive): Promise<void>;
    /**
     * This will return a url signed that points to : s3://bucket-of-cluster/tenant-data/{app}/{tenant-id}/attachments/{objectKey}
     * @param objectKey
     * @param ttlSeconds
     * @returns
     */
    static generateAttachmentUploadPresignedUrl(context: Context, objectKey: string, fileStorageSetting?: {
        ttlSeconds?: number;
    }): Promise<string>;
    /**
     * Usually before you want to generate a nanoid and use it will async context, see InfrastructureHelper.createAsyncContext
     * returns a presigned url that can be used to upload the file.
     * location would be : s3://cluster-bucket/dirty/tenants-data/{tenant-id}/uploads/{async-context-id}
     * @param context
     * @param objectKey
     * @param ttlSeconds
     * @returns
     */
    static getUploadPresignedUrl(context: Context, objectKey: string, urlTimeToLiveInSec?: number): Promise<string>;
    /**
     * This will return a url signed that points to : s3://cluster-bucket/tenants-data/{tenant-id}/{objectKey}
     * @param objectKey
     * @param ttlSeconds
     * @returns
     */
    static getDownloadPresignedUrl(context: Context, objectKey: string, fileStorageSetting?: {
        ttlSeconds?: number;
        filename?: string;
    }): Promise<string>;
    /**
     * Save under /cluster-bucket/tenant-data/tenantid/...
     * Description should be a tag
     * Expiration should be a tag (we will manage the cleanup later)
     *
     * e.g. objectKey: '/report-output/myReport.pdf'
     * @param context
     * @param objectKey
     * @param description
     * @param data
     * @param fileTimeToLive
     * @returns
     */
    static createFile(context: Context, objectKey: string, description: string, data: S3BodyPut, fileTimeToLive: FileTimeToLive): Promise<{
        ETag: string;
    }>;
    /**
     * Read result of async upload for the specified contextId
     * This is after virus scan is complete and the file is moved out of the dirty location to the upload location.
     * @param context
     * @param contextId
     * @param contextKind The contextKind determines the folder we read the file from
     * @returns
     */
    static readAsyncUploadResult(context: Context, contextId: string, contextKind: keyof typeof ASYNC_CONTEXT_CONST): Promise<{
        tagData: any;
        body: S3BodyGet;
    } | undefined>;
    /**
     * Read a file from the tenants folder on the clusters S3 bucket
     * /cluster-bucket/tenant-data/tenantid/objectKey
     * e.g. objectKey: '/uploads/myFile.txt'
     * @param context
     * @param objectKey
     * @returns
     */
    static readFile(context: Context, objectKey: string): Promise<{
        tagData: any;
        body: S3BodyGet;
    } | undefined>;
    /**
     * List the file for a given tenant on a given clusters bucket
     * Handle the max file(1000) with a next token (continuationToken)
     * @param context
     * @param continuationToken  to use only if you have more than 1000 values and get the following ones
     * @returns
     */
    static list(tenantDataType: TenantDataType, context: Context, continuationToken?: string): Promise<FileList>;
    static getTenantApps(tenantId: string): Promise<string[] | null>;
    static getTenantAppHelper(): Promise<TenantAppHelper | null>;
    static getEmailDomainOfTenant(tenantId: string): Promise<string | null>;
}
//# sourceMappingURL=infrastructure-helper.d.ts.map