# How to create, update or delete a workflow

## Overview of the available nodes and mutations

Workflows are implemented as 2 nodes:

- `WorkflowDefinition`: contains all the general properties of the workflow (`id`, `name`, `status`, ...),
  and a reference to a `WorkflowDiagram` component.
- `WorkflowDiagram`: contains the `version` and `data` of the diagram.

Workflows can be created, updated, or deleted with the standard `create`, `update` and `delete` mutations of the `WorkflowDefinition` node.

When creating or updating a workflow, the `WorkflowDefinition` payload that you pass to the create or update mutation encapsulates the `WorkflowDiagram` payload of the diagram component.
The actual JSON data of the diagram is in the `data` property of the diagram component.

The process to create or update a workflow is the following:

- call the `getApiDocumentation` and `getWorkflowDocumentation` tools to get all the documentation, if you hadn't called them already.
- generate a `WorkflowDefinition` `create` or `update` JSON payload, using this documentation. This payload will contain a `diagram` JSON object with a `data` object containing the diagram as a JSON string.
- generate a GraphQL request that calls the `create` or `update` mutation with this payload.
- call the `executeGraphql` tool with this GraphQL request.

Deleting a workflow is easier. You just have to pass the `_id` of the workflow to the `delete` mutation.

**Important**:

- Do not try to create, update or delete a workflow diagram via the `WorkflowDiagram` node,
  always use the mutations of the `WorkflowDefinition` node, and pass the diagram data in the `data` property of the `diagram` component.
- This documentation does not describe all the properties of the workflow nodes. To get them, call the `getMetadata` tool before generating the GraphQL query.
- The create, update and delete operations are performed with the _standard_ create, update and delete mutations, as described by the API documentation returned by the `getApiDocumentation` tool.
- Always get the metadata for the `WorkflowDefinition` and the `WorkflowDiagram` nodes with the `getMetadata` tool.

### Example

The following is an example of a GraphQL mutation that creates an empty workflow:

```graphql
mutation {
    xtremWorkflow {
        workflowDefinition {
            create(
                data: {
                    id: "DEMO1"
                    name: "Demo workflow #1"
                    diagram: {
                        data: """
                        {
                            "edges": [],
                            "nodes": []
                        }
                        """
                    }
                }
            ) {
                _id
            }
        }
    }
}
```

Notes:

- For a `create`, you only need to include the mandatory properties in the workflow definition payload: `id`, `name` and `diagram.data`. The other ones will be defaulted.
- For an `update`, you only need to pass the `_id` of the workflow definition and the `diagram.data`.
  You must pass the entire JSON for the new diagram data, though.
- For readability, the data JSON string should be a multiline string enclosed in 3 double quotes, like in the example above.

## Structure of the diagram data

The edges and nodes of the diagram data conforms to the React Flow types, as described in the following React Flow API Reference pages:

- [Edge](https://reactflow.dev/api-reference/types/edge)
- [Node](https://reactflow.dev/api-reference/types/node)

The React Flow node objects have the following properties:

| Name     | Type            | Description                                                                                                                                   |
| -------- | --------------- | --------------------------------------------------------------------------------------------------------------------------------------------- |
| type     | string          | Value is a string in a closed set of _node types_, that classifies the node.                                                                  |
| id       | string          | The node's ID. Must be unique inside a given diagram.                                                                                         |
| data     | object          | Value is an object that contains the configuration of the workflow node. The structure of this configuration object depends on the node type. |
| width    | integer         | The width of the node                                                                                                                         |
| height   | integer         | The height of the node                                                                                                                        |
| origin   | pair of numbers | Must be [0.5, 0]                                                                                                                              |
| position | point           | This position of the node's origin in the diagram as an { x: integer, y: integer } object                                                     |

The following sections document the structure of the `data` configuration object for the different node types that our workflow engine supports.

The node types can be grouped into 2 main categories: triggers and actions.
Triggers nodes start a workflow. They are followed by actions.

The diagram is a direct acyclic graph (DAG), with a single start node, its trigger.

Unless specified otherwise, the input handles of nodes are at the middle of their top edge,
Their output handles are at the middle of their bottom edge.
So diagrams flow vertically, from top to bottom.

## Trigger Nodes

Trigger nodes don't have any input handle.

Trigger nodes have a single output called `out`.

### Entity Created Trigger (type = 'entity-created/\_generic')

The `data` configuration object of an `entity-created` trigger node has the following properties:

| Name           | Type                         | Description                                                                                                                   | Example values                                    |
| -------------- | ---------------------------- | ----------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------- |
| localizedTitle | localized string             | The node title displayed in the graph                                                                                         | { "en": "Customer Created", "fr": "Client Créé" } |
| topic          | string                       | The event topic formatted as {{NodeName}}/created                                                                             | "Customer/created"                                |
| entityName     | string                       | The name of a node of the GraphQL API, prefixed by its package name                                                           | "@sage/xtrem-master-data/Customer"                |
| conditions     | array of workflow conditions | The conditions that determine if the workflow will be started or not. The workflow will only start if all conditions are true | See "Workflow Conditions" section below           |
| stepVariables  | array of workflow variables  | The workflow variables that are used by the trigger.                                                                          | See "Workflow Variables" section below            |

The output variable name is the GraphQL node name in camel case.

The `stepVariables` should always include the output variable, with the path of its `_id` property.
For example, the `data` object of a "customer created" trigger would be:

```json
{
    "topic": "Customer/created",
    "entityName": "@sage/xtrem-master-data/Customer",
    "conditions": [],
    "stepVariables": [
        {
            "title": "Customer / 🆔",
            "path": "customer._id",
            "type": "IntReference",
            "node": "Customer"
        }
    ]
}
```

**Important**: If a workflow should only be started when some conditions are met, the conditions should be placed in the trigger
rather than in a condition node after the trigger, so that the workflow engine does not create a process for a workflow that did not even start.

Notes:

- For the title of `_id` properties, use the `🆔` emoji instead of `_id`.
- A localized string is an object containing strings keyed by their locale name. For example `{ "en": "Customer", "fr": "Client" }`

### Entity Updated Trigger (type = 'entity-updated/\_generic')

The `data` configuration object of an `entity-updated` trigger has the same structure as the configuration object of an `entity-created` trigger.

The conditions of an entity-updated trigger can compare the new and old values of the modified record.
The old value can be obtained by prefixing the path of the new value with `$old.`.

Note that old values are only available for properties that are stored in the table.
They are not available for properties that are computed from other property values.

## Action Nodes

Action nodes have a single input handle called `in`.

Unless specified otherwise, action nodes have a single output handle called `out`.

### Send User Notification Action (type = 'send-user-notification/\_generic')

This action sends a notification to users of the application.
It cannot be used to notify people who do not have access to the application. Use the send email action for this.

The `data` configuration object of a `send-user-notification` action node has the following properties:

| Name                             | Type                        | Description                                         |
| -------------------------------- | --------------------------- | --------------------------------------------------- |
| localizedTitle                   | localized string            | The node title displayed in the graph               |
| localizedNotificationTitle       | localized handlebar string  | The localized titles of the notification.           |
| localizedNotificationDescription | localized handlebar string  | The localized descriptions of the notification.     |
| level                            | NotificationLevel enum      |
| icon                             | IconType enum               |
| shouldDisplayToast               | boolean                     | Should the notification display a toast in the UI?  |
| recipients                       | array of recipient objects  |
| actions                          | array of action objects     |
| stepVariables                    | array of workflow variables | The workflow variables that are used by the action. |

#### Notification Recipients

The recipients are objects with the following properties:

| Name          | Type    | Description                                                                                                           |
| ------------- | ------- | --------------------------------------------------------------------------------------------------------------------- |
| isManuallySet | boolean | controls what the user property contains                                                                              |
| user          | string  | if isManuallySet is true, user contains the email of the recipient, otherwise it contains the path of a user variable |

Example:

```json
{
    "recipients": [
        { "isManuallySet": true, "user": "john.doe@acme.com" },
        { "isManuallySet": false, "user": "customer._createUser._id" }
    ]
}
```

If `isManuallySet` is false, user must be a the path of a variable that references a User.

#### Notification Actions

The actions are objects with the following properties:

| Name           | Type                       | Description                                                         |
| -------------- | -------------------------- | ------------------------------------------------------------------- |
| title          | handlebar string           | The default title of the action                                     |
| localizedTitle | localized handlebar string | The localized title of the action                                   |
| link           | handlebar string           | The action's URL                                                    |
| style          | enum                       | The link style, one of 'primary', 'secondary', 'tertiary' or 'link' |

### Send Email Action (type = 'send-email/\_generic')

The `data` configuration object of a `send-email` action node has the following properties:

| Name             | Type                           | Description                                                                 |
| ---------------- | ------------------------------ | --------------------------------------------------------------------------- |
| localizedTitle   | localized string               | The node title displayed in the graph                                       |
| localizedSubject | localized handlebar string     | The subject of the email                                                    |
| addressList      | array of email address objects | The email address to whom the email will be sent                            |
| attachmentList   | array of attachment objects    | Optional list of email attachements                                         |
| report           | string                         | The name of the report for the email body.                                  |
| reportParameters | array of workflow parameters   | The parameters passed to the report (see Workflow Parameters section below) |
| from             | optional string                | The sender's email address for the outgoing email.                          |
| replyTo          | optional string                | Optional reply to address                                                   |
| stepVariables    | array of workflow variables    | The workflow variables that are used by the action.                         |

The body of the email is formatted from a report.
You should query the `Report` node to find a valid report name and its parameter definitions (called `variables` in the `Report` node).
If you cannot find any suitable report, you should cancel the generation of the workflow and inform the user.

#### Email Address

The email addresses are objects with the following properties:

| Name        | Type    | Description                              |
| ----------- | ------- | ---------------------------------------- |
| addressType | enum    | 'to', 'cc' or 'bcc'                      |
| isVariable  | boolean | Does the value comes from a variable?    |
| variable    | string  | The variable path if isVariable is true  |
| value       | string  | The email address if isVariable is false |

#### Email Attachement

The email attachments are objects with the following properties:

| Name           | Type             | Description                                                                                      |
| -------------- | ---------------- | ------------------------------------------------------------------------------------------------ |
| variable       | string           | The path of the variable that references the node instance containing the attachments            |
| tags           | array of strings | A list of tag names. Only the attachments that match these tags will be attached to the email |
| failIfNotFound | boolean          | If true, the sendMail action will fail if the attachment is not found.                           |

### Condition Action (type = 'condition/\_generic')

Condition actions have one input handle (`in`) and two output handles:

- `out-true` on the left side, for the actions that are executed when the condition is true.
- `out-false` on the right side, for the actions that are executed when the condition is false.

If there are no actions to perform when the condition is false, you can leave the `out-false` branch empty.
If there are no actions to perform when the condition is true, you can leave the `out-true` branch empty.

The `data` configuration object of a `condition` action node has the following properties:

| Name           | Type                         | Description                                         |
| -------------- | ---------------------------- | --------------------------------------------------- |
| localizedTitle | localized string             | The node title displayed in the graph               |
| conditions     | array of workflow conditions | See Workflow Conditions section below.              |
| stepVariables  | array of workflow variables  | The workflow variables that are used by the action. |

### Mutation Action (type = 'mutation/\_generic')

A `mutation` action executes a GraphQL mutation.

The `data` configuration object of a `mutation` action node has the following properties:

| Name               | Type                                 | Description                                               |
| ------------------ | ------------------------------------ | --------------------------------------------------------- |
| localizedTitle     | localized string                     | The node title displayed in the graph                     |
| mutationNaturalKey | string                               | '{{NodeName}}\|{{mutationName}}\|{{action}}'. More below. |
| mutationArguments  | array of workflow mutation arguments |
| selector           | string                               | The GraphQL selector for the mutation's result            |
| stepVariables      | array of workflow variables          | The workflow variables that are used by the action.       |

The mutationNaturalKey concatenates the node name (PascalCase), the mutationName (camelCase) and an optional action, separated by a '|' character.
The optional action is 'start' if the mutation is an async mutation, empty otherwise.

The selector is a GraphQL selector, for example `{ _id }` if the mutation returns a reference.

If the workflow has workflow variables that select other properties than `_id` from a reference returned by the mutation, you should not include them in the selector.
The workflow engine will automatically collect them and add them to the selector.

#### Mutation Argument

The mutation arguments are objects with the following properties:

| Name        | Type    | Description                                                        |
| ----------- | ------- | ------------------------------------------------------------------ |
| name        | string  | The name of the parameter                                          |
| type        | string  | One of the property types returned by the getMetadata tool.        |
| isMandatory | boolean | Is the value mandatory?                                            |
| node        | string  | If `type` is a reference, the node name                            |
| origin      | enum    | Where to get the value from: 'manual' or 'fromVariable'            |
| value       | any     | The variable path if origin is 'fromVariable', otherwise the value |

## Workflow Variables

A workflow variable is JSON object with the following properties:

| Name       | Type               |
| ---------- | ------------------ |
| title      | string             |
| path       | string             |
| type       | variable type enum |
| node       | string             |
| enumType   | string             |
| enumValues | array of strings   |
| isCustom   | boolean            |

The trigger node defines a root variable that all the other workflow nodes can use.

Action nodes may define an additional variable in which the result of the action will be stored.

A given workflow node may use any of the variables introduced by its upstream workflow nodes.
It may also access properties of such variables, at any depth.
For example, an action may use a variable with path customer.billToAddress.country.name if customer is a root variable.

Variables may be embedded into configuration properties (the ones typed as 'handlebar string').
For example, you may send an email subject like "Your order {{salesOrder.number}} has been delayed."

Each workflow node (trigger or action) must track all the variables used by its configuration (in handlebar strings, but also in conditions, etc.) and must include
them into the `stepVariables` array of its configuration.

### Variable types (only for workflow diagram `data` properties)

The workflow uses a different enum for variable types than what the getMetadata tool returns for properties.
Here is a mapping table to convert between them:

| Diagram variable type | Metadata property type | Comment                                                                         |
| --------------------- | ---------------------- | ------------------------------------------------------------------------------- |
| Boolean               | boolean                |
| String                | string                 |
| Int                   | integer                |
| Decimal               | decimal                |
| Date                  | date                   |
| datetime              | datetime               | Lowercase d in variable type is not a typo                                      |
| Enum                  | enum                   |
| Float                 | float                  |
| Id                    | string                 | Deprecated. use IntReference instead                                            |
| IntOrString           | string                 | Deprecated. use IntReference instead                                            |
| IntReference          | reference              | For `_id` properties and reference properties that contain the \_id of a record |
| \_InputTextStream     | textStream             |
| \_OutputTextStream    | textStream             |
| Json                  | json                   |

**Important**: Do not guess the `path` or `type` of a variable.
Call the `getMetadata` tool with the GraphQL node names to get the names and types of the properties that you need, and use the table above to convert the types.

## Workflow Parameters

Workflow parameters are configuration object for parameters passed to various operations, for example to render a report.

Workflow parameters are objects with the following properties:

| Name       | Type    | Description                                                                                  |
| ---------- | ------- | -------------------------------------------------------------------------------------------- |
| name       | string  | Name of the parameter                                                                        |
| isVariable | boolean | Does the value come from a variable?                                                         |
| value      | any     | The variable path if isVariable is true, otherwise the value (type depends on the parameter) |

## Workflow Conditions

Workflow conditions are objects with the following properties:

| Name         | Type    | Description                                                                    |
| ------------ | ------- | ------------------------------------------------------------------------------ |
| path         | string  | LHS of the condition: the path of the workflow variable being tested.          |
| operator     | enum    | See the condition operators section below                                      |
| useParameter | boolean | Does the value come from a variable? (naming is not very consistent here)      |
| value        | any     | RHS of the condition: a variable path if isVariable is true, otherwise a value |

### Condition operators

The `operator` can take one of the following values:

| Operators                                                  | LHS types            | RHS types            |
| ---------------------------------------------------------- | -------------------- | -------------------- |
| equals, notEqual                                           | All types            | Same as LHS          |
| lessThan, lessThanOrEqual, greaterThan, greaterThanOrEqual | All comparable types | Same as LHS          |
| matches, startsWith, endsWith                              | string               | string               |
| contains, notContains                                      | string, range of T   | string, T            |
| inRange                                                    | string, T            | string, range of T   |
| empty, notEmpty                                            | string, range        | N/A                  |
| set, multiNotEqual                                         | enum                 | array of enum values |

### Examples of conditions

| Condition                                    | JSON                                                                                                                      |
| -------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------- |
| customer name contains 'abc'                 | { "path": "customer.name", "operator": "contains", "useParameter": false, "value": "abc" }                                |
| process is 'waiting' or 'running'            | { "path": "process.status", "operator": "set", "useParameter": false, "value": ["waiting", "running"] }                   |
| custom's financial and stock sites are equal | { "path": "customer.financialSite.\_id", "operator": "equals", "useParameter": true, "value": "customer.stockSite.\_id" } |

## Important directives

Do not try to generate JSON for diagram nodes that are not documented above.
Simply bail out and tell the user that you cannot complete the request.
