"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.X3PackageGenerator = exports.logger = void 0;
const sync_1 = __importDefault(require("@prettier/sync"));
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const fs = __importStar(require("fs"));
const fsExtra = __importStar(require("fs-extra"));
const _ = __importStar(require("lodash"));
const os_1 = require("os");
const fsp = __importStar(require("path"));
const ts = __importStar(require("typescript"));
const dependencies_1 = require("../templates/dependencies");
const main_package_json_1 = require("../templates/main-package-json");
const tsconfig_artifacts_template_1 = require("../templates/tsconfig-artifacts-template");
const tsconfig_template_1 = require("../templates/tsconfig-template");
const x3_dictionary_helper_1 = require("./x3-dictionary-helper");
const x3_dictionary_interfaces_1 = require("./x3-dictionary-interfaces");
const x3_enum_generator_1 = require("./x3-enum-generator");
const x3_local_menu_dictionary_helper_1 = require("./x3-local-menu-dictionary-helper");
const x3_node_dictionary_helper_1 = require("./x3-node-dictionary-helper");
const x3_node_generator_1 = require("./x3-node-generator");
const x3_node_generator_helper_1 = require("./x3-node-generator-helper");
const x3_package_dictionary_helper_1 = require("./x3-package-dictionary-helper");
const x3_service_options_generator_1 = require("./x3-service-options-generator");
exports.logger = xtrem_log_1.Logger.getLogger(__filename, 'xtrem-package-generator');
class X3PackageGenerator {
    static { this.newline = ts.factory.createIdentifier('\n'); }
    static { this.devDependenciesNames = [
        '@sage/eslint-config-xtrem',
        '@types/chai',
        '@types/chai-as-promised',
        '@types/mocha',
        '@types/node',
        'chai',
        'chai-as-promised',
        'tsconfig-paths',
        'c8',
        'mocha',
        'typescript',
    ]; }
    constructor(dir, packageName, connPool, serviceName, x3FolderName) {
        this.dir = dir;
        this.packageName = packageName;
        this.connPool = connPool;
        this.serviceName = serviceName;
        this.x3FolderName = x3FolderName;
        /** list of package imports, by entity (node, enum, etc.) */
        this.imports = {};
        this.nodeGenerator = new x3_node_generator_1.X3NodeGenerator(this, this.connPool, this.x3FolderName);
    }
    static { this.allNodeObjects = []; }
    static { this.allNodesPerPackage = (0, xtrem_shared_1.createDictionary)(); }
    static get devDependencies() {
        return this.getPackageDependencies(X3PackageGenerator.devDependenciesNames);
    }
    static clearCache() {
        X3PackageGenerator.packages = undefined;
        X3PackageGenerator.allActivityCodes = undefined;
        X3PackageGenerator.allNodeObjects = [];
        X3PackageGenerator.allNodesPerPackage = (0, xtrem_shared_1.createDictionary)();
        X3PackageGenerator.localPackages = undefined;
        x3_dictionary_helper_1.X3DictionaryHelper.clearCache();
        x3_local_menu_dictionary_helper_1.X3LocalMenuDictionaryHelper.clearCache();
        x3_node_dictionary_helper_1.X3NodeDictionaryHelper.clearCache();
        x3_node_generator_helper_1.X3NodeGeneratorHelper.clearCache();
        x3_package_dictionary_helper_1.X3PackageDictionaryHelper.clearCache();
        x3_service_options_generator_1.X3ServiceOptionsGenerator.clearCache();
    }
    static async preLoadPackageData(connPool, x3FolderName, serviceName) {
        exports.logger.info('Preloading package metadata');
        this.packages = await X3PackageGenerator.getPackages(connPool, x3FolderName, serviceName, {});
        // Load enum ranges, local menues in advance, the first package generator instance will load it into memory
        exports.logger.info('Preloading local menu ranges');
        await x3_local_menu_dictionary_helper_1.X3LocalMenuDictionaryHelper.getCustomLocalMenuRanges(connPool, x3FolderName);
        exports.logger.info('Preloading local menus');
        await x3_local_menu_dictionary_helper_1.X3LocalMenuDictionaryHelper.loadLocalMenuHeaders(connPool, x3FolderName);
        exports.logger.info('Preloading node data');
        this.allNodeObjects = await x3_node_generator_helper_1.X3NodeGeneratorHelper.loadNodeObjects(connPool, x3FolderName);
        this.packages.forEach(pack => {
            exports.logger.info(`Preloading nodes for ${pack.name}`);
            this.allNodesPerPackage[pack.name] = this.allNodeObjects.filter(nodeData => nodeData.package === pack.name);
        });
        exports.logger.info('Preloading activity codes');
        this.allActivityCodes = await x3_service_options_generator_1.X3ServiceOptionsGenerator.loadActivityCodes(connPool, x3FolderName, serviceName);
        exports.logger.info('Finish preloading package metadata');
    }
    /**
     * Initialize package generator
     */
    async init() {
        // TODO: This will be an async method when we start reading the package definition from APACKAGE
        const packages = await X3PackageGenerator.getPackages(this.connPool, this.x3FolderName, this.serviceName, {});
        const pack = packages.find(p => p.name === this.packageName);
        if (!pack)
            throw new Error(`Package ${this.packageName} not registered in metadata.`);
        this._metadata = pack;
    }
    get metadata() {
        if (!this._metadata)
            throw new Error(`Package generator for ${this.packageName} is not initialized`);
        return this._metadata;
    }
    static readPackageJson(packagePath) {
        const filePath = fsp.join(packagePath, x3_dictionary_interfaces_1.packageJsonFileName);
        return JSON.parse(fs.readFileSync(filePath).toString());
    }
    /**
     * Format typescript node
     * @param node
     * @param sourcePath
     * @returns
     */
    static formatNodes(node, path, sourcePath = '<none>') {
        if (Array.isArray(node))
            return node.map(n => this.formatNodes(n, path, sourcePath)).join('');
        const resultFile = ts.createSourceFile(sourcePath, '', ts.ScriptTarget.Latest, false, ts.ScriptKind.TS);
        const printer = ts.createPrinter({ newLine: ts.NewLineKind.LineFeed });
        try {
            return printer.printNode(ts.EmitHint.Unspecified, node, resultFile);
        }
        catch (error) {
            const kind = ts.SyntaxKind[node.kind];
            exports.logger.error(`${path}: Error printing TS node ${kind}: \n ${error.stack}`);
            throw error;
        }
    }
    /**
     * Prettyify typescript file content
     * @param nodes
     * @param sourcePath
     * @returns
     */
    static prettyifyTypescript(nodes, sourcePath = '<none>', path = '') {
        let preformatSource = '';
        const prettierConfig = {
            parser: 'typescript',
            singleQuote: true,
            printWidth: 120,
            tabWidth: 4,
            useTabs: false,
            semi: true,
            trailingComma: 'all',
            arrowParens: 'avoid',
            endOfLine: 'lf',
            filepath: 'dummy.ts',
        };
        if (typeof nodes !== 'string') {
            preformatSource = X3PackageGenerator.formatNodes(nodes, path, sourcePath);
        }
        else {
            preformatSource = nodes;
        }
        preformatSource = preformatSource.replace(/([^\n])(\n *@)/g, '$1\n$2'); // ensure a newline is inserted before a decorator;
        return sync_1.default.format(preformatSource, prettierConfig);
    }
    /**
     * Prettyify JSON file content
     * @param jsonSource
     * @returns
     */
    static prettyifyJson(jsonSource) {
        const prettierConfig = {
            singleQuote: true,
            printWidth: 120,
            tabWidth: 4,
            useTabs: false,
            semi: true,
            trailingComma: 'all',
            endOfLine: 'lf',
            parser: 'json',
        };
        const source = sync_1.default.format(jsonSource, prettierConfig);
        // Windows environment : the code is generated using '\n' as newlines. To avoid a lot of
        // non meaningfull git differences, we replace these '\n' with '\r\n'.
        // When pushing to git, git will store the text with '\n'
        return os_1.EOL === '\r\n' ? source.replace(/\n/g, '\r\n') : source;
    }
    /**
     * Add an import to the list of imports this will be used later determined the dependencies of the package
     * @param type
     * @param name
     * @param importEntry
     */
    addImport(type, name, importEntry) {
        const key = `${type}~${name}`;
        if (!this.imports[key])
            this.imports[key] = {};
        const importMap = this.imports[key];
        if (importMap) {
            if (!importMap[importEntry.packageName]) {
                importMap[importEntry.packageName] = [importEntry.value];
            }
            else {
                const entries = importMap[importEntry.packageName] || [];
                if (!entries.includes(importEntry.value)) {
                    entries.push(importEntry.value);
                    importMap[importEntry.packageName] = entries;
                }
            }
            this.imports[key] = importMap;
        }
    }
    static sortImports(importsMap) {
        return Object.entries(importsMap)
            .slice()
            .sort((a, b) => {
            const keyA = a[0];
            const keyB = b[0];
            if (keyA === '..')
                return 1;
            if (keyB === '..')
                return -1;
            return keyA.localeCompare(keyB);
        });
    }
    /**
     * Get the package alias for the import * statement
     * @param packageName
     * @returns
     */
    static getPackageAlias(packageName) {
        return _.camelCase(packageName);
    }
    getDependencies() {
        const deps = [];
        Object.values(this.imports).forEach(importEntry => {
            deps.push(...Object.keys(importEntry).filter(dep => dep !== '..'));
        });
        return _.uniq(deps);
    }
    static getInternalDependencies(version, isCustom, serviceName) {
        // TODO: today all our versions are aligned but in the future we may have release cycle per product
        // therefore the platform package versions maybe different from the service packages and we will need
        // to deal with this.
        const packs = [
            '@sage/xtrem-async-helper',
            '@sage/xtrem-core',
            '@sage/xtrem-date-time',
            '@sage/xtrem-decimal',
            '@sage/xtrem-ui',
            '@sage/xtrem-shared',
            '@sage/xtrem-x3-gateway',
            '@sage/xtrem-x3-syracuse',
            '@sage/xtrem-x3-system-utils',
        ];
        if (isCustom) {
            packs.push('@sage/eslint-config-xtrem');
            packs.push('@sage/eslint-plugin-xtrem');
            packs.push(`@sage/${serviceName}-main`);
            return packs.reduce((r, k) => {
                const packVersion = this.devDependencies[k]?.replace(/[^0-9.]/g, '') || version;
                r[k] = `file:../../sage-packages/${k.replace('@sage/', 'sage-')}-${packVersion}.tgz`;
                return r;
            }, {});
        }
        return packs.reduce((r, k) => {
            r[k] = 'workspace:*';
            return r;
        }, {});
    }
    updatePackageDependencies() {
        const packageJsonPath = fsp.join(this.dir, x3_dictionary_interfaces_1.packageJsonFileName);
        if (!fs.existsSync(packageJsonPath)) {
            exports.logger.error(`${this.packageName}: package.json is missing in ${this.dir}`);
        }
        else {
            // add/update relevant dependencies
            const packageJsonContent = X3PackageGenerator.readPackageJson(this.dir);
            const deps = this.getDependencies();
            const sageDependencies = deps.filter(dependency => dependency.startsWith('@sage/'));
            const externalDependencies = deps.filter(dependency => !dependency.startsWith('@sage/'));
            const { version } = packageJsonContent;
            const internalDependencies = Object.keys(X3PackageGenerator.getInternalDependencies(version, !!this._metadata.isCustom, this.serviceName));
            const currentDependencies = Object.keys(!this.metadata.isCustom
                ? packageJsonContent.dependencies || {}
                : packageJsonContent.peerDependencies || {}).filter(dependency => !dependency.startsWith('@sage/'));
            if (!this.metadata.isCustom) {
                currentDependencies.push(...[...internalDependencies, ...deps]);
            }
            else {
                packageJsonContent.peerDependencies = [...internalDependencies, ...sageDependencies].reduce((r, k) => {
                    r[k] = `file:../../sage-packages/${k.replace('@sage/', 'sage-')}-${version}.tgz`;
                    return r;
                }, {});
                // Manage Sage API dependencies
                sageDependencies.forEach(sageDep => {
                    if (X3PackageGenerator.packages?.find(pack => pack.name === sageDep)) {
                        packageJsonContent.devDependencies[`${sageDep}-api`] =
                            `file:../../sage-packages/${sageDep.replace('@sage/', 'sage-')}-api-${version}.tgz`;
                    }
                });
                currentDependencies.push(...externalDependencies);
            }
            packageJsonContent.dependencies = packageJsonContent.dependencies || {};
            const depVersion = this.metadata.isCustom ? version : 'workspace:*';
            _.uniq(currentDependencies).forEach(dep => {
                packageJsonContent.dependencies[dep] = depVersion;
            });
            // sort dependencies
            packageJsonContent.dependencies = _.fromPairs(_.sortBy(_.toPairs(packageJsonContent.dependencies), 0));
            exports.logger.info(`${this.packageName}: Writing updated package.json to ${packageJsonPath}`);
            exports.logger.info(`${this.packageName}: Dependencies ${JSON.stringify(packageJsonContent.dependencies)}`);
            if (this.metadata.isCustom)
                exports.logger.info(`${this.packageName}: Peer dependencies \n ${JSON.stringify(packageJsonContent.peerDependencies)}`);
            fs.writeFileSync(packageJsonPath, X3PackageGenerator.prettyifyJson(JSON.stringify(packageJsonContent, null, 4)));
        }
    }
    static getFilesInPath(dirPath) {
        return fs.readdirSync(dirPath);
    }
    // Recursive function to traverse the file tree
    static traversePackageTree(rootPath) {
        let results = {};
        const files = this.getFilesInPath(rootPath);
        files.forEach(file => {
            const filePath = fsp.join(rootPath, file);
            const stat = fs.lstatSync(filePath);
            if (stat && stat.isDirectory() && !['node_modules', 'api', 'logs', 'build'].includes(file)) {
                results = { ...results, ...this.traversePackageTree(filePath) };
            }
            else if (file === x3_dictionary_interfaces_1.packageJsonFileName) {
                const content = X3PackageGenerator.readPackageJson(rootPath);
                const name = content.name;
                results[name] = rootPath;
            }
        });
        return results;
    }
    get rootPath() {
        return this.metadata.isCustom ? fsp.resolve(this.dir, '../..') : fsp.resolve(this.dir, '../../..');
    }
    getLocalPackages() {
        if (X3PackageGenerator.localPackages)
            return X3PackageGenerator.localPackages;
        X3PackageGenerator.localPackages = X3PackageGenerator.traversePackageTree(this.rootPath);
        return X3PackageGenerator.localPackages;
    }
    updatePackageReferences() {
        const tsConfigJsonPath = fsp.join(this.dir, 'tsconfig.json');
        const localPaths = this.getLocalPackages();
        if (!fs.existsSync(tsConfigJsonPath)) {
            exports.logger.warn(`Creating missing tsconfig.json to ${tsConfigJsonPath}`);
            fs.writeFileSync(tsConfigJsonPath, X3PackageGenerator.prettyifyJson(JSON.stringify(tsconfig_template_1.tsconfigTemplate)));
        }
        // add/update relevant dependencies
        const tsConfigJsonContent = JSON.parse(fs.readFileSync(tsConfigJsonPath, 'utf-8'));
        const deps = this.getDependencies();
        const sageDependencies = deps.filter(dependency => dependency.startsWith('@sage/'));
        const externalDependencies = deps.filter(dependency => !dependency.startsWith('@sage/'));
        const paths = [
            fsp.join(this.dir, '../../../platform/front-end/xtrem-ui'),
            fsp.join(this.dir, '../../../x3-services/platform/xtrem-x3-system-utils'),
            fsp.join(this.dir, '../../../x3-services/platform/xtrem-x3-syracuse'),
            fsp.join(this.dir, '../../../x3-services/platform/xtrem-x3-gateway'),
        ];
        const symLinks = {};
        const addApiToSymlink = (packageName, path) => {
            const api = `${packageName}-api`;
            const apiPath = fsp.join(path, 'api');
            // create api directory for symlink
            if (!fs.existsSync(apiPath))
                fs.mkdirSync(apiPath, { recursive: true });
            symLinks[api] = apiPath;
        };
        const getDependencyPath = (dep) => {
            exports.logger.info(`${this.packageName}: Resolving ${dep} for tsconfig.json, paths=${paths}`);
            const resolvedPath = localPaths[dep];
            if (!resolvedPath) {
                exports.logger.warn(`${this.packageName}: Cannot determine path for ${dep}`);
                return undefined;
            }
            symLinks[dep] = resolvedPath;
            addApiToSymlink(dep, resolvedPath);
            return { path: fsp.relative(this.dir, resolvedPath) };
        };
        if (this.metadata.isCustom) {
            tsConfigJsonContent.references = externalDependencies.map(getDependencyPath).filter(ref => ref);
            addApiToSymlink(this.packageName, this.dir);
        }
        else {
            const internalDependencies = Object.keys(X3PackageGenerator.getInternalDependencies('', false, this.serviceName));
            tsConfigJsonContent.references = _.uniq([...internalDependencies, ...sageDependencies])
                .map(getDependencyPath)
                .filter(ref => ref);
        }
        exports.logger.info(`${this.packageName}: Writing updated tsconfig.json to ${tsConfigJsonPath}`);
        exports.logger.info(`${this.packageName}: references \n ${JSON.stringify(tsConfigJsonContent.references)}`);
        fs.writeFileSync(tsConfigJsonPath, X3PackageGenerator.prettyifyJson(JSON.stringify(tsConfigJsonContent)));
        if (this._metadata.isCustom && Object.keys(symLinks).length > 0) {
            Object.entries(symLinks).forEach(([key, path]) => {
                const target = fsp.join(this.dir, 'node_modules', key);
                if (!fs.existsSync(target)) {
                    fsExtra.createSymlinkSync(path, target, 'junction');
                }
                else {
                    exports.logger.warn(`${this.packageName}: ${target} already exists, skipping symlink creation`);
                }
            });
        }
    }
    static getPackageDependencies(deps) {
        return deps.reduce((result, dep) => {
            if (dependencies_1.dependencies[dep]) {
                result[dep] = dependencies_1.dependencies[dep];
            }
            return result;
        }, {});
    }
    /**
     * Build the package.json structure
     * @param packageName
     * @param version
     * @param isCustom
     * @returns
     */
    static getPackageJsonContent(packageJsonAtriibuteValues) {
        const { packageName, version, isCustom, isHidden } = packageJsonAtriibuteValues;
        const getRootDependencies = (deps) => this.getPackageDependencies(deps);
        const packageJsonContent = {
            name: packageName,
            description: _.startCase(packageName).replace(' X 3 ', ' X3 '),
            version,
            author: packageName.split('/')[0].replace('@', ''),
            license: 'UNLICENSED',
            xtrem: {
                packageName,
                isHidden,
            },
            keywords: isCustom
                ? ['xtrem-x3-services-specific-application-package']
                : ['xtrem-x3-services-application-package'],
            main: 'build/index.js',
            files: [
                'build',
                'api',
                'lib/pages',
                'lib/page-extensions',
                'lib/widgets',
                'lib/page-fragments',
                'lib/stickers',
                'lib/client-functions',
                'lib/shared-functions',
                'README.md',
                'CHANGELOG.md',
            ],
            typings: 'build/package-definition.d.ts',
            devDependencies: this.devDependencies,
            scripts: {
                'generate-package': `xtrem x3-dev generate-package --service-name ${packageJsonAtriibuteValues.serviceName}`,
                'generate-migrate-translations': 'xtrem x3-dev generate-migrate-translations',
                'generate-translations': `xtrem x3-dev generate-translations --service-name ${packageJsonAtriibuteValues.serviceName}`,
                'build:api': 'xtrem build --only-api-client',
                clean: 'rm -rf build',
                start: 'xtrem start',
            },
            c8: {
                cache: false,
                all: true,
                extension: ['.ts', '.tsx'],
                sourceMap: true,
                instrument: true,
                reporter: ['text-summary', 'clover', 'json', 'lcov'],
                include: ['lib/**/*.ts'],
                exclude: ['test/**/*', 'data/**/*'],
            },
        };
        // Standard packages
        if (!isCustom) {
            packageJsonContent.publishConfig = {
                registry: 'https://pkgs.dev.azure.com/Sage-LiveServices/_packaging/Sage-ERP/npm/registry/',
            };
            packageJsonContent.repository = {
                type: 'git',
                url: 'git://github.com/Sage-ERP-X3/xtrem.git',
            };
            packageJsonContent.dependencies = this.getInternalDependencies(version, isCustom, packageJsonAtriibuteValues.serviceName);
            // eslint, xtrem-cli and xtrem-cli-bundle-x3 is a devDependency in standard packages
            packageJsonContent.devDependencies['@sage/eslint-plugin-xtrem'] = 'workspace:*';
            packageJsonContent.devDependencies['@sage/xtrem-cli'] = 'workspace:*';
            packageJsonContent.devDependencies['@sage/xtrem-cli-bundle-x3'] = 'workspace:*';
            packageJsonContent.scripts.build = 'xtrem compile';
            packageJsonContent.scripts.lint = 'xtrem lint';
            packageJsonContent.scripts.test = 'xtrem test --unit --graphql';
            packageJsonContent.scripts['test:ci'] = 'xtrem test --unit --ci';
            packageJsonContent.scripts['build:binary'] = 'pnpm run clean && pnpm run build';
        }
        else {
            // The package is a specific or add-on package
            packageJsonContent.peerDependencies = this.getInternalDependencies(version, isCustom, packageJsonAtriibuteValues.serviceName);
            packageJsonContent.devDependencies[`${packageName}-api`] = `^${version}`;
            packageJsonContent.scripts.build = 'xtrem compile';
            packageJsonContent.scripts.lint = 'xtrem lint';
            packageJsonContent.scripts.test = 'xtrem test --unit --graphql';
            packageJsonContent.scripts['test:ci'] = 'xtrem test --unit --ci';
            // Add required eslint plugin dependencies
            packageJsonContent.dependencies = {
                ...packageJsonContent.dependencies,
                ...getRootDependencies([
                    '@sage/eslint-config-xtrem',
                    'eslint-config-prettier',
                    'eslint-plugin-import',
                    'eslint-plugin-jsx-a11y',
                    'eslint-plugin-mocha',
                    'eslint-plugin-react',
                    'eslint-plugin-react-hooks',
                    'eslint-plugin-unicorn',
                ]),
            };
        }
        // xtrem needs to be the last script
        packageJsonContent.scripts.xtrem = 'xtrem';
        return packageJsonContent;
    }
    static updatePackageJsonFile(pathToPackage, packageDetails) {
        const { isCustom, serviceName } = packageDetails;
        const packageJsonPath = fsp.join(pathToPackage, x3_dictionary_interfaces_1.packageJsonFileName);
        let packageJsonContent;
        if (!fs.existsSync(packageJsonPath)) {
            packageJsonContent = this.getPackageJsonContent({
                packageName: packageDetails.name,
                version: packageDetails.version,
                serviceName,
                isCustom,
                isHidden: packageDetails.isHidden,
            });
            packageJsonContent.version = packageDetails.version;
        }
        else {
            // add/update relevant dependencies
            packageJsonContent = this.readPackageJson(pathToPackage);
            const internalDependencies = this.getInternalDependencies(packageDetails.version, isCustom, serviceName);
            if (isCustom) {
                packageJsonContent.peerDependencies = {
                    ...(packageJsonContent.peerDependencies || {}),
                    ...internalDependencies,
                };
                packageJsonContent.devDependencies[`${packageDetails.name}-api`] = `^${packageDetails.version}`;
            }
            else {
                packageJsonContent.dependencies = {
                    ...(packageJsonContent.dependencies || {}),
                    ...internalDependencies,
                };
            }
            packageJsonContent.xtrem = packageJsonContent.xtrem || {};
            packageJsonContent.xtrem.packageName = packageDetails.name;
            packageJsonContent.xtrem.isHidden = packageDetails.isHidden;
            packageJsonContent.version = packageDetails.version;
            if (packageJsonContent.scripts['generate-package'] &&
                !packageJsonContent.scripts['generate-translations']) {
                const tmpScripts = {};
                Object.keys(packageJsonContent.scripts).forEach(key => {
                    tmpScripts[key] = packageJsonContent.scripts[key];
                    if (key === 'generate-package' && !isCustom) {
                        tmpScripts['generate-translations'] = 'xtrem x3-dev generate-translations';
                    }
                });
                packageJsonContent.scripts = tmpScripts;
            }
            if (packageJsonContent.scripts['generate-package'] &&
                !packageJsonContent.scripts['generate-migrate-translations']) {
                const tmpScripts = {};
                Object.keys(packageJsonContent.scripts).forEach(key => {
                    tmpScripts[key] = packageJsonContent.scripts[key];
                    if (key === 'generate-package') {
                        tmpScripts['generate-migrate-translations'] = 'xtrem x3-dev generate-migrate-translations';
                    }
                });
                packageJsonContent.scripts = tmpScripts;
            }
        }
        exports.logger.info(`${packageDetails.name}: Writing package.json to ${packageJsonPath}`);
        fs.writeFileSync(packageJsonPath, this.prettyifyJson(JSON.stringify(packageJsonContent)));
    }
    static writePackageFile(packageName, filePath, content) {
        if (fs.existsSync(filePath)) {
            exports.logger.info(`${packageName}: Deleting ${filePath}`);
            fs.unlinkSync(filePath);
        }
        exports.logger.info(`${packageName}: Writing ${filePath}`);
        fs.writeFileSync(filePath, content);
    }
    /**
     * @param rootPackagePath
     * @param packageParentFolder parent folder of package, standard packages => x3-services/packages and specific package => add-ons
     * @param packageDetails
     */
    static ensurePackageStructureExists(packageParentFolder, packageDetails) {
        if (!packageDetails.isCustom &&
            !packageDetails.name.split('/')[1].startsWith(`${packageDetails.serviceName}-`)) {
            exports.logger.verbose(() => `${packageDetails.name}: package will not be generated as it does not belong to the service ${packageDetails.serviceName}`);
            return;
        }
        const pathToPackage = fsp.join(packageParentFolder, fsp.basename(packageDetails.name));
        if (!fs.existsSync(pathToPackage)) {
            exports.logger.info(`${packageDetails.name}: Creating directory ${pathToPackage}`);
            fs.mkdirSync(pathToPackage, { recursive: true });
        }
        // check package.json
        this.updatePackageJsonFile(pathToPackage, packageDetails);
        const eslintConfigPath = fsp.join(pathToPackage, 'eslint.config.mjs');
        if (!fs.existsSync(eslintConfigPath)) {
            const eslintConfigContent = this.prettyifyTypescript(fs.readFileSync(fsp.join(__dirname, '../templates/eslint.config.mjs'), 'utf-8'));
            this.writePackageFile(packageDetails.name, eslintConfigPath, eslintConfigContent);
        }
        const eslintFilenameConfigPath = fsp.join(pathToPackage, 'eslint-filename.config.mjs');
        const eslintFilenameConfigContent = this.prettyifyTypescript(fs.readFileSync(fsp.join(__dirname, '../templates/eslint-filename.config.mjs'), 'utf-8'));
        this.writePackageFile(packageDetails.name, eslintFilenameConfigPath, eslintFilenameConfigContent);
        const tsConfigPath = fsp.join(pathToPackage, 'tsconfig.json');
        if (!fs.existsSync(tsConfigPath)) {
            exports.logger.info(`${packageDetails.name}: Writing tsconfig.json to ${tsConfigPath}`);
            fs.writeFileSync(tsConfigPath, this.prettyifyJson(JSON.stringify(tsconfig_template_1.tsconfigTemplate)));
        }
        const tsConfigArtifactsPath = fsp.join(pathToPackage, 'tsconfig-artifacts.json');
        if (!fs.existsSync(tsConfigArtifactsPath)) {
            exports.logger.info(`${packageDetails.name}: Writing tsconfig-artifacts.json to ${tsConfigArtifactsPath}`);
            fs.writeFileSync(tsConfigArtifactsPath, this.prettyifyJson(JSON.stringify(tsconfig_artifacts_template_1.tsconfigArtifactTemplate)));
        }
        const pathToPackageLib = fsp.join(pathToPackage, 'lib');
        if (!fs.existsSync(pathToPackageLib)) {
            exports.logger.info(`${packageDetails.name}: Creating directory ${pathToPackageLib}`);
            fs.mkdirSync(pathToPackageLib, { recursive: true });
        }
        const pathToRootIndex = fsp.join(pathToPackage, 'index.ts');
        if (!fs.existsSync(pathToRootIndex)) {
            exports.logger.info(`${packageDetails.name}: Creating ${pathToRootIndex}`);
            fs.writeFileSync(pathToRootIndex, X3PackageGenerator.prettyifyTypescript("export * from './lib/index';"));
        }
        const writeDummyIndex = (apiComponent) => {
            const dummyContent = 'export type Dummy = void;';
            const pathToApiComponent = fsp.join(pathToPackageLib, apiComponent);
            const pathToApiComponentIndex = fsp.join(pathToApiComponent, 'index.ts');
            if (!fs.existsSync(pathToApiComponent)) {
                exports.logger.info(`${packageDetails.name}: Creating directory ${pathToApiComponent}`);
                fs.mkdirSync(pathToApiComponent, { recursive: true });
            }
            if (!fs.existsSync(pathToApiComponentIndex)) {
                exports.logger.info(`${packageDetails.name}: Creating ${pathToApiComponentIndex}`);
                fs.writeFileSync(pathToApiComponentIndex, X3PackageGenerator.prettyifyTypescript(dummyContent));
            }
        };
        const apiComponents = ['nodes', 'node-extensions', 'enums', 'service-options'];
        apiComponents.forEach(writeDummyIndex);
        const exportComponents = [...apiComponents];
        fs.readdirSync(pathToPackageLib)
            .filter((f) => fs.statSync(fsp.join(pathToPackageLib, f)).isDirectory())
            .forEach((f) => {
            const base = fsp.basename(f);
            if (apiComponents.includes(base))
                return;
            if (fs.existsSync(fsp.join(pathToPackageLib, f, 'index.ts')))
                exportComponents.push(base);
        });
        const pathToLibIndex = fsp.join(pathToPackageLib, 'index.ts');
        exports.logger.info(`${packageDetails.name}: Creating ${pathToLibIndex}`);
        fs.writeFileSync(pathToLibIndex, X3PackageGenerator.prettyifyTypescript(exportComponents
            .slice()
            // Filter out client folders
            .filter(exportComponent => !['client-functions', 'pages', 'widgets', 'stickers', 'page-extensions'].includes(exportComponent))
            .sort((a, b) => a.localeCompare(b))
            .map(apiComponent => `export * as ${_.camelCase(apiComponent)} from './${apiComponent}';`)
            .join('\n')));
    }
    static async getPackages(connPool, x3Folder, serviceName, options) {
        return (await x3_package_dictionary_helper_1.X3PackageDictionaryHelper.getSortedPackages(connPool, x3Folder, serviceName)).filter(pack => {
            if (options?.specificOnly && !pack.isCustom)
                return false;
            if (options?.scope)
                return pack.name.startsWith(`${options.scope}/`);
            return true;
        });
    }
    /**
     * Main package generation method
     */
    async generate() {
        if (!this.metadata.isCustom && !this.metadata.name.split('/')[1].startsWith(`${this.serviceName}-`)) {
            exports.logger.verbose(() => `${this.metadata.name}: package will not be generated as it does not belong to the service ${this.serviceName}`);
            return;
        }
        await x3_enum_generator_1.X3EnumGenerator.generatePackageEnums(this);
        await x3_service_options_generator_1.X3ServiceOptionsGenerator.generatePackageServiceOptions(this);
        await this.nodeGenerator.generateNodes();
        await this.nodeGenerator.generateNodeExtensions();
        // update package dependencies
        this.updatePackageDependencies();
        // updated tsconfig references
        this.updatePackageReferences();
    }
    static generateMainPackage(dir, version, serviceName) {
        const mainPackageName = `${serviceName}-main`;
        const mainDir = fsp.join(dir, 'main', mainPackageName);
        fs.mkdirSync(fsp.join(dir, 'main', mainPackageName), { recursive: true });
        const mainPackageJsonContent = {
            name: `@sage/${serviceName}-main`,
            description: `${serviceName.toUpperCase()} Services main package`,
            ..._.cloneDeep(main_package_json_1.mainPackageJson),
        };
        // Set to repo version
        mainPackageJsonContent.version = version;
        // Initialize package dependencies list to just the application packages
        const mainDependencies = this.packages?.filter(p => !p.isCustom && p.type === 'application');
        // List of non-application packages that are not in the dependency tree
        const standaloneSharedPackages = this.packages?.filter(p => !p.isCustom &&
            p.type !== 'application' &&
            !this.packages?.some(pack => pack.dependsOn?.includes(p.name)));
        // Add non-application packages that are not dependencies on the application packages
        mainDependencies?.push(...(standaloneSharedPackages ?? []));
        // If the main package has no dependencies there is an issue on the metadata, throw an error
        if (mainDependencies?.length === 0)
            throw new Error('No sage packages to add to main package dependencies.');
        // initialize the dependencies with cli
        mainPackageJsonContent.dependencies = {
            '@sage/xtrem-cli': 'workspace:*',
            '@sage/xtrem-cli-bundle-x3': 'workspace:*',
            '@sage/xtrem-standalone': 'workspace:*',
            '@sage/xtrem-x3-adc-ui': 'workspace:*',
            '@sage/xtrem-x3-copilot': 'workspace:*',
            '@sage/xtrem-x3-pages-ui': 'workspace:*',
            '@sage/xtrem-x3-interop': 'workspace:*',
        };
        mainDependencies?.forEach(pack => {
            mainPackageJsonContent.dependencies[pack.name] = 'workspace:*';
        });
        mainPackageJsonContent.dependencies = _.fromPairs(_.sortBy(_.toPairs(mainPackageJsonContent.dependencies), 0));
        // Write the main package.json
        fs.writeFileSync(fsp.join(mainDir, 'package.json'), this.prettyifyJson(JSON.stringify(mainPackageJsonContent, null, 4)));
        exports.logger.info(`Enum issues: \n ${Object.values(x3_local_menu_dictionary_helper_1.X3LocalMenuDictionaryHelper.localMenuHeaders)
            .filter(header => !!header.issue)
            .map(header => header.issue)
            .join('\n')}`);
    }
}
exports.X3PackageGenerator = X3PackageGenerator;
//# sourceMappingURL=x3-package-generator.js.map