"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.X3NodeGenerator = void 0;
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_x3_gateway_1 = require("@sage/xtrem-x3-gateway");
const fs = __importStar(require("fs"));
const _ = __importStar(require("lodash"));
const path = __importStar(require("path"));
const ts = __importStar(require("typescript"));
const x3_dictionary_interfaces_1 = require("./x3-dictionary-interfaces");
const x3_expression_parser_1 = require("./x3-expression-parser/x3-expression-parser");
const x3_local_menu_dictionary_helper_1 = require("./x3-local-menu-dictionary-helper");
const x3_node_class_generator_1 = require("./x3-node-class-generator");
const x3_node_dictionary_helper_1 = require("./x3-node-dictionary-helper");
const x3_node_generator_helper_1 = require("./x3-node-generator-helper");
const x3_package_generator_1 = require("./x3-package-generator");
class X3NodeGenerator {
    constructor(packageGenerator, connPool, x3FolderName) {
        this.packageGenerator = packageGenerator;
        this.connPool = connPool;
        this.x3FolderName = x3FolderName;
        this.nodeDefinitions = {};
        this.nodeExtensionDefinitions = {};
        this.x3NodeClassGenerator = new x3_node_class_generator_1.X3NodeClassGenerator(this.packageGenerator, this.connPool, this.x3FolderName);
    }
    addImport(name, importEntry, isExtension) {
        this.packageGenerator.addImport(isExtension ? `${this.packageGenerator.metadata.name}/node-extension` : 'node', name, importEntry);
    }
    /**
     * Generate const join object that will be passed as a parameter to the X3StorageManagerExtension constructor
     * @param nodeObject
     * @param packageName package of proeprties to generate joins on
     * @returns
     */
    async generateNodeExtensionJoinsObject(nodeExtensionObject, packageName) {
        this.addImport(nodeExtensionObject.nodeName, { packageName: x3_dictionary_interfaces_1.platformPackages.x3Gateway, value: 'Joins' }, true);
        const joins = await this.generateNodeJoinsObjectProperties(nodeExtensionObject, packageName, true);
        return ts.factory.createPropertyDeclaration([ts.factory.createModifier(ts.SyntaxKind.ConstKeyword)], ts.factory.createIdentifier('joins'), undefined, ts.factory.createTypeReferenceNode(ts.factory.createIdentifier('Joins'), [
            ts.factory.createTypeReferenceNode(x3_node_generator_helper_1.X3NodeGeneratorHelper.getNodeExtensionName(nodeExtensionObject.nodeName)),
        ]), ts.factory.createObjectLiteralExpression(joins, undefined));
    }
    /**
     * Create the node from metadata
     * @param nodeObject
     * @returns
     */
    async createNodeExtension(nodeExtensionObject) {
        const key = `${nodeExtensionObject.package}/${nodeExtensionObject.nodeName}`;
        if (this.nodeExtensionDefinitions[key])
            return this.nodeExtensionDefinitions[key];
        let profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating node extension class on ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName}`);
        const tsNodeObject = await this.x3NodeClassGenerator.generateNodeExtensionClass(nodeExtensionObject);
        profiler.success(`${this.packageGenerator.packageName}: Node extension on ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating joins for node extension ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName}`);
        const tsNodeJoinsObject = await this.generateNodeExtensionJoinsObject(nodeExtensionObject, nodeExtensionObject.package);
        profiler.success(`${this.packageGenerator.packageName}: Joins for node extension ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating composite references for node extension ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName}`);
        const tsNodeCompositeReferenceObject = X3NodeGenerator.generateCompositeNodeReferenceObject(nodeExtensionObject, nodeExtensionObject.package, true);
        profiler.success(`${this.packageGenerator.packageName}: Composite reference for node extension ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating node extension module declaration ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName}`);
        const tsNodeExtensionModule = x3_node_generator_helper_1.X3NodeGeneratorHelper.generateNodeExtensionModule(nodeExtensionObject);
        profiler.success(`${this.packageGenerator.packageName}: module declaration ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating imports for node extension on ${nodeExtensionObject.nodeName}`);
        const tsImportNodeObject = this.generateImportForNodeClass(nodeExtensionObject.nodeName, nodeExtensionObject.package, true);
        profiler.success(`${this.packageGenerator.packageName}: Imports for node extension on ${nodeExtensionObject.package}/${nodeExtensionObject.nodeName} complete`);
        const tsNodes = [
            ...tsImportNodeObject,
            x3_package_generator_1.X3PackageGenerator.newline,
            x3_package_generator_1.X3PackageGenerator.newline,
            tsNodeJoinsObject,
            x3_package_generator_1.X3PackageGenerator.newline,
            x3_package_generator_1.X3PackageGenerator.newline,
        ];
        if (tsNodeCompositeReferenceObject) {
            tsNodes.push(tsNodeCompositeReferenceObject);
            tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
            tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        }
        tsNodes.push(tsNodeObject);
        tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        tsNodes.push(tsNodeExtensionModule);
        tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        this.nodeExtensionDefinitions[key] = x3_package_generator_1.X3PackageGenerator.prettyifyTypescript(tsNodes, undefined, `${nodeExtensionObject.package}.node-extension.${nodeExtensionObject.nodeName}`);
        return this.nodeExtensionDefinitions[key];
    }
    static getExpressionFromPath(targetNode, propertyPath) {
        if (propertyPath.length === 0)
            throw new Error(`${targetNode}: cannot resolve an empty path`);
        const reversedPath = propertyPath.slice().reverse();
        const pathWalked = [];
        let currentPropertyName = reversedPath.pop();
        if (!currentPropertyName)
            throw new Error(`${targetNode}: cannot resolve path`);
        pathWalked.push(currentPropertyName);
        let propertyChain = ts.factory.createPropertyAccessExpression(ts.factory.createThis(), ts.factory.createIdentifier(currentPropertyName));
        currentPropertyName = reversedPath.pop();
        let isNullable = false;
        while (currentPropertyName) {
            const previousProperty = x3_node_dictionary_helper_1.X3NodeDictionaryHelper.getPropertyFromPath(targetNode, pathWalked);
            // as soon as one property in the path is nullable we need identify the remainder as nullable so that the chaining
            // of the ? is correct
            isNullable = isNullable || !!previousProperty.isNullable;
            pathWalked.push(currentPropertyName);
            propertyChain = ts.factory.createPropertyAccessChain(ts.factory.createParenthesizedExpression(ts.factory.createAwaitExpression(propertyChain)), isNullable ? ts.factory.createToken(ts.SyntaxKind.QuestionDotToken) : undefined, currentPropertyName);
            currentPropertyName = reversedPath.pop();
        }
        return propertyChain;
    }
    static generateNodeJoinPropertyPathTypeExpression(nodeObject, propName, value) {
        const propertyPath = value.split('.');
        if (propertyPath.length === 0)
            throw new Error(`${nodeObject.nodeName}.${propName} wrong property path declaration`);
        return X3NodeGenerator.getExpressionFromPath(nodeObject.nodeName, x3_node_dictionary_helper_1.X3NodeDictionaryHelper.expandPropertyPath(nodeObject.nodeName, propertyPath));
    }
    /**
     * Generate the join value for a literal type
     * @param joinValue
     * @param propName
     * @param nodeObject
     * @returns
     */
    async generateNodeJoinTypeExpression(joinValue, propName, nodeObject, isExtension) {
        const { value } = joinValue;
        switch (joinValue.type) {
            case 'string':
                if (!value)
                    return { expression: ts.factory.createNull() };
                return { expression: ts.factory.createStringLiteral(value) };
            case 'integer':
            case 'decimal':
                return { expression: ts.factory.createNumericLiteral(value) };
            case 'date': {
                const nullDate = xtrem_x3_gateway_1.SqlResolver.getNullDate(this.connPool.dialect);
                const nullDateString = `${nullDate.year}${String(nullDate.month).padStart(2, '0')}${String(nullDate.day).padStart(2, '0')}`;
                if (nullDateString === value)
                    return { expression: ts.factory.createNull() };
                // date.parse(YYYYMMDD
                // date.parse('20200229', 'en-US','YYYYMMDD')
                this.addImport(nodeObject.nodeName, {
                    packageName: x3_dictionary_interfaces_1.platformPackages.core,
                    value: 'date',
                }, isExtension);
                return {
                    expression: ts.factory.createCallExpression(ts.factory.createPropertyAccessExpression(ts.factory.createIdentifier('date'), 'parse'), undefined, [
                        ts.factory.createStringLiteral(value),
                        ts.factory.createStringLiteral('en-US'),
                        ts.factory.createStringLiteral('YYYYMMDD'),
                    ]),
                };
            }
            case 'enum': {
                if (value === '0')
                    return { expression: ts.factory.createNull() };
                const property = nodeObject.properties.find(p => p.name === propName);
                if (!property)
                    throw new Error(`${nodeObject.nodeName}.${propName}: cannot find property.`);
                const { targetNode } = property;
                if (!targetNode)
                    throw new Error(`${nodeObject.nodeName}.${propName}: target node missing.`);
                const targetNodeData = await x3_node_dictionary_helper_1.X3NodeDictionaryHelper.findNodeData(this.connPool, this.x3FolderName, targetNode);
                const targetNodeProperty = (await x3_node_generator_helper_1.X3NodeGeneratorHelper.getNodeProperties(this.connPool, this.x3FolderName, targetNodeData)).find(p => p.name === joinValue.targetPropertyName);
                if (!targetNodeProperty)
                    throw new Error(`${targetNode}.${joinValue.targetPropertyName}: cannot find property.`);
                if (!targetNodeProperty.localMenuNumber)
                    throw new Error(`${nodeObject.nodeName}.${propName}: enum join found for non-enum property (${targetNode}.${joinValue.targetPropertyName}).`);
                const localMenuDefinition = await x3_local_menu_dictionary_helper_1.X3LocalMenuDictionaryHelper.getLocalMenuDefinition(this.connPool, this.x3FolderName, targetNodeProperty.localMenuNumber, this.packageGenerator.metadata);
                const localMenuValue = localMenuDefinition.values.find(lmValue => lmValue.valueNumber === Number(value));
                if (!localMenuValue)
                    throw new Error(`${nodeObject.nodeName}.${propName}: ${value} is not a value in local menu ${property.localMenuNumber}.`);
                x3_package_generator_1.logger.info(`${nodeObject.nodeName}.${propName}: local menu join value ${value} maps to ${localMenuValue}`);
                return { expression: ts.factory.createStringLiteral(localMenuValue.name) };
            }
            case 'property':
            case 'propertyPath': {
                // property path example:
                // site.legalCompany.legislation.code -> return (await (await (await this.site).legalCompany).legislation).code
                if (!value)
                    return { expression: ts.factory.createNull() };
                const hasAwait = value.split('.').length > 0;
                return {
                    expression: X3NodeGenerator.generateNodeJoinPropertyPathTypeExpression(nodeObject, propName, value),
                    hasAwait,
                };
            }
            case 'expression': {
                if (!value)
                    return { expression: ts.factory.createNull() };
                const property = nodeObject.properties.find(p => p.name === propName);
                if (!property)
                    throw new Error(`${nodeObject.nodeName}.${propName}: cannot find property.`);
                const expression = new x3_expression_parser_1.X3ExpressionParser(nodeObject.nodeName).convertFunction(value);
                if (property.type !== 'boolean' && expression.hasBooleanOperators) {
                    throw new Error(`${nodeObject.nodeName}.${propName}: expression has a boolean operator and is not a boolean property.`);
                }
                return { expression: expression.newExpression, hasAwait: expression.hasAwait };
            }
            default:
                throw new Error(`${nodeObject.nodeName}.${propName} invalid join type ${joinValue.type}.`);
        }
    }
    /**
     * Resolve the join value based on the type.
     * @param nodeObject
     * @param propertyName
     * @param joinValue
     * @param useArrowFunctions
     * @returns
     */
    async resolveJoinValue(nodeObject, propertyName, joinValue, isExtension, useArrowFunctions = false) {
        // Property join
        // e.g. { code: 'company' }
        if (joinValue.type === 'property') {
            return ts.factory.createStringLiteral(joinValue.value);
        }
        const { expression, hasAwait } = await this.generateNodeJoinTypeExpression(joinValue, propertyName, nodeObject, isExtension);
        const body = ts.factory.createBlock([ts.factory.createReturnStatement(expression)], true);
        // An arrow function or method declaration
        // e.g.
        // value = 909
        // becomes;
        // glossaryId: ()=>{ return 909; }
        // or glossaryId(){ return 909; }
        const functionModifiers = hasAwait ? [ts.factory.createModifier(ts.SyntaxKind.AsyncKeyword)] : undefined;
        return useArrowFunctions
            ? ts.factory.createArrowFunction(undefined, undefined, [], undefined, undefined, expression)
            : ts.factory.createMethodDeclaration(functionModifiers, undefined, joinValue.targetPropertyName, undefined, undefined, [], undefined, body);
    }
    async resolveFilterJoinValue(nodeObject, propertyName, joinValue, isExtension) {
        const { expression, hasAwait } = await this.generateNodeJoinTypeExpression(joinValue, propertyName, nodeObject, isExtension);
        switch (joinValue.type) {
            case 'property':
            case 'propertyPath': {
                const body = ts.factory.createBlock([ts.factory.createReturnStatement(expression)], true);
                // An arrow function or method declaration
                // e.g.
                // value = 'company'
                // becomes;
                // company: ()=>{ return this.company; }
                // value = 'company.code'
                // becomes;
                // or company(){ return (await this.company).code; }
                const functionModifiers = hasAwait
                    ? [ts.factory.createModifier(ts.SyntaxKind.AsyncKeyword)]
                    : undefined;
                return ts.factory.createMethodDeclaration(functionModifiers, undefined, joinValue.targetPropertyName, undefined, undefined, [], undefined, body);
            }
            default:
                return expression;
        }
    }
    /**
     * Get reference or collection join
     * @param nodeObject
     * @param packageName
     * @param propertyType
     * @returns
     */
    async getForeignJoins(nodeObject, packageName, propertyType, isExtension) {
        // company: {code:'company'}
        const foreignJoins = {};
        await (0, xtrem_core_1.asyncArray)(nodeObject.properties.filter(property => property.type === propertyType && property.joinValues?.length)).forEach(async (property) => {
            const propName = property.name;
            // Property join not relevant for this package
            if (property.packageName !== packageName)
                return;
            foreignJoins[propName] = {};
            await (0, xtrem_core_1.asyncArray)(property.joinValues || []).forEach(async (joinValue) => {
                const join = await this.resolveJoinValue(nodeObject, propName, joinValue, isExtension, false);
                foreignJoins[propName][joinValue.targetPropertyName] = join;
            });
        });
        return foreignJoins;
    }
    static getFunctionFromExpression(nodeObject, expresionString, functionName) {
        const expression = new x3_expression_parser_1.X3ExpressionParser(nodeObject.nodeName);
        const { newExpression, hasAwait } = expression.convertFunction(expresionString);
        const body = ts.factory.createBlock([ts.factory.createReturnStatement(newExpression)], true);
        const functionModifiers = hasAwait ? [ts.factory.createModifier(ts.SyntaxKind.AsyncKeyword)] : undefined;
        return ts.factory.createMethodDeclaration(functionModifiers, undefined, functionName, undefined, undefined, [], undefined, body);
    }
    /**
     * Generate all localized string joins
     * @param nodeObject
     * @param packageName
     * @returns
     */
    async getLocalizedStringJoins(nodeObject, packageName, isExtension) {
        const joinPropertyExpressions = {};
        await (0, xtrem_core_1.asyncArray)(nodeObject.properties.filter(property => property.type === 'localizedString')).forEach(async (property) => {
            // Property join not relevant for this package
            if (property.packageName !== packageName || !property.localizedStringInfo) {
                return;
            }
            const { tableName, columnName } = property.localizedStringInfo;
            const getKeyEntries = (keyName) => {
                const keyJoinValues = property.joinValues?.filter(joinValue => joinValue.targetPropertyName === keyName);
                if (keyJoinValues?.some(keyJoinValue => keyJoinValue.type === 'property' &&
                    !nodeObject.properties.find(p => p.name === keyJoinValue.value))) {
                    x3_package_generator_1.logger.error(`${packageName}: not all ${keyName} join properties exist in ${nodeObject.nodeName} for localized string property ${property.name}`);
                    return undefined;
                }
                return (0, xtrem_core_1.asyncArray)(keyJoinValues || [])
                    .map(joinValue => {
                    return this.resolveJoinValue(nodeObject, property.name, joinValue, isExtension, true);
                })
                    .toArray();
            };
            const key1 = await getKeyEntries('key1');
            const key2 = await getKeyEntries('key2');
            if (tableName && columnName) {
                if (key1?.length) {
                    const propertyExpressions = [];
                    propertyExpressions.push(X3NodeGenerator.getFunctionFromExpression(nodeObject, tableName, 'tableName'));
                    propertyExpressions.push(X3NodeGenerator.getFunctionFromExpression(nodeObject, columnName, 'columnName'));
                    propertyExpressions.push(ts.factory.createPropertyAssignment('key1', ts.factory.createArrayLiteralExpression(key1, true)));
                    if (key2?.length)
                        propertyExpressions.push(ts.factory.createPropertyAssignment('key2', ts.factory.createArrayLiteralExpression(key2, true)));
                    joinPropertyExpressions[property.name] = ts.factory.createObjectLiteralExpression(propertyExpressions, true);
                }
                else {
                    x3_package_generator_1.logger.error(`${packageName}: Cannot generate localized string join for  ${nodeObject.nodeName}.${property.name}, key1 not povided`);
                }
            }
        });
        if (Object.keys(joinPropertyExpressions).length) {
            return ts.factory.createObjectLiteralExpression(Object.keys(joinPropertyExpressions).map(key => {
                return ts.factory.createPropertyAssignment(key, joinPropertyExpressions[key]);
            }), true);
        }
        return undefined;
    }
    /**
     * Generate the reference, collection and localized string joins
     * @param nodeObject
     * @param packageName package property belongs to
     * @returns
     */
    async generateNodeJoinsObjectProperties(nodeObject, packageName, isExtension) {
        const joinsLiteralElements = [];
        await (0, xtrem_core_1.asyncArray)(['referenceJoins', 'localizedStrings', 'collectionJoins']).forEach(async (joinType) => {
            const addJoins = (joins) => {
                if (Object.keys(joins).length > 0) {
                    const joinsObject = x3_node_generator_helper_1.X3NodeGeneratorHelper.makeJoinObjectLiteral(joins);
                    if (joinsObject)
                        joinsLiteralElements.push(ts.factory.createPropertyAssignment(ts.factory.createStringLiteral(String(joinType)), joinsObject));
                }
            };
            switch (joinType) {
                case 'referenceJoins': {
                    // { company: {code:'company'} }
                    // { company: {code(){ return 'FOO' }}, currency: {code: 'currency'} }
                    const joinValue = await this.getForeignJoins(nodeObject, packageName, 'reference', isExtension);
                    addJoins(joinValue);
                    break;
                }
                case 'localizedStrings': {
                    const localizedStringJoins = await this.getLocalizedStringJoins(nodeObject, packageName, isExtension);
                    if (!localizedStringJoins)
                        return;
                    joinsLiteralElements.push(ts.factory.createPropertyAssignment(ts.factory.createStringLiteral(String(joinType)), localizedStringJoins));
                    return;
                }
                case 'collectionJoins': {
                    const joinValue = await this.getForeignJoins(nodeObject, packageName, 'collection', isExtension);
                    addJoins(joinValue);
                    break;
                }
                default:
                    break;
            }
        });
        return joinsLiteralElements;
    }
    /**
     * Generate const join object that will be passed as a parameter to the X3StorageManager constructor
     * @param nodeObject
     * @param packageName package of proeprties to generate joins on
     * @returns
     */
    async generateNodeJoinsObject(nodeObject, packageName) {
        this.addImport(nodeObject.nodeName, { packageName: x3_dictionary_interfaces_1.platformPackages.x3Gateway, value: 'Joins' }, false);
        const joins = await this.generateNodeJoinsObjectProperties(nodeObject, packageName, false);
        return ts.factory.createPropertyDeclaration([ts.factory.createModifier(ts.SyntaxKind.ConstKeyword)], ts.factory.createIdentifier('joins'), undefined, ts.factory.createTypeReferenceNode(ts.factory.createIdentifier('Joins'), [
            ts.factory.createTypeReferenceNode(nodeObject.nodeName),
        ]), ts.factory.createObjectLiteralExpression(joins, undefined));
    }
    static getCompositeReferencesObjLiteralExpression(compositeReferences) {
        const compRefObjLitExp = [];
        Object.entries(compositeReferences).forEach(([key, val]) => {
            const compRefEntries = [];
            Object.entries(val).forEach(([eKey, eVal]) => {
                compRefEntries.push(ts.factory.createPropertyAssignment(ts.factory.createIdentifier(String(eKey)), ts.factory.createStringLiteral(String(eVal))));
            });
            compRefObjLitExp.push(ts.factory.createPropertyAssignment(ts.factory.createIdentifier(String(key)), ts.factory.createObjectLiteralExpression(compRefEntries, true)));
        });
        return compRefObjLitExp;
    }
    static generateCompositeNodeReferenceObject(nodeObject, packageName, isExtension) {
        // composite reference in storage manager
        const compositeReferences = {};
        nodeObject.properties
            .filter(prop => (prop.packageName === packageName ||
            (prop.packageName === null && nodeObject.package === packageName && !isExtension)) &&
            prop.composite?.referenceProperty &&
            prop.composite?.propertyPath)
            .forEach(prop => {
            if (prop.composite?.referenceProperty && prop.composite?.propertyPath) {
                _.set(compositeReferences, [prop.composite?.referenceProperty, prop.name], prop.composite?.propertyPath);
            }
        });
        if (Object.keys(compositeReferences).length > 0) {
            return ts.factory.createPropertyDeclaration([ts.factory.createModifier(ts.SyntaxKind.ConstKeyword)], ts.factory.createIdentifier('compositeReferences'), undefined, undefined, ts.factory.createObjectLiteralExpression(this.getCompositeReferencesObjLiteralExpression(compositeReferences), true));
        }
        return undefined;
    }
    /**
     * Generate const denormalized object that will be passed as a parameter to the X3StorageManager constructor
     * @param nodeObject
     * @param packageName package of proeprties to generate joins on
     * @returns
     */
    generateNodeDenormalizedObject(nodeObject) {
        this.addImport(nodeObject.nodeName, { packageName: x3_dictionary_interfaces_1.platformPackages.x3Gateway, value: 'Denormalized' }, false);
        const denormalizedElements = [];
        if (nodeObject.denormalized?.dimension) {
            // Example:
            // {
            //     maxRepeat: 3,
            //
            // };
            denormalizedElements.push(ts.factory.createPropertyAssignment(ts.factory.createStringLiteral(String('maxRepeat')), ts.factory.createNumericLiteral(nodeObject.denormalized.dimension)));
        }
        else if (nodeObject.denormalized?.activityCode) {
            // Example:
            // {
            //     maxRepeat: (context: Context) => x3System.functions.getSizingFromActivityCode(context, 'PER'),
            //
            // };
            this.addImport(nodeObject.nodeName, { packageName: x3_dictionary_interfaces_1.platformPackages.systemUtils, value: '*' }, false);
            this.addImport(nodeObject.nodeName, { packageName: x3_dictionary_interfaces_1.platformPackages.core, value: 'Context' }, false);
            const systemPackageAlias = x3_package_generator_1.X3PackageGenerator.getPackageAlias(x3_dictionary_interfaces_1.platformPackages.systemUtils);
            denormalizedElements.push(ts.factory.createPropertyAssignment(ts.factory.createStringLiteral(String('maxRepeat')), ts.factory.createArrowFunction(undefined, undefined, [
                ts.factory.createParameterDeclaration(undefined, undefined, ts.factory.createIdentifier('context'), undefined, ts.factory.createTypeReferenceNode(ts.factory.createIdentifier('Context'), undefined), undefined),
            ], undefined, ts.factory.createToken(ts.SyntaxKind.EqualsGreaterThanToken), ts.factory.createCallExpression(ts.factory.createPropertyAccessExpression(ts.factory.createIdentifier(systemPackageAlias), ts.factory.createIdentifier('getSizingFromActivityCode')), undefined, [
                ts.factory.createIdentifier('context'),
                ts.factory.createStringLiteral(nodeObject.denormalized.activityCode),
            ]))));
        }
        return ts.factory.createPropertyDeclaration([ts.factory.createModifier(ts.SyntaxKind.ConstKeyword)], ts.factory.createIdentifier('denormalized'), undefined, ts.factory.createTypeReferenceNode(ts.factory.createIdentifier('Denormalized'), []), ts.factory.createObjectLiteralExpression(denormalizedElements, undefined));
    }
    /**
     * Import statements of a node files
     * @param name
     * @param currentPackageName
     * @returns
     */
    generateImportForNodeClass(name, currentPackageName, isExtension = false) {
        const nodeKey = isExtension ? `${currentPackageName}/node-extension~${name}` : `node~${name}`;
        const importsMap = this.packageGenerator.imports[nodeKey] || {};
        const result = {};
        const nodeImport = isExtension ? 'NodeExtension' : 'Node';
        x3_package_generator_1.X3PackageGenerator.sortImports(importsMap).forEach(entry => {
            const key = entry[0];
            const value = entry[1];
            const importEntries = key === x3_dictionary_interfaces_1.platformPackages.core ? [...['decorators', nodeImport], ...value] : value;
            if (importEntries.includes('*')) {
                const alias = x3_package_generator_1.X3PackageGenerator.getPackageAlias(key === '..' ? currentPackageName : key);
                result[`${key}.*`] = ts.factory.createImportDeclaration(undefined, ts.factory.createImportClause(false, undefined, ts.factory.createNamespaceImport(ts.factory.createIdentifier(alias))), ts.factory.createStringLiteral(key));
            }
            const packageImports = importEntries.filter(importValue => importValue !== '*');
            if (packageImports.length > 0)
                result[key] = ts.factory.createImportDeclaration(undefined, ts.factory.createImportClause(false, undefined, ts.factory.createNamedImports(packageImports.map(imp => ts.factory.createImportSpecifier(false, undefined, ts.factory.createIdentifier(imp))))), ts.factory.createStringLiteral(key));
        });
        if (!result[x3_dictionary_interfaces_1.platformPackages.core]) {
            result[x3_dictionary_interfaces_1.platformPackages.core] = ts.factory.createImportDeclaration(undefined, ts.factory.createImportClause(false, undefined, ts.factory.createNamedImports([
                ts.factory.createImportSpecifier(false, undefined, ts.factory.createIdentifier('decorators')),
                ts.factory.createImportSpecifier(false, undefined, ts.factory.createIdentifier(nodeImport)),
            ])), ts.factory.createStringLiteral(x3_dictionary_interfaces_1.platformPackages.core));
        }
        return Object.values(result);
    }
    /**
     * Create the node from metadata
     * @param nodeObject
     * @returns
     */
    async createNode(nodeObject) {
        if (this.nodeDefinitions[nodeObject.nodeName])
            return this.nodeDefinitions[nodeObject.nodeName];
        let profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating node ${nodeObject.package}/${nodeObject.nodeName}`);
        const tsNodeObject = await this.x3NodeClassGenerator.generateNodeClass(nodeObject);
        profiler.success(`${this.packageGenerator.packageName}: Generating node ${nodeObject.package}/${nodeObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating joins for node ${nodeObject.nodeName}`);
        const tsNodeJoinsObject = await this.generateNodeJoinsObject(nodeObject, nodeObject.package);
        profiler.success(`${this.packageGenerator.packageName}: Generating joins for node ${nodeObject.nodeName} complete`);
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating composite references for node ${nodeObject.package}/${nodeObject.nodeName}`);
        const tsNodeCompositeReferenceObject = X3NodeGenerator.generateCompositeNodeReferenceObject(nodeObject, nodeObject.package, false);
        profiler.success(`${this.packageGenerator.packageName}: Composite reference for node ${nodeObject.package}/${nodeObject.nodeName} complete`);
        let tsNodeDenormalizedObject = ts.factory.createEmptyStatement();
        if (nodeObject.denormalized) {
            profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating denormalized parameter for node ${nodeObject.nodeName}`);
            tsNodeDenormalizedObject = this.generateNodeDenormalizedObject(nodeObject);
            profiler.success(`${this.packageGenerator.packageName}: Generating denormalized parameter for node ${nodeObject.nodeName} complete`);
        }
        profiler = x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Generating imports for node ${nodeObject.nodeName}`);
        const tsImportNodeObject = this.generateImportForNodeClass(nodeObject.nodeName, nodeObject.package);
        profiler.success(`${this.packageGenerator.packageName}: Generating imports for node ${nodeObject.nodeName} complete`);
        const tsNodes = [
            ...tsImportNodeObject,
            x3_package_generator_1.X3PackageGenerator.newline,
            x3_package_generator_1.X3PackageGenerator.newline,
            tsNodeDenormalizedObject,
            x3_package_generator_1.X3PackageGenerator.newline,
            x3_package_generator_1.X3PackageGenerator.newline,
            tsNodeJoinsObject,
            x3_package_generator_1.X3PackageGenerator.newline,
            x3_package_generator_1.X3PackageGenerator.newline,
        ];
        if (tsNodeCompositeReferenceObject) {
            tsNodes.push(tsNodeCompositeReferenceObject);
            tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
            tsNodes.push(x3_package_generator_1.X3PackageGenerator.newline);
        }
        tsNodes.push(tsNodeObject);
        this.nodeDefinitions[nodeObject.nodeName] = x3_package_generator_1.X3PackageGenerator.prettyifyTypescript(tsNodes, undefined, `${nodeObject.package}.node.${nodeObject.nodeName}`);
        return this.nodeDefinitions[nodeObject.nodeName];
    }
    /**
     * Generate the nodes of the package
     * @param nodeClassName
     */
    async generateNodes() {
        const packageName = this.packageGenerator.metadata.name;
        // get all data from X3 database
        const nodeObjects = x3_package_generator_1.X3PackageGenerator.allNodesPerPackage[packageName].filter(node => {
            // TODO: we need to refine this to check operations, mutations and transient node when implemented
            if (node.properties.length === 0 && node.operations?.length === 0) {
                x3_package_generator_1.logger.error(`${packageName}: Cannot generate without node ${node.nodeName} it has no memebers`);
                return false;
            }
            if (node.properties.length > 0 && node.keyPropertyNames.length === 0) {
                x3_package_generator_1.logger.error(`${packageName}: Cannot generate without node ${node.nodeName} it has no keyProperties`);
                return false;
            }
            return true;
        });
        const nodeFiles = [];
        const pathToNodes = path.join(this.packageGenerator.dir, 'lib/nodes');
        if (fs.existsSync(pathToNodes))
            fs.rmSync(pathToNodes, { recursive: true });
        fs.mkdirSync(pathToNodes, { recursive: true });
        await (0, xtrem_core_1.asyncArray)(nodeObjects).forEach(async (nObj) => {
            const apiNode = await this.createNode(nObj);
            const filename = `${_.kebabCase(nObj.nodeName)}.ts`;
            x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Create node file -> ${filename}`);
            nodeFiles.push(filename);
            fs.writeFileSync(path.join(this.packageGenerator.dir, 'lib/nodes', filename), apiNode);
        });
        const nodeIndexFilename = path.join(pathToNodes, 'index.ts');
        let indexContent = 'export type Dummy = void;';
        if (nodeFiles.length > 0)
            indexContent = nodeFiles
                .slice()
                .sort((a, b) => a.localeCompare(b))
                .map(nodeFile => `export * from './${path.parse(nodeFile).name}';`)
                .join('\n');
        fs.writeFileSync(nodeIndexFilename, x3_package_generator_1.X3PackageGenerator.prettyifyTypescript(indexContent));
    }
    /**
     * Generate the nodes of the package
     * @param nodeClassName
     */
    async generateNodeExtensions() {
        const packageName = this.packageGenerator.metadata.name;
        // get all data from X3 database
        const nodeExtensionObjects = await x3_node_generator_helper_1.X3NodeGeneratorHelper.getNodeExtensions(this.connPool, this.x3FolderName, packageName);
        const nodeExtensionsFiles = [];
        const pathToNodeExtensions = path.join(this.packageGenerator.dir, 'lib/node-extensions');
        if (fs.existsSync(pathToNodeExtensions))
            fs.rmSync(pathToNodeExtensions, { recursive: true });
        fs.mkdirSync(pathToNodeExtensions, { recursive: true });
        await (0, xtrem_core_1.asyncArray)(nodeExtensionObjects).forEach(async (nodeExtensionObject) => {
            const apiNodeExtension = await this.createNodeExtension(nodeExtensionObject);
            const filename = `${_.kebabCase(x3_node_generator_helper_1.X3NodeGeneratorHelper.getNodeExtensionName(nodeExtensionObject.nodeName))}.ts`;
            x3_package_generator_1.logger.info(`${this.packageGenerator.packageName}: Create node extension file -> ${filename}`);
            nodeExtensionsFiles.push(filename);
            fs.writeFileSync(path.join(pathToNodeExtensions, filename), apiNodeExtension);
        });
        const nodeIndexFilename = path.join(pathToNodeExtensions, 'index.ts');
        let indexContent = 'export type Dummy = void;';
        if (nodeExtensionsFiles.length > 0)
            indexContent = nodeExtensionsFiles
                .slice()
                .sort((a, b) => a.localeCompare(b))
                .map(nodeFile => `export * from './${path.parse(nodeFile).name}';`)
                .join('\n');
        fs.writeFileSync(nodeIndexFilename, x3_package_generator_1.X3PackageGenerator.prettyifyTypescript(indexContent));
    }
}
exports.X3NodeGenerator = X3NodeGenerator;
//# sourceMappingURL=x3-node-generator.js.map