"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.X3NodeDictionaryHelper = void 0;
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_x3_gateway_1 = require("@sage/xtrem-x3-gateway");
const _ = __importStar(require("lodash"));
const x3_dictionary_helper_1 = require("./x3-dictionary-helper");
const logger = xtrem_log_1.Logger.getLogger(__filename, 'x3-dictionary-helper');
class X3NodeDictionaryHelper {
    // Dimension of the join values
    static { this.numberOfJoins = 20; }
    static { this.anodePropertiesData = {}; }
    static { this.nodeData = {}; }
    static { this.denormalizeNodeData = {}; }
    static clearCache() {
        X3NodeDictionaryHelper.anodePropertiesData = {};
        X3NodeDictionaryHelper.globalVariableData = undefined;
        X3NodeDictionaryHelper.nodeData = {};
        X3NodeDictionaryHelper.denormalizeNodeData = {};
        X3NodeDictionaryHelper.denormalizedNodeDbData = undefined;
        X3NodeDictionaryHelper.propertyDataDb = undefined;
        X3NodeDictionaryHelper.nodeIndexDefinitions = undefined;
        X3NodeDictionaryHelper.nodeExtensionData = undefined;
        X3NodeDictionaryHelper.baseNodeOperations = undefined;
        X3NodeDictionaryHelper.nodeOperationProperties = undefined;
    }
    static async getDenormalizedNodeData(connPool, x3Folder, nodeName) {
        if (this.denormalizedNodeDbData)
            return this.denormalizedNodeDbData.filter((data) => x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', data?.PARENT_0) === nodeName);
        const filter = [{ key: 'DENCOLGR_0', operator: '<>', value: ' ' }];
        const propertyData = await this.getPropertyData(connPool, x3Folder);
        this.denormalizedNodeDbData = (await connPool
            .createTableReader(await x3_dictionary_helper_1.X3DictionaryHelper.getTableDefinition(connPool, x3Folder, 'ANODE'), filter)
            .readAll()).map((dbData) => {
            const properties = propertyData.filter((row) => {
                const dataDenormalizedNodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', row.DENCOLNOD_0);
                return dataDenormalizedNodeName === dbData.ANODE_0;
            });
            // We need to determine the package of the denormalized node from the properties
            // Today there is a control on X3 that stops denormalized properties being in different packages
            // If this changes the we need to review this code.
            if (properties.length) {
                dbData.APACK_0 = properties[0].APACK_0;
            }
            else {
                dbData.APACK_0 = null;
            }
            return dbData;
        });
        return this.denormalizedNodeDbData.filter((data) => x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', data?.PARENT_0) === nodeName);
    }
    static async findDenormalizedNodeData(connPool, x3Folder, parent) {
        const { nodeName } = parent;
        if (this.denormalizeNodeData[nodeName])
            return Object.values(this.denormalizeNodeData[nodeName]);
        this.denormalizeNodeData[nodeName] = {};
        const result = await this.getDenormalizedNodeData(connPool, x3Folder, nodeName);
        result.forEach((record) => {
            const name = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record.ANODE_0);
            const nodeData = {
                packageName: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record.APACK_0) || parent.packageName,
                nodeName: name,
                nodeBinding: parent.nodeBinding,
                tableName: parent.tableName,
                activityCode: parent.activityCode,
                isPublished: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', record.APUBFLG_0),
                denormalized: {
                    collectionPropertyName: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record.DENCOLGR_0),
                    parentNode: nodeName,
                    activityCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record.CODACT_0),
                    dimension: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', record.DIME_0),
                },
                isVitalCollectionChild: true,
            };
            if (!nodeData.denormalized?.dimension && !nodeData.denormalized?.activityCode) {
                logger.error(() => `${name}: cannot determine size of denormalized node. Dimension and activity code not provided. Node skipped.`);
                throw new Error(`${name}: cannot determine size of denormalized node. Dimension and activity code not provided. Node skipped.`);
                // return;
            }
            this.nodeData[name] = nodeData;
            this.denormalizeNodeData[nodeName][name] = this.nodeData[name];
        });
        return Object.values(this.denormalizeNodeData[nodeName] || {});
    }
    static async findNodeData(connPool, x3Folder, nodeName) {
        if (!Object.keys(this.nodeData).length)
            await this.getAllNodeData(connPool, x3Folder);
        if (!this.nodeData[nodeName])
            throw new Error(`${nodeName}: node not found.`);
        return this.nodeData[nodeName];
    }
    /**
     * Fills the node data in the dictionary.
     *
     * @param connPool The connection pool.
     * @param x3Folder The X3 folder.
     * @param nodeBindingType The node binding type.
     * @returns A promise that resolves when the node data is filled.
     */
    static async fillNodeData(connPool, x3Folder, nodeBindingType) {
        const filter = [{ key: 'ABDGTYP_0', operator: '=', value: nodeBindingType }];
        const result = await connPool
            .createTableReader(await x3_dictionary_helper_1.X3DictionaryHelper.getTableDefinition(connPool, x3Folder, 'ANODEBDG'), filter)
            .readAll();
        await (0, xtrem_core_1.asyncArray)(result).forEach(async (record) => {
            const name = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.ANODE_0);
            let tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.MAINTABLE_0);
            // This is the node binding
            if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', record?.ABDGTYP_0) === 5) {
                tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.AX3KEY1_0);
            }
            this.nodeData[name] = {
                packageName: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.APACKAGE_0),
                nodeName: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.ANODE_0),
                nodeBinding: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.AREFBIND_0),
                tableName,
                activityCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.CODACT_0),
                tableFilters: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.TABFLT_0),
                authorizationCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', record?.FNC_0),
                isPublished: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', record?.APUBFLG_0),
            };
            this.nodeData[name].denormalizedNodes = await this.findDenormalizedNodeData(connPool, x3Folder, this.nodeData[name]);
        });
    }
    static async getAllNodeData(connPool, x3Folder) {
        if (Object.keys(this.nodeData).length)
            return Object.values(this.nodeData);
        // Fill table node data
        await this.fillNodeData(connPool, x3Folder, '2');
        // Fill view node data
        await this.fillNodeData(connPool, x3Folder, '5');
        return Object.values(this.nodeData);
    }
    static getPropertyKind(propertyKind) {
        switch (Number(propertyKind)) {
            case 1:
                return 'reference';
            case 3:
                return 'collection';
            default:
                return undefined;
        }
    }
    static getPropertyType(propertyType, nodeName, propertyName, propertyKind) {
        // 1 - boolean
        // 2 - integer
        // 3 - decimal
        // 4 - string
        // 5 - date
        // 6 - Node
        // 7 - localized string
        // 8 - enum
        // 9 - object
        // 10 - blob binaryStream
        // 11 - clob textStream
        // 12 - datetime
        switch (propertyType) {
            case 1:
                return 'boolean';
            case 2:
                return 'integer';
            case 3:
                return 'decimal';
            case 4:
                return 'string';
            case 5:
                return 'date';
            case 6: {
                const resolvedPropertyKind = this.getPropertyKind(propertyKind ?? '1');
                if (resolvedPropertyKind == null)
                    throw new Error(`Invalid node property kind ${nodeName}.${propertyName} - ${propertyKind}`);
                return resolvedPropertyKind;
            }
            case 7:
                return 'localizedString';
            case 8:
                return 'enum';
            case 9:
                return 'object';
            case 10:
                return 'binaryStream';
            case 11:
                return 'textStream';
            case 12:
                return 'datetime';
            default:
                throw new Error(`Invalid property type ${nodeName}.${propertyName} - ${propertyType}`);
        }
    }
    static async getReferenceColumnType(connPool, x3Folder, property) {
        if (property.type !== 'reference')
            return undefined;
        const { targetNode, joinValues } = property;
        if (!targetNode || !joinValues)
            throw new Error(`${property.nodeName}.${property.name}: property target not set.`);
        if (property.computedProperty) {
            // column type is not required for a computed property
            return undefined;
        }
        const primaryIndex = await this.getNodePrimaryIndex(connPool, x3Folder, targetNode);
        const indexNodeKeys = primaryIndex.properties;
        const nodeKeys = primaryIndex.properties.filter(key => joinValues.find(join => join.targetPropertyName === key.propertyName && join.type === 'property'));
        if (nodeKeys.length === 0) {
            // fallback type is either the target node default type or string
            let fallback = indexNodeKeys.length ? indexNodeKeys[indexNodeKeys.length - 1].columnType : 'string';
            const propertyJoinElement = joinValues.find(j => j.type === 'property' && j.value === property.name);
            if (propertyJoinElement) {
                const nodeKeyOfCurrentProperty = indexNodeKeys.find(key => propertyJoinElement.targetPropertyName === key.propertyName);
                if (nodeKeyOfCurrentProperty)
                    fallback = nodeKeyOfCurrentProperty.columnType;
            }
            logger.error(`${property.nodeName}.${property.name}: no element of the reference join maps to property on the target node. Target node key ${JSON.stringify(indexNodeKeys, null, 2)}. Property join ${JSON.stringify(property.joinValues, null, 2)}. Falling back to ${fallback}. This may be incorrect`);
            return fallback;
        }
        // We need to get the last property on the key, the type of this property is the type of the reference column
        return nodeKeys[nodeKeys.length - 1].columnType;
    }
    static async getGlobalVariables(connPool, x3Folder) {
        if (this.globalVariableData)
            return this.globalVariableData;
        const selectGlobalVariable = `SELECT VARVALUE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('VARVALUE_0', connPool.dialect)},
                                             VARTYP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('VARTYP_0', connPool.dialect)},
                                             CODVAR_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODVAR_0', connPool.dialect)},
                                             CODZONE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODZONE_0', connPool.dialect)},
                                             CODFIC_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODFIC_0', connPool.dialect)}
                                        FROM ${x3Folder}.ANODGVAR`;
        const results = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, selectGlobalVariable).readAll();
        });
        this.globalVariableData = results.reduce((r, k) => {
            const glVarName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', k.CODVAR_0);
            const columnName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', k.CODZONE_0);
            const tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', k.CODFIC_0);
            const key = `${glVarName}~${columnName}~${tableName}`;
            r[key] = k;
            return r;
        }, {});
        return this.globalVariableData ?? {};
    }
    static async getGlobalVariableValue(connPool, x3Folder, glVarName, columnName, tableName) {
        const key = `${glVarName}~${columnName}~${tableName}`;
        return (await this.getGlobalVariables(connPool, x3Folder))[key];
    }
    static async getReferenceJoinValues(connPool, x3Folder, property, dbProp) {
        if (!property.targetNode)
            throw new Error(`${property.nodeName}.${property.name}: reference property target node not set.`);
        const primaryIndex = await this.getNodePrimaryIndex(connPool, x3Folder, property.targetNode);
        const indexNodeKeys = primaryIndex.properties;
        const joinValues = [];
        const useDbProp = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.JOINCOD_0) != null;
        if (indexNodeKeys.length || useDbProp) {
            for (let index = 0; index < this.numberOfJoins; index += 1) {
                const joinValueColumn = `JOINVAL_${index}`;
                const joinTypeColumn = `JOINTYP_${index}`;
                const joinPropertyName = `JOINCOD_${index}`;
                if (!dbProp[joinTypeColumn] || dbProp[joinTypeColumn] === '0')
                    break;
                if (indexNodeKeys[index] == null && !useDbProp) {
                    logger.error(`${property.nodeName}.${property.name}: join key mismatch at index ${index}. Join value provided but no key available for value.`);
                    break;
                }
                const targetPropertyName = useDbProp
                    ? x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp[joinPropertyName])
                    : x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', indexNodeKeys[index].propertyName);
                const joinValue = {
                    value: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp[joinValueColumn]),
                    type: this.getJoinType(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp[joinTypeColumn])),
                    targetPropertyName,
                };
                // if joinType is globalVar
                if (joinValue.type === 'globalVariable' &&
                    property.columnName &&
                    x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.MAINTABLE_0)) {
                    const glVarVal = await this.getGlobalVariableValue(connPool, x3Folder, joinValue.value, property.columnName, x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.MAINTABLE_0));
                    if (glVarVal) {
                        joinValue.type = this.getJoinType(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', glVarVal.VARTYP_0));
                        joinValue.value = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', glVarVal.VARVALUE_0);
                    }
                    else {
                        logger.error(`${property.nodeName}.${property.name}: join key mismatch at index ${index}. Could not find global variable mapping.`);
                        break;
                    }
                }
                joinValues.push(joinValue);
            }
        }
        else {
            logger.error(`${property.nodeName}.${property.name}: node reference missing for join to ${property.targetNode}.`);
        }
        return joinValues;
    }
    static async getJoinValues(connPool, x3Folder, property, dbProp) {
        const joinValues = [];
        for (let index = 0; index < this.numberOfJoins; index += 1) {
            const joinValueColumn = `JOINVAL_${index}`;
            const joinTypeColumn = `JOINTYP_${index}`;
            const joinPropertyName = `JOINCOD_${index}`;
            if (dbProp[joinValueColumn] === ' ')
                break;
            const joinValue = {
                value: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp[joinValueColumn]),
                type: this.getJoinType(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp[joinTypeColumn])),
                targetPropertyName: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp[joinPropertyName]),
            };
            // if joinType is globalVar
            if (joinValue.type === 'globalVariable' &&
                property.columnName &&
                x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.MAINTABLE_0)) {
                const glVarVal = await this.getGlobalVariableValue(connPool, x3Folder, joinValue.value, property.columnName, x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.MAINTABLE_0));
                if (glVarVal) {
                    joinValue.type = this.getJoinType(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', glVarVal.VARTYP_0));
                    joinValue.value = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', glVarVal.VARVALUE_0);
                }
                else {
                    logger.error(`${property.nodeName}.${property.name}: join key mismatch at index ${index}. Could not find global variable mapping.`);
                    break;
                }
            }
            joinValues.push(joinValue);
        }
        return joinValues;
    }
    static async getDenormalizedNodeKeyProperties(connPool, x3Folder, nodeData, nodePackage) {
        const { nodeName, nodeBinding } = nodeData;
        const keyProperties = [
            {
                nodeName,
                nodeBinding,
                name: 'denormalizedIndex',
                packageName: nodeData.packageName || nodePackage,
                nodePackage,
                type: 'integer',
                isPublished: true,
                keyOrder: 2,
                keyPart: 1,
                denomColumnIndex: 0,
                lookupAccess: true,
            },
        ];
        if (!nodeData.denormalized?.parentNode)
            throw new Error(`${nodeName}: denormalized node missing parent node.`);
        const parentKeyProperties = (await this.getNodeProperties(connPool, x3Folder, nodeData.denormalized.parentNode, nodeBinding, nodePackage)).filter(prop => !!prop.keyPart);
        parentKeyProperties.forEach(parentKeyProperty => {
            const keyProperty = { ...parentKeyProperty, nodeName };
            keyProperty.type = keyProperty.columnType || keyProperty.type;
            if (keyProperty.joinKind)
                delete keyProperty.joinKind;
            if (keyProperty.targetNode)
                delete keyProperty.targetNode;
            if (keyProperty.joinValues)
                delete keyProperty.joinValues;
            if (keyProperty.isNullable)
                delete keyProperty.isNullable;
            if (keyProperty.isMutable)
                delete keyProperty.isMutable;
            if (keyProperty.provides)
                delete keyProperty.provides;
            if (keyProperty.keyPart)
                keyProperty.keyPart += 1;
            keyProperty.packageName = nodeData.packageName || nodePackage;
            keyProperties.push(keyProperty);
        });
        const vitalParentReference = {
            nodeName,
            nodeBinding,
            packageName: nodeData.packageName || nodePackage,
            nodePackage,
            name: '_denormalizedParent',
            type: 'reference',
            joinKind: 'reference',
            isVitalParent: true,
            isStored: true,
            columnType: 'string',
            targetNode: nodeData.denormalized?.parentNode,
            joinValues: parentKeyProperties.map(k => ({
                targetPropertyName: k.name,
                type: 'property',
                value: k.name,
            })),
            denomColumnIndex: 0,
        };
        keyProperties.push(vitalParentReference);
        return keyProperties;
    }
    static async getDenormalizedCollectionPropertyData(connPool, x3Folder, currentNode, propertyName) {
        const propertyData = await this.getPropertyData(connPool, x3Folder);
        const denormalizedProp = propertyData.find(prop => prop.ANODE_0 === currentNode && prop.DENCOLGR_0 === propertyName);
        const denormalizedNodeData = (await this.getDenormalizedNodeData(connPool, x3Folder, currentNode)).find((d) => d.DENCOLGR_0 === propertyName);
        if (denormalizedNodeData) {
            const parentNode = await this.findNodeData(connPool, x3Folder, x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedNodeData.PARENT_0));
            return {
                AREFBIND_0: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedProp.AREFBIND_0),
                ANODE_0: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedProp.ANODE_0),
                APROPNAM_0: propertyName,
                APACK_0: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedNodeData.APACK_0),
                KINDREF_0: 3,
                ATYPE_0: 6,
                ATARGET_0: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedNodeData.ANODE_0),
                MAINTABLE_0: parentNode.tableName,
                CODACT_0: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', denormalizedNodeData.CODACT_0),
            };
        }
        return undefined;
    }
    /**
     * Retrieve the property data at the end of a path
     * This walk searches and returns the raw DB property data
     * @param targetNode
     * @param path
     * @returns
     */
    static async getPropertyDataFromPath(connPool, x3Folder, targetNode, path) {
        let currentNode = targetNode;
        let property;
        const propertyData = await this.getPropertyData(connPool, x3Folder);
        await (0, xtrem_core_1.asyncArray)(path).forEach(async (k, i) => {
            if (propertyData.some(prop => prop.ANODE_0 === currentNode && prop.DENCOLGR_0 === k)) {
                // property is a denormalized collection we need to set property manually.
                property = await this.getDenormalizedCollectionPropertyData(connPool, x3Folder, currentNode, k);
                if (path.length - 1 !== i)
                    throw Error(`${targetNode}.${path.join('.')}=>${k}: invalid property in path. Denormalized collection is only allowed at the end of the path.`);
            }
            else {
                property = propertyData.find(prop => prop.ANODE_0 === currentNode && prop.APROPNAM_0 === k);
                if (property.DENCOLGR_0 != null && property.DENCOLGR_0 !== ' ')
                    throw Error(`${targetNode}.${path.join('.')}=>${k}: invalid property in path. Denormalized property is not allowed in the path. Collection group = '${property.DENCOLGR_0}', property in path = '${k}'`);
            }
            if (!property)
                throw Error(`${currentNode}.${k}: property data not found.`);
            if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', property.KINDREF_0) !== 1 && path.length - 1 !== i) {
                throw Error(`${targetNode}.${path.join('.')}=>${k}: invalid property in path.`);
            }
            currentNode = property.ATARGET_0 || '';
        });
        if (!property)
            throw Error(`${targetNode}.${path.join('.')}: property path not found.`);
        return property;
    }
    /**
     * This walk searches and returns the list of activity codes for the given path.
     * @param targetNode
     * @param path
     * @returns
     */
    static async getActivityCodesFromPath(connPool, x3Folder, nodeClassPropObject) {
        const compositeReference = this.anodePropertiesData[nodeClassPropObject.nodeName].find(prop => prop.name === nodeClassPropObject.composite?.referenceProperty);
        if (!compositeReference)
            throw Error(`${nodeClassPropObject.nodeName}.${nodeClassPropObject.composite?.referenceProperty}: property not found.`);
        const { targetNode, activityCode } = compositeReference;
        if (!targetNode)
            throw Error(`${nodeClassPropObject.nodeName}.${nodeClassPropObject.composite?.referenceProperty}: target node not set.`);
        let currentNode = targetNode;
        const path = nodeClassPropObject.composite ? nodeClassPropObject.composite.propertyPath.split('.') : [];
        let property;
        const activityCodes = [];
        if (activityCode)
            activityCodes.push(activityCode);
        await (0, xtrem_core_1.asyncArray)(path).forEach(async (k, i) => {
            const nodeData = await this.findNodeData(connPool, x3Folder, currentNode);
            if (nodeData.activityCode && !activityCodes.includes(nodeData.activityCode))
                activityCodes.push(nodeData.activityCode);
            const propertyData = await this.getPropertyData(connPool, x3Folder);
            if (propertyData.some(prop => prop.ANODE_0 === currentNode && prop.DENCOLGR_0 === k)) {
                // property is a denormalized collection we need to set property manually.
                property = await this.getDenormalizedCollectionPropertyData(connPool, x3Folder, currentNode, k);
            }
            else {
                property = propertyData.find(prop => prop.ANODE_0 === currentNode && prop.APROPNAM_0 === k);
            }
            if (!property)
                throw Error(`${currentNode}.${k}: property data not found in ${path.join()}.`);
            const propertyActivityCode = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', property.CODACT_0);
            if (!activityCodes.includes(propertyActivityCode))
                activityCodes.push(propertyActivityCode);
            if (i === path.length - 1)
                return;
            currentNode = property.ATARGET_0 || '';
        });
        return activityCodes;
    }
    static getPropertyDirection(direction) {
        switch (direction) {
            case 2:
                return 'output';
            case 3:
                return 'both';
            default:
                return 'input';
        }
    }
    /**
     * Determines the boolean value of a property argument's lookup access.
     *
     * This method evaluates the provided `mLookupAccess` and `aLookupAccess` arguments to determine
     * which value should be converted to a boolean using the `fromSql` method. If `mLookupAccess` is
     * neither `'0'` nor `null`, it is used for the conversion. Otherwise, if `mLookupAccess` is `'0'`
     * or `null`, `aLookupAccess` is used. If neither condition is met, the method returns `false`.
     *
     * @param mLookupAccess - The manual lookup access value, which may be a string or null.
     * @param aLookupAccess - The automatic lookup access value, used as a fallback when conditions for `mLookupAccess` are not met.
     * @returns A boolean indicating the resolved lookup access.
     */
    static getPropertyArgumentLookupAccess(mLookupAccess, aLookupAccess) {
        if (mLookupAccess !== '0' && mLookupAccess !== null)
            return x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', mLookupAccess);
        if (mLookupAccess === '0' || mLookupAccess === null)
            return x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', aLookupAccess);
        return false;
    }
    static async getCompositePropertyDerivedInfo(connPool, x3Folder, dbProp, nodePackage, propertyPackage) {
        const propertyPath = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APROPATH_0);
        const referenceProperty = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.ANODLNK_0);
        const type = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.ATYPE_0);
        let joinKind = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.KINDREF_0);
        let derivedLocalMenuNumber;
        let packageName = propertyPackage;
        const propertyData = await this.getPropertyData(connPool, x3Folder);
        const lookupAccess = this.getPropertyArgumentLookupAccess(dbProp.MLOOKUP_0, dbProp.ALOOKUP_0);
        const targetReferenceProperty = propertyData.find(prop => prop.ANODE_0 === x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.ANODE_0) &&
            prop.APROPNAM_0 === referenceProperty);
        if (!targetReferenceProperty || x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', targetReferenceProperty.KINDREF_0) !== 1)
            throw new Error(`${dbProp.ANODE_0}.${referenceProperty}: invalid composite reference`);
        const compositeTargetNodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', targetReferenceProperty.ATARGET_0);
        const compositePath = propertyPath.split('.');
        const targetProperty = await this.getPropertyDataFromPath(connPool, x3Folder, compositeTargetNodeName, compositePath);
        const targetType = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', targetProperty.ATYPE_0);
        const propertyType = targetType ?? type;
        if (propertyType === 6) {
            joinKind = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', targetProperty.KINDREF_0);
        }
        if (propertyType === 8)
            derivedLocalMenuNumber = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', targetProperty.JOINVAL_0);
        const compositeTargetNode = await this.findNodeData(connPool, x3Folder, compositeTargetNodeName);
        const targetPropertyPackage = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', targetProperty.APACK_0);
        if (packageName === nodePackage &&
            targetPropertyPackage &&
            targetPropertyPackage !== compositeTargetNode.packageName) {
            packageName = targetPropertyPackage;
        }
        const columnName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', targetProperty.ABIND_0);
        const nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', targetProperty.ANODE_0);
        const targetNode = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', targetProperty.ATARGET_0);
        if (packageName) {
            const destinationNodeData = await this.findNodeData(connPool, x3Folder, nodeName);
            // Destination property package does not match the node package (node extension)
            // and the destination property is different from the current composed property package
            // Print a warning that this may cause a cyclic dependencies or other unexpected errors.
            if (destinationNodeData.packageName !== packageName && nodePackage !== packageName) {
                logger.warn(() => `The composite property ${nodePackage}/${dbProp.ANODE_0}.${dbProp.APROPNAM_0} relates to the property path ${propertyPath}.
                The target property is from a node extension of ${nodeName} in ${packageName}.
                This composed property may cause unexpected errors such as cyclic package dependencies, errors dues to inactive packages/modules,etc,
                if it is not assigned to the correct package.`);
            }
        }
        return {
            nodeName,
            type: propertyType,
            joinKind,
            localMenuNuumber: derivedLocalMenuNumber,
            packageName,
            columnName,
            targetNode,
            lookupAccess,
        };
    }
    static computedDependsOnSplit(dbProp) {
        const depends = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APIDPD_0);
        if (depends) {
            return depends.split(',').map(d => _.trim(d));
        }
        return undefined;
    }
    static fillComputedProperty(dbProp, property) {
        if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.APIEVE_0) &&
            x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.APIEVE_0) > 0) {
            const computedEvent = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.APIEVE_0);
            property.computedProperty = {
                event: computedEvent,
            };
            property.computedProperty.depends = this.computedDependsOnSplit(dbProp);
            property.computedProperty.expression = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APIEXX_0);
            property.computedProperty.functionPackage = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.FNCPACK_0);
            property.computedProperty.functionPath = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APIFNCPATH_0);
            property.isNullable = true;
        }
    }
    static fillEnumProperty(dbProp, property, derivedLocalMenuNumber) {
        if (property.type === 'enum') {
            property.localMenuNumber =
                derivedLocalMenuNumber ?? x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.JOINVAL_0);
            if (!property.localMenuNumber)
                throw new Error(`${property.nodeName}.${property.name}: no local menu provided for enum property`);
        }
    }
    static fillPropertyProvides(dbProp, property) {
        if (property.name === dbProp.ACTRLFCY_0) {
            if (property.type !== 'reference') {
                throw new Error(`${property.nodeName}.${property.name}: property is not a reference, cannot set provides site.`);
            }
            property.provides = 'site';
        }
        if (property.name === dbProp.ACTRLACS_0) {
            if (property.type !== 'reference') {
                throw new Error(`${property.nodeName}.${property.name}: property is not a reference, cannot set provides accessCode.`);
            }
            property.provides = 'accessCode';
        }
    }
    static async fillReferenceProperty(connPool, x3Folder, dbProp, property, options) {
        if (property.type === 'reference' && !property.composite?.propertyPath) {
            if (!options?.fromOperation) {
                property.joinValues = await this.getReferenceJoinValues(connPool, x3Folder, property, dbProp);
                property.columnType = await this.getReferenceColumnType(connPool, x3Folder, property);
                if (!property.columnType)
                    logger.error(`${property.nodeName}.${property.name}: cannot determine reference column type`);
            }
            // Start with references for now, we may include a verification for collection joins later.
            // Composite references may be a problem in the future if we don't throw an error
            if (property.columnType &&
                Object.keys(property.joinValues || {}).length === 0 &&
                !property.computedProperty) {
                logger.error(`${property.nodeName}.${property.name}: cannot determine join for ${property.type} property, falling back to column type. Composite references may be a problem in the future if we don't throw an error`);
                property.type = property.columnType;
            }
        }
    }
    static fillReferenceLookup(property) {
        if (property.type === 'reference') {
            const scalarJoins = property.joinValues &&
                property.joinValues.filter(j => !['property', 'propertyPath', 'globalVariable', 'expression'].includes(j.type));
            // TODO: We need to remember -> In the future handle excluded types in two places look for X3NodeClassGenerator.setReferencePropertyDecoratorAttributes
            if (scalarJoins && scalarJoins.length > 0) {
                property.filters = property.filters ?? {};
                property.filters.lookup = property.filters.lookup || [];
                property.filters.lookup.push(...scalarJoins);
                property.filters.control = property.filters.lookup || [];
                property.filters.control.push(...scalarJoins);
            }
        }
    }
    static async buildPropertyFromDbDefinition(connPool, x3Folder, dbProp, nodePackage, options) {
        const propertyPath = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APROPATH_0);
        const referenceProperty = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.ANODLNK_0);
        let type = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.ATYPE_0);
        let joinKind = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.KINDREF_0) ?? '1';
        let derivedLocalMenuNumber;
        let columnName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.ABIND_0);
        const packageName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.APACK_0) || nodePackage;
        let targetNode = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.ATARGET_0);
        let lookupAccess = this.getPropertyArgumentLookupAccess(dbProp.MLOOKUP_0, dbProp.ALOOKUP_0);
        /**
         * If the property is a reference or collection, and has a composite path
         * we need to determine the joinKind (KINDREF) from the target property of the path
         */
        if (propertyPath) {
            const derivedValues = await this.getCompositePropertyDerivedInfo(connPool, x3Folder, dbProp, nodePackage, packageName);
            columnName = derivedValues.columnName;
            derivedLocalMenuNumber = derivedValues.localMenuNuumber;
            joinKind = derivedValues.joinKind;
            type = derivedValues.type;
            targetNode = derivedValues.targetNode;
            lookupAccess = derivedValues.lookupAccess;
        }
        const property = {
            nodeName: dbProp.ANODE_0,
            nodeBinding: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.AREFBIND_0),
            name: dbProp.APROPNAM_0,
            packageName,
            nodePackage,
            type: this.getPropertyType(type, dbProp.ANODE_0, dbProp.APROPNAM_0, joinKind),
            columnName,
            targetNode,
            isNullable: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', dbProp.AISNULL_0),
            isArray: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', dbProp.AISARRAY_0),
            keyOrder: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.KEYORDER_0) || 2,
            keyPart: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.KEYPART_0),
            joinKind: this.getPropertyKind(joinKind),
            activityCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.CODACT_0)
                ? x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.CODACT_0)
                : undefined,
            composite: {
                referenceProperty,
                propertyPath,
            },
            isMutable: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', dbProp.AISMUTABLE_0),
            isTransientInput: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', dbProp.ATRANSPROP_0),
            parentProperty: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.PARENTPROP_0),
            direction: this.getPropertyDirection(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.ASENS_0)),
            denomColumnIndex: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.INDEX_0),
            lookupAccess,
        };
        // Values for vitalValue ==> 1=None, 2=vitalParent, 3=vitalChild
        const vitalValue = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', dbProp.VITALREL_0);
        if (vitalValue === 2) {
            property.isVitalParent = true;
        }
        else if (vitalValue === 3) {
            property.isVital = true;
        }
        this.fillComputedProperty(dbProp, property);
        this.fillPropertyProvides(dbProp, property);
        if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.DENCOLNOD_0)) {
            property.nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.DENCOLNOD_0);
        }
        await this.fillReferenceProperty(connPool, x3Folder, dbProp, property, options);
        if (['collection', 'binaryStream', 'textStream'].includes(property.type) && property.isNullable !== undefined) {
            delete property.isNullable;
        }
        if ((property.type === 'localizedString' || property.type === 'collection') &&
            !property.composite?.propertyPath) {
            property.localizedStringInfo = {};
            property.localizedStringInfo.tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.LOCTAB_0);
            property.localizedStringInfo.columnName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', dbProp.LOCFLD_0);
            property.joinValues = await this.getJoinValues(connPool, x3Folder, property, dbProp);
        }
        this.fillEnumProperty(dbProp, property, derivedLocalMenuNumber);
        if (((property.columnName && property.type !== 'collection') || property.type === 'reference') &&
            !property.isTransientInput &&
            !property.computedProperty)
            property.isStored = true;
        property.isPublished = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', dbProp.APROPPUB_0);
        if ((property.type === 'reference' || property.type === 'collection') && property.isPublished) {
            const targetNodeDataModel = await this.findNodeData(connPool, x3Folder, targetNode);
            property.isPublished = targetNodeDataModel?.isPublished;
        }
        this.fillReferenceLookup(property);
        return property;
    }
    static async getPropertyData(connPool, x3Folder) {
        if (this.propertyDataDb)
            return this.propertyDataDb;
        const joinsColumns = [];
        for (let index = 0; index < this.numberOfJoins; index += 1) {
            joinsColumns.push(`ANODPBDG.JOINVAL_${index} ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias(`JOINVAL_${index}`, connPool.dialect)}`);
            joinsColumns.push(`ANODPBDG.JOINTYP_${index} ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias(`JOINTYP_${index}`, connPool.dialect)}`);
            joinsColumns.push(`ANODPBDG.JOINCOD_${index} ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias(`JOINCOD_${index}`, connPool.dialect)}`);
        }
        const selectNodeProp = `SELECT
                ANODEBDG.AREFBIND_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREFBIND_0', connPool.dialect)},
                ANODPROP.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                ANODPBDG.ABIND_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ABIND_0', connPool.dialect)},
                ANODPROP.APROPNAM_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPNAM_0', connPool.dialect)},
                ANODPBDG.KEYPART_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KEYPART_0', connPool.dialect)},
                ANODPBDG.KEYORDER_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KEYORDER_0', connPool.dialect)},
                ANODPBDG.KINDREF_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KINDREF_0', connPool.dialect)},
                ANODPROP.ATYPE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATYPE_0', connPool.dialect)},
                ANODPROP.ATARGET_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATARGET_0', connPool.dialect)},
                ANODPROP.AISNULL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AISNULL_0', connPool.dialect)},
                ANODPROP.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                ANODPBDG.LOCTAB_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('LOCTAB_0', connPool.dialect)},
                ANODPBDG.LOCFLD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('LOCFLD_0', connPool.dialect)},
                ANODPROP.DENCOLNOD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('DENCOLNOD_0', connPool.dialect)},
                ANODE.DENCOLGR_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('DENCOLGR_0', connPool.dialect)},
                ANODPROP.CODACT_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODACT_0', connPool.dialect)},
                ANODEBDG.MAINTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MAINTABLE_0', connPool.dialect)},
                ANODPROP.ANODLNK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODLNK_0', connPool.dialect)},
                ANODPROP.APROPATH_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPATH_0', connPool.dialect)},
                ANODPROP.AISMUTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AISMUTABLE_0', connPool.dialect)},
                ANODEBDG.FNC_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FNC_0', connPool.dialect)},
                ANODEBDG.ACTRLACS_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ACTRLACS_0', connPool.dialect)},
                ANODEBDG.ACTRLFCY_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ACTRLFCY_0', connPool.dialect)},
                ANODPROP.APIEVE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIEVE_0', connPool.dialect)},
                ANODPROP.FNCPACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FNCPACK_0', connPool.dialect)},
                ANODPROP.APIFNCPATH_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIFNCPATH_0', connPool.dialect)},
                ANODPROP.APIDPD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIDPD_0', connPool.dialect)},
                ANODPROP.APIEXX_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIEXX_0', connPool.dialect)},
                ANODPROP.ATRANSPROP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATRANSPROP_0', connPool.dialect)},
                ANODPROP.APROPPUB_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPPUB_0', connPool.dialect)},
                ANODPBDG.INDEX_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('INDEX_0', connPool.dialect)},
                ANODPRADD.MLOOKUP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MLOOKUP_0', connPool.dialect)},
                ANODPRADD.ALOOKUP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ALOOKUP_0', connPool.dialect)},
                ANODPRADD.VITALREL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('VITALREL_0', connPool.dialect)},
                ${joinsColumns.join()}
                FROM ` +
            `${x3Folder}.ANODPROP ` +
            `JOIN ${x3Folder}.ANODEBDG ON ${x3Folder}.ANODPROP.ANODE_0 = ${x3Folder}.ANODEBDG.ANODE_0 AND ${x3Folder}.ANODEBDG.ABDGTYP_0 IN (2,5) ` +
            `LEFT JOIN ${x3Folder}.ANODE ON ${x3Folder}.ANODPROP.DENCOLNOD_0 = ${x3Folder}.ANODE.ANODE_0 ` +
            `LEFT JOIN ${x3Folder}.ANODPRADD ON ${x3Folder}.ANODPROP.ANODE_0 = ${x3Folder}.ANODPRADD.ANODE_0 AND ${x3Folder}.ANODPROP.APROPNAM_0 = ${x3Folder}.ANODPRADD.APROPNAM_0 AND ((${x3Folder}.ANODE.DENCOLGR_0 IS NULL AND ${x3Folder}.ANODPRADD.DENCOLGR_0 = ' ') OR ${x3Folder}.ANODPRADD.DENCOLGR_0 = ${x3Folder}.ANODE.DENCOLGR_0)` +
            `LEFT JOIN  ${x3Folder}.ANODPBDG ON ${x3Folder}.ANODPBDG.APROPNAM_0 = ${x3Folder}.ANODPROP.APROPNAM_0 AND ${x3Folder}.ANODPROP.ANODE_0 = ${x3Folder}.ANODPBDG.ANODE_0 AND ${x3Folder}.ANODEBDG.AREFBIND_0 = ${x3Folder}.ANODPBDG.AREFBIND_0 AND ((${x3Folder}.ANODE.DENCOLGR_0 IS NULL AND ${x3Folder}.ANODPBDG.DENCOLGR_0 = ' ') OR ${x3Folder}.ANODPBDG.DENCOLGR_0 = ${x3Folder}.ANODE.DENCOLGR_0) ` +
            'WHERE ANODPROP.APROPNAM_0 NOT IN (\'_nodeStatus\') ' +
            'ORDER BY ANODPBDG.AREFBIND_0,ANODPBDG.ANODE_0, CASE WHEN ANODPBDG.KEYPART_0=0 THEN 9999 ELSE ANODPBDG.KEYPART_0 END, ANODPBDG.NUMLIG_0';
        // We filter out the _nodeStatus property as it is not a real property, we inject it into the API payload back to X3
        // but we do not want to expose it as a property in the graphql schema.
        this.propertyDataDb = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, selectNodeProp).readAll();
        });
        // we do a minor control here to restrict the use of properties starting with an underscore
        // we cannot user camelCase here as they are properties with numbers and consecutive uppercase letters
        const invalidPropertyNames = this.propertyDataDb
            .filter(prop => String(prop.APROPNAM_0).startsWith('_') &&
            !['_x3Transaction', '_x3Auuid'].includes(prop.APROPNAM_0))
            .map(iv => `${iv.ANODE_0}.${iv.APROPNAM_0} `)
            .join('\n');
        if (invalidPropertyNames) {
            throw new Error(`Invalid property name(s): \n${invalidPropertyNames}`);
        }
        return this.propertyDataDb;
    }
    static async getNodeProperties(connPool, x3Folder, nodeName, nodeBinding, nodePackage) {
        if (this.anodePropertiesData[nodeName] != null)
            return this.anodePropertiesData[nodeName];
        const propertyData = await this.getPropertyData(connPool, x3Folder);
        const result = propertyData.filter(row => {
            const dataNodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', row.ANODE_0);
            const dataDenormalizedNodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', row.DENCOLNOD_0);
            return ((dataNodeName === nodeName && dataDenormalizedNodeName == null) || dataDenormalizedNodeName === nodeName);
        });
        const nodeData = await this.findNodeData(connPool, x3Folder, nodeName);
        this.anodePropertiesData[nodeName] = [];
        if (nodeData.denormalized?.parentNode) {
            //  We need to inject the denormalized index and key properties of the parent for denormalized nodes
            this.anodePropertiesData[nodeName].push(...(await this.getDenormalizedNodeKeyProperties(connPool, x3Folder, nodeData, nodePackage)));
        }
        this.anodePropertiesData[nodeName].push(...(await (0, xtrem_core_1.asyncArray)(result)
            .map(dbProp => this.buildPropertyFromDbDefinition(connPool, x3Folder, dbProp, nodePackage))
            .toArray()));
        if (nodeData.denormalizedNodes && nodeData.denormalizedNodes.length > 0) {
            nodeData.denormalizedNodes.forEach(denormalizedNode => {
                logger.verbose(() => `Detected denormalized node ${denormalizedNode.nodeName} for ${nodeName}`);
                if (denormalizedNode.denormalized) {
                    logger.verbose(() => `Adding denormalized collection property ${denormalizedNode.denormalized?.collectionPropertyName} to ${nodeName}`);
                    this.anodePropertiesData[nodeName].push({
                        nodeName,
                        nodeBinding,
                        name: denormalizedNode.denormalized.collectionPropertyName,
                        packageName: denormalizedNode.packageName || nodePackage,
                        nodePackage,
                        type: 'collection',
                        targetNode: denormalizedNode.nodeName,
                        isVital: true,
                        isPublished: true,
                        reverseReference: '_denormalizedParent',
                        activityCode: denormalizedNode.denormalized.activityCode,
                        denomColumnIndex: 0,
                    });
                }
            });
        }
        return this.anodePropertiesData[nodeName];
    }
    static getJoinType(result) {
        switch (Number(result)) {
            case 1:
                return 'property';
            case 2:
                return 'string';
            case 3:
                return 'integer';
            case 4:
                return 'date';
            case 5:
                return 'decimal';
            case 6:
                return 'enum';
            case 7:
                return 'propertyPath';
            case 8:
                return 'globalVariable';
            case 9:
                return 'expression';
            default:
                throw new Error(`Invalid join type ${result}(${typeof result})`);
        }
    }
    static async getNodeIndexDefinitions(connPool, x3Folder) {
        if (this.nodeIndexDefinitions)
            return this.nodeIndexDefinitions;
        const joinsColumns = [];
        for (let index = 0; index < this.numberOfJoins; index += 1) {
            joinsColumns.push(`ANODREF.CODREF_${index} ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias(`CODREF_${index}`, connPool.dialect)}`);
            joinsColumns.push(`ANODREF.TYPREF_${index} ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias(`TYPREF_${index}`, connPool.dialect)}`);
        }
        const selectNodeJoinRef = `SELECT ANODREF.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                                          ANODREF.ISUNIQUE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ISUNIQUE_0', connPool.dialect)},
                                          ANODREF.ISPRIMARY_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ISPRIMARY_0', connPool.dialect)},
                                          ${joinsColumns.join()}
                                    FROM ${x3Folder}.ANODREF ORDER BY ANODREF.ANODE_0, ANODREF.CODIND_0`;
        const results = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, selectNodeJoinRef).readAll();
        });
        this.nodeIndexDefinitions = {};
        if (results.length === 0) {
            return this.nodeIndexDefinitions;
        }
        results.forEach(refJoin => {
            if (this.nodeIndexDefinitions && !this.nodeIndexDefinitions[refJoin.ANODE_0])
                this.nodeIndexDefinitions[refJoin.ANODE_0] = [];
            const joinProperties = [];
            for (let index = 0; index < this.numberOfJoins; index += 1) {
                if (refJoin[`CODREF_${index}`] === ' ')
                    break;
                const propertyName = refJoin[`CODREF_${index}`];
                let columnType = this.getJoinType(refJoin[`TYPREF_${index}`]);
                if (columnType === 'property' ||
                    columnType === 'propertyPath' ||
                    columnType === 'globalVariable' ||
                    columnType === 'expression') {
                    logger.error(() => `${refJoin.ANODE_0}.${propertyName}: invalid type in ANODREF, '${columnType}' is not allowed. Falling back to string as columnType.`);
                    columnType = 'string';
                }
                else if (columnType === 'enum') {
                    columnType = 'integer';
                }
                if (propertyName === ' ')
                    break;
                joinProperties.push({ propertyName, columnType });
            }
            this.nodeIndexDefinitions?.[refJoin.ANODE_0].push({
                properties: joinProperties,
                isUnique: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', refJoin.ISUNIQUE_0),
                isPrimary: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', refJoin.ISPRIMARY_0),
            });
        });
        logger.info(`Loaded ${results.length} indexes`);
        return this.nodeIndexDefinitions;
    }
    static async getNodeIndexes(connPool, x3Folder, nodeName) {
        const nodeIndexDefinitions = await this.getNodeIndexDefinitions(connPool, x3Folder);
        return nodeIndexDefinitions[nodeName] ?? [];
    }
    static async getNodePrimaryIndex(connPool, x3Folder, nodeName) {
        const nodeIndexDefinitions = await this.getNodeIndexDefinitions(connPool, x3Folder);
        const primaryIndex = nodeIndexDefinitions[nodeName].find(index => index.isPrimary);
        if (primaryIndex != null)
            return primaryIndex;
        throw new Error(`${nodeName}: missing reference join.`);
    }
    static async getNodeExtensionData(connPool, x3Folder) {
        if (this.nodeExtensionData)
            return this.nodeExtensionData;
        const sql = `SELECT DISTINCT anb.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                            anpb.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                            anb.AREFBIND_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREFBIND_0', connPool.dialect)},
                            anb.APACKAGE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('NODE_PACKAGE', connPool.dialect)},
                            anb.MAINTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MAINTABLE_0', connPool.dialect)},
                            anb.ABDGTYP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ABDGTYP_0', connPool.dialect)},
                            anb.AX3KEY1_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AX3KEY1_0', connPool.dialect)}
                        FROM
                            ${x3Folder}.ANODEBDG anb
                        JOIN ${x3Folder}.ANODPROP anpb ON
                            anb.ANODE_0 = anpb.ANODE_0
                            AND anb.APACKAGE_0 <> anpb.APACK_0
                            AND anpb.APACK_0 <> ' '
                        WHERE anb.ABDGTYP_0  IN (2,5)
                    UNION
                    SELECT DISTINCT anb.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                            anpb.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                            anb.AREFBIND_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREFBIND_0', connPool.dialect)},
                            anb.APACKAGE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('NODE_PACKAGE', connPool.dialect)},
                            anb.MAINTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MAINTABLE_0', connPool.dialect)},
                            anb.ABDGTYP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ABDGTYP_0', connPool.dialect)},
                            anb.AX3KEY1_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AX3KEY1_0', connPool.dialect)}
                        FROM
                            ${x3Folder}.ANODEBDG anb
                        JOIN ${x3Folder}.ANODPROP anpb ON
                            anb.ANODE_0 = anpb.ATARGET_0
                            AND anb.APACKAGE_0 <> anpb.APACK_0
                            AND anpb.APACK_0 <> ' '
                        LEFT JOIN ${x3Folder}.ANODE ON anpb.DENCOLNOD_0 = ${x3Folder}.ANODE.ANODE_0
                        JOIN ${x3Folder}.ANODPRADD anpad ON
                            anb.ANODE_0 = anpad.ANODE_0
                            AND anpb.APROPNAM_0 = anpad.APROPNAM_0
                            AND ((${x3Folder}.ANODE.DENCOLGR_0 IS NULL
                                AND anpad.DENCOLGR_0 = ' ')
                            OR anpad.DENCOLGR_0 = ${x3Folder}.ANODE.DENCOLGR_0)
                            AND anpad.VITALREL_0 = 2
                        WHERE anb.ABDGTYP_0 IN (2,5)
                     UNION
                    SELECT DISTINCT anb.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                           op.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                           anb.AREFBIND_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREFBIND_0', connPool.dialect)},
                           anb.APACKAGE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('NODE_PACKAGE', connPool.dialect)},
                           anb.MAINTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MAINTABLE_0', connPool.dialect)},
                           anb.ABDGTYP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ABDGTYP_0', connPool.dialect)},
                           anb.AX3KEY1_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AX3KEY1_0', connPool.dialect)}
                        FROM
                            ${x3Folder}.ANODEBDG anb
                        JOIN ${x3Folder}.APIOPE op ON
                            anb.ANODE_0 = op.APIREF_0
                            AND anb.APACKAGE_0 <> op.APACK_0
                            AND op.APACK_0 <> ' '
                            AND op.APIOPEOVW_0 = ' '
                        WHERE anb.ABDGTYP_0  IN (2,5) ORDER BY ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)}`;
        // Get a distinct nodes with extened properties
        const result = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, sql).readAll();
        });
        this.nodeExtensionData = result.map((nodeExtension) => {
            let tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', nodeExtension.MAINTABLE_0);
            // This is the node binding
            if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', nodeExtension.ABDGTYP_0) === 5) {
                tableName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', nodeExtension.AX_3_KEY_1_0);
            }
            return {
                nodeName: nodeExtension.ANODE_0,
                nodePackage: nodeExtension.NODE_PACKAGE,
                nodeBinding: nodeExtension.AREFBIND_0,
                package: nodeExtension.APACK_0,
                tableName,
            };
        });
        return this.nodeExtensionData;
    }
    static expandPropertyPath(targetNode, path) {
        let currentNode = this.nodeData[targetNode];
        const resultPath = [];
        path.forEach((p, i) => {
            let property = this.anodePropertiesData[currentNode.nodeName].find(prop => prop.name === p);
            // We don't find the property, if the current node a denormalized node, check if the property exists in the parent node
            if (currentNode.denormalized?.parentNode) {
                if (!property || property?.keyPart) {
                    currentNode = this.nodeData[currentNode.denormalized?.parentNode];
                    resultPath.push('_denormalizedParent');
                    property = this.anodePropertiesData[currentNode.nodeName].find(prop => prop.name === p);
                }
            }
            if (!property)
                throw new Error(`${targetNode}.${resultPath.join('.')}.${p}: property not found.`);
            if (property.type !== 'reference' && path.length - 1 !== i) {
                throw Error(`${targetNode}.${path.join('.')}=>${p}: invalid property in path (${property.type}).`);
            }
            resultPath.push(property.name);
            currentNode = this.nodeData[property.targetNode || ''];
        });
        return resultPath;
    }
    static getPropertyFromPath(targetNode, path) {
        let currentNode = targetNode;
        const expandedPath = this.expandPropertyPath(targetNode, path);
        let property;
        expandedPath.forEach((k, i) => {
            property = this.anodePropertiesData[currentNode].find(prop => prop.name === k);
            if (!property)
                throw Error(`${currentNode}.${k}: property not found.`);
            if (property.type !== 'reference' && expandedPath.length - 1 !== i) {
                throw Error(`${targetNode}.${expandedPath.join('.')}=>${k}: invalid property in path.`);
            }
            currentNode = property.targetNode || '';
        });
        if (!property)
            throw Error(`${targetNode}.${expandedPath.join('.')}: property path not found.`);
        return property;
    }
    static getComposedPropertyFromPath(composedProperty) {
        let property = { ...composedProperty };
        if (property.composite?.propertyPath) {
            const compositeReference = this.anodePropertiesData[property.nodeName].find(prop => prop.name === property.composite?.referenceProperty);
            if (!compositeReference)
                throw Error(`${property.nodeName}.${property.composite?.referenceProperty}: property not found.`);
            if (compositeReference.type !== 'reference')
                throw Error(`${property.nodeName}.${property.composite?.referenceProperty}: invalid composite reference on ${property.nodeName}.${property.name}`);
            const { targetNode } = compositeReference;
            property = this.getPropertyFromPath(targetNode || '', property.composite?.propertyPath.split('.'));
        }
        if (property.composite?.propertyPath)
            throw new Error(`${composedProperty.nodeName}.${composedProperty.name}: multi level composition is not allowed, please use set compiste path to end on a non-composed property`);
        return property;
    }
    static getOperationMethod(value) {
        switch (value) {
            case 1:
                return 'WINDOW';
            case 2:
                return 'IMPORT';
            case 3:
                return 'CLASS';
            case 4:
                return 'QUERY';
            case 5:
                return 'SCRIPT';
            case 6:
                return 'NODE_SCRIPT';
            default:
                throw new Error(`Invalid operation method ${value}`);
        }
    }
    static getOperationType(value) {
        switch (value) {
            case 1:
                return 'custom';
            case 2:
                return 'create';
            case 3:
                return 'update';
            case 4:
                return 'delete';
            case 5:
                return 'create/update/delete';
            default:
                throw new Error(`Invalid operation type ${value}`);
        }
    }
    static async getBaseNodeOperations(connPool, x3Folder) {
        if (this.baseNodeOperations)
            return this.baseNodeOperations;
        const sql = `SELECT
                            op.ANODE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('OP_ANODE_0', connPool.dialect)},
                            op.APIREF_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIREF_0', connPool.dialect)},
                            op.ANODOP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODOP_0', connPool.dialect)},
                            op.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                            op.AREQBDG_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREQBDG_0', connPool.dialect)},
                            op.AX3OPE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AX3OPE_0', connPool.dialect)},
                            op.OPEMUTTYP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('OPEMUTTYP_0', connPool.dialect)},
                            p.ANODE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('P_ANODE_0', connPool.dialect)},
                            p.ABDGTYP_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('P_ABDGTYP_0', connPool.dialect)},
                            op.AREQRSP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREQRSP_0', connPool.dialect)},
                            r.ANODE_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('R_ANODE_0', connPool.dialect)},
                            r.ABDGTYP_0  ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('R_ABDGTYP_0', connPool.dialect)},
                            op.CODACT_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODACT_0', connPool.dialect)},
                            op.FNC_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FNC_0', connPool.dialect)},
                            op.OPT_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('OPT_0', connPool.dialect)},
                            op.CANCRE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CANCRE_0', connPool.dialect)},
                            op.CANUPD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CANUPD_0', connPool.dialect)},
                            op.CANDEL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CANDEL_0', connPool.dialect)},
                            op.FUNCPACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FUNCPACK_0', connPool.dialect)},
                            op.FUNCPATH_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FUNCPATH_0', connPool.dialect)},
                            op.FUNCRNULL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FUNCRNULL_0', connPool.dialect)}
                        FROM ${x3Folder}.APIOPE op
                        JOIN ${x3Folder}.ANODEBDG p ON p.AREFBIND_0 = op.AREQBDG_0
                        LEFT JOIN ${x3Folder}.ANODEBDG r ON r.AREFBIND_0 = op.AREQRSP_0
                        WHERE op.APIOPEOVW_0 = ' ' ORDER BY op.ANODOP_0 ASC`;
        const result = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, sql).readAll();
        });
        this.baseNodeOperations = result.map((operation) => {
            const requestNodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.OP_ANODE_0);
            const nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.APIREF_0);
            const operationName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.ANODOP_0);
            const packageName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.APACK_0);
            const method = this.getOperationMethod(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', operation.AX_3_OPE_0));
            const type = this.getOperationType(x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', operation.OPEMUTTYP_0));
            const createFlag = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', operation.CANCRE_0);
            const updateFlag = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', operation.CANUPD_0);
            const deleteFlag = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', operation.CANDEL_0);
            // Default the parameter to an instance on the current main node
            const parameter = {
                type: 'node',
                nodeName,
                isMainNode: true,
            };
            // Default the return parameter to the keys of the main node
            const returnParam = {
                type: 'keys',
                nodeName,
                isMainNode: true,
            };
            if (type === 'custom') {
                if (method === 'SCRIPT' || method === 'NODE_SCRIPT') {
                    const getParamType = (t) => {
                        switch (t) {
                            case 2:
                            case 3:
                            case 5:
                                return 'node';
                            default:
                                return 'constructed';
                        }
                    };
                    parameter.nodeBindingType = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', operation.P_ABDGTYP_0);
                    parameter.type = getParamType(parameter.nodeBindingType);
                    parameter.isMainNode = false;
                    parameter.nodeBinding = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.AREQBDG_0);
                    parameter.nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.P_ANODE_0);
                    if (x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.AREQRSP_0) != null) {
                        returnParam.nodeBinding = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.AREQRSP_0);
                        returnParam.nodeBindingType = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', operation.R_ABDGTYP_0);
                        returnParam.nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.R_ANODE_0);
                    }
                    else {
                        returnParam.nodeBinding = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.AREQBDG_0);
                        returnParam.nodeBindingType = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('integer', operation.P_ABDGTYP_0);
                        returnParam.nodeName = x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.P_ANODE_0);
                    }
                    returnParam.type = getParamType(returnParam.nodeBindingType);
                    returnParam.isMainNode = false;
                }
            }
            return {
                nodeName,
                packageName,
                operationName,
                type,
                method,
                parameter,
                returnParam,
                activityCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.CODACT_0),
                authorizationCode: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.FNC_0),
                authorizationOption: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.OPT_0),
                isPublished: true,
                requestNodeName,
                createFlag,
                updateFlag,
                deleteFlag,
                functionPackage: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.FUNCPACK_0),
                functionPath: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('string', operation.FUNCPATH_0),
                functionReturnNull: x3_dictionary_helper_1.X3DictionaryHelper.fromSql('boolean', operation.FUNCRNULL_0),
            };
        });
        return this.baseNodeOperations;
    }
    static async getOperationNodeProperties(connPool, x3Folder, nodeBinding) {
        if (this.nodeOperationProperties)
            return this.nodeOperationProperties.filter(prop => prop.nodeBinding === nodeBinding);
        const sql = `SELECT
                        ANODEBDG.AREFBIND_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AREFBIND_0', connPool.dialect)},
                        ANODPROP.ANODE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODE_0', connPool.dialect)},
                        ANODPBDG.ABIND_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ABIND_0', connPool.dialect)},
                        ANODPROP.APROPNAM_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPNAM_0', connPool.dialect)},
                        ANODPBDG.KEYPART_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KEYPART_0', connPool.dialect)},
                        ANODPBDG.KEYORDER_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KEYORDER_0', connPool.dialect)},
                        ANODPBDG.KINDREF_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('KINDREF_0', connPool.dialect)},
                        ANODPROP.ATYPE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATYPE_0', connPool.dialect)},
                        ANODPROP.ATARGET_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATARGET_0', connPool.dialect)},
                        ANODPROP.AISNULL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AISNULL_0', connPool.dialect)},
                        ANODPROP.AISARRAY_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AISARRAY_0', connPool.dialect)},
                        ANODPROP.APACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APACK_0', connPool.dialect)},
                        ANODPBDG.LOCTAB_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('LOCTAB_0', connPool.dialect)},
                        ANODPBDG.LOCFLD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('LOCFLD_0', connPool.dialect)},
                        ANODPROP.DENCOLNOD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('DENCOLNOD_0', connPool.dialect)},
                        ANODPROP.CODACT_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('CODACT_0', connPool.dialect)},
                        ANODEBDG.MAINTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('MAINTABLE_0', connPool.dialect)},
                        ANODPROP.ANODLNK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODLNK_0', connPool.dialect)},
                        ANODPROP.APROPATH_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPATH_0', connPool.dialect)},
                        ANODPROP.AISMUTABLE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('AISMUTABLE_0', connPool.dialect)},
                        ANODPROP.ASENS_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ASENS_0', connPool.dialect)},
                        ANODPBDG.PARENTPROP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('PARENTPROP_0', connPool.dialect)},
                        ANODPBDG.JOINVAL_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('JOINVAL_0', connPool.dialect)},
                        ANODEBDG.FNC_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FNC_0', connPool.dialect)},
                        ANODEBDG.ACTRLACS_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ACTRLACS_0', connPool.dialect)},
                        ANODEBDG.ACTRLFCY_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ACTRLFCY_0', connPool.dialect)},
                        ANODPROP.APIEVE_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIEVE_0', connPool.dialect)},
                        ANODPROP.FNCPACK_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('FNCPACK_0', connPool.dialect)},
                        ANODPROP.APIFNCPATH_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIFNCPATH_0', connPool.dialect)},
                        ANODPROP.APIDPD_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIDPD_0', connPool.dialect)},
                        ANODPROP.APIEXX_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APIEXX_0', connPool.dialect)},
                        ANODPROP.ATRANSPROP_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ATRANSPROP_0', connPool.dialect)},
                        ANODPROP.APROPPUB_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('APROPPUB_0', connPool.dialect)}
                    FROM
                        ${x3Folder}.ANODPROP
                    JOIN ${x3Folder}.ANODPBDG ON
                        ${x3Folder}.ANODPBDG.APROPNAM_0 = ${x3Folder}.ANODPROP.APROPNAM_0
                        AND ${x3Folder}.ANODPROP.ANODE_0 = ${x3Folder}.ANODPBDG.ANODE_0
                    JOIN ${x3Folder}.ANODEBDG ON
                        ${x3Folder}.ANODPROP.ANODE_0 = ${x3Folder}.ANODEBDG.ANODE_0
                        AND ${x3Folder}.ANODEBDG.AREFBIND_0 = ${x3Folder}.ANODPBDG.AREFBIND_0
                    WHERE EXISTS (SELECT 1 FROM ${x3Folder}.APIOPE WHERE APIOPE.AREQBDG_0 = ANODEBDG.AREFBIND_0 OR APIOPE.AREQRSP_0 = ANODEBDG.AREFBIND_0)
                    ORDER BY
                        ANODPBDG.AREFBIND_0,ANODPBDG.NUMLIG_0`;
        // Get a distinct nodes with extened properties
        const result = await connPool.withConnection(cnx => {
            return connPool.createReader(cnx, sql).readAll();
        });
        this.nodeOperationProperties = await (0, xtrem_core_1.asyncArray)(result)
            .map(dbProp => {
            return this.buildPropertyFromDbDefinition(connPool, x3Folder, dbProp, '', { fromOperation: true });
        })
            .toArray();
        return this.nodeOperationProperties.filter(prop => prop.nodeBinding === nodeBinding);
    }
}
exports.X3NodeDictionaryHelper = X3NodeDictionaryHelper;
//# sourceMappingURL=x3-node-dictionary-helper.js.map