/** @module @sage/xtrem-x3-sql */
import { AnyValue, AsyncReader, AsyncResponse } from '@sage/xtrem-async-helper';
import { LoggerInterface } from '@sage/xtrem-shared';
import { PoolConfig, PoolReaderOptions, SqlExecuteOptions, SqlReadTableSchemaOptions, TableDefinition, TableFilter } from './types';
export type AnyLob = AnyValue;
export type Dialect = 'postgres' | 'oracle' | 'sqlServer';
export declare abstract class Pool<ConnectionT extends AnyValue = {}> {
    protected logger?: LoggerInterface | undefined;
    usePool: boolean;
    config: PoolConfig;
    protected connections: ConnectionT[];
    constructor(config: PoolConfig, logger?: LoggerInterface | undefined);
    /**
     * Getter method for dialect
     */
    abstract get dialect(): Dialect;
    /**
     * Returns string containing parameter with passed index i
     * @param i
     */
    abstract param(i: number): string;
    /**
     * Verifies that the supplied table name exists in the supplied schema
     * @param name
     * @param schema
     */
    abstract tableExists(name: string, schema: string): Promise<boolean>;
    /**
     * Verifies that the supplied table name exists in the supplied schema
     * @param name
     * @param schema
     */
    abstract viewExists(name: string, schema: string): Promise<boolean>;
    /**
     * Function that returns callback function withConnection to execute supplied SQL command, callback takes a ConnectionT as a parameter
     * @param sql SQL command
     * @param args SQL arguments
     * @param opts SQL execution options
     */
    abstract execute<T extends AnyValue>(sql?: string, args?: AnyValue[], opts?: SqlExecuteOptions): Promise<T>;
    /**
     * Function that execute SQL command on the provided connection and with the provided arguments and options.
     * @param cnx Database connection
     * @param sql SQL command
     * @param args SQL arguments
     * @param opts SQL options
     */
    abstract execute<T extends AnyValue>(cnx?: ConnectionT, sql?: string, args?: AnyValue[], opts?: SqlExecuteOptions): Promise<T>;
    /**
     * Function to execute passed callback function passing the current allocated connection to the callback and then releasing the connection
     * @param body
     */
    withConnection<T extends AnyValue>(body: (cnx: ConnectionT) => AsyncResponse<T>): Promise<T>;
    /**
     * Return the full qualified table name based on the passed table definition
     * @param def
     */
    abstract getFullTableDefName(def: TableDefinition): string;
    /**
     *  Return the full qualified table name based on the passed schema, table name and alias
     * @param schemaName
     * @param tableName
     * @param alias
     */
    abstract getFullTableName(schemaName: string, tableName: string, alias?: string): string;
    /**
     * Checks the connection passed in, if not connected then returns undefined
     * else returns the connection
     * @param cnx
     */
    protected abstract checkConnection(cnx: ConnectionT | undefined): ConnectionT | undefined;
    /**
     * Closes all connections
     * TODO: Review: this method is not called anywhere
     */
    closeConnections(): Promise<void>;
    /**
     * Gets the current database connection, if one does not exists then a connection is created
     */
    allocConnection(): Promise<ConnectionT>;
    /**
     * Closes supplied database connection
     * @param cnx
     * @param err
     */
    releaseConnection(cnx: ConnectionT, err?: Error): Promise<undefined>;
    /**
     * Creates a new connection
     */
    protected abstract createConnection(): Promise<ConnectionT>;
    /**
     * Closes supplied connection
     * @param cnx
     */
    protected abstract closeConnection(cnx: ConnectionT): AsyncResponse<void>;
    /**
     * Returns table definition for the passed in schema and table
     * @param schemaName
     * @param tableName
     * @param trackerOrOptions
     */
    abstract readTableSchema(schemaName: string, tableName: string, trackerOrOptions?: SqlReadTableSchemaOptions): Promise<TableDefinition>;
    /**
     * Returns a reader that could be used to read all the records of the table
     * @param tableDef is an object that has been created by readTableSchema()
     * @param filters
     */
    createTableReader<T extends AnyValue>(tableDef: TableDefinition, filters: TableFilter[]): AsyncReader<T>;
    /**
     * Creates a reader for the provided connection and SQL statement
     * @param cnx
     * @param sql
     * @param args
     * @param opts
     */
    abstract createReader<T extends AnyValue>(cnx: ConnectionT, sql: string, args?: AnyValue[], opts?: PoolReaderOptions): AsyncReader<T>;
}
//# sourceMappingURL=pool.d.ts.map