import { Command } from '@ckeditor/ckeditor5-core';
/**
 * The table row attribute command.
 *
 * This command is a base command for other table property commands.
 */
export default class QueryTableRowPropertyCommand extends Command {
    /**
     * Creates a new `TableRowPropertyCommand` instance.
     *
     * @param editor An editor in which this command will be used.
     * @param attributeName Table cell attribute name.
     * @param defaultValue The default value of the attribute.
     */
    constructor(editor, attributeName, defaultValue) {
        super(editor);
        this.attributeName = attributeName;
        this._defaultValue = defaultValue;
    }
    /**
     * @inheritDoc
     */
    refresh() {
        const editor = this.editor;
        const selection = editor.model.document.selection;
        const queryTableRow = selection.getFirstPosition()?.findAncestor('queryTableRow');
        this.isEnabled = !!queryTableRow;
        this.value = queryTableRow ? this._getValue(queryTableRow) : undefined;
    }
    /**
     * Executes the command.
     *
     * @fires execute
     * @param options.value If set, the command will set the attribute on the selected table.
     * If not set, the command will remove the attribute from the selected table.
     * @param options.batch Pass the model batch instance to the command to aggregate changes,
     * for example, to allow a single undo step for multiple executions.
     */
    execute(options = {}) {
        const model = this.editor.model;
        const selection = model.document.selection;
        const { value, batch } = options;
        const queryTableRow = selection.getFirstPosition()?.findAncestor('queryTableRow');
        if (!queryTableRow) {
            return;
        }
        const valueToSet = this._getValueToSet(value);
        model.enqueueChange(batch, writer => {
            if (valueToSet) {
                writer.setAttribute(this.attributeName, valueToSet, queryTableRow);
            }
            else {
                writer.removeAttribute(this.attributeName, queryTableRow);
            }
        });
    }
    /**
     * Returns the attribute value for a table.
     */
    _getValue(tableRow) {
        if (!tableRow) {
            return undefined;
        }
        return tableRow.getAttribute(this.attributeName);
    }
    /**
     * Returns the proper model value. It can be used to add a default unit to numeric values.
     */
    _getValueToSet(value) {
        return value;
    }
}
//# sourceMappingURL=query-table-row-property-command.js.map