import { COMMAND_CELL_BORDER_WIDTH_QUERY_TABLE } from '../../../constants';
import { addDefaultUnitToNumericValue, getSingleValue } from '../query-table-editing-utils';
import QueryTableCellPropertyCommand from './query-table-cell-property-command';
/**
 * The table cell border width command.
 *
 * The command is registered by the {@link module:table/tablecellproperties/tablecellpropertiesediting~TableCellPropertiesEditing} as
 * the `'tableCellBorderWidth'` editor command.
 *
 * To change the border width of selected cells, execute the command:
 *
 * ```ts
 * editor.execute( 'tableCellBorderWidth', {
 *   value: '5px'
 * } );
 * ```
 *
 * **Note**: This command adds the default `'px'` unit to numeric values. Executing:
 *
 * ```ts
 * editor.execute( 'tableCellBorderWidth', {
 *   value: '5'
 * } );
 * ```
 *
 * will set the `borderWidth` attribute to `'5px'` in the model.
 */
export default class QueryTableCellBorderWidthCommand extends QueryTableCellPropertyCommand {
    /**
     * Creates a new `TableCellBorderWidthCommand` instance.
     *
     * @param editor An editor in which this command will be used.
     * @param defaultValue The default value of the attribute.
     */
    constructor(editor, defaultValue) {
        super(editor, COMMAND_CELL_BORDER_WIDTH_QUERY_TABLE, defaultValue);
    }
    /**
     * @inheritDoc
     */
    _getAttribute(tableCell) {
        if (!tableCell) {
            return undefined;
        }
        return getSingleValue(tableCell.getAttribute(this.attributeName));
    }
    /**
     * @inheritDoc
     */
    _getValueToSet(value) {
        return addDefaultUnitToNumericValue(value, 'px');
    }
}
//# sourceMappingURL=query-table-cell-border-width-command.js.map