import { COMMAND_BORDER_WIDTH_QUERY_TABLE } from '../../../constants';
import { addDefaultUnitToNumericValue, getSingleValue } from '../query-table-editing-utils';
import QueryTablePropertyCommand from './query-table-property-command';
/**
 * The table width border command.
 *
 * The command is registered by the {@link module:table/tableproperties/tablepropertiesediting~TablePropertiesEditing} as
 * the `'tableBorderWidth'` editor command.
 *
 * To change the border width of the selected table, execute the command:
 *
 * ```ts
 * editor.execute( 'tableBorderWidth', {
 *   value: '5px'
 * } );
 * ```
 *
 * **Note**: This command adds the default `'px'` unit to numeric values. Executing:
 *
 * ```ts
 * editor.execute( 'tableBorderWidth', {
 *   value: '5'
 * } );
 * ```
 *
 * will set the `borderWidth` attribute to `'5px'` in the model.
 */
export default class QueryTableBorderWidthCommand extends QueryTablePropertyCommand {
    /**
     * Creates a new `TableBorderWidthCommand` instance.
     *
     * @param editor An editor in which this command will be used.
     * @param defaultValue The default value of the attribute.
     */
    constructor(editor, defaultValue) {
        super(editor, COMMAND_BORDER_WIDTH_QUERY_TABLE, defaultValue);
    }
    /**
     * @inheritDoc
     */
    _getValue(table) {
        if (!table) {
            return undefined;
        }
        return getSingleValue(table.getAttribute(this.attributeName));
    }
    /**
     * @inheritDoc
     */
    _getValueToSet(value) {
        const newValue = addDefaultUnitToNumericValue(value, 'px');
        return newValue;
    }
}
//# sourceMappingURL=query-table-border-width-command.js.map