import { MultiRootEditor } from '@ckeditor/ckeditor5-editor-multi-root';
import { InsertListPropertyDialog } from './components/insert-list-property-dialog/insert-list-property-dialog';
import * as React from 'react';
import { COMMAND_ZOOM_IN, COMMAND_ZOOM_OUT, OBJECT_TYPE_GLOBAL_PROPERTIES, OBJECT_TYPE_ROOT } from './constants';
import { LeftPanel } from './components/editor-panel/left-panel';
import { RightPanel } from './components/editor-panel/right-panel';
import { ConditionDialog } from './components/condition-dialog';
import ToggleDataPropertiesPanelUi from './plugins/toggle-data-properties-panel/toggle-data-properties-panel-ui';
import ToggleRightPanelUi from './plugins/toggle-right-panel/toggle-right-panel-ui';
import { useMultiRootEditor } from '@ckeditor/ckeditor5-react';
import ToggleGlobalPropertiesPanelUi from './plugins/toggle-global-properties-panel/toggle-global-properties-panel-ui';
import { getEditorAttributes, getEditorData, getStaticEditorConfig } from './utils';
import { round } from 'lodash';
export { OBJECT_TYPE_GLOBAL_PROPERTIES, OBJECT_TYPE_ROOT };
export * from './context-provider';
export * as constants from './constants';
const bodyFragments = ['header', 'body', 'footer'];
const normalizeHtmlData = (processor, data) => {
    return processor.toData(processor.toView(data));
};
export function XtremDocumentEditor({ carbonLocale, contextProvider, footerValue = '', headerValue = '', isDisabled, locale, localize, localizeEnumMember, marginBottom = 2, marginLeft = 2, marginRight = 2, marginTop = 2, onBlur, onChange, onDisplayNotification, onFocus, onFooterChange, onHeaderChange, onReady, onTelemetryEvent: onTelemetryEventProp, paperOrientation = 'portrait', paperSize = 'a4', value, }) {
    const [zoomLevel, setZoomLevel] = React.useState(parseFloat(window.localStorage.getItem('DOCUMENT_EDITOR_ZOOM_LEVEL_KEY') || '1'));
    const onTelemetryEvent = React.useCallback(
    // eslint-disable-next-line no-console
    (event, data) => onTelemetryEventProp?.(event, data) ?? console.log({ event, data }), [onTelemetryEventProp]);
    const [lookupRequest, setLookupRequest] = React.useState(null);
    const [conditionDialogRequest, setConditionDialogRequest] = React.useState(null);
    const [isRightPanelOpen, setRightPanelOpen] = React.useState(false);
    const [isDataPropertiesPanelOpen, setDataPropertiesPanelOpen] = React.useState(false);
    const [isGlobalPropertiesPanelOpen, setGlobalPropertiesPanelOpen] = React.useState(false);
    React.useEffect(() => {
        window.localStorage.setItem('DOCUMENT_EDITOR_ZOOM_LEVEL_KEY', zoomLevel.toString());
    }, [zoomLevel]);
    const onGlobalPropertiesPanelOpenChange = React.useCallback((isOpen) => {
        if (isOpen) {
            setDataPropertiesPanelOpen(false);
        }
        setGlobalPropertiesPanelOpen(isOpen);
    }, []);
    const onDataPropertiesPanelOpenChange = React.useCallback((isOpen) => {
        if (isOpen) {
            setGlobalPropertiesPanelOpen(false);
        }
        setDataPropertiesPanelOpen(isOpen);
    }, []);
    const onZoomIn = React.useCallback(() => {
        if (zoomLevel > 1.6) {
            onDisplayNotification(localize('@sage/xtrem-document-editor/max-zoom-level-reached', 'Maximum zoom level reached'), 'warning');
            return;
        }
        setZoomLevel(round(zoomLevel + 0.1, 2));
    }, [localize, onDisplayNotification, zoomLevel]);
    const onZoomOut = React.useCallback(() => {
        if (zoomLevel < 0.5) {
            onDisplayNotification(localize('@sage/xtrem-document-editor/min-zoom-level-reached', 'Minimum zoom level reached'), 'warning');
            return;
        }
        setZoomLevel(round(zoomLevel - 0.1, 2));
    }, [localize, onDisplayNotification, zoomLevel]);
    const onEditorValueChange = React.useCallback((event, editor) => {
        const sourceDocument = event.source;
        if (sourceDocument.differ.getChanges().length === 0) {
            return;
        }
        const rootName = sourceDocument.selection.getFirstRange()?.start.root.rootName;
        const newData = editor.getData({ rootName });
        const processor = editor.data.processor;
        if (rootName === 'header' && onHeaderChange) {
            const normalizedNewData = normalizeHtmlData(processor, newData);
            const normalizedHeaderValue = normalizeHtmlData(processor, headerValue);
            if (normalizedNewData !== normalizedHeaderValue) {
                onHeaderChange(newData);
            }
        }
        if (rootName === 'footer' && onFooterChange) {
            const normalizedNewData = normalizeHtmlData(processor, newData);
            const normalizedFooterValue = normalizeHtmlData(processor, footerValue);
            if (normalizedNewData !== normalizedFooterValue) {
                onFooterChange(newData);
            }
        }
        if (rootName === 'body') {
            const normalizedNewData = normalizeHtmlData(processor, newData);
            const normalizedPreviousValue = normalizeHtmlData(processor, value);
            if (normalizedNewData !== normalizedPreviousValue) {
                onChange(newData);
            }
        }
    }, [onChange, onFooterChange, onHeaderChange, footerValue, headerValue, value]);
    const { data, editor: editorRef, toolbarElement, setData, setAttributes, editableElements, } = useMultiRootEditor({
        editor: MultiRootEditor,
        data: getEditorData({ paperOrientation, value, headerValue, footerValue }),
        onReady,
        onBlur,
        onFocus,
        rootsAttributes: getEditorAttributes({ paperOrientation, marginLeft, marginRight }),
        onChange: onEditorValueChange,
        config: {
            ...getStaticEditorConfig(locale),
            reportEditorConfig: {
                contextProvider,
                openLookup: setLookupRequest,
                onOpenConditionDialog: setConditionDialogRequest,
                onRightPanelOpenChange: setRightPanelOpen,
                onDataPropertiesPanelOpenChange,
                onGlobalPropertiesPanelOpenChange,
                onDisplayNotification,
                localize,
            },
        },
    });
    React.useEffect(() => {
        if (editorRef) {
            const dataPropertiesPanelPlugin = editorRef.plugins.get(ToggleDataPropertiesPanelUi.pluginName);
            const globalPropertiesPanelPlugin = editorRef.plugins.get(ToggleGlobalPropertiesPanelUi.pluginName);
            const rightPanelPlugin = editorRef.plugins.get(ToggleRightPanelUi);
            dataPropertiesPanelPlugin.view.isEnabled = !isDisabled;
            globalPropertiesPanelPlugin.view.isEnabled = !isDisabled;
            rightPanelPlugin.view.isEnabled = !isDisabled;
            if (isDisabled) {
                setRightPanelOpen(false);
                setDataPropertiesPanelOpen(false);
                setGlobalPropertiesPanelOpen(false);
            }
        }
    }, [editorRef, isDisabled]);
    React.useEffect(() => {
        if (editorRef) {
            editorRef.on(COMMAND_ZOOM_IN, onZoomIn);
            editorRef.on(COMMAND_ZOOM_OUT, onZoomOut);
        }
        return () => {
            if (editorRef) {
                editorRef.off(COMMAND_ZOOM_IN, onZoomIn);
                editorRef.off(COMMAND_ZOOM_OUT, onZoomOut);
            }
        };
    }, [editorRef, zoomLevel, onZoomIn, onZoomOut]);
    React.useEffect(() => {
        if (!editorRef) {
            return;
        }
        // This hook should only trigger on external value updates, that's why the `data` is excluded from the dependency list
        if (normalizeHtmlData(editorRef.data.processor, value) !==
            normalizeHtmlData(editorRef.data.processor, data.body) ||
            normalizeHtmlData(editorRef.data.processor, headerValue) !==
                normalizeHtmlData(editorRef.data.processor, data.header) ||
            normalizeHtmlData(editorRef.data.processor, footerValue) !==
                normalizeHtmlData(editorRef.data.processor, data.footer)) {
            setData(getEditorData({ paperOrientation, value, headerValue, footerValue }));
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [setData, footerValue, headerValue, value, paperOrientation]);
    React.useEffect(() => {
        setAttributes(getEditorAttributes({ paperOrientation, marginLeft, marginRight }));
    }, [setAttributes, marginLeft, marginRight, paperOrientation]);
    const footerLabel = React.useMemo(() => localize('@sage/xtrem-document-editor/footer-label', 'Footer'), [localize]);
    const containerStyle = React.useMemo(() => {
        const tableBodyLabel = localize('@sage/xtrem-document-editor/table-body-label', 'Table body');
        const tableGroup1FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 1 });
        const tableGroup2FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 2 });
        const tableGroup3FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 3 });
        const tableGroup4FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 4 });
        const tableGroup5FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 5 });
        const tableGroup6FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 6 });
        const tableGroup7FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 7 });
        const tableGroup8FooterLabel = localize('@sage/xtrem-document-editor/table-group-footer-label', 'Group {{level}} footer', { level: 8 });
        const translationKeysStyle = {
            '--tableBodyLabel': `"${tableBodyLabel}"`,
            '--tableFooterLabel': `"${footerLabel}"`,
            '--tableGroup1FooterLabel': `"${tableGroup1FooterLabel}"`,
            '--tableGroup2FooterLabel': `"${tableGroup2FooterLabel}"`,
            '--tableGroup3FooterLabel': `"${tableGroup3FooterLabel}"`,
            '--tableGroup4FooterLabel': `"${tableGroup4FooterLabel}"`,
            '--tableGroup5FooterLabel': `"${tableGroup5FooterLabel}"`,
            '--tableGroup6FooterLabel': `"${tableGroup6FooterLabel}"`,
            '--tableGroup7FooterLabel': `"${tableGroup7FooterLabel}"`,
            '--tableGroup8FooterLabel': `"${tableGroup8FooterLabel}"`,
        };
        return zoomLevel === 1
            ? { ...translationKeysStyle }
            : {
                transform: `scale(${zoomLevel})`,
                marginTop: `${(zoomLevel - 1) * 10}cm`,
                marginBottom: `${(zoomLevel - 1) * 10}cm`,
                ...translationKeysStyle,
            };
    }, [localize, footerLabel, zoomLevel]);
    return (React.createElement("div", { className: "document-editor" },
        React.createElement(InsertListPropertyDialog, { carbonLocale: carbonLocale, contextProvider: contextProvider, locale: locale, localize: localize, localizeEnumMember: localizeEnumMember, lookupRequest: lookupRequest, onDisplayNotification: onDisplayNotification, onClose: () => {
                setLookupRequest(null);
            } }),
        React.createElement(ConditionDialog, { carbonLocale: carbonLocale, conditionRequest: conditionDialogRequest, contextProvider: contextProvider, editorRef: editorRef, locale: locale, localize: localize, localizeEnumMember: localizeEnumMember, onClose: () => {
                setConditionDialogRequest(null);
            } }),
        React.createElement("div", { className: "document-editor__toolbar" }, toolbarElement),
        React.createElement("div", { className: `document-editor__editable-container document-editor__editable-container-${paperOrientation}` },
            React.createElement(LeftPanel, { contextProvider: contextProvider, editorRef: editorRef, isOpen: isDataPropertiesPanelOpen, localize: localize, onOpenLookupDialog: setLookupRequest, rootContext: OBJECT_TYPE_ROOT }),
            React.createElement(LeftPanel, { contextProvider: contextProvider, editorRef: editorRef, isOpen: isGlobalPropertiesPanelOpen, localize: localize, onOpenLookupDialog: setLookupRequest, rootContext: OBJECT_TYPE_GLOBAL_PROPERTIES }),
            React.createElement(RightPanel, { carbonLocale: carbonLocale, contextProvider: contextProvider, editorRef: editorRef, isOpen: isRightPanelOpen, locale: locale, localize: localize, localizeEnumMember: localizeEnumMember, onTelemetryEvent: onTelemetryEvent, setConditionDialogRequest: setConditionDialogRequest }),
            React.createElement("div", { style: containerStyle, className: `ck-page-body ck-page-body-orientation-${paperOrientation} ck-page-body-size-${paperSize}` }, editableElements.map((e, index) => {
                const key = paperOrientation === 'fullScreen' ? 'body' : bodyFragments[index];
                const style = {};
                if (key === 'header') {
                    style.height = `${marginTop}cm`;
                }
                if (key === 'footer') {
                    style.height = `${marginBottom}cm`;
                }
                return (React.createElement("div", { className: `ck-page-fragment ck-page-fragment-${key}`, style: style, key: key },
                    key === 'header' && (React.createElement("div", { className: "ck-page-fragment-label-header" }, localize('@sage/xtrem-document-editor/header-label', 'Header'))),
                    e,
                    key === 'footer' && React.createElement("div", { className: "ck-page-fragment-label-footer" }, footerLabel)));
            })))));
}
export default XtremDocumentEditor;
//# sourceMappingURL=index.js.map