import { ActionPopover, ActionPopoverDivider, ActionPopoverItem, ActionPopoverMenuButton } from 'carbon-react/lib/components/action-popover';
import { Tabs, Tab } from 'carbon-react/lib/components/tabs';
import _, { cloneDeepWith, isString, get, findLastIndex, last, omit, isEqual, memoize, isEmpty, cloneDeep, noop as noop$1, camelCase, merge } from 'lodash';
import * as React from 'react';
import React__default, { useRef, useState, useEffect, PureComponent, useLayoutEffect, useCallback } from 'react';
import styled from 'styled-components';
import axios from 'axios';
import { ResponsiveContainer, LineChart, Line, CartesianGrid, XAxis, YAxis, Treemap, RadialBarChart, PolarAngleAxis, RadialBar, ComposedChart, Tooltip as Tooltip$1, Legend as Legend$1, Bar, PieChart, Pie, Cell, Sector } from 'recharts';
import Icon from 'carbon-react/lib/components/icon/icon.component';
import Tooltip from 'carbon-react/lib/components/tooltip';
import Typography from 'carbon-react/lib/components/typography';
import Decimal from 'decimal.js';
import Image from 'carbon-react/lib/components/image';
import Pill from 'carbon-react/lib/components/pill';
import { v4 } from 'uuid';
import Link from 'carbon-react/lib/components/link';
import Icon$1 from 'carbon-react/lib/components/icon';
import { Checkbox } from 'carbon-react/lib/components/checkbox';
import { InfiniteLoader, AutoSizer, ColumnSizer, Table as Table$1, List, CellMeasurerCache, CellMeasurer } from 'react-virtualized';
import Portrait from 'carbon-react/lib/components/portrait';
import * as tokens from '@sage/design-tokens/js/base/common';
import IconButton from 'carbon-react/lib/components/icon-button';
import Loader from 'carbon-react/lib/components/loader';
import Button from 'carbon-react/lib/components/button';
import ButtonMinor from 'carbon-react/lib/components/button-minor';
import { ButtonToggleGroup, ButtonToggle } from 'carbon-react/lib/components/button-toggle';
import { FilterableSelect, Option } from 'carbon-react/lib/components/select';
import { Chart } from 'react-charts';
import useLatestWhen from 'react-charts/lib/hooks/useLatestWhen';
import { VisualProcessEditorComponent } from '@sage/visual-process-editor';
import CarbonPreview from 'carbon-react/lib/components/preview';
import Box from 'carbon-react/lib/components/box';
import Textarea from 'carbon-react/lib/components/textarea';
import { flushSync } from 'react-dom';
import scrollIntoView from 'scroll-into-view-if-needed';
import ReactGridLayout, { WidthProvider, Responsive } from 'react-grid-layout';

const defaultLiterals = {
    action_clone: 'Clone',
    action_close: 'Close',
    action_delete: 'Delete',
    action_edit: 'Edit',
    action_share: 'Share',
    actions: 'Actions',
    add_contact: 'Add contact',
    add_note: 'Add note',
    add_site: 'Add address',
    address_name: 'Address name',
    address: 'Address',
    addresses: 'Addresses',
    cancel: 'Cancel',
    choose_date: 'Choose another date range',
    clear_filter: 'Clear filter',
    contact_card_empty_text: 'This widget has no contacts yet.',
    contact_image: 'Contact image',
    contact_type: 'Contact type',
    contact: 'Contact',
    contacts: 'Contacts',
    current_date_filter: 'Current date filter:',
    dashboard_settings: 'Dashboard settings',
    day: 'Day',
    delete: 'Delete',
    edit: 'Edit',
    email: 'Email',
    expand_row: 'Expand row',
    general_finish_editing: 'Finish Editing',
    general_invalid_json: 'Json not valid for current widget type',
    general_welcome_page: 'Welcome Page',
    month_1: 'January',
    month_10: 'October',
    month_11: 'November',
    month_12: 'December',
    month_2: 'February',
    month_3: 'March',
    month_4: 'April',
    month_5: 'May',
    month_6: 'June',
    month_7: 'July',
    month_8: 'August',
    month_9: 'September',
    month: 'Month',
    more_options: 'More options',
    next_period: 'Select next period',
    next: 'Next',
    no_change: 'No change',
    note: 'Note',
    notes: 'Notes',
    phone_number: 'Phone number',
    position: 'Position',
    previous_period: 'Select previous period',
    previous: 'Previous',
    quarter: 'Quarter',
    reload: 'Reload',
    save: 'Save',
    scroll_left: 'Scroll left',
    scroll_right: 'Scroll right',
    select_address: 'Select address',
    select_contact: 'Select contact',
    site_card_empty_text: 'This widget has no addresses yet.',
    site: 'Site',
    table_compare_with_previous_period: 'Compare with previous period',
    table_filter_menu: 'Filter menu',
    table_select_all: 'Select All',
    table_select: 'Select row',
    table_switch_to_card_view: 'Switch to card view',
    table_switch_to_chart_view: 'Switch to chart view',
    table_switch_to_contact_view: 'Switch to contact view',
    table_switch_to_site_view: 'Switch to site view',
    table_switch_to_table_view: 'Switch to table view',
    table_variant_area: 'Area View',
    table_variant_card: 'Card View',
    table_variant_table: 'Table View',
    view_switch: 'View switch',
    week: 'Week',
    year: 'Year',
    ai_generated_warning: 'This widget was generated with the help of AI. Please review its content for accuracy.',
};

const tokenReplace = '######';
function translateObject(obj, stringsLiterals) {
    if (!obj) {
        return obj;
    }
    const translated = cloneDeepWith(obj, objValue => {
        if (isString(objValue)) {
            const arrayVal = objValue
                .split('{')
                .join(`${tokenReplace}{`)
                .split('}')
                .join(`}${tokenReplace}`)
                .split(tokenReplace);
            const arrayValTranslated = arrayVal.map(val => {
                const valWithoutArc = val.split('{').join('').split('}').join('');
                return get(stringsLiterals, valWithoutArc) ?? val;
            });
            return arrayValTranslated.join('');
        }
    });
    return translated;
}
function resolveTranslationKey(stringLiterals, key) {
    if (key == null) {
        return key;
    }
    const valWithoutArc = key.split('{').join('').split('}').join('');
    const resolvedValue = get(stringLiterals, valWithoutArc);
    if (resolvedValue != null) {
        return resolvedValue;
    }
    return key;
}

function useAnchors(initialStateAnchors, store, stringLiterals) {
    const initialAnchors = useRef(translateObject(initialStateAnchors, stringLiterals));
    const [myAnchors, setMyAnchors] = useState(initialAnchors.current);
    const anchorDistanceLightUp = 100;
    useEffect(() => {
        if (initialAnchors.current.some(anchorItem => !document.getElementById(anchorItem.id))) {
            return;
        }
        const newAnchorList = initialAnchors.current
            .filter(anchorItem => document.getElementById(anchorItem.id) !== null)
            .map(anchorItem => {
            const element = document.getElementById(anchorItem.id);
            return { ...anchorItem, currentPosition: element.getBoundingClientRect().top };
        });
        let selectAnchorLight;
        for (let i = 0; i < newAnchorList.length; i++) {
            if (newAnchorList[i].currentPosition <= anchorDistanceLightUp)
                selectAnchorLight = newAnchorList[i].id;
        }
        const setAnchorLight = newAnchorList.map(anchorItem => {
            let isVisible = false;
            anchorItem.id === selectAnchorLight ? (isVisible = true) : (isVisible = false);
            return { ...anchorItem, visible: isVisible };
        });
        setMyAnchors(setAnchorLight);
    }, [store.renderOnChange]);
    return myAnchors;
}

const getWidgetAnchors = (storeStateWidgets, displayAnchors) => {
    const widgetList = storeStateWidgets
        .filter(widget => widget.type == 'separator' && displayAnchors)
        .map(widget => ({ title: widget.title, id: widget.i, y: widget.y, visible: false }));
    return widgetList.sort((a, b) => (a.y > b.y ? 1 : b.y > a.y ? -1 : 0));
};

class HttpService {
    source;
    instance;
    constructor() {
        this.instance = axios.create();
        this.source = axios.CancelToken.source();
    }
    get axiosInstance() {
        return this.instance;
    }
    resetCancelTokens() {
        this.source.cancel();
        this.source = axios.CancelToken.source();
    }
    get(url, config) {
        return this.instance.get(url, config);
    }
    delete(url, config) {
        return this.instance.delete(url, config);
    }
    post(url, data, config) {
        return this.instance.post(url, data, config);
    }
    put(url, data, config) {
        return this.instance.put(url, data, config);
    }
    patch(url, data, config) {
        return this.instance.patch(url, data, config);
    }
}
var httpService = new HttpService();

class DashboardHttpService {
    set requestInterceptor(requestInterceptor) {
        httpService.axiosInstance.interceptors.request.use(requestInterceptor?.onFulfilled, requestInterceptor?.onRejected);
    }
    set responseInterceptor(responseInterceptor) {
        httpService.axiosInstance.interceptors.response.use(responseInterceptor?.onFulfilled, responseInterceptor?.onRejected);
    }
}
var dashboard_http_service = new DashboardHttpService();

const NEW_WIDGET_ID = '__NEW_WIDGET';
const PENDING_WIDGET_ID = '__PENDING_ITEM';

function LineGraph({ data }) {
    return (React__default.createElement("div", { className: "db-widget-content widgetLineGraph" },
        React__default.createElement("div", { className: "fg1" },
            React__default.createElement(ResponsiveContainer, { width: "100%", aspect: 3 },
                React__default.createElement(LineChart, { width: 600, height: 300, data: data, margin: { top: 5, right: 20, bottom: 5, left: 0 } },
                    React__default.createElement(Line, { type: "monotone", dataKey: "uv", stroke: "#8884d8" }),
                    React__default.createElement(CartesianGrid, { stroke: "#ccc", strokeDasharray: "5 5" }),
                    React__default.createElement(XAxis, { dataKey: "name" }),
                    React__default.createElement(YAxis, null)))),
        React__default.createElement("div", { className: "db-widget-skeletonFooter" }, "Sticky footer of the content")));
}

const currenciesMapping = {
    'en-US': 'USD',
    'en-GB': 'GBP',
    'fr-FR': 'EUR',
    'es-ES': 'EUR',
    'pt-PT': 'EUR',
    'pt-BR': 'BRL',
    'ar-SA': 'SAR',
    'de-DE': 'EUR',
    'it-IT': 'EUR',
    'pl-PL': 'PLN',
    'zh-CN': 'CNY',
};
const formatNumber = (num, locale = 'en-US', precision = 2) => {
    const numberFormat = new Intl.NumberFormat(locale, {
        maximumFractionDigits: precision,
        minimumFractionDigits: precision,
    });
    return numberFormat.format(num);
};
const formatCurrencyNumber = (num, locale = 'en-US', precision = 2, currency) => {
    const numberFormat = new Intl.NumberFormat(locale, {
        maximumFractionDigits: precision,
        minimumFractionDigits: precision,
        useGrouping: true, // Es legislation not showing grouping for Nbrs under 10000
        style: 'currency',
        currency: currency ?? currenciesMapping[locale],
    });
    return numberFormat.format(num);
};

const TableWidget = ({ data, locale, stringLiterals, }) => {
    const [comparing, setComparing] = useState(true);
    const isPositiveValue = (element) => {
        return parseInt(getPercentage(element.value, element.previousValue, false)) > 0;
    };
    const generateRows = (elements) => {
        const rows = [];
        for (let index = 0; index < elements.length; index++) {
            const element = elements[index];
            rows.push(React__default.createElement("tr", null,
                React__default.createElement("td", { className: "values taRight" }, index + 1 + '.'),
                React__default.createElement("td", null,
                    React__default.createElement("a", { href: element.url, target: "_blank", rel: "noreferrer" }, element.name)),
                React__default.createElement("td", { className: comparing ? 'values' : 'hidden' },
                    formatNumber(element.previousValue, locale),
                    "\u20AC"),
                React__default.createElement("td", { className: "values" },
                    formatNumber(element.value, locale),
                    "\u20AC"),
                React__default.createElement("td", { className: comparing ? 'smallText taRight' : 'hidden' },
                    getPercentage(element.value, element.previousValue, true),
                    "%",
                    ' ',
                    React__default.createElement("svg", { version: "1.1", xmlns: "http://www.w3.org/2000/svg", x: "0px", y: "0px", className: isPositiveValue(element) ? 'arrow arrowUp' : 'arrow arrowDown', viewBox: "0 0 213.333 213.333" },
                        React__default.createElement("g", null,
                            React__default.createElement("g", null,
                                React__default.createElement("polygon", { points: "0,53.333 106.667,160 213.333,53.333 \t\t" }))))),
                React__default.createElement("td", { className: "taRight" },
                    element.share,
                    "%")));
        }
        return rows;
    };
    function getPercentage(value, previousValue, withoutSign = false) {
        let difference = value - previousValue;
        if (withoutSign && difference < 0) {
            difference = difference * -1;
        }
        return ((difference * 100) / previousValue).toFixed(2);
    }
    try {
        return (React__default.createElement(React__default.Fragment, null, data ? (React__default.createElement(React__default.Fragment, null,
            React__default.createElement("table", { className: "tableWidget fg1" },
                React__default.createElement("thead", null,
                    React__default.createElement("tr", null,
                        React__default.createElement("th", { className: "numberColumn taRight" }, "#"),
                        React__default.createElement("th", null, "Name"),
                        React__default.createElement("th", { className: comparing ? 'taRight' : 'hidden' }, "April"),
                        React__default.createElement("th", { className: "taRight" }, "Purchases May \u20AC"),
                        React__default.createElement("th", { className: comparing ? '' : 'hidden' }),
                        React__default.createElement("th", { className: "taRight" }, "Share"))),
                generateRows(data.data),
                React__default.createElement("tr", null,
                    React__default.createElement("td", null),
                    React__default.createElement("td", null,
                        React__default.createElement("a", { href: "https://www.google.com", target: "_blank", rel: "noreferrer" }, data.others.name)),
                    React__default.createElement("td", { className: comparing ? 'values' : 'hidden' },
                        formatNumber(data.others.previousValue, locale),
                        "\u20AC"),
                    React__default.createElement("td", { className: "values" },
                        formatNumber(data.others.value, locale),
                        "\u20AC"),
                    React__default.createElement("td", { className: comparing ? '' : 'hidden' }),
                    React__default.createElement("td", { className: "taRight" },
                        data.others.share,
                        "%"))),
            ' ',
            React__default.createElement("div", { className: "mt2 mb1" },
                React__default.createElement("label", null,
                    React__default.createElement("input", { type: "checkbox", checked: comparing, onChange: () => setComparing(!comparing) }),
                    "\u00A0 Compare with previous period")))) : (React__default.createElement(React__default.Fragment, null))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
};
TableWidget.displayName = 'TableWidget';

class CustomizedContent extends PureComponent {
    render() {
        const { x, y, width, height, size, locale } = this.props;
        return (React__default.createElement("g", null,
            React__default.createElement("rect", { x: x, y: y, width: width, height: height, style: {
                    stroke: '#FFFFFF',
                    strokeWidth: 5,
                } }),
            React__default.createElement("text", { x: x + 15, y: y + 15 + 14, textAnchor: "left", fill: "#00000", style: {
                    stroke: '#000000',
                    strokeWidth: 1,
                }, fontSize: 14 },
                size ? formatNumber(size, locale) : React__default.createElement(React__default.Fragment, null),
                "\u20AC"),
            React__default.createElement("text", { x: x + 15, y: y + 35 + 14, textAnchor: "left", fill: "#6292c3", style: {
                    stroke: '#6292c3',
                    strokeWidth: 1,
                }, fontSize: 14 }, name)));
    }
}
const TreemapWidget = ({ data, locale, stringLiterals, }) => {
    try {
        return (React__default.createElement("div", { className: "tableWidget treemapWidget fg1" },
            React__default.createElement(ResponsiveContainer, { width: "100%" },
                React__default.createElement(Treemap, { data: data.tree, dataKey: "size", stroke: "#FFFFFF", fill: "#e8ecec", content: React__default.createElement(CustomizedContent, { locale: locale }) }))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
};
TreemapWidget.displayName = 'TreemapWidget';

const TableTreeMap = ({ data, stringLiterals, locale, }) => {
    const [mode, setMode] = useState('table');
    const [rowsNumber, setRowsNumber] = useState(10);
    const [tableData, setTableData] = useState();
    const [treemapData, setTreemapData] = useState();
    useEffect(() => {
        if (rowsNumber === 10) {
            const elements = getNElements(data.elements, 10);
            setTableData(elements);
            setTreemapData(formatDataToTreemap(data.elements, 10));
        }
        else if (rowsNumber === 20) {
            const elements = getNElements(data.elements, 20);
            setTableData(elements);
            setTreemapData(formatDataToTreemap(data.elements, 20));
        }
        else if (rowsNumber === 30) {
            setTableData(data.elements);
            setTreemapData(formatDataToTreemap(data.elements, 30));
        }
    }, [data.elements, rowsNumber]);
    const getNElements = (data, numberOfElements) => {
        const slicedData = data?.data?.slice(0, numberOfElements);
        const elements = { ...data, data: slicedData };
        return elements;
    };
    const formatDataToTreemap = (data, numberOfElements) => {
        const slicedData = data?.data?.slice(0, numberOfElements);
        const treeData = slicedData?.map(e => {
            return {
                name: e.name,
                size: e.value,
            };
        });
        const elements = { ...data, tree: treeData };
        return elements;
    };
    const renderWidget = () => {
        if (mode === 'table') {
            return React__default.createElement(TableWidget, { data: tableData, locale: locale, stringLiterals: stringLiterals });
        }
        else {
            return React__default.createElement(TreemapWidget, { data: treemapData, locale: locale, stringLiterals: stringLiterals });
        }
    };
    try {
        return (React__default.createElement("div", { className: "db-widget-content widgetTreeMap" },
            React__default.createElement("div", { className: "mb1" },
                React__default.createElement("select", { className: "topSelector", name: "top", id: "top", onChange: e => setRowsNumber(parseInt(e.target.value)) },
                    React__default.createElement("option", { value: "10" }, "Top 10"),
                    React__default.createElement("option", { value: "20" }, "Top 20"),
                    React__default.createElement("option", { value: "30" }, "Top 30")),
                React__default.createElement("select", { className: "modeSelector", name: "mode", id: "mode", onChange: e => setMode(e.target.value) },
                    React__default.createElement("option", { value: "table" }, "Table"),
                    React__default.createElement("option", { value: "area" }, "Area"))),
            tableData ? renderWidget() : React__default.createElement(React__default.Fragment, null)));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
};
TableTreeMap.displayName = 'TableTreeMap';

function ProgressTracker({ title = '', progress, current, total, currency = '€', locale, stringLiterals, }) {
    const progressArray = [{ value: progress, name: 'value' }];
    const circleSize = 125;
    const getColor = (value) => {
        if (value <= 33) {
            return 'red';
        }
        else if (value > 33 && value <= 69) {
            return '#FFB500';
        }
        return '#009900';
    };
    try {
        return (React__default.createElement("div", { className: "db-widget-content widgetProgressTracker" },
            React__default.createElement("h2", { className: "mt0" }, resolveTranslationKey(stringLiterals, title)),
            React__default.createElement("div", { className: "responsiveContainer" },
                React__default.createElement(RadialBarChart, { width: circleSize, height: circleSize, cx: circleSize / 2, cy: circleSize / 2, innerRadius: circleSize / 2 - 10, outerRadius: circleSize - 10, barSize: 5, data: progressArray, startAngle: 90, endAngle: -270 },
                    React__default.createElement(PolarAngleAxis, { type: "number", domain: [0, 100], angleAxisId: 0, tick: false }),
                    React__default.createElement(RadialBar, { background: true, dataKey: "value", cornerRadius: circleSize / 2, fill: getColor(progressArray[0].value) }),
                    React__default.createElement("text", { x: circleSize / 2, y: circleSize / 2, textAnchor: "middle", dominantBaseline: "middle", className: "progress-label", fontSize: "2em", fontWeight: "bold", fill: getColor(progressArray[0].value) },
                        progress,
                        "%"))),
            React__default.createElement("div", { className: "mt1 mb1" },
                React__default.createElement("div", { className: "budget" },
                    React__default.createElement("a", null,
                        resolveTranslationKey(stringLiterals, total.name),
                        ":"),
                    React__default.createElement("a", { className: "data" },
                        formatNumber(total.value, locale),
                        " ",
                        currency)),
                React__default.createElement("div", { className: "balance" },
                    React__default.createElement("a", null,
                        resolveTranslationKey(stringLiterals, current.name),
                        ": "),
                    React__default.createElement("a", { className: "data" },
                        formatNumber(current.value, locale),
                        " ",
                        currency)))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
}

const DEFAULT_HEIGHT = 48;
const DEFAULT_WIDTH = 48;
const getImageUri = (serializedSVG) => `data:image/svg+xml;utf-8,${encodeURIComponent(serializedSVG)}`;
const onImageLoad = ({ canvas, width, height }) => (e) => {
    const image = e.target;
    const context = canvas.getContext('2d');
    if (!context) {
        return;
    }
    context.drawImage(image, 0, 0, width, height);
};
const fixHtmlColor = (html, color) => {
    if (!color)
        return html;
    return html.replace(/fill:[^;]*/gm, `fill:${color}`).replace(/fill="[^"]*"/gm, `fill="${color}"`);
};
const fetchSVG = async ({ canvasRef, url, color, width, height }) => {
    try {
        const response = await fetch(url);
        const html = await response.text();
        const parser = new DOMParser();
        const htmlWithColorFixed = fixHtmlColor(html, color);
        const doc = parser.parseFromString(htmlWithColorFixed, 'text/html');
        const svg = doc.body.children[0];
        const xmlSerializer = new XMLSerializer();
        const customSVGAttributes = {
            preserveAspectRatio: 'none',
        };
        Object.keys(customSVGAttributes).forEach(key => svg.setAttribute(key, customSVGAttributes[key]));
        const canvas = canvasRef.current;
        svg.setAttribute('width', String(width));
        svg.setAttribute('height', String(height));
        const serializedSVG = xmlSerializer.serializeToString(svg);
        const image = new window.Image();
        image.crossOrigin = 'Anonymous';
        image.onload = onImageLoad({ canvas, width, height });
        image.src = getImageUri(serializedSVG);
    }
    catch (error) {
        console.warn(`Could not load image from '${url}' due to the following error: `, error);
    }
};
const SvgIcon = ({ url, color, name }) => {
    const canvasRef = useRef(null);
    useEffect(() => {
        if (name)
            return;
        fetchSVG({ canvasRef, url, color, width: DEFAULT_WIDTH, height: DEFAULT_HEIGHT });
    }, [url, color, name]);
    return name ? (React__default.createElement(Icon, { type: name, color: color })) : (React__default.createElement("canvas", { height: DEFAULT_HEIGHT, width: DEFAULT_WIDTH, ref: canvasRef, className: "db-draggable-element-indicator" }));
};

function AiGeneratedIcon() {
    return (React__default.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", width: "20", height: "20", viewBox: "0 0 20 20", fill: "none" },
        React__default.createElement("g", { clipPath: "url(#clip0_5414_18518)" },
            React__default.createElement("path", { d: "M16.3795 9.79983L12.6031 8.31052C12.1868 8.14654 11.8575 7.81684 11.693 7.40058L10.2035 3.62467C9.66584 2.2615 7.73653 2.2615 7.19884 3.62467L5.70934 7.40058C5.54534 7.81684 5.21559 8.14611 4.79928 8.31052L1.02288 9.79983C-0.340471 10.3374 -0.340471 12.2665 1.02288 12.8041L4.79928 14.2934C5.21559 14.4574 5.5449 14.7871 5.70934 15.2034L7.19884 18.9793C7.73653 20.3424 9.66584 20.3424 10.2035 18.9793L11.693 15.2034C11.857 14.7871 12.1868 14.4578 12.6031 14.2934L16.3795 12.8041C17.7428 12.2665 17.7428 10.3374 16.3795 9.79983Z", fill: "black", fillOpacity: "0.9" }),
            React__default.createElement("path", { d: "M17.1733 5.65549C18.735 5.65549 20.001 4.38968 20.001 2.82823C20.001 1.26678 18.735 0.000976562 17.1733 0.000976562C15.6117 0.000976562 14.3457 1.26678 14.3457 2.82823C14.3457 4.38968 15.6117 5.65549 17.1733 5.65549Z", fill: "#00D639" })),
        React__default.createElement("defs", null,
            React__default.createElement("clipPath", { id: "clip0_5414_18518" },
                React__default.createElement("rect", { width: "20", height: "20", fill: "white", transform: "translate(0.000976562 0.000976562)" })))));
}

function AiGeneratedWarning({ stringLiterals }) {
    return (React__default.createElement("div", { className: "db-widget-skeleton-ai-generated-warning" },
        React__default.createElement(Tooltip, { message: stringLiterals.ai_generated_warning, position: "top" },
            React__default.createElement("div", null,
                React__default.createElement(AiGeneratedIcon, null)))));
}

const DifferenceValue = ({ differenceValue, locale }) => {
    return (React__default.createElement(React__default.Fragment, null,
        formatNumber(differenceValue, locale),
        "%",
        React__default.createElement("svg", { version: "1.1", xmlns: "http://www.w3.org/2000/svg", x: "0px", y: "0px", className: differenceValue > 0 ? 'arrow arrowUp' : 'arrow arrowDown', viewBox: "0 0 213.333 213.333" },
            React__default.createElement("g", null,
                React__default.createElement("g", null,
                    React__default.createElement("polygon", { points: "0,53.333 106.667,160 213.333,53.333 \t\t" }))))));
};
const SimpleCurrencyIndicator = ({ currencyCode, description = '', differenceValue, indicator, isAiGenerated, locale, stringLiterals, title = '', }) => {
    return (React__default.createElement("div", { className: "indicator-widget-content", "data-indicator-type": "currency" },
        title && (React__default.createElement("h3", { className: "indicatorWidgetTitle db-draggable-element-indicator" },
            title,
            " ",
            isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: stringLiterals }))),
        differenceValue !== undefined && !isNaN(differenceValue) ? (React__default.createElement("div", { className: "differenceValue" },
            React__default.createElement(DifferenceValue, { locale: locale, differenceValue: differenceValue }))) : null,
        indicator?.value !== undefined && !isNaN(indicator.value) ? (React__default.createElement("div", { className: "mainIndicatorValue" }, formatCurrencyNumber(indicator.value, locale, 2, currencyCode))) : null,
        indicator?.amount !== undefined && !isNaN(indicator.amount) ? (React__default.createElement("div", { className: "mainIndicatorAmount" }, formatNumber(indicator.amount, locale, 0))) : null,
        React__default.createElement("div", { className: "mainIndicatorTitle db-draggable-element-indicator" }, indicator?.title),
        React__default.createElement("div", { className: "description db-draggable-element-indicator" },
            React__default.createElement("a", null, description))));
};

const SimpleNumberIndicator = ({ description = '', locale, prefix, suffix, subtitle = '', value, }) => {
    const hasPrefix = typeof prefix === 'string';
    const hasSuffix = typeof suffix === 'string';
    return (React__default.createElement("div", { className: "indicator-widget-content", "data-indicator-type": "number" },
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line1 db-draggable-element-indicator" },
            hasPrefix && (React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--prefix", "data-testid": "indicator-widget-prefix" }, prefix)),
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--value", "data-testid": "indicator-widget-value" }, formatNumber(value, locale)),
            hasSuffix && (React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--suffix", "data-testid": "indicator-widget-suffix" }, suffix))),
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line2 db-draggable-element-indicator" },
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--subtitle", "data-testid": "indicator-widget-subtitle" }, subtitle)),
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line3 db-draggable-element-indicator" },
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--description", "data-testid": "indicator-widget-description" }, description))));
};
SimpleNumberIndicator.displayName = 'SimpleNumberIndicator';

const SimpleTextIndicator = ({ description = '', isAiGenerated, stringLiterals, subtitle = '', title, }) => {
    return (React__default.createElement("div", { className: "indicator-widget-content", "data-indicator-type": "text" },
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line1 db-draggable-element-indicator" },
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--title", "data-testid": "indicator-widget-title" },
                title,
                isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: stringLiterals }))),
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line2 db-draggable-element-indicator" },
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--subtitle", "data-testid": "indicator-widget-subtitle" }, subtitle)),
        React__default.createElement("div", { className: "indicator-widget-content__line indicator-widget-content__line--line3 db-draggable-element-indicator" },
            React__default.createElement("span", { className: "indicator-widget-content__line-content indicator-widget-content__line-content--description", "data-testid": "indicator-widget-description" }, description))));
};
SimpleTextIndicator.displayName = 'SimpleTextIndicator';

const getIndicatorProps = (type = 'currency', props) => {
    switch (type) {
        case 'currency':
            return {
                currencyCode: props.currencyCode,
                description: resolveTranslationKey(props.stringLiterals, props.description),
                differenceValue: props.differenceValue,
                indicator: props.mainIndicator,
                locale: props.locale,
                title: props.title,
                isAiGenerated: props.isAiGenerated,
                stringLiterals: props.stringLiterals,
            };
        case 'number': {
            const indicator = props.mainIndicator;
            return {
                description: resolveTranslationKey(props.stringLiterals, props.description),
                locale: props.locale,
                prefix: indicator.prefix,
                suffix: indicator.suffix,
                subtitle: indicator.subtitle,
                value: indicator.value,
                isAiGenerated: props.isAiGenerated,
                stringLiterals: props.stringLiterals,
            };
        }
        case 'text': {
            const indicator = props.mainIndicator;
            return {
                description: resolveTranslationKey(props.stringLiterals, props.description),
                subtitle: indicator.subtitle,
                title: indicator.title,
                isAiGenerated: props.isAiGenerated,
                stringLiterals: props.stringLiterals,
            };
        }
        default:
            return {
                currencyCode: props.currencyCode,
                description: props.description,
                differenceValue: props.differenceValue,
                indicator: props.mainIndicator,
                locale: props.locale,
                title: props.title,
                isAiGenerated: props.isAiGenerated,
                stringLiterals: props.stringLiterals,
            };
    }
};
function SimpleIndicator(props) {
    const indicatorProps = getIndicatorProps(props.type, props);
    try {
        return (React__default.createElement("div", { className: "db-widget-content widgetSimpleIndicator" },
            props.icon ? (React__default.createElement("div", { className: "indicatorWidgetIcon" },
                React__default.createElement(SvgIcon, { ...props.icon }))) : null,
            React__default.createElement("div", { className: "indicatorWidgetTexts" },
                (props.type === undefined || props.type === 'currency') && (React__default.createElement(SimpleCurrencyIndicator, { ...indicatorProps })),
                props.type === 'number' && (React__default.createElement(SimpleNumberIndicator, { ...indicatorProps })),
                props.type === 'text' && (React__default.createElement(SimpleTextIndicator, { ...indicatorProps })))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(props.stringLiterals, 'general_invalid_json'));
    }
}

const WidgetMainContainer = styled.div `
  display: flex;
  flex-direction: column;
  height: 100%;
  & p {
    margin-bottom: 0;
  }
`;
const ContainerComponent = ({ children, className }) => {
    return React__default.createElement("div", { className: className }, children);
};
const SummaryContainer = styled(ContainerComponent) `
  display: flex;
  justify-content: space-between;
  align-items: center;
`;
const MainTitleContainer = styled(ContainerComponent) `
  display: flex;
  gap: 2rem;
  align-items: center;
`;
const IndicatorTitleContainer = styled(ContainerComponent) `
  margin-bottom: 8px;
  & p {
    color: rgba(0, 0, 0, 0.55);
  }
`;
const IndicatorAmountContainer = styled(ContainerComponent) `
  margin-bottom: 8px;
`;
const DescriptionContainer = styled(ContainerComponent) `
  margin-top: 4px;
  & p {
    color: rgba(0, 0, 0, 0.55);
  }
`;
const IndicatorsContainerComponent = ({ children, className, }) => {
    return React__default.createElement("div", { className: className }, children);
};
const IndicatorsContainer = styled(IndicatorsContainerComponent) `
  display: flex;
  justify-content: space-around;
  margin-top: auto;
  & > div:not(:first-child):not(:last-child) {
    border-left: 0.5px solid #c6c6c6;
    border-right: 0.5px solid #c6c6c6;
  }
  ${({ indicatorsAmount }) => indicatorsAmount > 1
    ? `
    & > div:last-child {
      border-left: 0.5px solid #C6C6C6;
    }
    & > div:first-child {
      border-right: 0.5px solid #C6C6C6;
    }`
    : ''}
  & > div {
    width: ${({ indicatorsAmount }) => (indicatorsAmount > 0 ? 100 / indicatorsAmount : 100)}%;
  }
`;
const BodyContainer = styled(ContainerComponent) `
  display: flex;
  justify-content: center;
  margin: 8px 0px;
  flex-grow: 1;
  align-items: center;
  min-height: 34px;
`;
const PercentageTypography = styled(Typography) `
  font-family: 'Sage Headline', sans-serif;
  text-align: center;
`;
const IndicatorContainer = styled(ContainerComponent) `
  display: flex;
  flex-direction: column;
  align-items: flex-end;
`;
const RightContainer = styled.div `
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 2rem;
`;
const RoundedImage = styled(Image) `
  border-radius: 9999px;
`;
const computeTotal = (indicators) => indicators
    .reduce((acc, currentIndicator) => currentIndicator.value ? Decimal.add(acc, currentIndicator.value) : acc, new Decimal(0))
    .toNumber();
const getTotalAmount$1 = (indicators) => indicators
    .reduce((acc, currentIndicator) => currentIndicator.amount ? Decimal.add(acc, currentIndicator.amount) : acc, new Decimal(0))
    .toNumber();
const Indicator = ({ indicator, locale, currencyCode }) => {
    const { title, value, amount } = indicator;
    return (React__default.createElement(IndicatorContainer, { className: "indicator-container" },
        React__default.createElement(IndicatorTitleContainer, { className: "indicator-title-container" },
            React__default.createElement(Typography, { mr: "18px", fontWeight: "500", fontSize: "12px", lineHeight: "18px", className: "indicator-title" }, title)),
        amount !== undefined && !isNaN(amount) ? (React__default.createElement(IndicatorAmountContainer, { className: "amount-container" },
            React__default.createElement(Typography, { mr: "18px", fontWeight: "700", fontSize: "16px", lineHeight: "19px", className: "indicator-amount" }, formatNumber(amount, locale, 0)))) : null,
        value !== undefined && !isNaN(value) ? (React__default.createElement(Typography, { mr: "18px", fontWeight: "700", fontSize: "16px", lineHeight: "19px", className: "indicator-value" }, currencyCode
            ? formatCurrencyNumber(value, locale, 2, currencyCode)
            : formatCurrencyNumber(value, locale))) : null));
};
const MultipleIndicators = ({ currencyCode, customTotal, customTotalAmount, description, icon, image, indicators = [], isTotalAmountDisplayed, isTotalDisplayed, locale, pillColor, pillTextContent, title, percentageIndicatorValue, isPercentageIndicatorDisplayed = false, }) => {
    if (!indicators.length) {
        return React__default.createElement(React__default.Fragment, null);
    }
    return (React__default.createElement(WidgetMainContainer, null,
        React__default.createElement(SummaryContainer, { className: "summary-container", isPercentageIndicatorDisplayed: isPercentageIndicatorDisplayed },
            React__default.createElement(MainTitleContainer, { className: "title-container" },
                image !== undefined && (React__default.createElement(RoundedImage, { src: `data:image;base64,${image}`, alt: "Image", size: 48 })),
                image === undefined && icon && Object.keys(icon).length > 0 && React__default.createElement(SvgIcon, { ...icon }),
                React__default.createElement("div", null,
                    React__default.createElement(Typography, { className: "title-value", fontWeight: "700", fontSize: "22px", lineHeight: "26.4px" },
                        title,
                        isTotalAmountDisplayed
                            ? ` (${formatNumber(customTotalAmount ?? getTotalAmount$1(indicators), locale, 0)})`
                            : ''),
                    React__default.createElement(DescriptionContainer, { className: "description-container" },
                        React__default.createElement(Typography, { className: "description-value", fontWeight: "400", fontSize: "14px", lineHeight: "16.8px" }, description)))),
            React__default.createElement(RightContainer, null,
                isTotalDisplayed && (React__default.createElement(Typography, { className: "total-value", fontWeight: "700", fontSize: "22px", lineHeight: "26.4px" }, currencyCode
                    ? formatCurrencyNumber(customTotal ?? computeTotal(indicators), locale, 2, currencyCode)
                    : formatCurrencyNumber(customTotal ?? computeTotal(indicators), locale))),
                pillTextContent && (React__default.createElement(Pill, { borderColor: pillColor, size: "S", fill: true }, pillTextContent)))),
        React__default.createElement(BodyContainer, { className: "body-container" }, isPercentageIndicatorDisplayed && (React__default.createElement(PercentageTypography, { className: "percentage-indicator", fontWeight: "900", fontSize: "36px", lineHeight: "45px" },
            percentageIndicatorValue,
            "%"))),
        React__default.createElement(IndicatorsContainer, { className: "indicators-container", indicatorsAmount: indicators.length }, indicators.map((indicator, indicatorIndex) => (React__default.createElement(Indicator, { key: indicatorIndex, indicator: indicator, locale: locale, currencyCode: currencyCode }))))));
};

const DivWithIndent = ({ children, className }) => {
    return React__default.createElement("div", { className: className }, children);
};
const LinkContainer = styled(DivWithIndent) `
  margin-left: ${({ indent }) => indent * 16 + 16}px;
  & button {
    text-decoration: none;
  }
  & button:hover {
    text-decoration: underline;
  }
  margin-top: 4px;
  margin-bottom: 4px;
`;
const LabelContainer = styled(DivWithIndent) `
  margin-left: ${({ indent }) => indent * 16 + 16}px;
  margin-top: 4px;
  margin-bottom: 4px;
`;
const LeftLabelContainer = styled.div `
  margin-top: 4px;
  margin-bottom: 4px;
`;
const ListAndTotalContainer = styled.div `
  display: flex;
  flex-direction: column;
`;
const ListContainer = styled.div `
  display: flex;
  flex-direction: column;
`;
const ListLineContainer = styled.div `
  display: flex;
  flex-direction: row;
  justify-content: space-between;
`;
const ListLine = ({ outstanding, linksCallback }) => {
    const { label, value, indent, link } = outstanding;
    return (React__default.createElement(ListLineContainer, null,
        link ? (React__default.createElement(LinkContainer, { indent: indent },
            React__default.createElement(Link, { ...(linksCallback
                    ? {
                        onClick: () => {
                            linksCallback(link);
                        },
                    }
                    : { href: link }) }, label))) : (React__default.createElement(LabelContainer, { indent: indent }, label)),
        value ? React__default.createElement(LeftLabelContainer, null, value) : null));
};
const TotalLine = ({ total }) => {
    const { label, value } = total;
    return (React__default.createElement(ListLineContainer, null,
        React__default.createElement(Typography, { variant: "strong", ml: 16, mt: 16, mb: "4px" }, label ?? 'Total'),
        React__default.createElement(Typography, { variant: "strong", mb: "4px", mt: 16 }, value)));
};
const SimpleList = ({ simpleListData, linksCallback }) => {
    const { outstandings, total } = simpleListData;
    return (React__default.createElement(ListAndTotalContainer, null,
        React__default.createElement(ListContainer, null,
            outstandings?.map?.(outstanding => (React__default.createElement(ListLine, { key: v4(), outstanding: outstanding, linksCallback: linksCallback }))) ?? null,
            total ? React__default.createElement(TotalLine, { total: total }) : null)));
};

const HorizontalBar = ({ title, values, totalText, currency, locale, stringLiterals, }) => {
    const total = getTotalAmount(values);
    const totalAbsolue = getTotalAmountAbsolute(values);
    const divChart = useRef(null);
    const [canDisplay, setCanDisplay] = useState(false);
    const intervalId = useRef(null);
    function generateTooltipMessage(element, stringLiterals, currency) {
        return `${resolveTranslationKey(stringLiterals, element.tooltipTitle)}\n${getFormatValue(element.amount, locale, currency)}\n${resolveTranslationKey(stringLiterals, element.tooltipInfo)}`;
    }
    function generateBar(amountElements, stringLiterals, currency) {
        const elements = [];
        const chartWidth = divChart.current?.offsetWidth ?? 0;
        amountElements?.forEach((element, i) => {
            const percentage = getPercentage(Math.abs(element.amount), totalAbsolue);
            const colourClass = getColourClass(element.id);
            let overlay = null;
            if (colourClass === 'oldest' ||
                colourClass === 'positive' ||
                colourClass === 'late' ||
                colourClass === 'latest') {
                overlay = generateSummaryOverlay(element, colourClass, stringLiterals, locale, currency);
            }
            getWidthBar(Math.abs(element.amount), totalAbsolue, chartWidth);
            elements.push(React__default.createElement(Tooltip, { key: i, message: generateTooltipMessage(element, stringLiterals, currency) },
                React__default.createElement("div", { className: colourClass + ' overlay', style: {
                        width: percentage,
                    } }, overlay ? overlay : React__default.createElement(React__default.Fragment, null))));
        });
        if (amountElements.length > 0 && !intervalId.current) {
            intervalId.current = setInterval(checkChartContent, 200);
        }
        return elements;
    }
    function calculateOverlayStyle() {
        const chartRight = divChart?.current?.getBoundingClientRect().right;
        if (chartRight) {
            const arrayOverly = ['positive', 'late', 'latest'];
            arrayOverly.forEach(overly => {
                const currentOverly = divChart?.current?.querySelector(`div[class^='${overly}']`);
                if (currentOverly) {
                    const lastspan = currentOverly.querySelector('span:last-child');
                    const lastSpanRight = lastspan?.getBoundingClientRect()?.right ?? 0;
                    if (lastspan && lastSpanRight > chartRight) {
                        lastspan.style.right = '0px';
                        lastspan.style.left = 'unset';
                    }
                }
            });
            // logic not used for now
            /*     const oldestBar = document.querySelector(`div[class^='oldest']`);
            let previousSize = { left: 0, right: 0, bottom: 120 };
            if (oldestBar) {
              const oldestLastspan: HTMLElement | null = oldestBar.querySelector('span:last-child');
              if (oldestLastspan) {
                const { left, right } = oldestLastspan.getBoundingClientRect() ?? { left: 0, right: 0 };
                previousSize = { left, right, bottom: 120 };
              }
            }
            arrayOverly.forEach((overly) => {
              const currentOverly: HTMLElement | null = document.querySelector(`div[class^='${overly}']`);
              if (currentOverly) {
                const lastspan: HTMLElement | null = currentOverly.querySelector('span:last-child');
                if (lastspan) {
                  const currentSize = {
                    left: lastspan.getBoundingClientRect().left,
                    right: lastspan.getBoundingClientRect().right,
                  };
                  if (currentSize.left < previousSize.right) {
                    lastspan.style.bottom = `${previousSize.bottom === 120 ? 180 : 120}%`;
                    previousSize = { ...previousSize, bottom: previousSize.bottom };
                  } else {
                    previousSize = { ...currentSize, bottom: 120 };
                  }
                }
              }
            }); */
            const middlespans = Array.from(document.querySelectorAll('.overlay'));
            if (middlespans.length > 1) {
                const filtredSpans = middlespans.filter(span => !span.classList.contains('oldest') && !span.classList.contains('late'));
                filtredSpans?.forEach(span => {
                    const lastspan = span.querySelector('span:last-child');
                    lastspan && (lastspan.style.top = '120%');
                });
            }
        }
        if (divChart.current) {
            divChart.current.style.opacity = '1';
        }
    }
    const checkChartContent = () => {
        if (divChart.current && divChart.current.children.length > 0) {
            if (intervalId.current) {
                clearInterval(intervalId.current);
                intervalId.current = null;
            }
            calculateOverlayStyle();
        }
    };
    useEffect(() => {
        setCanDisplay(true);
        return () => {
            if (intervalId.current) {
                clearInterval(intervalId.current);
                intervalId.current = null;
            }
        };
    }, []);
    try {
        return (React__default.createElement("div", { className: "db-widget-content widgetHorizontalBar" },
            React__default.createElement("div", { className: "responsiveContainer" },
                title && (React__default.createElement(Typography, { variant: "h4", mt: 0, mb: 3 }, resolveTranslationKey(stringLiterals, title))),
                React__default.createElement("div", { ref: divChart, className: "chart" }, canDisplay && generateBar(values, stringLiterals, currency)),
                React__default.createElement("div", { className: "total mt2" },
                    getFormatValue(total, locale, currency),
                    ' ',
                    resolveTranslationKey(stringLiterals, totalText)))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
};
function getColourClass(id) {
    switch (id) {
        case 1:
            return 'oldest';
        case 2:
            return 'older';
        case 3:
            return 'justPast';
        case 4:
            return 'positive';
        case 5:
            return 'next';
        case 6:
            return 'shortLong';
        case 7:
            return 'late';
        case 8:
            return 'moreThanLate';
        case 9:
            return 'superLate';
        case 10:
            return 'latest';
    }
}
// prettier-ignore
const symbolToCurrencyMap = {
    '$': 'USD',
    '€': 'EUR',
    '£': 'GBP',
    '¥': 'JPY',
};
const getFormatValue = (value, locale, currency) => {
    const normalizedCurrency = currency && currency in symbolToCurrencyMap
        ? symbolToCurrencyMap[currency]
        : currency;
    return formatCurrencyNumber(value, locale, 2, normalizedCurrency);
};
function getTotalAmount(amountElements) {
    if (amountElements) {
        return amountElements?.reduce((a, b) => a + b.amount, 0);
    }
    return 0;
}
function getTotalAmountAbsolute(amountElements) {
    if (amountElements) {
        return amountElements?.reduce((a, b) => a + Math.abs(b.amount), 0);
    }
    return 0;
}
function getPercentage(amount, total) {
    return (amount * 100) / total + '%';
}
function getWidthBar(amount, total, chartWidth) {
    return chartWidth * (amount / total);
}
function generateSummaryOverlay(element, colourClass, stringLiterals, locale, currency) {
    let text = '';
    if (element.overlayText) {
        if (colourClass === 'oldest')
            text = ' ' + resolveTranslationKey(stringLiterals, element.overlayText);
        else if (colourClass === 'late')
            text = ' ' + resolveTranslationKey(stringLiterals, element.overlayText) + ' ';
        else if (colourClass === 'latest')
            text = '  ' + resolveTranslationKey(stringLiterals, element.overlayText) + ' ';
    }
    return (React__default.createElement("span", { className: "overlayInfoText" }, `${getFormatValue(element.amount, locale, currency)} ${text}`));
}
HorizontalBar.displayName = 'HorizontalBar';

function StatusReport({ data, elements, locale, stringLiterals, }) {
    const [legend, setLegend] = useState(false);
    const getElement = (component) => {
        switch (component.type) {
            case 'bar':
                return React__default.createElement(Bar, { dataKey: component.name, barSize: 20, className: component.class });
            case 'line':
                return (React__default.createElement(Line, { type: "linear", dataKey: component.name, dot: false, strokeWidth: 3, stroke: "#ffb500", className: component.class }));
            default:
                return React__default.createElement(React__default.Fragment, null);
        }
    };
    const getChartComponents = () => {
        const chartComponents = [];
        elements.forEach(element => {
            chartComponents.push(getElement(element));
        });
        return chartComponents;
    };
    const customLegend = React__default.useCallback(() => {
        const items = [];
        elements?.forEach((element) => {
            let total = 0;
            data?.forEach((col) => {
                total += col[element.name];
            });
            const className = `legendItem ${element.class}`;
            items.push(React__default.createElement("div", { className: className },
                React__default.createElement("a", { className: "legendItemValue" },
                    formatNumber(total, locale),
                    "\u20AC"),
                React__default.createElement("a", { className: "legendItemLabel" }, element.name)));
        });
        setLegend(React__default.createElement("div", { className: "legendBody" }, items));
    }, [data, elements, locale]);
    const getLegend = () => {
        return legend;
    };
    useEffect(() => {
        customLegend();
    }, [customLegend]);
    try {
        return (React__default.createElement("div", { className: "db-widget-content widgetStatusReport", style: { width: '100%' } }, getLegend() ? (React__default.createElement(ResponsiveContainer, { width: "100%", aspect: 3 },
            React__default.createElement(ComposedChart, { data: data, barGap: 0 },
                React__default.createElement(CartesianGrid, { vertical: false }),
                React__default.createElement(XAxis, { dataKey: "name", tickLine: false, fontWeight: "bold", tickMargin: 15 }),
                React__default.createElement(YAxis, { tickMargin: 15 }),
                React__default.createElement(Tooltip$1, null),
                React__default.createElement(Legend$1, { content: getLegend, align: "right", verticalAlign: "middle", layout: "vertical" }),
                React__default.createElement(CartesianGrid, { stroke: "#f5f5f5" }),
                getChartComponents()))) : (React__default.createElement(React__default.Fragment, null))));
    }
    catch (error) {
        return React__default.createElement("p", null, resolveTranslationKey(stringLiterals, 'general_invalid_json'));
    }
}

const hexToRgb = (hex) => {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})?$/i.exec(hex);
    return result
        ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16),
        }
        : null;
};

function TileIndicator({ color, currencyIndicator, icon, isAiGenerated, locale, onClick, stringLiterals, title = '', value = '', valueType, }) {
    const getBgColor = () => {
        if (color?.startsWith('#')) {
            const components = hexToRgb(color);
            if (components == null) {
                throw new Error(`Cannot convert ${color} to hex.`);
            }
            return `rgba(${components.r}, ${components.g}, ${components.b}, 0.2)`;
        }
        return color;
    };
    const body = (React__default.createElement(React__default.Fragment, null,
        React__default.createElement("div", { className: "iconContainer", style: { background: getBgColor() } },
            React__default.createElement(Icon$1, { type: icon, color: color, fontSize: "large" })),
        React__default.createElement("div", { className: "tileIndicatorBody" },
            React__default.createElement(Typography, { fontWeight: "700", lineHeight: "30px", fontSize: "30px", mb: "2", "data-test-id": "db-tile-indicator-value", className: "tile-indicator-value" },
                valueType === 'currency'
                    ? formatCurrencyNumber(Number(currencyIndicator?.value), locale, 2, currencyIndicator?.currencyCode)
                    : value,
                isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: stringLiterals })),
            React__default.createElement(Typography, { fontSize: "18px", mb: 0, color: "blackOpacity65", className: "tile-indicator-description" }, title))));
    return onClick ? (React__default.createElement("button", { className: "db-widget-content widgetTileIndicator db-draggable-element-indicator", onClick: onClick }, body)) : (React__default.createElement("div", { className: "db-widget-content widgetTileIndicator db-draggable-element-indicator" }, body));
}

const noop = () => {
    // Intentionally left empty.
};
const resolveValueOrCallback = (property, i) => {
    if (!property) {
        return null;
    }
    if (property instanceof Function) {
        return property(i);
    }
    return property;
};

const baseColorVariants = ['neutral', 'negative', 'positive', 'warning'];
function isBaseCorlorVariant(value) {
    return baseColorVariants.includes(value);
}
function getPillStyleAndVariant(variant) {
    const colorVariant = isBaseCorlorVariant(variant) ? variant : undefined;
    const style = variant == null || isBaseCorlorVariant(variant) ? undefined : colorfulPillPattern[variant];
    return {
        colorVariant,
        style,
    };
}
const colorfulPillPattern = {
    filledNeutral: {
        backgroundColor: tokens.colorsSemanticNeutral500,
        borderColor: tokens.colorsSemanticNeutral500,
        color: tokens.colorsSemanticNeutralYang100,
    },
    filledInformation: {
        backgroundColor: tokens.colorsSemanticInfo500,
        borderColor: tokens.colorsSemanticInfo500,
        color: tokens.colorsSemanticInfoYang100,
    },
    filledPositive: {
        backgroundColor: tokens.colorsSemanticPositive500,
        borderColor: tokens.colorsSemanticPositive500,
        color: tokens.colorsSemanticPositiveYang100,
    },
    filledNegative: {
        backgroundColor: tokens.colorsSemanticNegative500,
        borderColor: tokens.colorsSemanticNegative500,
        color: tokens.colorsSemanticNegativeYang100,
    },
    filledCaution: {
        backgroundColor: tokens.colorsSemanticCaution500,
        borderColor: tokens.colorsSemanticCaution500,
        color: tokens.colorsSemanticCautionYin090,
    },
    filledFocus: {
        backgroundColor: tokens.colorsSemanticFocus500,
        borderColor: tokens.colorsSemanticFocus500,
        color: tokens.colorsSemanticNeutralYang100, // colorsSemanticFocusYang100 doesn't exist
    },
    filledClosing: {
        backgroundColor: tokens.colorsGray1000,
        borderColor: tokens.colorsGray1000,
        color: tokens.colorsYang100,
    },
    outlinedNeutral: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticNeutral500,
        color: tokens.colorsSemanticNeutralYin090,
    },
    outlinedInformation: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticInfo500,
        color: tokens.colorsSemanticInfoYin090,
    },
    outlinedPositive: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticPositive500,
        color: tokens.colorsSemanticPositiveYin090,
    },
    outlinedNegative: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticNegative500,
        color: tokens.colorsSemanticNegativeYin090,
    },
    outlinedCaution: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticCaution500,
        color: tokens.colorsSemanticCautionYin090,
    },
    outlinedFocus: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsSemanticFocus500,
        color: tokens.colorsSemanticNeutralYin090, // colorsSemanticFocusYin090 doesn't exist
    },
    outlinedClosing: {
        backgroundColor: tokens.colorsYang100,
        borderColor: tokens.colorsGray1000,
        color: tokens.colorsSemanticNeutralYin090, // colorsSemanticClosingYin090 doesn't exist
    },
    default: {
        backgroundColor: tokens.colorsUtilityMajor200,
        borderColor: tokens.colorsUtilityMajor200,
        color: tokens.colorsSemanticNeutralYang100,
    },
    transparent: {
        backgroundColor: tokens.colorsTransparent,
        borderColor: tokens.colorsTransparent,
        color: tokens.colorsTransparent,
    },
    textBox: {
        backgroundColor: tokens.colorsTransparent,
        borderColor: tokens.colorsTransparent,
        color: tokens.colorsGray000,
    },
};

const StyledLinkForTable = styled(Link) `
  line-height: 40px;
  display: block;
  width: 100%;
  button {
    width: 100%;
    span {
      width: 100%;
      overflow: hidden;
      white-space: nowrap;
      text-overflow: ellipsis;
      display: block;
      text-align: left;
    }
  }
`;
const StyledLink = styled(Link) `
  line-height: var(--spacing150);
  display: block;
  width: 100%;
  button {
    font-size: 14px;
    width: 100%;
    span {
      width: 100%;
      overflow: hidden;
      white-space: nowrap;
      text-overflow: ellipsis;
      display: block;
      text-align: left;
    }
  }
`;
const StyledLargeLink = styled(Link) `
  line-height: 18px;
  button {
    font-size: 16px;
    font-weight: ${tokens.fontWeights500};
  }
`;
const colorVariantIconMapping = {
    neutral: 'var(--colorsSemanticNeutral500)',
    negative: 'var(--colorsSemanticNegative500)',
    positive: 'var(--colorsSemanticPositive500)',
    warning: 'var(--colorsSemanticCaution400)',
};
/**
 * Renders a value of the table or card view with formatting options
 */
const CollectionItemValueComponent = React__default.memo(({ value, definition, locale, isLeftPosition: isLeftPosition, i, isInTable, isBold, propertyName: key, }) => {
    let formattedValue = value;
    if (definition?.valueFormat === 'number') {
        formattedValue =
            value === '' ? '' : formatNumber(Number(value), locale, definition?.decimalDigits);
    }
    const onClick = () => definition?.onClick?.(i);
    const tooltipMessage = resolveValueOrCallback(definition?.displayOptions?.tooltipText, i);
    const testId = `db-table-value-${key}`;
    const isLabelDisplayed = definition?.displayOptions?.isLabelDisplayedOnCard;
    const labelValue = definition?.title;
    const getValueBody = () => {
        const renderedAs = resolveValueOrCallback(definition?.renderedAs, i);
        const colorVariant = resolveValueOrCallback(definition?.displayOptions?.colorVariant, i);
        switch (renderedAs) {
            case 'icon':
                return ((colorVariant && (React__default.createElement("span", { className: "db-table-icon-wrapper" },
                    isLabelDisplayed && !isInTable && labelValue + ': ',
                    React__default.createElement(Icon$1, { type: formattedValue, color: colorVariantIconMapping[colorVariant], "data-testid": testId })))) || React__default.createElement(React__default.Fragment, null));
            case 'pill':
                return (
                // The span is needed here so the tooltip component can apply on the pill.
                React__default.createElement("span", { className: "db-table-pill-wrapper" },
                    isLabelDisplayed && !isInTable && labelValue + ': ',
                    React__default.createElement(Pill, { size: "S", onClick: onClick, pillRole: "status", fill: !!colorVariant, "data-testid": testId, ...getPillStyleAndVariant(colorVariant) }, formattedValue)));
            case 'link':
                if (isInTable) {
                    return (React__default.createElement(StyledLinkForTable, { "data-testid": testId, onClick: onClick, variant: "neutral" },
                        isLabelDisplayed && !isInTable && labelValue + ': ',
                        " ",
                        formattedValue));
                }
                return isLeftPosition ? (React__default.createElement(StyledLargeLink, { "data-testid": testId, onClick: onClick, variant: "neutral" },
                    isLabelDisplayed && !isInTable && labelValue + ': ',
                    " ",
                    formattedValue)) : (React__default.createElement(StyledLink, { "data-testid": testId, onClick: onClick, variant: "neutral" },
                    isLabelDisplayed && !isInTable && labelValue + ': ',
                    " ",
                    formattedValue));
            default:
                const classes = ['db-table-value'];
                if (isBold) {
                    classes.push('db-table-value-bold');
                }
                return (React__default.createElement("span", { className: classes.join(' '), "data-testid": testId },
                    isLabelDisplayed && !isInTable && labelValue + ': ',
                    " ",
                    formattedValue));
        }
    };
    if (tooltipMessage) {
        return React__default.createElement(Tooltip, { message: tooltipMessage }, getValueBody());
    }
    return getValueBody();
});
CollectionItemValueComponent.displayName = 'TableValueComponent';

const useIsomorphicLayoutEffect = typeof window !== 'undefined' ? useLayoutEffect : useEffect;

function useEventListener(eventName, handler, element, options) {
    // Create a ref that stores handler
    const savedHandler = useRef(handler);
    useIsomorphicLayoutEffect(() => {
        savedHandler.current = handler;
    }, [handler]);
    useEffect(() => {
        // Define the listening target
        const targetElement = element?.current || window;
        if (!(targetElement && targetElement.addEventListener)) {
            return;
        }
        // Create event listener that calls handler function stored in ref
        const eventListener = event => savedHandler.current(event);
        targetElement.addEventListener(eventName, eventListener, options);
        // Remove event listener on cleanup
        return () => {
            targetElement.removeEventListener(eventName, eventListener);
        };
    }, [eventName, element, options]);
}

function useElementSize() {
    // Mutable values like 'ref.current' aren't valid dependencies
    // because mutating them doesn't re-render the component.
    // Instead, we use a state as a ref to be reactive.
    const [ref, setRef] = useState(null);
    const [size, setSize] = useState({
        width: 0,
        height: 0,
    });
    // Prevent too many rendering using useCallback
    const handleSize = useCallback(() => {
        setSize({
            width: ref?.offsetWidth || 0,
            height: ref?.offsetHeight || 0,
        });
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [ref?.offsetHeight, ref?.offsetWidth]);
    useEventListener('resize', handleSize);
    useIsomorphicLayoutEffect(() => {
        handleSize();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [ref?.offsetHeight, ref?.offsetWidth]);
    return [setRef, size];
}

const MIN_COLUMN_WIDTH = 100;
const ROW_HEIGHT$2 = 40;
const SELECT_COLUMN_WIDTH = 40;
const ROW_ACTIONS_COLUMN_WIDTH = 50;

const renderActionPopoverButton = (stringLiterals) => function ActionPopoverButton() {
    return (React__default.createElement("div", { "data-component": "action-popover-button", "data-element": "action-popover-button" },
        React__default.createElement(Icon$1, { type: "ellipsis_vertical", tooltipMessage: stringLiterals.more_options, color: tokens.colorsUtilityMajor450, "data-testid": "db-card-list-row-actions" })));
};

const CollectionActionPopoverComponent = React__default.memo(({ rowActions, row, rowActionTestId, rowActionsTestId, stringLiterals, }) => {
    return (React__default.createElement(ActionPopover, { "aria-label": stringLiterals.more_options, "data-testid": rowActionsTestId, renderButton: renderActionPopoverButton(stringLiterals) }, rowActions
        ?.slice(rowActions.findIndex(a => !a.isMenuSeparator), findLastIndex(rowActions, a => !a.isMenuSeparator) + 1)
        ?.filter((action, i, allActions) => {
        const previousAction = i > 0 ? allActions[i - 1] : null;
        if (action.isMenuSeparator &&
            (previousAction === null || previousAction.isMenuSeparator)) {
            return false;
        }
        if (action.isHidden && typeof action.isHidden === 'function'
            ? action.isHidden(row.i, row)
            : action.isHidden) {
            return false;
        }
        return true;
    })
        .filter((action, i, allActions) => {
        if (action.isMenuSeparator) {
            const nextVisibleAction = allActions.slice(i + 1).find(a => !a.isMenuSeparator);
            return nextVisibleAction !== undefined;
        }
        return true;
    })
        .reduce((actions, action) => {
        // Filter out consecutive menu separators
        // If the last action is a menu separator, we don't need to add another one
        if (action.isMenuSeparator &&
            last(actions.filter(v => !v.isHidden || v.isMenuSeparator))?.isMenuSeparator) {
            return actions;
        }
        return [...actions, action];
    }, [])
        .map((a, i) => {
        if (a.isMenuSeparator) {
            return (React__default.createElement(ActionPopoverDivider, { key: `separator-${i}`, "data-testid": `${rowActionTestId}-${a.id}` }));
        }
        const action = a;
        const className = action.isDestructive ? 'db-table-row-action-destructive' : undefined;
        return (React__default.createElement(ActionPopoverItem
        // @ts-expect-error className is actually a valid prop
        , { 
            // @ts-expect-error className is actually a valid prop
            className: className, "data-pendoid": `${rowActionTestId}-${action.label}`, "data-testid": `${rowActionTestId}-${action.label}`, icon: action.icon, key: i, onClick: () => action.onClick(row.i, row), disabled: action.isDisabled && typeof action.isDisabled === 'function'
                ? action.isDisabled(row.i, row)
                : Boolean(action.isDisabled) }, action.label));
    })));
});
CollectionActionPopoverComponent.displayName = 'CollectionActionPopoverComponent';

const TableRow = React__default.memo(({ canSelect, hasIcon, hasImage, index, isRowExpanded, isSelected, locale, onRowExpandToggle, onRowResized, onSelectionChange, row, rowActions, rowDefinition, stringLiterals, style: incomingStyle = {}, }) => {
    const [ref, { height }] = useElementSize();
    React__default.useEffect(() => {
        onRowResized(index, height);
    }, [height, index, onRowResized]);
    const rowDefs = React__default.useMemo(() => Object.keys(rowDefinition).filter(k => k !== 'icon' && k !== 'image' && k !== 'longContent'), [rowDefinition]);
    const style = React__default.useMemo(() => omit(incomingStyle, 'columnWidth'), [incomingStyle]);
    const onExpandButtonClick = React__default.useCallback(() => {
        onRowExpandToggle(index, !isRowExpanded);
    }, [index, isRowExpanded, onRowExpandToggle]);
    const hasLongContent = rowDefinition.longContent && row.longContent;
    return (React__default.createElement("div", { key: row.i, className: "db-table-table-row ReactVirtualized__Table__row", "data-testid": "db-table-table-row", style: style, role: "row", "aria-level": rowDefinition.longContent ? 1 : undefined, "aria-expanded": rowDefinition.longContent ? isRowExpanded : undefined },
        canSelect && (React__default.createElement("div", { key: "__selection", className: "db-table-table-selection-column db-table-table-cell", style: {
                width: SELECT_COLUMN_WIDTH,
                minWidth: SELECT_COLUMN_WIDTH,
                maxWidth: SELECT_COLUMN_WIDTH,
                verticalAlign: 'middle',
            }, role: "cell" },
            React__default.createElement(Checkbox, { checked: isSelected, onChange: () => onSelectionChange(row.i), "data-testid": `db-table-table-select-${row.i}`, "aria-label": resolveTranslationKey(stringLiterals, 'table_select') }))),
        rowDefinition.longContent && (React__default.createElement("div", { key: "__expand-button", className: "db-table-table-expand-column db-table-table-cell", style: { verticalAlign: 'middle' }, role: "cell" }, hasLongContent && (React__default.createElement(IconButton, { onClick: onExpandButtonClick, "aria-label": resolveTranslationKey(stringLiterals, 'expand_row'), "data-testid": "db-table-table-expansion-row-toggle-button" },
            React__default.createElement(Icon$1, { type: isRowExpanded ? 'caret_down' : 'caret_right' }))))),
        rowDefs.map((k, i) => {
            const customMinWidth = rowDefinition[k]?.displayOptions?.columnWidth ?? 0;
            return (React__default.createElement("div", { key: k, role: "cell", className: rowDefinition?.[k]?.valueFormat === 'number'
                    ? 'db-table-table-cell db-table-table-numeric-column'
                    : 'db-table-table-cell', style: {
                    width: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : incomingStyle.columnWidth,
                    minWidth: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : incomingStyle.columnWidth,
                    maxWidth: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : incomingStyle.columnWidth,
                    verticalAlign: 'middle',
                } },
                React__default.createElement("div", { className: "db-table-table-cell-wrapper" },
                    i === 0 && hasImage && (React__default.createElement("span", { className: "db-table-table-image" },
                        React__default.createElement(Portrait, { size: "XS", src: row.image, initials: String(row[k]), alt: String(row[k]) }))),
                    i === 0 && hasIcon && (React__default.createElement("span", { className: "db-table-table-icon" },
                        React__default.createElement(Icon$1, { fontSize: "small", type: row.icon ?? 'none' }))),
                    React__default.createElement(CollectionItemValueComponent, { i: row.i, propertyName: k, value: row[k], definition: rowDefinition[k], locale: locale, isInTable: true, isBold: i === 0 && isRowExpanded }))));
        }),
        rowActions && rowActions.length > 0 && (React__default.createElement("div", { key: "__actions", role: "cell", className: "db-table-table-cell db-table-table-actions-column", "data-testid": "db-table-row-actions", style: {
                width: ROW_ACTIONS_COLUMN_WIDTH,
                minWidth: ROW_ACTIONS_COLUMN_WIDTH,
                maxWidth: ROW_ACTIONS_COLUMN_WIDTH,
                textAlign: 'left', // Added to align content to the left
                paddingLeft: 0,
            } },
            React__default.createElement(CollectionActionPopoverComponent, { row: row, rowActions: rowActions, rowActionTestId: "db-table-row-action", rowActionsTestId: "db-table-table-row-actions", stringLiterals: stringLiterals }))),
        hasLongContent && isRowExpanded && (React__default.createElement("div", { ref: ref, key: `${row.i}-expanded`, className: "db-table-table-expansion-row", "data-testid": "db-table-table-expansion-row", role: "row", "aria-level": 2 }, row.longContent))));
});
TableRow.displayName = 'TableRow';

var WidgetCollectionItemStatus;
(function (WidgetCollectionItemStatus) {
    WidgetCollectionItemStatus[WidgetCollectionItemStatus["LOADING"] = 0] = "LOADING";
    WidgetCollectionItemStatus[WidgetCollectionItemStatus["LOADED"] = 1] = "LOADED";
    WidgetCollectionItemStatus[WidgetCollectionItemStatus["FAILED"] = 2] = "FAILED";
    WidgetCollectionItemStatus[WidgetCollectionItemStatus["STALE"] = 3] = "STALE";
})(WidgetCollectionItemStatus || (WidgetCollectionItemStatus = {}));

const Table = React__default.memo(({ canSelect, data, hasIcon, hasImage, height, loadMoreRows: loadMore, locale, onSelectAllChange, onSelectionChange, rowActions, rowDefinition, selectedRows, setHasAllDataLoaded, stringLiterals, }) => {
    const [expansionMap, setExpansionMap] = React__default.useState({});
    const [expansionHeightMap, setExpansionHeightMap] = React__default.useState({});
    const infiniteLoaderRef = React__default.useRef(null);
    const [loadedMap, setLoadedMap] = React__default.useState(data.map(() => WidgetCollectionItemStatus.LOADED));
    React__default.useEffect(() => {
        setLoadedMap(data.map(() => WidgetCollectionItemStatus.LOADED));
        infiniteLoaderRef.current?.resetLoadMoreRowsCache();
        setHasAllDataLoaded(false);
    }, [data, setHasAllDataLoaded]);
    const isRowLoaded = React__default.useCallback(({ index }) => {
        return loadedMap[index] != null;
    }, [loadedMap]);
    const onRowExpanded = React__default.useCallback((index, isExpanded) => {
        setExpansionMap(current => ({ ...current, [index]: isExpanded }));
    }, []);
    const onRowResized = React__default.useCallback((index, height) => {
        setExpansionHeightMap(current => ({ ...current, [index]: height }));
    }, []);
    const getRowHeight = React__default.useCallback(({ index }) => {
        if (expansionMap[index] && expansionHeightMap[index]) {
            return ROW_HEIGHT$2 + expansionHeightMap[index];
        }
        return ROW_HEIGHT$2;
    }, [expansionHeightMap, expansionMap]);
    const reload = React__default.useCallback(() => {
        infiniteLoaderRef.current?.resetLoadMoreRowsCache(true);
    }, []);
    const rowRenderer = React__default.useCallback(({ rowData, style, index }) => {
        if (loadedMap[index] === WidgetCollectionItemStatus.LOADING) {
            return (React__default.createElement("div", { style: style, className: "db-row-loading" },
                React__default.createElement(Loader, { size: "small" })));
        }
        if (loadedMap[index] === WidgetCollectionItemStatus.FAILED) {
            return (React__default.createElement("div", { style: style, className: "db-row-error" },
                React__default.createElement(Button, { buttonType: "tertiary", fullWidth: true, destructive: true, iconType: "refresh", onClick: reload }, resolveTranslationKey(stringLiterals, 'reload'))));
        }
        if (!rowData) {
            return null;
        }
        return (React__default.createElement(TableRow, { canSelect: canSelect, hasIcon: hasIcon, hasImage: hasImage, index: index, isRowExpanded: expansionMap[index], isSelected: selectedRows.includes(rowData.i), key: rowData.i, locale: locale, onRowExpandToggle: onRowExpanded, onRowResized: onRowResized, onSelectionChange: onSelectionChange, row: rowData, rowActions: rowActions, rowDefinition: rowDefinition, stringLiterals: stringLiterals, style: style }));
    }, [
        canSelect,
        expansionMap,
        hasIcon,
        hasImage,
        loadedMap,
        locale,
        onRowExpanded,
        onRowResized,
        onSelectionChange,
        reload,
        rowActions,
        rowDefinition,
        selectedRows,
        stringLiterals,
    ]);
    const loadMoreRows = React__default.useCallback(async ({ startIndex, stopIndex }) => {
        const pageSize = stopIndex - startIndex + 1;
        const pageCount = Math.ceil((startIndex + 1) / pageSize);
        let after = data[startIndex - 1];
        let first = pageSize;
        if (loadedMap[startIndex - 1] !== WidgetCollectionItemStatus.LOADED) {
            const idx = loadedMap
                .slice(0, startIndex)
                .reverse()
                .findIndex(item => item === WidgetCollectionItemStatus.LOADED);
            if (idx !== -1) {
                const newStartIndex = startIndex - idx - 1;
                after = data[newStartIndex];
                first = stopIndex - newStartIndex + 1;
            }
        }
        try {
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    clone[i] = WidgetCollectionItemStatus.LOADING;
                }
                return clone;
            });
            const newRows = await loadMore?.({
                startIndex,
                stopIndex,
                pageCount,
                pageSize,
                after,
                first,
            });
            // No rows returned or same rows as before
            if (newRows?.length === 0 || isEqual(newRows, data)) {
                setHasAllDataLoaded(true);
                setLoadedMap(current => {
                    const clone = [...current];
                    for (let i = startIndex; i <= stopIndex; i++) {
                        clone[i] = WidgetCollectionItemStatus.STALE;
                    }
                    return clone;
                });
                return;
            }
            // Less rows than pageSize
            if (newRows && newRows?.length < pageSize) {
                setHasAllDataLoaded(true);
                setLoadedMap(current => {
                    const clone = [...current];
                    for (let i = startIndex; i <= stopIndex; i++) {
                        clone[i] =
                            i >= startIndex + newRows.length
                                ? WidgetCollectionItemStatus.STALE
                                : WidgetCollectionItemStatus.LOADED;
                    }
                    return clone;
                });
                return;
            }
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    if (newRows?.[i - startIndex]) {
                        clone[i] = WidgetCollectionItemStatus.LOADED;
                    }
                    else {
                        clone[i] = WidgetCollectionItemStatus.STALE;
                    }
                }
                return clone;
            });
        }
        catch (err) {
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    clone[i] = WidgetCollectionItemStatus.FAILED;
                }
                return clone;
            });
        }
    }, [data, loadMore, loadedMap, setHasAllDataLoaded]);
    const rowDefs = React__default.useMemo(() => Object.keys(rowDefinition).filter(k => k !== 'icon' && k !== 'image' && k !== 'longContent'), [rowDefinition]);
    const headerRenderer = React__default.useCallback(({ style }) => {
        const { columnWidth } = style;
        return (React__default.createElement("div", { className: "db-table-table-header", style: omit(style, 'columnWidth', 'paddingRight'), role: "rowgroup" },
            React__default.createElement("div", { className: "db-table-table-header-row", role: "row" },
                canSelect && (React__default.createElement("div", { key: "__selection", className: "db-table-table-selection-column-select-all db-table-header-cell", role: "columnheader", style: {
                        width: SELECT_COLUMN_WIDTH,
                        minWidth: SELECT_COLUMN_WIDTH,
                        maxWidth: SELECT_COLUMN_WIDTH,
                    } },
                    React__default.createElement(Checkbox, { "data-testid": "db-table-table-select-all", checked: selectedRows.length === data?.length, onChange: ev => onSelectAllChange(ev.target.checked), "aria-label": resolveTranslationKey(stringLiterals, 'table_select_all') }))),
                rowDefinition.longContent && (React__default.createElement("div", { key: "__expand-button-header", className: "db-table-table-expand-column-header", role: "columnheader" })),
                rowDefs.map(k => {
                    const customMinWidth = rowDefinition[k]?.displayOptions?.columnWidth ?? 0;
                    return (React__default.createElement("div", { key: k, style: {
                            width: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : columnWidth,
                            minWidth: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : columnWidth,
                            maxWidth: customMinWidth > MIN_COLUMN_WIDTH ? customMinWidth : columnWidth,
                        }, "data-testid": "db-table-table-selection-column-header", role: "columnheader", className: rowDefinition[k]?.valueFormat === 'number'
                            ? 'db-table-header-cell db-table-table-selection-column-header-number'
                            : 'db-table-header-cell' },
                        React__default.createElement("span", { style: { boxSizing: 'border-box' } }, rowDefinition[k]?.title)));
                }),
                rowActions && rowActions.length > 0 && (React__default.createElement("div", { className: "db-table-header-cell", role: "columnheader", "data-testid": "db-table-row-actions-header", style: {
                        width: ROW_ACTIONS_COLUMN_WIDTH,
                        minWidth: ROW_ACTIONS_COLUMN_WIDTH,
                        maxWidth: ROW_ACTIONS_COLUMN_WIDTH,
                    } })))));
    }, [
        canSelect,
        data?.length,
        onSelectAllChange,
        rowActions,
        rowDefinition,
        rowDefs,
        selectedRows.length,
        stringLiterals,
    ]);
    const rowGetter = React__default.useCallback(({ index }) => {
        return data[index];
    }, [data]);
    const rowCount = React__default.useMemo(() => {
        return loadedMap.filter(l => l === WidgetCollectionItemStatus.LOADED).length;
    }, [loadedMap]);
    const columnCount = React__default.useMemo(() => rowDefs.length, [rowDefs.length]);
    return (React__default.createElement("div", { "data-testid": "db-table-table-representation", className: "db-table-table-representation", style: { maxHeight: height || '90%', overflowY: 'hidden' } },
        React__default.createElement(InfiniteLoader, { ref: infiniteLoaderRef, isRowLoaded: isRowLoaded, loadMoreRows: loadMoreRows, rowCount: Number.MAX_SAFE_INTEGER, minimumBatchSize: 20, threshold: 8 }, ({ onRowsRendered, registerChild: registerInfiniteLoaderChild }) => (React__default.createElement(AutoSizer, { disableHeight: true, "data-testid": "db-table-table-representation", defaultHeight: height || 100 }, ({ width: autoWidth }) => {
            const customWidthsSum = Object.keys(rowDefinition).reduce((acc, k) => {
                const customMinWidth = rowDefinition[k]?.displayOptions?.columnWidth ?? 0;
                if (customMinWidth > MIN_COLUMN_WIDTH) {
                    acc += customMinWidth;
                }
                return acc;
            }, 0);
            const customWidthColumnsLength = Object.keys(rowDefinition).filter(k => {
                const customWidth = rowDefinition[k]?.displayOptions?.columnWidth;
                return customWidth !== undefined && customWidth > MIN_COLUMN_WIDTH;
            }).length;
            const columnSizerWidth = Math.max(autoWidth, rowDefs.length * MIN_COLUMN_WIDTH + (canSelect ? SELECT_COLUMN_WIDTH : 0));
            return (React__default.createElement(ColumnSizer, { columnMaxWidth: undefined, columnMinWidth: MIN_COLUMN_WIDTH, columnCount: columnCount, width: columnSizerWidth }, ({ adjustedWidth, registerChild: registerColumnSizerChild, columnWidth }) => {
                columnWidth =
                    rowActions && rowActions.length > 0
                        ? columnWidth - ROW_ACTIONS_COLUMN_WIDTH / columnCount
                        : columnWidth;
                columnWidth = rowDefinition.longContent
                    ? columnWidth - 76 / columnCount
                    : columnWidth;
                const colWidth = canSelect
                    ? columnWidth - SELECT_COLUMN_WIDTH / columnCount
                    : columnWidth;
                const tableWidth = adjustedWidth - customWidthColumnsLength * colWidth + customWidthsSum;
                return (React__default.createElement(Table$1, { ref: r => {
                        registerInfiniteLoaderChild(r);
                        registerColumnSizerChild(r);
                    }, className: "db-table-table", containerStyle: { width: '100%' }, estimatedColumnSize: rowDefs.length, estimatedRowSize: 20, gridStyle: { width: '100%', paddingBottom: '15px' }, headerHeight: 40, headerRowRenderer: headerRenderer, headerStyle: { columnWidth: colWidth }, height: height || 100, onRowsRendered: onRowsRendered, role: rowDefinition.longContent ? 'treegrid' : 'grid', rowCount: rowCount, rowGetter: rowGetter, rowHeight: getRowHeight, rowRenderer: rowRenderer, rowStyle: { columnWidth: colWidth }, width: tableWidth, style: {
                        width: tableWidth,
                        boxSizing: 'border-box',
                    } }));
            }));
        })))));
});
Table.displayName = 'TableBody';

/**
 * Returns the value passed to the hook on previous render.
 *
 * Yields `undefined` on first render.
 *
 * @param value Value to yield on next render
 */
function usePrevious(value) {
    const previous = useRef();
    useEffect(() => {
        previous.current = value;
    });
    return previous.current;
}

var DateFilterPeriodType;
(function (DateFilterPeriodType) {
    DateFilterPeriodType["DAY"] = "DAY";
    DateFilterPeriodType["WEEK"] = "WEEK";
    DateFilterPeriodType["MONTH"] = "MONTH";
    DateFilterPeriodType["QUARTER"] = "QUARTER";
    DateFilterPeriodType["YEAR"] = "YEAR";
})(DateFilterPeriodType || (DateFilterPeriodType = {}));

const COLUMN_COUNT = 3;
const ROW_HEIGHT$1 = 50;
const PERIOD_SELECTOR_HEIGHT = 340;
function getWeekOfTheYear(startDate, locale) {
    const date = new Date(startDate.getFullYear(), startDate.getMonth(), startDate.getDate());
    const startOfWeek = locale === 'en-US' ? 0 : 1;
    const firstDayOfYear = new Date(date.getFullYear(), 0, 1);
    const dayOfWeek = firstDayOfYear.getDay();
    const daysOffset = (startOfWeek - dayOfWeek + 7) % 7;
    const firstWeekStart = new Date(date.getFullYear(), 0, 1 + daysOffset);
    const diffInDays = Math.floor((date.getTime() - firstWeekStart.getTime()) / (1000 * 60 * 60 * 24));
    const weekNumber = Math.floor(diffInDays / 7) + 1;
    return weekNumber;
}
const defaultMinDate = new Date(1970, 0, 1);
const defaultMaxDate = new Date(2099, 11, 31);
/** This functions generates an ISO date stamp with UTC timezone */
const convertToIsoDateStamp = (date) => {
    return date.toISOString();
};
function getLocalCurrentDateString() {
    const now = new Date().toISOString();
    return now;
}
/**
 * This function computes all potential periods between the max and min dates depending on the type of period
 * @param type
 * @param dateFilter
 */
const getPeriods = memoize(({ type, locale, dateFilter, stringLiterals }) => {
    const minDate = dateFilter?.minDate || new Date(defaultMinDate);
    const maxDate = dateFilter?.maxDate || new Date(defaultMaxDate);
    // Here we adjust the min date to the beginning of the period depending on its type
    switch (type) {
        case DateFilterPeriodType.DAY:
            minDate.setHours(0, 0, 0, 0);
            maxDate.setHours(23, 59, 59, 999); // needed?
            break;
        case DateFilterPeriodType.WEEK:
            // Set the min date to the beginning of the week depending on the locale, if the locale is en-us, the week starts on sunday
            if (locale === 'en-US') {
                minDate.setDate(minDate.getDate() - minDate.getDay());
            }
            else {
                minDate.setDate(minDate.getDate() - (minDate.getDay() === 0 ? 6 : minDate.getDay() - 1));
            }
            minDate.setHours(0, 0, 0, 0);
            break;
        case DateFilterPeriodType.MONTH:
            minDate.setDate(1);
            minDate.setHours(0, 0, 0, 0);
            break;
        case DateFilterPeriodType.QUARTER:
            minDate.setMonth(Math.floor(minDate.getMonth() / 3) * 3);
            minDate.setDate(1);
            minDate.setHours(0, 0, 0, 0);
            break;
        case DateFilterPeriodType.YEAR:
            minDate.setMonth(0);
            minDate.setDate(1);
            minDate.setHours(0, 0, 0, 0);
            break;
    }
    const periods = [];
    let currentPeriod = new Date(minDate);
    const dayDateFormat = new Intl.DateTimeFormat(locale, {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        timeZone: Intl.DateTimeFormat().resolvedOptions().timeZone,
    });
    const monthDateFormat = new Intl.DateTimeFormat(locale, {
        year: 'numeric',
        month: 'long',
        timeZone: Intl.DateTimeFormat().resolvedOptions().timeZone,
    });
    const monthPickerFormat = new Intl.DateTimeFormat(locale, {
        month: 'short',
        timeZone: Intl.DateTimeFormat().resolvedOptions().timeZone,
    });
    while (currentPeriod <= maxDate) {
        const monthName = stringLiterals[`month_${currentPeriod.getMonth() + 1}`];
        switch (type) {
            case DateFilterPeriodType.DAY:
                const endOfDay = new Date(currentPeriod);
                endOfDay.setHours(23, 59, 59, 999);
                periods.push({
                    label: currentPeriod.getDate().toString(),
                    startDate: convertToIsoDateStamp(new Date(currentPeriod)),
                    endDate: convertToIsoDateStamp(endOfDay),
                    group: `${monthName} ${currentPeriod.getFullYear()}`,
                    headerLabel: dayDateFormat.format(currentPeriod),
                    type: DateFilterPeriodType.DAY,
                });
                currentPeriod = new Date(currentPeriod.setDate(currentPeriod.getDate() + 1));
                break;
            case DateFilterPeriodType.WEEK:
                const lastDayOfWeek = new Date(currentPeriod);
                lastDayOfWeek.setDate(lastDayOfWeek.getDate() + 6);
                lastDayOfWeek.setHours(23, 59, 59, 999); // Set to the end of the day
                const weekNumber = getWeekOfTheYear(currentPeriod, locale);
                periods.push({
                    label: `${stringLiterals.week} ${weekNumber}`,
                    startDate: convertToIsoDateStamp(currentPeriod),
                    endDate: convertToIsoDateStamp(lastDayOfWeek),
                    group: `${monthName} ${currentPeriod.getFullYear()}`,
                    headerLabel: `${stringLiterals.week} ${weekNumber}, ${currentPeriod.getFullYear()}`,
                    type: DateFilterPeriodType.WEEK,
                });
                currentPeriod = new Date(currentPeriod.setDate(currentPeriod.getDate() + 7));
                break;
            case DateFilterPeriodType.MONTH:
                // Get the last day of the month
                const lastDayOfMonth = new Date(currentPeriod);
                lastDayOfMonth.setMonth(lastDayOfMonth.getMonth() + 1);
                lastDayOfMonth.setDate(0);
                lastDayOfMonth.setHours(23, 59, 59, 999); // Set to the end of the day
                periods.push({
                    label: monthPickerFormat.format(currentPeriod),
                    startDate: convertToIsoDateStamp(currentPeriod),
                    endDate: convertToIsoDateStamp(lastDayOfMonth),
                    group: currentPeriod.getFullYear().toString(),
                    headerLabel: monthDateFormat.format(currentPeriod),
                    type: DateFilterPeriodType.MONTH,
                });
                currentPeriod = new Date(currentPeriod.setMonth(currentPeriod.getMonth() + 1));
                break;
            case DateFilterPeriodType.QUARTER:
                // Get last day of the quarter
                const lastDayOfQuarter = new Date(currentPeriod);
                lastDayOfQuarter.setMonth(lastDayOfQuarter.getMonth() + 3);
                lastDayOfQuarter.setDate(0);
                lastDayOfQuarter.setHours(23, 59, 59, 999);
                periods.push({
                    label: `Q${currentPeriod.getMonth() / 3 + 1}`,
                    startDate: convertToIsoDateStamp(currentPeriod),
                    endDate: convertToIsoDateStamp(lastDayOfQuarter),
                    group: currentPeriod.getFullYear().toString(),
                    headerLabel: `Q${currentPeriod.getMonth() / 3 + 1} ${currentPeriod.getFullYear()}`,
                    type: DateFilterPeriodType.QUARTER,
                });
                currentPeriod = new Date(currentPeriod.setMonth(currentPeriod.getMonth() + 3));
                break;
            case DateFilterPeriodType.YEAR:
                // Get last day of the year
                const lastDayOfYear = new Date(currentPeriod);
                lastDayOfYear.setMonth(12);
                lastDayOfYear.setDate(0);
                lastDayOfYear.setHours(23, 59, 59, 999);
                periods.push({
                    label: currentPeriod.getFullYear().toString(),
                    startDate: convertToIsoDateStamp(currentPeriod),
                    endDate: convertToIsoDateStamp(lastDayOfYear),
                    headerLabel: currentPeriod.getFullYear().toString(),
                    type: DateFilterPeriodType.YEAR,
                });
                currentPeriod = new Date(currentPeriod.setFullYear(currentPeriod.getFullYear() + 1));
                break;
        }
    }
    return periods;
});
const getRowsFromPeriods = (periods) => periods.reduce((acc, period, index, prds) => {
    const prevValue = prds[index - 1];
    if (period.group && (!prevValue || prevValue.group !== period.group)) {
        acc.push(period.group);
    }
    const lastRow = acc[acc.length - 1];
    if (lastRow && !isString(lastRow) && lastRow.length < COLUMN_COUNT) {
        lastRow.push(period);
    }
    else {
        acc.push([period]);
    }
    return acc;
}, []);
function findPeriodByDate(dateString, type, locale, stringLiterals) {
    return (getPeriods({ type, locale, stringLiterals }).find(period => {
        return period.startDate <= dateString && period.endDate >= dateString;
    }) || null);
}
function getNeighboringPeriod(currentPeriod, type, locale, stringLiterals, direction) {
    const periods = getPeriods({ type, locale, stringLiterals });
    const index = periods.findIndex(period => period.startDate === currentPeriod.startDate && period.endDate == currentPeriod.endDate);
    const newIndex = index + direction;
    return periods[newIndex] || periods[index] || null;
}
function getInitialRowScrollPosition(selectedPeriod, rows) {
    return selectedPeriod
        ? Math.max(rows.findIndex(row => !isString(row) &&
            row.find(r => r.startDate === selectedPeriod.startDate && r.endDate === selectedPeriod.endDate)) - 3, 0)
        : 0;
}

function DateFilterPeriodSelector({ type, locale, stringLiterals, dateFilter, setSelectedPeriod, selectedPeriod, }) {
    const periods = getPeriods({ type, locale, dateFilter, stringLiterals });
    const rows = getRowsFromPeriods(periods);
    const initialRow = getInitialRowScrollPosition(selectedPeriod, rows);
    const [scrollPosition, setScrollPosition] = React__default.useState(Math.max(initialRow, 0) * ROW_HEIGHT$1);
    const handleScroll = React__default.useCallback(({ scrollTop }) => {
        setScrollPosition(scrollTop);
    }, []);
    const onPeriodSelect = React__default.useCallback((period) => () => {
        setSelectedPeriod(period);
    }, [setSelectedPeriod]);
    const rowRenderer = ({ index, key, style }) => {
        const row = rows[index];
        const isActiveGroup = (isString(row) && row === selectedPeriod?.group) ||
            (!isString(row) && row[0].group === selectedPeriod?.group);
        if (isString(row)) {
            const separatorClasses = ['db-date-filter-selector-row-separator'];
            if (!isActiveGroup) {
                separatorClasses.push('db-date-filter-selector-row-inactive');
            }
            return (React__default.createElement("div", { className: "db-date-filter-selector-row", key: key, style: style },
                React__default.createElement("div", { className: separatorClasses.join(' ') }, row)));
        }
        return (React__default.createElement("div", { className: "db-date-filter-selector-row", key: key, style: style }, row.map((period) => {
            const isSelected = period.startDate === selectedPeriod?.startDate &&
                period.endDate === selectedPeriod?.endDate;
            const rowClasses = ['db-date-filter-selector-row-element'];
            if (isSelected) {
                rowClasses.push('db-date-filter-selector-row-element-selected');
            }
            if (!isActiveGroup) {
                rowClasses.push('db-date-filter-selector-row-inactive');
            }
            return (React__default.createElement("button", { className: rowClasses.join(' '), key: period.startDate, onClick: onPeriodSelect(period), "aria-label": period.headerLabel, "data-testid": `db-date-filter-selector-period-${period.startDate}` }, period.label));
        })));
    };
    return (React__default.createElement("div", { className: "db-date-filter-selector", "data-testid": "db-date-filter-selector", style: { height: PERIOD_SELECTOR_HEIGHT }, role: "dialog" },
        React__default.createElement(AutoSizer, { defaultHeight: PERIOD_SELECTOR_HEIGHT }, ({ height, width }) => (React__default.createElement(List, { height: height, onScroll: handleScroll, overscanRowCount: 40, rowCount: rows.length, rowHeight: ROW_HEIGHT$1, rowRenderer: rowRenderer, scrollTop: scrollPosition, width: width })))));
}

const isPeriodTypeEnabled = (dateFilter, periodType) => dateFilter.periodTypes.includes(periodType);
function DateFilter({ dateFilter, onDateFilterChanged, locale, stringLiterals, selectedPeriod, }) {
    const [periodType, setPeriodType] = React__default.useState(selectedPeriod?.type || dateFilter.defaultPeriodType || dateFilter.periodTypes[0]);
    const [isPeriodSelectorOpen, setIsPeriodSelectorOpen] = React__default.useState(false);
    const onPeriodTypeChange = React__default.useCallback((_event, selectedPeriodType) => {
        if (!selectedPeriodType || !selectedPeriod) {
            return;
        }
        _event?.preventDefault();
        const newPeriod = findPeriodByDate(selectedPeriod.startDate, selectedPeriodType, locale, stringLiterals);
        if (!newPeriod) {
            return;
        }
        setPeriodType(selectedPeriodType);
        onDateFilterChanged(newPeriod);
    }, [locale, onDateFilterChanged, selectedPeriod, stringLiterals]);
    const onPeriodSelectorButtonClick = React__default.useCallback((event) => {
        event.preventDefault();
        setIsPeriodSelectorOpen(!isPeriodSelectorOpen);
    }, [isPeriodSelectorOpen]);
    const onPeriodSelected = React__default.useCallback((p) => {
        setIsPeriodSelectorOpen(false);
        onDateFilterChanged(p);
    }, [onDateFilterChanged]);
    const onPeriodStepperClicked = React__default.useCallback((direction) => () => {
        if (!selectedPeriod)
            return;
        const nextPeriod = getNeighboringPeriod(selectedPeriod, periodType, locale, stringLiterals, direction);
        if (!nextPeriod)
            return;
        onDateFilterChanged(nextPeriod);
    }, [locale, onDateFilterChanged, periodType, selectedPeriod, stringLiterals]);
    React__default.useEffect(() => {
        if (selectedPeriod) {
            return;
        }
        const startDate = getLocalCurrentDateString();
        const defaultPeriod = findPeriodByDate(startDate, periodType, locale, stringLiterals);
        if (!defaultPeriod) {
            return;
        }
        onDateFilterChanged(defaultPeriod);
    }, [periodType, selectedPeriod, stringLiterals, locale, onDateFilterChanged]);
    return (React__default.createElement("div", { className: "db-date-filter" },
        React__default.createElement("div", { className: "db-date-filter-stepper" },
            React__default.createElement(ButtonMinor, { iconType: "chevron_left", "aria-label": stringLiterals.previous_period, size: "small", mr: "16px", mt: "4px", mb: "4px", onClick: onPeriodStepperClicked(-1), "data-testid": "db-table-date-previous-period" }),
            React__default.createElement(ButtonMinor, { iconType: "chevron_right", "aria-label": stringLiterals.next_period, size: "small", mt: "4px", mb: "4px", onClick: onPeriodStepperClicked(1), "data-testid": "db-table-date-next-period" })),
        React__default.createElement("div", { className: "db-date-filter-dropdown" },
            React__default.createElement("button", { className: "db-date-filter-dropdown-button", onClick: (event) => {
                    onPeriodSelectorButtonClick(event);
                }, "data-testid": "db-date-filter-dropdown-button", "aria-label": `${stringLiterals.current_date_filter}: ${selectedPeriod?.headerLabel}. ${stringLiterals.choose_date}` },
                React__default.createElement("span", { className: "db-date-filter-dropdown-button-label" }, selectedPeriod?.headerLabel || ' '),
                React__default.createElement(Icon$1, { type: "dropdown" }))),
        isPeriodSelectorOpen && selectedPeriod?.type === periodType && (React__default.createElement(DateFilterPeriodSelector, { dateFilter: dateFilter, locale: locale, stringLiterals: stringLiterals, type: periodType, selectedPeriod: selectedPeriod || null, setSelectedPeriod: onPeriodSelected })),
        React__default.createElement("div", { className: "db-date-filter-period-type" },
            React__default.createElement(ButtonToggleGroup, { id: "date-filter-period-type", onChange: onPeriodTypeChange, "aria-label": "Date filter period type", value: periodType, "data-testid": "db-date-filter-period-type" },
                isPeriodTypeEnabled(dateFilter, DateFilterPeriodType.DAY) && (React__default.createElement(ButtonToggle, { value: DateFilterPeriodType.DAY }, stringLiterals.day)),
                isPeriodTypeEnabled(dateFilter, DateFilterPeriodType.WEEK) && (React__default.createElement(ButtonToggle, { value: DateFilterPeriodType.WEEK }, stringLiterals.week)),
                isPeriodTypeEnabled(dateFilter, DateFilterPeriodType.MONTH) && (React__default.createElement(ButtonToggle, { value: DateFilterPeriodType.MONTH }, stringLiterals.month)),
                isPeriodTypeEnabled(dateFilter, DateFilterPeriodType.QUARTER) && (React__default.createElement(ButtonToggle, { value: DateFilterPeriodType.QUARTER }, stringLiterals.quarter)),
                isPeriodTypeEnabled(dateFilter, DateFilterPeriodType.YEAR) && (React__default.createElement(ButtonToggle, { value: DateFilterPeriodType.YEAR }, stringLiterals.year))))));
}

function FilterLine({ filter, toggle, selectAll, stringLiterals, isFilterFullWidth, onFilterHeightChange, }) {
    const selectInputRef = React__default.useRef(null);
    const [filterRef, { height: filterRefHeight }] = useElementSize();
    React__default.useEffect(() => {
        onFilterHeightChange?.(filterRefHeight);
    }, [filterRefHeight, onFilterHeightChange]);
    const onXButtonClick = React__default.useCallback(() => {
        filter?.onChange(null);
        if (selectInputRef.current) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const nativeInputValueSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
            nativeInputValueSetter.call(selectInputRef.current, '');
            selectInputRef.current.dispatchEvent(new Event('input', { bubbles: true }));
        }
    }, [filter]);
    const leftChildren = React__default.useMemo(() => filter?.value && (React__default.createElement("div", { className: "db-filter-cross-icon-wrapper", "data-testid": "db-filter-cross-icon-wrapper" },
        React__default.createElement(IconButton, { onClick: onXButtonClick, "aria-label": stringLiterals.clear_filter },
            React__default.createElement(Icon$1, { type: "cross_circle", color: "#9AACB7" })))), [filter?.value, onXButtonClick, stringLiterals.clear_filter]);
    const onToggleButtonClick = React__default.useCallback((ev) => {
        ev.preventDefault();
    }, []);
    if (_.isEmpty(filter?.options) && _.isEmpty(toggle?.options) && !selectAll) {
        return null;
    }
    return (React__default.createElement("div", { "data-testid": "db-filter-line", ref: filterRef, "data-variant": filter?.variant ?? 'default', className: filter?.representation === 'tabs'
            ? 'db-filter-line'
            : 'db-filter-line db-filter-line-dropdown-mode' },
        selectAll && (React__default.createElement("div", { className: "db-select-all-container" },
            React__default.createElement(Checkbox, { "data-testid": "db-card-list-select-all", checked: !!selectAll.value, onChange: ev => selectAll.onChange(ev.target.checked), mt: "6px", mr: "16px", label: resolveTranslationKey(stringLiterals, 'table_select_all') }))),
        React__default.createElement("div", { className: "db-filter-container" }, filter && (React__default.createElement(Filter, { filter: filter, leftChildren: leftChildren, isFilterFullWidth: isFilterFullWidth }))),
        toggle && (React__default.createElement("div", { "data-testid": "db-toggle-container", className: "db-toggle-container" },
            React__default.createElement(ButtonToggleGroup, { "aria-label": toggle.ariaLabel, id: toggle.id, onChange: (_ev, value) => toggle.onChange(String(value)), value: toggle.value || Object.keys(toggle.options)[0] }, Object.keys(toggle.options).map(option => (React__default.createElement(ButtonToggle, { "aria-label": toggle.options[option].label, buttonIcon: toggle.options[option].icon, "data-element": `filter-line-toggle-${option}`, key: option, size: "small", value: option, onClick: onToggleButtonClick }))))))));
}
const Filter = ({ filter: { ariaLabel, onChange: controlledOnChange, options, value: controlledValue, isEmptyAllowed, representation, placeholder, }, leftChildren: controlledLeftChildren, isFilterFullWidth, }) => {
    const optionKeys = React__default.useMemo(() => Object.keys(options), [options]);
    const leftChildren = React__default.useMemo(() => (isEmptyAllowed ? controlledLeftChildren : undefined), [
        isEmptyAllowed,
        controlledLeftChildren,
    ]);
    const value = React__default.useMemo(() => controlledValue || '', [controlledValue]);
    const onChange = React__default.useCallback(ev => {
        controlledOnChange(ev.target.value);
    }, [controlledOnChange]);
    const onTabChange = React__default.useCallback(newTabValue => {
        controlledOnChange(newTabValue);
    }, [controlledOnChange]);
    const style = React__default.useMemo(() => (!isFilterFullWidth ? { maxWidth: '230px' } : {}), [
        isFilterFullWidth,
    ]);
    if (optionKeys.length === 0) {
        return null;
    }
    if (representation === 'tabs') {
        return (React__default.createElement(Tabs, { align: "left", position: "top", onTabChange: onTabChange }, optionKeys.map(option => (React__default.createElement(Tab, { tabId: option, key: option, title: options[option] })))));
    }
    return (React__default.createElement("div", { style: style },
        React__default.createElement(FilterableSelect, { placeholder: placeholder, "aria-label": ariaLabel, "data-testid": "filter-line-dropdown", leftChildren: leftChildren, onChange: onChange, size: "small", value: value }, optionKeys.map(option => (React__default.createElement(Option, { key: option, text: options[option], value: option }))))));
};

const TitleHeader = ({ getTotalCountText, isAiGenerated, stringLiterals, title = '', }) => {
    return (React__default.createElement("div", { className: "db-table-header-title-line" },
        React__default.createElement("div", { className: "db-table-header-title-container db-draggable-element-indicator" },
            React__default.createElement("div", { className: "db-table-header-title-wrapper" },
                React__default.createElement(Typography, { variant: "h3", "data-testid": "db-table-title" }, title),
                isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: stringLiterals }),
                React__default.createElement("span", null,
                    React__default.createElement(Pill, { pillRole: "status", size: "S", fill: true, ml: "1", "data-element": "db-table-count" }, getTotalCountText().toString()))))));
};

const TileTableHeader = ({ getTotalCountText, iconSrc, isAiGenerated, stringLiterals, title = '', }) => {
    return (React__default.createElement("div", { className: "db-custom-table-header db-draggable-element-indicator" },
        React__default.createElement("div", { className: "db-custom-table-header-title-line", "data-testid": "db-table-header-title-line" },
            iconSrc && (React__default.createElement("img", { className: "db-custom-header-icon", "data-testid": "db-custom-header-icon", alt: "icon", src: iconSrc })),
            React__default.createElement("div", { className: "db-custom-header-body", "data-testid": "db-custom-header-body" },
                React__default.createElement("div", null,
                    React__default.createElement(Typography, { fontWeight: "700", lineHeight: "30px", fontSize: "30px", variant: "span", mb: "2", className: "db-custom-header-value", "data-element": "db-table-count" }, getTotalCountText())),
                React__default.createElement("div", null,
                    React__default.createElement(Typography, { fontSize: "18px", mb: 0, variant: "span", color: "blackOpacity65", className: "db-custom-header-title" }, title),
                    isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: stringLiterals }))))));
};

const TableHeader = ({ allRowsSelected, canChangeViewMode, filterMenu, filterMenuType, getTotalCountText, hasCardSelectAllCheckbox, iconSrc, isAiGenerated, isCardHeaderLayout, onFilterMenuChanged, onModeChanged, onSelectAllChange, representationMode, stringLiterals, title, }) => {
    const [filterValue, setFilterValue] = React__default.useState(Object.keys(filterMenu ?? {})[0]);
    const onChange = React__default.useCallback(newValue => {
        setFilterValue(newValue ?? undefined);
        onFilterMenuChanged(newValue);
    }, [onFilterMenuChanged]);
    const filterMenuOptions = React__default.useMemo(() => filterMenu && !isEmpty(filterMenu)
        ? {
            representation: filterMenuType,
            options: filterMenu,
            onChange,
            ariaLabel: resolveTranslationKey(stringLiterals, 'table_filter_menu'),
            value: filterValue,
        }
        : undefined, [filterMenu, filterMenuType, filterValue, onChange, stringLiterals]);
    const toggleOptions = React__default.useMemo(() => canChangeViewMode
        ? {
            ariaLabel: resolveTranslationKey(stringLiterals, 'view_switch'),
            id: 'table-filter-toggle',
            onChange: newValue => {
                onModeChanged(newValue);
            },
            value: representationMode,
            options: {
                table: {
                    icon: 'list_view',
                    label: resolveTranslationKey(stringLiterals, 'table_switch_to_table_view'),
                },
                card: {
                    icon: 'card_view',
                    label: resolveTranslationKey(stringLiterals, 'table_switch_to_card_view'),
                },
            },
            isHidden: !canChangeViewMode,
        }
        : undefined, [canChangeViewMode, onModeChanged, representationMode, stringLiterals]);
    const selectAllOptions = React__default.useMemo(() => {
        return hasCardSelectAllCheckbox
            ? {
                value: allRowsSelected,
                onChange: onSelectAllChange,
            }
            : undefined;
    }, [allRowsSelected, hasCardSelectAllCheckbox, onSelectAllChange]);
    return (React__default.createElement(React__default.Fragment, null,
        React__default.createElement("div", { className: "db-table-header" },
            React__default.createElement("div", { className: "db-table-title-wrapper" }, isCardHeaderLayout ? (React__default.createElement(TileTableHeader, { title: title, getTotalCountText: getTotalCountText, iconSrc: iconSrc, stringLiterals: stringLiterals, isAiGenerated: isAiGenerated })) : (React__default.createElement(TitleHeader, { title: title, getTotalCountText: getTotalCountText, stringLiterals: stringLiterals, isAiGenerated: isAiGenerated }))),
            (selectAllOptions || filterMenuOptions || toggleOptions) && (React__default.createElement("div", { className: `db-table-header-filter-line ${filterMenuType === 'tabs' ? 'db-table-header-filter-line-tabs-mode' : ''}` },
                React__default.createElement(FilterLine, { filter: filterMenuOptions, toggle: toggleOptions, selectAll: selectAllOptions, stringLiterals: stringLiterals }))))));
};

const CardRow = React__default.memo(({ canSelect, hasIcon, hasImage, isSelected, locale, measure, onSelectionChange, registerChild, row, rowActions, rowDefinition, stringLiterals, style, }) => {
    const hasLongContent = React__default.useMemo(() => rowDefinition.longContent && row.longContent, [
        row.longContent,
        rowDefinition.longContent,
    ]);
    return (React__default.createElement("div", { ref: registerChild, onLoad: measure, style: style, key: row.i, className: isSelected ? 'db-card-list-row db-table-selected' : 'db-card-list-row', "data-testid": `db-card-list-row-${row.i}`, role: "row" },
        canSelect && (React__default.createElement("span", { className: "db-card-list-select", role: "gridcell" },
            React__default.createElement(Checkbox, { "data-testid": `db-card-list-select-${row.i}`, checked: isSelected, onChange: () => onSelectionChange(row.i), "aria-label": resolveTranslationKey(stringLiterals, 'table_select') }))),
        (hasImage || hasIcon) && (React__default.createElement("span", { className: "db-card-list-image", role: "gridcell" },
            hasImage && (React__default.createElement(Portrait, { size: "M", src: row.image, initials: String(row.title), alt: String(row.title) })),
            hasIcon && React__default.createElement(Icon$1, { fontSize: "large", type: row.icon ?? 'none' }))),
        React__default.createElement("div", { className: "db-card-list-content" },
            React__default.createElement("div", { className: "db-card-list-row-row" },
                React__default.createElement("span", { className: "db-card-list-title", "data-testid": `db-card-list-row-field-${row.i}-title`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.title, locale: locale, definition: rowDefinition.title, i: row.i, isLeftPosition: true, propertyName: "title" })),
                row.titleRight && (React__default.createElement("span", { className: "db-card-list-row-row-right", "data-testid": `db-card-list-row-field-${row.i}-title-right`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.titleRight, locale: locale, definition: rowDefinition.titleRight, i: row.i, isLeftPosition: false, propertyName: "titleRight" })))),
            hasLongContent && (React__default.createElement("div", { className: "db-card-list-row-long-content", "data-testid": `db-card-list-row-field-${row.i}-long-content`, role: "gridcell" }, row.longContent)),
            React__default.createElement("div", { className: "db-card-list-row-row" },
                row.line2 && (React__default.createElement("div", { className: "db-card-list-line-2", "data-testid": `db-card-list-row-field-${row.i}-line2`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.line2, locale: locale, definition: rowDefinition.line2, i: row.i, isLeftPosition: true, propertyName: "line2" }))),
                row.line2Right && (React__default.createElement("span", { className: "db-card-list-row-row-right", "data-testid": `db-card-list-row-field-${row.i}-line2-right`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.line2Right, locale: locale, definition: rowDefinition.line2Right, i: row.i, isLeftPosition: false, propertyName: "line2Right" })))),
            React__default.createElement("div", { className: "db-card-list-row-row" },
                row.line3 && (React__default.createElement("div", { className: "db-card-list-line-3", "data-testid": `db-card-list-row-row-field-${row.i}-line3`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.line3, locale: locale, definition: rowDefinition.line3, i: row.i, propertyName: "line3" }))),
                row.line3Right && (React__default.createElement("span", { className: "db-card-list-row-row-right", "data-testid": `db-card-list-row-field-${row.i}-line3-right`, role: "gridcell" },
                    React__default.createElement(CollectionItemValueComponent, { value: row.line3Right, locale: locale, definition: rowDefinition.line3Right, i: row.i, propertyName: "line3Right" }))))),
        rowActions && rowActions.length > 0 && (React__default.createElement("div", { className: row.line3 || row.line3Right
                ? 'db-row-actions-container-3-lines'
                : 'db-row-actions-container' },
            React__default.createElement(CollectionActionPopoverComponent, { row: row, rowActions: rowActions, rowActionTestId: "db-card-row-action", rowActionsTestId: "db-card-list-row-actions", stringLiterals: stringLiterals })))));
});
CardRow.displayName = 'CardRow';

const ROW_HEIGHT = 86;
const CardList = React__default.memo(({ canSelect, data, hasIcon, hasImage, height, loadMoreRows: loadMore, locale, onSelectionChange, rowDefinition, rowActions, selectedRows, setHasAllDataLoaded, stringLiterals, }) => {
    const cache = React__default.useMemo(() => new CellMeasurerCache({
        defaultHeight: ROW_HEIGHT,
        fixedWidth: true,
        minHeight: ROW_HEIGHT,
    }), []);
    const infiniteLoaderRef = React__default.useRef(null);
    const [loadedMap, setLoadedMap] = React__default.useState(data.map(() => WidgetCollectionItemStatus.LOADED));
    const isRowLoaded = React__default.useCallback(({ index }) => {
        return loadedMap[index] != null;
    }, [loadedMap]);
    const rowCount = React__default.useMemo(() => {
        return loadedMap.filter(l => l === WidgetCollectionItemStatus.LOADED).length;
    }, [loadedMap]);
    const reload = React__default.useCallback(() => {
        infiniteLoaderRef.current?.resetLoadMoreRowsCache(true);
    }, []);
    React__default.useEffect(() => {
        setLoadedMap(data.map(() => WidgetCollectionItemStatus.LOADED));
        infiniteLoaderRef.current?.resetLoadMoreRowsCache();
        setHasAllDataLoaded(false);
    }, [data, setHasAllDataLoaded]);
    const rowRenderer = React__default.useCallback(({ style, index, columnIndex, key, parent }) => {
        if (loadedMap[index] === WidgetCollectionItemStatus.LOADING) {
            return (React__default.createElement("div", { style: style, className: "db-row-loading" },
                React__default.createElement(Loader, { size: "small" })));
        }
        if (loadedMap[index] === WidgetCollectionItemStatus.FAILED) {
            return (React__default.createElement("div", { style: style, className: "db-row-error" },
                React__default.createElement(Button, { buttonType: "tertiary", fullWidth: true, destructive: true, iconType: "refresh", onClick: reload }, resolveTranslationKey(stringLiterals, 'reload'))));
        }
        const rowData = data[index];
        if (!rowData) {
            return null;
        }
        return (React__default.createElement(CellMeasurer, { cache: cache, columnIndex: columnIndex, key: key, parent: parent, rowIndex: index }, ({ measure, registerChild }) => (React__default.createElement(CardRow, { measure: measure, registerChild: registerChild, canSelect: canSelect, hasIcon: hasIcon, hasImage: hasImage, isSelected: selectedRows.includes(rowData.i), locale: locale, onSelectionChange: onSelectionChange, row: rowData, rowActions: rowActions, rowDefinition: rowDefinition, stringLiterals: stringLiterals, style: omit(style, ['columnWidth']) }))));
    }, [
        loadedMap,
        data,
        cache,
        reload,
        canSelect,
        hasIcon,
        hasImage,
        selectedRows,
        locale,
        onSelectionChange,
        rowActions,
        rowDefinition,
        stringLiterals,
    ]);
    const loadMoreRows = React__default.useCallback(async ({ startIndex, stopIndex }) => {
        const pageSize = stopIndex - startIndex + 1;
        const pageCount = Math.ceil((startIndex + 1) / pageSize);
        let after = data[startIndex - 1];
        let first = pageSize;
        if (loadedMap[startIndex - 1] !== WidgetCollectionItemStatus.LOADED) {
            const idx = loadedMap
                .slice(0, startIndex)
                .reverse()
                .findIndex(item => item === WidgetCollectionItemStatus.LOADED);
            if (idx !== -1) {
                const newStartIndex = startIndex - idx - 1;
                after = data[newStartIndex];
                first = stopIndex - newStartIndex + 1;
            }
        }
        try {
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    clone[i] = WidgetCollectionItemStatus.LOADING;
                }
                return clone;
            });
            const newRows = await loadMore?.({
                startIndex,
                stopIndex,
                pageCount,
                pageSize,
                after,
                first,
            });
            // No rows returned or same rows as before
            if (newRows?.length === 0 || isEqual(newRows, data)) {
                setHasAllDataLoaded(true);
                setLoadedMap(current => {
                    const clone = [...current];
                    for (let i = startIndex; i <= stopIndex; i++) {
                        clone[i] = WidgetCollectionItemStatus.STALE;
                    }
                    return clone;
                });
                return;
            }
            // Less rows than pageSize
            if (newRows && newRows?.length < pageSize) {
                setHasAllDataLoaded(true);
                setLoadedMap(current => {
                    const clone = [...current];
                    for (let i = startIndex; i <= stopIndex; i++) {
                        clone[i] =
                            i >= startIndex + newRows.length
                                ? WidgetCollectionItemStatus.STALE
                                : WidgetCollectionItemStatus.LOADED;
                    }
                    return clone;
                });
                return;
            }
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    if (newRows?.[i - startIndex]) {
                        clone[i] = WidgetCollectionItemStatus.LOADED;
                    }
                    else {
                        clone[i] = WidgetCollectionItemStatus.STALE;
                    }
                }
                return clone;
            });
        }
        catch (err) {
            setLoadedMap(current => {
                const clone = [...current];
                for (let i = startIndex; i <= stopIndex; i++) {
                    clone[i] = WidgetCollectionItemStatus.FAILED;
                }
                return clone;
            });
        }
    }, [loadMore, data, loadedMap, setHasAllDataLoaded]);
    return (React__default.createElement("div", { className: "db-card-list-representation", "data-testid": "db-card-list-representation", style: { height: height || '90%' } },
        React__default.createElement(InfiniteLoader, { ref: infiniteLoaderRef, isRowLoaded: isRowLoaded, loadMoreRows: loadMoreRows, rowCount: Number.MAX_SAFE_INTEGER, minimumBatchSize: 20 }, ({ onRowsRendered, registerChild: registerInfiniteLoaderChild }) => (React__default.createElement(AutoSizer, { disableHeight: true, "data-testid": "db-table-table-representation", defaultHeight: height || 100 }, ({ width }) => {
            return (React__default.createElement(List, { ref: registerInfiniteLoaderChild, containerStyle: { width: '100%' }, deferredMeasurementCache: cache, estimatedRowSize: 20, height: height || 100, onRowsRendered: onRowsRendered, rowCount: rowCount, rowHeight: cache.rowHeight, rowRenderer: rowRenderer, width: width }));
        })))));
});
CardList.displayName = 'CardBody';

/* eslint-disable react/prop-types */
function defaultLoadMoreRows() {
    return Promise.resolve([]);
}
const TableComponent = ({ callToActions, canChangeViewMode, canSelect, data, dateFilter, emptyStateMessage = 'No items to display.', externalSelectedRows, filterMenu, filterMenuType = 'dropdown', iconSrc, isAiGenerated, loadMoreRows = defaultLoadMoreRows, locale, mode, onDateFilterChanged, onFilterMenuChanged = noop, onRepresentationModeChanged = noop, onSelectionChanged = noop, pageSize, rowActions, rowDefinition, selectedPeriod, stringLiterals, title, totalCount, }) => {
    const [representationMode, setRepresentationMode] = React__default.useState(mode ?? 'table');
    const [hasAllDataLoaded, setHasAllDataLoaded] = React__default.useState(false);
    const [selectedRows, setSelectedRows] = React__default.useState(externalSelectedRows ?? []);
    React__default.useEffect(() => {
        setSelectedRows(externalSelectedRows ?? []);
    }, [externalSelectedRows]);
    const [headerRef, { height: headerHeight }] = useElementSize();
    const [tableComponentRef, { height: tableComponentHeight }] = useElementSize();
    const hasCallToActions = React__default.useMemo(() => Boolean(callToActions &&
        Object.keys(callToActions).filter(a => !callToActions[a].isHidden).length > 0), [callToActions]);
    const hadCallToActions = usePrevious(hasCallToActions);
    const hasData = React__default.useMemo(() => (data ?? []).length > 0, [data]);
    const hasCardSelectAllCheckbox = React__default.useMemo(() => hasData && Boolean(canSelect) && representationMode === 'card', [canSelect, hasData, representationMode]);
    const hasImage = React__default.useMemo(() => hasData && Boolean(data?.find(d => d.image)), [
        data,
        hasData,
    ]);
    const hasIcon = React__default.useMemo(() => hasData && !hasImage && Boolean(data?.find(d => d.icon)), [data, hasData, hasImage]);
    const allRowsSelected = React__default.useMemo(() => {
        return data && data?.length > 0 && selectedRows.length === data.length;
    }, [data, selectedRows]);
    const onSelectionChange = React__default.useCallback((i) => {
        const index = selectedRows.indexOf(i);
        const currentSelection = [...selectedRows];
        if (index === -1) {
            setSelectedRows([...selectedRows, i]);
            onSelectionChanged([...selectedRows, i]);
        }
        else {
            currentSelection.splice(index, 1);
            setSelectedRows(currentSelection);
            onSelectionChanged([...currentSelection]);
        }
    }, [onSelectionChanged, selectedRows]);
    const onSelectAllChange = React__default.useCallback((newState) => {
        const newSelection = newState ? data?.map(d => d.i) ?? [] : [];
        setSelectedRows([...newSelection]);
        onSelectionChanged([...newSelection]);
    }, [data, onSelectionChanged]);
    const onModeChanged = React__default.useCallback((mode) => {
        if (onRepresentationModeChanged) {
            onRepresentationModeChanged(mode);
        }
        setRepresentationMode(mode);
    }, [onRepresentationModeChanged]);
    const bodyHeight = React__default.useMemo(() => {
        const callToActionCompensation = (Number(hasCallToActions || false) - Number(hadCallToActions || false)) * 47;
        const dateFilterCompensation = dateFilter && onDateFilterChanged ? 74 : 0;
        return tableComponentHeight - headerHeight - dateFilterCompensation - callToActionCompensation;
    }, [
        dateFilter,
        hadCallToActions,
        hasCallToActions,
        headerHeight,
        onDateFilterChanged,
        tableComponentHeight,
    ]);
    const dataLength = React__default.useMemo(() => data?.length || 0, [data?.length]);
    const notDivisibleByPageSize = React__default.useMemo(() => (pageSize ? dataLength % pageSize !== 0 : false), [dataLength, pageSize]);
    const isCardHeaderLayout = React__default.useMemo(() => {
        return iconSrc !== undefined || totalCount !== undefined;
    }, [totalCount, iconSrc]);
    const getTotalCountText = React__default.useCallback(() => {
        if (totalCount !== undefined && totalCount !== null) {
            return totalCount;
        }
        if (hasAllDataLoaded || dataLength === 0 || notDivisibleByPageSize) {
            return dataLength;
        }
        return `${dataLength} +`;
    }, [dataLength, hasAllDataLoaded, notDivisibleByPageSize, totalCount]);
    return (React__default.createElement("div", { className: "db-table-component", ref: tableComponentRef },
        React__default.createElement("div", { className: "db-table-header", ref: headerRef },
            React__default.createElement(TableHeader, { allRowsSelected: allRowsSelected, canChangeViewMode: canChangeViewMode, filterMenu: filterMenu, filterMenuType: filterMenuType, getTotalCountText: getTotalCountText, hasCardSelectAllCheckbox: hasCardSelectAllCheckbox, iconSrc: iconSrc, isCardHeaderLayout: isCardHeaderLayout, onFilterMenuChanged: onFilterMenuChanged, onModeChanged: onModeChanged, onSelectAllChange: onSelectAllChange, representationMode: representationMode, stringLiterals: stringLiterals, title: title, isAiGenerated: isAiGenerated })),
        dateFilter && onDateFilterChanged && (React__default.createElement(DateFilter, { dateFilter: dateFilter, onDateFilterChanged: onDateFilterChanged, stringLiterals: stringLiterals, locale: locale, selectedPeriod: selectedPeriod })),
        React__default.createElement("div", { className: "db-table-body" },
            hasData && representationMode === 'card' && (React__default.createElement(CardList, { canSelect: canSelect, data: data ?? [], hasIcon: hasIcon, hasImage: hasImage, height: bodyHeight, loadMoreRows: loadMoreRows, locale: locale, onSelectAllChange: onSelectAllChange, onSelectionChange: onSelectionChange, rowActions: rowActions, rowDefinition: rowDefinition, selectedRows: selectedRows, setHasAllDataLoaded: setHasAllDataLoaded, stringLiterals: stringLiterals })),
            hasData && representationMode === 'table' && (React__default.createElement(Table, { canSelect: canSelect, data: data ?? [], hasIcon: hasIcon, hasImage: hasImage, height: bodyHeight, loadMoreRows: loadMoreRows, locale: locale, onSelectAllChange: onSelectAllChange, onSelectionChange: onSelectionChange, rowActions: rowActions, rowDefinition: rowDefinition, selectedRows: selectedRows, setHasAllDataLoaded: setHasAllDataLoaded, stringLiterals: stringLiterals })),
            !hasData && (React__default.createElement("div", { className: "db-table-empty-state", "data-testid": "db-table-empty-state" }, emptyStateMessage && (React__default.createElement(Typography, { variant: "h4", color: "blackOpacity55", margin: "auto" }, emptyStateMessage)))))));
};

const Legend = ({ details, style, type }) => {
    return (React__default.createElement("div", { className: "db-chart-component__legend", style: style }, Object.entries(details).map(([key, item]) => {
        return (React__default.createElement("div", { key: key, className: "db-chart-component__legend_wrapper" },
            React__default.createElement("div", { style: {
                    width: 16,
                    height: 16,
                    marginRight: '0.5rem',
                    backgroundColor: item.color,
                    borderRadius: type === 'line' ? 8 : 0,
                } }),
            React__default.createElement("span", null, item.label)));
    })));
};

const triangleSize = 7;
const getBackgroundColor = () => '#335C6D';
function TooltipRenderer(props) {
    const latestFit = useLatestWhen(props.anchor.fit, !!props.anchor.fit);
    if (!props.focusedDatum) {
        return null;
    }
    const { focusedDatum } = props;
    const { dark } = props.getOptions();
    const groupDatums = props.focusedDatum?.tooltipGroup ?? [];
    groupDatums.length;
    // Get the focused series' index
    groupDatums.findIndex(d => d === focusedDatum);
    const finalAlign = `${latestFit?.side}-${latestFit?.align}`;
    let arrowPosition;
    let triangleStyles;
    if (!arrowPosition) {
        if (finalAlign === 'left-center') {
            arrowPosition = 'right';
        }
        else if (finalAlign === 'right-center') {
            arrowPosition = 'left';
        }
        else if (finalAlign === 'top-center') {
            arrowPosition = 'bottom';
        }
        else if (finalAlign === 'bottom-center') {
            arrowPosition = 'top';
        }
        else if (finalAlign === 'right-start') {
            arrowPosition = 'bottomLeft';
        }
        else if (finalAlign === 'right-end') {
            arrowPosition = 'topLeft';
        }
        else if (finalAlign === 'left-start') {
            arrowPosition = 'bottomRight';
        }
        else if (finalAlign === 'left-end') {
            arrowPosition = 'topRight';
        }
    }
    const backgroundColor = props.backgroundColor || getBackgroundColor();
    if (arrowPosition === 'bottom') {
        triangleStyles = {
            top: '100%',
            left: '50%',
            transform: 'translate3d(-50%, 0%, 0)',
            borderLeft: `${triangleSize * 0.8}px solid transparent`,
            borderRight: `${triangleSize * 0.8}px solid transparent`,
            borderTop: `${triangleSize}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'top') {
        triangleStyles = {
            top: '0%',
            left: '50%',
            transform: 'translate3d(-50%, -100%, 0)',
            borderLeft: `${triangleSize * 0.8}px solid transparent`,
            borderRight: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'right') {
        triangleStyles = {
            top: '50%',
            left: '100%',
            transform: 'translate3d(0%, -50%, 0)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderLeft: `${triangleSize}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'left') {
        triangleStyles = {
            top: '50%',
            left: '0%',
            transform: 'translate3d(-100%, -50%, 0)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderRight: `${triangleSize}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'topRight') {
        triangleStyles = {
            top: '0%',
            left: '100%',
            transform: 'translate3d(-50%, -50%, 0) rotate(-45deg)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderLeft: `${triangleSize * 2}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'bottomRight') {
        triangleStyles = {
            top: '100%',
            left: '100%',
            transform: 'translate3d(-50%, -50%, 0) rotate(45deg)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderLeft: `${triangleSize * 2}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'topLeft') {
        triangleStyles = {
            top: '0%',
            left: '0%',
            transform: 'translate3d(-50%, -50%, 0) rotate(45deg)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderRight: `${triangleSize * 2}px solid ${backgroundColor}`,
        };
    }
    else if (arrowPosition === 'bottomLeft') {
        triangleStyles = {
            top: '100%',
            left: '0%',
            transform: 'translate3d(-50%, -50%, 0) rotate(-45deg)',
            borderTop: `${triangleSize * 0.8}px solid transparent`,
            borderBottom: `${triangleSize * 0.8}px solid transparent`,
            borderRight: `${triangleSize * 2}px solid ${backgroundColor}`,
        };
    }
    else {
        triangleStyles = {
            opacity: 0,
        };
    }
    return (React__default.createElement("div", { style: {
            position: 'relative',
            // MC: custom font size
            fontSize: '14px',
            // MC: custom padding
            padding: '8px 12px',
            background: backgroundColor,
            color: dark ? 'black' : 'white',
        } },
        React__default.createElement("div", { style: {
                position: 'absolute',
                width: 0,
                height: 0,
                ...triangleStyles,
            } }),
        React__default.createElement("div", null, props.children)));
}

const DEFAULT_COLORS = [
    '#003349', // https://www.figma.com/file/QHlyy9vYTmapfgsnl8INRP/%5BDelivery%5D-XTreeM-Dashboard-Widgets?node-id=373%3A9751
    '#0f83ab',
    '#faa43a',
    '#fd6868',
    '#53cfc9',
    '#a2d925',
    '#decf3f',
    '#734fe9',
    '#cd82ad',
    '#006d92',
    '#de7c00',
    '#f33232',
    '#3f9a80',
    '#53c200',
    '#d7af00',
    '#4c26c9',
    '#d44d99',
];
const CIRCLE_RADIUS = 5;
const ChartComponent = ({ data: inputData, options, primaryAxis: primary, primaryAxisLabel, secondaryAxes: secondary, secondaryAxisLabel, stringLiterals, }) => {
    const areAxesSwapped = options.type === 'bar' && options.areAxesSwapped;
    const areAxesStacked = options.type === 'bar' && options.areAxesStacked;
    const isHistogram = options.type === 'bar' && options.isHistogram;
    const data = React__default.useMemo(() => {
        return secondary.map(({ title, color, bind }) => ({
            id: bind,
            label: resolveTranslationKey(stringLiterals, title),
            /**
             * Currently this option is not taken into account (see https://github.com/TanStack/react-charts/issues/252)
             * To set the appropriate color "seriesColors" & "getSeriesStyle" are used instead.
             */
            color,
            data: inputData.map(d => ({ primary: get(d, primary.bind), secondary: get(d, bind) })),
        }));
    }, [primary, secondary, inputData, stringLiterals]);
    const primaryAxis = React__default.useMemo(() => ({
        position: areAxesSwapped ? 'left' : 'bottom',
        getValue: datum => datum.primary,
        showGrid: options.type === 'line',
        ...(isHistogram && { innerBandPadding: 0.005, innerSeriesBandPadding: 0.005 }),
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        scaleType: 'band',
        tickLabelRotationDeg: 0,
    }), [areAxesSwapped, isHistogram, options.type]);
    const max = React__default.useMemo(() => secondary
        .flatMap(({ bind }) => inputData.map(d => get(d, bind)))
        .reduce((acc, cur) => Math.max(Number(cur), acc), Number.NEGATIVE_INFINITY), [inputData, secondary]);
    const secondaryAxes = React__default.useMemo(() => [
        {
            position: areAxesSwapped ? 'bottom' : 'left',
            getValue: datum => datum.secondary,
            min: 0,
            max,
            showDatumElements: true,
            stacked: areAxesStacked,
            scaleType: 'linear',
            tickLabelRotationDeg: 0,
            /* eslint-disable @typescript-eslint/no-explicit-any */
            ...(options.type === 'line' && {
                elementType: 'line',
                curve: function (ctx) {
                    return {
                        areaStart: function () {
                            this._line = 0;
                        },
                        areaEnd: function () {
                            this._line = NaN;
                        },
                        lineStart: function () {
                            this._point = 0;
                        },
                        lineEnd: function () {
                            if (this._line || (this._line !== 0 && this._point === 1))
                                ctx.closePath();
                            this._line = 1 - this._line;
                        },
                        point: function (x, y) {
                            (x = +x), (y = +y);
                            switch (this._point) {
                                case 0:
                                    this._point = 1;
                                    this._line ? ctx.lineTo(x, y) : ctx.moveTo(x, y);
                                    break;
                                case 1:
                                    this._point = 2; // falls through
                                default:
                                    ctx.lineTo(x, y);
                                    break;
                            }
                        },
                    };
                },
            }),
            /* eslint-enable @typescript-eslint/no-explicit-any */
        },
    ], [areAxesSwapped, max, areAxesStacked, options.type]);
    // Dict of series by series ID (i.e. 'bind')
    const seriesMap = React__default.useMemo(() => {
        return secondary.reduce((acc, curr, index) => {
            acc[curr.bind] = { ...curr, color: curr.color || DEFAULT_COLORS[index] };
            return acc;
        }, {});
    }, [secondary]);
    // Dict of colors by series title
    const colors = React__default.useMemo(() => {
        return secondary.reduce((acc, curr, index) => {
            acc[curr.bind] = {
                color: curr.color || DEFAULT_COLORS[index],
                label: curr.title ?? curr.bind,
            };
            return acc;
        }, {});
    }, [secondary]);
    const getSeriesStyle = React__default.useCallback((series) => {
        return {
            fill: seriesMap[series.id].color,
            stroke: seriesMap[series.id].color,
            transition: 'none',
        };
    }, [seriesMap]);
    const tooltipRenderer = args => {
        if (!args.focusedDatum) {
            return null;
        }
        const { tooltipContent, color } = seriesMap[args.focusedDatum.seriesId];
        if (!tooltipContent) {
            return null;
        }
        const { primaryValue, secondaryValue, index } = args.focusedDatum;
        if (primaryValue === undefined || secondaryValue === undefined || index === undefined) {
            return;
        }
        return (React__default.createElement(TooltipRenderer, { ...args, backgroundColor: color },
            React__default.createElement("p", { style: { whiteSpace: 'pre-wrap' } }, tooltipContent({ primaryValue, secondaryValue, record: cloneDeep(inputData[index]) }))));
    };
    const getDatumStyle = () => ({
        // "r" is an internal property of react-charts used to determine the circle radius
        circle: { r: CIRCLE_RADIUS, strokeWidth: 2, fill: 'white' },
    });
    const onClickDatum = (args, event) => {
        const { primaryValue, secondaryValue, index, seriesId } = args || {};
        if (primaryValue === undefined ||
            secondaryValue === undefined ||
            index === undefined ||
            seriesId === undefined) {
            return;
        }
        const seriesDefinition = seriesMap[seriesId];
        if (seriesDefinition.onClick === undefined) {
            return;
        }
        const clickedPoint = { x: event.pageX, y: event.pageY };
        /**
         * Click event is always triggered on Voronoi rectangle so the following is needed to determine
         * if clicked point actually belongs to the chart.
         */
        if (Array.from(args?.element?.parentElement?.querySelectorAll(options.type === 'line' ? 'circle' : 'rect') ?? [])
            .map(rect => rect.getBoundingClientRect())
            .some(rect => {
            return (clickedPoint.x >= rect.x &&
                clickedPoint.x <= rect.x + rect.width &&
                clickedPoint.y >= rect.y &&
                clickedPoint.y <= rect.y + rect.height);
        })) {
            seriesDefinition.onClick({
                primaryValue,
                secondaryValue,
                record: cloneDeep(inputData[index]),
            });
        }
    };
    const hasLegend = secondary.length > 1;
    return (React__default.createElement("div", { className: "db-chart-component__y-wrapper" },
        React__default.createElement("div", { className: "db-chart-component", datatype: options.type, style: {
                // need to set a fixed height for the chart to work
                // subtract [ (legend's height + gap) + (primary axis height + gap) ]
                height: `calc(100% - ${hasLegend ? 17 + 32 : 0}px - ${primaryAxisLabel ? 17 + 8 : 0}px)`,
            } },
            secondaryAxisLabel && (React__default.createElement("div", { className: "db-chart-component__y-axis-title", style: {
                    // (x axis - chart bottom) + (primary axix label height) + (chart bottom - primary axis label)
                    marginBottom: `calc(24.8px + ${primaryAxisLabel ? 17 : 0}px + 0.5rem)`,
                } }, secondaryAxisLabel)),
            React__default.createElement("div", { className: "db-chart-component__x-wrapper" },
                React__default.createElement(Chart, { className: `db-chart-component__chart`, options: {
                        data,
                        defaultColors: DEFAULT_COLORS,
                        primaryAxis,
                        secondaryAxes,
                        tooltip: {
                            render: tooltipRenderer,
                        },
                        primaryCursor: { show: false, showLabel: false, showLine: false },
                        secondaryCursor: { show: false, showLabel: false, showLine: false },
                        getSeriesStyle,
                        getDatumStyle,
                        onClickDatum,
                        useIntersectionObserver: false,
                    } }),
                primaryAxisLabel && (React__default.createElement("div", { className: "db-chart-component__x-axis-title" }, primaryAxisLabel)))),
        hasLegend && (React__default.createElement(Legend, { details: colors, type: options.type, style: secondaryAxisLabel ? { marginLeft: '2rem' } : undefined }))));
};

const BarChartComponent = props => {
    const { areAxesSwapped = false, areAxesStacked = false, isHistogram = false } = props;
    return (React__default.createElement(ChartComponent, { ...props, options: { type: 'bar', areAxesSwapped, areAxesStacked, isHistogram } }));
};

const basicResolveImageUrl = (collectionId, resourceId) => `/assets/icons/${collectionId}/${resourceId}.svg`;
const VisualProcess = ({ value, onLinkClick, resolveImageUrl, }) => {
    return (React__default.createElement(VisualProcessEditorComponent, { canEdit: false, onLinkClick: onLinkClick || noop$1, icons: {}, name: "vp", getData: () => Promise.resolve({
            label: 'noop',
            value: 'noop',
        }), resolveImageUrl: resolveImageUrl || basicResolveImageUrl, onChange: noop$1, document: value }));
};

const LineChartComponent = props => {
    return React__default.createElement(ChartComponent, { ...props, options: { type: 'line' } });
};

var PieChartViewMode;
(function (PieChartViewMode) {
    PieChartViewMode["CHART"] = "chart";
    PieChartViewMode["GRID"] = "grid";
})(PieChartViewMode || (PieChartViewMode = {}));

const START_ANGLE = 90;
const END_ANGLE = -270;
const PADDING_ANGLE = 3;
const RADIAN = Math.PI / 180;
const DEFAULT_PIE_PART_COLORS = ['#00884A', '#0E74A7', '#007776', '#757575', '#C15708'];
const renderCustomizedLabel = ({ cx, cy, midAngle, innerRadius, outerRadius, percent, }) => {
    const radius = innerRadius + (outerRadius - innerRadius) * 0.5;
    const x = cx + radius * Math.cos(-midAngle * RADIAN);
    const y = cy + radius * Math.sin(-midAngle * RADIAN);
    return (React__default.createElement("text", { x: x, y: y, fill: "white", textAnchor: "middle", dominantBaseline: "central" }, `${(percent * 100).toFixed(0)}%`));
};
const getCellColor = ({ piePartIndex, defaultColors, piePart, }) => {
    if (piePart.color)
        return piePart.color;
    return defaultColors[piePartIndex % defaultColors.length];
};
const PieChartWidget = ({ pieChartData, isDonut, stringLiterals, chartLabel, hasCardView = false, collectionItemDefinition: tableDataDefinition, locale, callToActions, filter: controlledFilter, }) => {
    const [viewMode, setViewMode] = React__default.useState(PieChartViewMode.CHART);
    const [filterHeight, setFilterHeight] = React__default.useState(0);
    const [pieChartComponentRef, { height: pieChartComponentHeight }] = useElementSize();
    const hasCallToActions = callToActions && Object.keys(callToActions).filter(a => !callToActions[a].isHidden).length > 0;
    const hadCallToActions = usePrevious(hasCallToActions);
    const bodyHeight = React__default.useMemo(() => {
        const callToActionCompensation = (Number(hasCallToActions || false) - Number(hadCallToActions || false)) * 47;
        return pieChartComponentHeight - filterHeight - callToActionCompensation;
    }, [hadCallToActions, hasCallToActions, filterHeight, pieChartComponentHeight]);
    const [filterValue, setFilterValue] = React__default.useState(Object.keys(controlledFilter?.options ?? {})[0]);
    const filter = React__default.useMemo(() => controlledFilter
        ? {
            ...controlledFilter,
            value: filterValue ?? '',
            onChange: newValue => {
                setFilterValue(newValue ?? undefined);
                controlledFilter?.onChange(newValue);
            },
        }
        : undefined, [controlledFilter, filterValue]);
    const toggle = React__default.useMemo(() => {
        if (hasCardView && tableDataDefinition) {
            return {
                ariaLabel: resolveTranslationKey(stringLiterals, 'view_switch'),
                id: 'piechart-view-mode',
                onChange: newValue => setViewMode(newValue),
                value: viewMode,
                options: {
                    [PieChartViewMode.CHART]: {
                        icon: 'chart_pie',
                        label: resolveTranslationKey(stringLiterals, 'table_switch_to_chart_view'),
                    },
                    [PieChartViewMode.GRID]: {
                        icon: 'list_view',
                        label: resolveTranslationKey(stringLiterals, 'table_switch_to_card_view'),
                    },
                },
            };
        }
        return undefined;
    }, [hasCardView, stringLiterals, tableDataDefinition, viewMode]);
    return (React__default.createElement("div", { className: "widget-pie-chart", ref: pieChartComponentRef },
        React__default.createElement(FilterLine, { filter: filter, toggle: toggle, stringLiterals: stringLiterals, isFilterFullWidth: true, onFilterHeightChange: setFilterHeight }),
        chartLabel && (React__default.createElement(Typography, { variant: "h2" }, resolveTranslationKey(stringLiterals, chartLabel))),
        viewMode === PieChartViewMode.CHART && (React__default.createElement(ResponsiveContainer, { height: "100%", width: "100%" },
            React__default.createElement(PieChart, null,
                React__default.createElement(Pie, { innerRadius: isDonut ? '50%' : 0, dataKey: "value", nameKey: "title", data: pieChartData.map(d => ({
                        ...d,
                        title: d.legendLabel
                            ? resolveTranslationKey(stringLiterals, d.legendLabel)
                            : resolveTranslationKey(stringLiterals, String(d.title)),
                    })), label: renderCustomizedLabel, labelLine: false, paddingAngle: PADDING_ANGLE, startAngle: START_ANGLE, endAngle: END_ANGLE }, pieChartData.map((piePart, piePartIndex) => (React__default.createElement(Cell, { key: `cell-${piePartIndex}`, fill: getCellColor({
                        piePartIndex,
                        defaultColors: DEFAULT_PIE_PART_COLORS,
                        piePart,
                    }) })))),
                React__default.createElement(Legend$1, { align: "center", verticalAlign: "bottom", iconType: "square" })))),
        viewMode === PieChartViewMode.GRID && tableDataDefinition && (React__default.createElement("div", { className: "db-table-component pie-chart-card-table" },
            React__default.createElement("div", { className: "db-table-body" },
                React__default.createElement(CardList, { hasImage: !!tableDataDefinition?.image, hasIcon: !!tableDataDefinition?.icon, canSelect: false, setHasAllDataLoaded: noop$1, onSelectAllChange: noop$1, onSelectionChange: noop$1, selectedRows: [], rowDefinition: tableDataDefinition, locale: locale, height: bodyHeight, stringLiterals: stringLiterals, loadMoreRows: () => Promise.resolve([]), data: pieChartData.map(r => ({ ...r, i: r._id })) }))))));
};

const EvolutionIndicator = ({ evolutionUnit, evolutionValue, stringLiterals, scale, locale, }) => {
    const classes = ['db-evolution-indicator'];
    let displayValue = '';
    let iconType = 'none';
    let iconColor = '';
    const formattedValue = formatNumber(Math.abs(evolutionValue), locale, scale);
    if (evolutionValue < 0) {
        displayValue = `- ${formattedValue} ${evolutionUnit}`;
        iconType = 'caret_down';
        iconColor = '#b1420c';
        classes.push('db-evolution-indicator-negative');
    }
    else if (evolutionValue > 0) {
        displayValue = `+ ${formattedValue} ${evolutionUnit}`;
        iconType = 'caret_up';
        iconColor = '#00884A';
        classes.push('db-evolution-indicator-positive');
    }
    else {
        displayValue = resolveTranslationKey(stringLiterals, 'no_change');
    }
    return (React__default.createElement("div", { className: classes.join(' ') },
        evolutionValue !== 0 && React__default.createElement(Icon$1, { type: iconType, color: iconColor }),
        React__default.createElement("span", null, displayValue)));
};

const renderActiveShape = props => {
    const { cx, cy, innerRadius, outerRadius, startAngle, endAngle, fill, payload } = props;
    return (React__default.createElement("g", null,
        React__default.createElement("text", { x: cx, y: cy, dy: 8, textAnchor: "middle", fill: fill, fontFamily: "Sage UI", fontSize: 42, fontWeight: 500 }, payload.name),
        React__default.createElement(Sector, { cx: cx, cy: cy, innerRadius: innerRadius, outerRadius: outerRadius, startAngle: startAngle, endAngle: endAngle, fill: fill })));
};
const Gauge = ({ 
// TODO: Replace with design token
color = tokens.colorsSemanticInfo500, data, evolutionUnit = '%', locale, scale = 0, stringLiterals, valueUnit = '%', }) => {
    const [containerRef, { width, height }] = useElementSize();
    const totalValue = data.totalValue || 100;
    const remainingValue = totalValue - data.value;
    const formattedValue = formatNumber(data.value, locale, scale);
    const chartData = [
        { name: 'RemainingValue', value: remainingValue, color: tokens.colorsUtilityDisabled400 },
        { name: `${formattedValue} ${valueUnit}`, value: data.value, color },
    ];
    const outerRadius = Math.min(height || 60, width || 60) / 2 - 12;
    return (React__default.createElement("div", { className: "db-widget-content widgetGauge" },
        data.evolutionValue != null && (React__default.createElement(EvolutionIndicator, { stringLiterals: stringLiterals, locale: locale, evolutionUnit: evolutionUnit, evolutionValue: data.evolutionValue, scale: scale })),
        React__default.createElement("div", { className: "db-gauge-container", ref: containerRef },
            React__default.createElement(PieChart, { width: width, height: height - 24 },
                React__default.createElement(Pie, { startAngle: -270, activeShape: renderActiveShape, activeIndex: 1, data: chartData, cx: "50%", cy: "50%", innerRadius: outerRadius - 30, outerRadius: outerRadius, fill: "#8884d8", paddingAngle: 2, dataKey: "value" }, chartData.map((entry, index) => (React__default.createElement(Cell, { key: `cell-${index}`, fill: entry.color }))))))));
};

function useParentElementSize() {
    // Mutable values like 'ref.current' aren't valid dependencies
    // because mutating them doesn't re-render the component.
    // Instead, we use a state as a ref to be reactive.
    const [ref, setRef] = useState(null);
    const [size, setSize] = useState({
        width: 0,
        height: 0,
    });
    // Prevent too many rendering using useCallback
    const handleSize = useCallback(() => {
        setSize({
            width: ref?.parentElement?.offsetWidth || 0,
            height: ref?.parentElement?.offsetHeight || 0,
        });
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [ref?.parentElement?.offsetHeight, ref?.parentElement?.offsetWidth]);
    useEventListener('resize', handleSize);
    useIsomorphicLayoutEffect(() => {
        handleSize();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [ref?.parentElement?.offsetHeight, ref?.parentElement?.offsetWidth]);
    return [setRef, size];
}

function Preview() {
    const [containerRef, { height }] = useParentElementSize();
    return (React__default.createElement("div", { ref: containerRef, className: "db-preview-wrapper" },
        React__default.createElement(CarbonPreview, { height: `${height + 6}px` })));
}

function CustomWidget({ component = React__default.createElement(React__default.Fragment, null) }) {
    return component;
}

function QuickLinks({ links, linksCallback, stringLiterals }) {
    return (React__default.createElement("div", { className: "db-widget-content quicklinks", style: {
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'flex-start',
            gap: '16px',
        } }, links?.map(link => (React__default.createElement(Link, { key: link.link, className: "quicklinks-link", ...(linksCallback && !link.link.toLowerCase().startsWith('http')
            ? {
                onClick: () => {
                    linksCallback(link.link);
                },
            }
            : { href: link.link, target: '_blank', icon: 'link', iconAlign: 'right' }) }, resolveTranslationKey(stringLiterals, link.label))))));
}

const ContactCardHeader = ({ numberOfContacts, numberOfAddresses, iconSrc, stringLiterals }) => {
    return (React__default.createElement(Box, { alignItems: "center", alignSelf: "flex-start", backgroundColor: "var(--color-background)", boxSizing: "border-box", display: "flex", gap: "var(--spacing200)", justifyContent: "center", padding: "var(--spacing200)", className: "db-contact-card-body-header db-draggable-element-indicator" },
        iconSrc && (React__default.createElement("img", { className: "db-contact-card-icon", src: iconSrc, alt: resolveTranslationKey(stringLiterals, 'contact_image') })),
        React__default.createElement(Box, { boxSizing: "border-box", display: "flex", flexDirection: "column", gap: "var(--spacing050)", justifyContent: "center", flex: "1" },
            React__default.createElement(Box, { display: "flex", gap: "var(--spacing100)", alignItems: "flex-end" },
                React__default.createElement(Typography, { variant: "span", fontWeight: "700", fontSize: "16px", lineHeight: "24px" }, numberOfContacts),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, resolveTranslationKey(stringLiterals, 'contacts'))),
            React__default.createElement(Box, { display: "flex", gap: "var(--spacing100)", alignItems: "flex-end" },
                React__default.createElement(Typography, { variant: "span", fontWeight: "700", fontSize: "16px", lineHeight: "24px" }, numberOfAddresses),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, resolveTranslationKey(stringLiterals, 'addresses'))))));
};

function useDeepCompareMemoize(value) {
    const ref = React__default.useRef(value);
    const signalRef = React__default.useRef(0);
    if (!isEqual(value, ref.current)) {
        ref.current = value;
        signalRef.current += 1;
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
    return React__default.useMemo(() => ref.current, [signalRef.current]);
}
function useDeepCompareEffect(callback, dependencies) {
    // eslint-disable-next-line react-hooks/exhaustive-deps
    return React__default.useEffect(callback, useDeepCompareMemoize(dependencies));
}
function useDeepCompareMemo(factory, dependencies) {
    // eslint-disable-next-line react-hooks/exhaustive-deps
    return React__default.useMemo(factory, useDeepCompareMemoize(dependencies));
}

const ContactCardSelectionBar = ({ addresses, contacts, contactType, onContactTypeSwitchChanged, onSelectedContactChanged, onSelectedAddressChanged, selectedAddressId, selectedContactId, setContactType, stringLiterals, }) => {
    const selectedItemId = React__default.useMemo(() => (contactType === 'contact' ? selectedContactId : selectedAddressId), [contactType, selectedAddressId, selectedContactId]);
    const isControlled = React__default.useMemo(() => selectedItemId !== undefined, [selectedItemId]);
    const items = useDeepCompareMemo(() => (contactType === 'contact' ? contacts : addresses ?? {}), [
        addresses,
        contactType,
        contacts,
    ]);
    const onSelectedItemChange = React__default.useCallback((id) => {
        if (contactType === 'contact') {
            onSelectedContactChanged?.(id);
        }
        else {
            onSelectedAddressChanged?.(id);
        }
    }, 
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [contactType]);
    const [internalFilterValue, setInternalFilterValue] = React__default.useState(() => isControlled ? selectedItemId ?? undefined : Object.keys(items)[0]);
    const filterValue = React__default.useMemo(() => (isControlled ? selectedItemId ?? undefined : internalFilterValue), [internalFilterValue, isControlled, selectedItemId]);
    const onFilterChange = React__default.useCallback(newValue => {
        if (!isControlled) {
            setInternalFilterValue(newValue ?? undefined);
        }
        onSelectedItemChange?.(newValue || null);
    }, [isControlled, onSelectedItemChange]);
    useDeepCompareEffect(() => {
        const selected = isControlled ? selectedItemId : Object.keys(items)[0];
        if (selected === undefined) {
            return;
        }
        if (!isControlled) {
            setInternalFilterValue(selected ?? undefined);
        }
        onFilterChange(selected);
    }, [isControlled, selectedItemId, items, filterValue, onFilterChange]);
    const filter = useDeepCompareMemo(() => ({
        options: items,
        onChange: onFilterChange,
        isEmptyAllowed: true,
        value: filterValue,
        ariaLabel: resolveTranslationKey(stringLiterals, contactType === 'contact' ? 'select_contact' : 'select_address'),
        placeholder: resolveTranslationKey(stringLiterals, contactType === 'contact' ? 'select_contact' : 'select_address'),
    }), [items, filterValue, onFilterChange, stringLiterals, contactType]);
    const onToggleChange = React__default.useCallback(newValue => {
        const newContactType = newValue;
        setContactType(newContactType);
        onContactTypeSwitchChanged?.(newContactType);
    }, 
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [setContactType]);
    const toggle = useDeepCompareMemo(() => ({
        id: 'contact-type-toggle',
        ariaLabel: resolveTranslationKey(stringLiterals, 'contact_type'),
        onChange: onToggleChange,
        value: contactType,
        options: {
            contact: {
                icon: 'person_tick',
                label: resolveTranslationKey(stringLiterals, 'table_switch_to_contact_view'),
            },
            site: {
                icon: 'factory',
                label: resolveTranslationKey(defaultLiterals, 'table_switch_to_site_view'),
            },
        },
    }), [contactType, onToggleChange, stringLiterals]);
    return (React__default.createElement(FilterLine, { stringLiterals: stringLiterals, isFilterFullWidth: true, filter: filter, toggle: toggle }));
};

const ContactCardBody = ({ address, addressFunction, addressRole, canAddNotes, contactEmailAddress, contactImage, contactName, contactPhoneNumber, contactPosition, contactRole, contacts, contactType, notes, onSelectedContactChanged, selectedContactId, stringLiterals, }) => {
    const isControlled = React__default.useMemo(() => selectedContactId !== undefined, [selectedContactId]);
    const [internalFilterValue, setInternalFilterValue] = React__default.useState(isControlled ? selectedContactId ?? undefined : Object.keys(contacts)[0]);
    const filterValue = React__default.useMemo(() => (isControlled ? selectedContactId ?? undefined : internalFilterValue), [internalFilterValue, isControlled, selectedContactId]);
    const onFilterChange = React__default.useCallback(newValue => {
        if (!isControlled) {
            setInternalFilterValue(newValue ?? undefined);
        }
        onSelectedContactChanged?.(newValue || null);
    }, [isControlled, onSelectedContactChanged]);
    useDeepCompareEffect(() => {
        const selected = isControlled ? selectedContactId : Object.keys(contacts)[0];
        if (selected === undefined) {
            return;
        }
        if (!isControlled) {
            setInternalFilterValue(selected ?? undefined);
        }
        onFilterChange(selected);
    }, [isControlled, selectedContactId, contacts, filterValue, onFilterChange]);
    const filter = useDeepCompareMemo(() => ({
        options: contacts,
        onChange: onFilterChange,
        isEmptyAllowed: true,
        value: filterValue,
        ariaLabel: resolveTranslationKey(stringLiterals, 'select_contact'),
        variant: 'ghost',
        placeholder: resolveTranslationKey(stringLiterals, 'select_contact'),
    }), [contacts, filterValue, onFilterChange, stringLiterals]);
    const isEmptyContact = React__default.useMemo(() => !contactName &&
        !contactPosition &&
        !contactPhoneNumber &&
        !contactEmailAddress &&
        !contactImage &&
        !contactRole, [
        contactEmailAddress,
        contactImage,
        contactName,
        contactPhoneNumber,
        contactPosition,
        contactRole,
    ]);
    const isEmptyAddress = React__default.useMemo(() => !address && !addressRole && !addressFunction, [
        address,
        addressFunction,
        addressRole,
    ]);
    return (!isEmptyContact && contactType === 'contact') || contactType === 'site' ? (React__default.createElement(Box, { marginX: "var(--spacing200)", marginTop: "var(--spacing200)", paddingBottom: "var(--spacing200)", display: "flex", flexDirection: "column", justifyContent: "center", gap: "var(--spacing200)", boxSizing: "border-box", className: `db-contact-card-body${canAddNotes === false && (notes?.length ?? 0) === 0 ? ' no-border' : ''}`, "data-element": contactType },
        contactType === 'site' && (React__default.createElement(Box, { display: "flex", flexDirection: "column", gap: "var(--spacing150)", className: "db-contact-card-body-site" },
            addressFunction && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)", maxWidth: `calc(100% - 50px)` },
                React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "factory", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'address_name')}:`),
                React__default.createElement(Pill, { pillRole: "status", size: "S" }, addressFunction))),
            address && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)", maxWidth: `calc(100% - 50px)` },
                React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "marker", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'address')}:`),
                React__default.createElement(Box, { display: "flex", flexDirection: "column", gap: "var(--spacing025)" },
                    React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, address),
                    addressRole && (React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "20px" }, addressRole))))),
            !isEmpty(contacts) && (React__default.createElement(Box, { className: "db-contact-card-body-contacts", pt: "var(--spacing075)", "data-component": isEmptyAddress ? 'empty' : undefined },
                React__default.createElement(FilterLine, { stringLiterals: stringLiterals, isFilterFullWidth: true, filter: filter }))))),
        React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)" },
            contactName && (React__default.createElement(React__default.Fragment, null,
                React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "person_tick", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'contact')}:`),
                React__default.createElement("div", { className: "db-contact-card-body-contact-details" },
                    React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" },
                        contactName,
                        React__default.createElement("br", null)),
                    React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" }, contactRole)))),
            contactImage && (React__default.createElement("img", { className: "db-contact-card-body-avatar", height: "50px", width: "50px", src: contactImage, alt: "Contact image" }))),
        contactPosition && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)" },
            React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "business", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
            React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'position')}:`),
            React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" }, contactPosition))),
        contactType === 'contact' && address && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)" },
            React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "marker", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
            React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'address')}:`),
            React__default.createElement(Box, { display: "flex", flexDirection: "column", gap: "var(--spacing025)" },
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, address),
                addressRole && (React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "20px" }, addressRole))))),
        contactPhoneNumber && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)" },
            React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "call", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
            React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'phone_number')}:`),
            React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" },
                React__default.createElement("a", { href: `tel:${contactPhoneNumber}` }, contactPhoneNumber)))),
        contactEmailAddress && (React__default.createElement(Box, { display: "flex", gap: "var(--spacing075)" },
            React__default.createElement(Icon$1, { color: "var(--colorsActionMinor400)", type: "email", mr: "var(--spacing025)", className: "db-contact-card-icon" }),
            React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, `${resolveTranslationKey(stringLiterals, 'email')}:`),
            React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" },
                React__default.createElement("a", { href: `mailto:${contactEmailAddress}` }, contactEmailAddress)))))) : null;
};

const MAX_NOTE_LENGTH = 200;
function reducer(currentState, action) {
    switch (action.type) {
        case 'addEmptyNote':
            return { mode: 'adding', newNoteText: '' };
        case 'addNote':
            return { mode: 'idle' };
        case 'editNote':
            return { mode: currentState.mode === 'editing' ? 'idle' : 'editing', note: action.note };
        case 'editNewNoteText':
            return { mode: 'adding', newNoteText: action.text.slice(0, MAX_NOTE_LENGTH) };
        case 'editLocalNote':
            return {
                mode: 'editing',
                note: { ...action.note, text: action.note.text.slice(0, MAX_NOTE_LENGTH) },
            };
        case 'deleteNote':
            return { mode: 'idle' };
        case 'cancel':
            return { mode: 'idle' };
    }
}
const ContactCardNotes = ({ notes, onNoteAdded, onNoteDeleted, onNoteEdited, stringLiterals, canAddNotes }) => {
    const [state, dispatch] = React__default.useReducer(reducer, { mode: 'idle' });
    const textareaRef = React__default.useRef(null);
    const saveButtonRef = React__default.useRef(null);
    const onCancel = React__default.useCallback(() => {
        dispatch({ type: 'cancel' });
    }, []);
    const onAddEmptyNote = React__default.useCallback(() => {
        flushSync(() => {
            dispatch({ type: 'addEmptyNote' });
        });
        if (saveButtonRef.current) {
            scrollIntoView(saveButtonRef.current, {
                block: 'nearest',
                inline: 'nearest',
                scrollMode: 'if-needed',
            });
        }
        textareaRef.current?.focus();
    }, []);
    const onAddNote = React__default.useCallback(() => {
        if (state.mode !== 'adding') {
            return;
        }
        if (!state.newNoteText) {
            return;
        }
        onNoteAdded?.(state.newNoteText);
        dispatch({ type: 'addNote', text: state.newNoteText });
    }, [onNoteAdded, state]);
    const onEditNote = React__default.useCallback(note => () => {
        flushSync(() => {
            dispatch({ type: 'editNote', note });
        });
        saveButtonRef.current?.scrollIntoView();
        textareaRef.current?.focus();
    }, []);
    const onDeleteNote = React__default.useCallback(note => () => onNoteDeleted?.(note._id), [onNoteDeleted]);
    const onEditNoteSave = React__default.useCallback(() => {
        if (state.mode !== 'editing') {
            return;
        }
        if (!state.note.text) {
            return;
        }
        onNoteEdited?.(state.note._id, state.note.text);
        dispatch({ type: 'editNote', note: state.note });
    }, [onNoteEdited, state]);
    const onEditNewNote = React__default.useCallback(({ target: { value } }) => {
        dispatch({ type: 'editNewNoteText', text: value });
    }, []);
    const onEditLocalNote = React__default.useCallback(({ target: { value: text } }) => {
        if (state.mode !== 'editing') {
            return;
        }
        dispatch({ type: 'editLocalNote', note: { ...state.note, text } });
    }, [state]);
    return (React__default.createElement(Box, { padding: "var(--spacing200)", display: "flex", flexDirection: "column", alignSelf: "flex-start", className: "db-contact-card-notes", width: "100%", boxSizing: "border-box" },
        React__default.createElement(Box, { display: "flex", alignItems: "center", justifyContent: "space-between", boxSizing: "border-box", marginBottom: "var(--spacing100)" },
            ((notes ?? []).length > 0 || state.mode === 'adding') && (React__default.createElement(Box, { display: "flex", alignItems: "center", gap: "var(--spacing100)" },
                React__default.createElement(Icon$1, { color: "var(--colorsActionMinor500)", type: "draft" }),
                React__default.createElement(Typography, { variant: "span", fontWeight: "500", fontSize: "14px", lineHeight: "21px" }, state.mode === 'adding' || state.mode === 'editing'
                    ? `${resolveTranslationKey(stringLiterals, 'note')}:`
                    : `${resolveTranslationKey(stringLiterals, 'notes')}:`))),
            canAddNotes !== false && (React__default.createElement(Button, { buttonType: "tertiary", iconType: "add", size: "small", onClick: onAddEmptyNote, disabled: state.mode === 'adding' || state.mode === 'editing' }, resolveTranslationKey(stringLiterals, 'add_note')))),
        state.mode === 'idle' && (React__default.createElement("ul", null, (notes ?? []).map(note => {
            return (React__default.createElement("li", { key: note._id },
                React__default.createElement(Box, { padding: "var(--spacing100)", gap: "var(--spacing150)", display: "flex", alignItems: "center", justifyContent: "space-between", width: "100%", boxSizing: "border-box" },
                    React__default.createElement(Box, { flex: 1 },
                        React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" }, note.text)),
                    React__default.createElement(Typography, { variant: "span", fontWeight: "400", fontSize: "14px", lineHeight: "21px" }, new Intl.DateTimeFormat(undefined, {
                        dateStyle: 'short',
                    }).format(note.timestamp)),
                    React__default.createElement(ActionPopover, { "aria-label": stringLiterals.more_options, "data-testid": `db-contact-card-notes-popover-${note._id}`, renderButton: renderActionPopoverButton(stringLiterals) },
                        React__default.createElement(ActionPopoverItem, { disabled: canAddNotes === false, icon: "edit", onClick: onEditNote(note) }, resolveTranslationKey(stringLiterals, 'edit')),
                        React__default.createElement(ActionPopoverDivider, null),
                        React__default.createElement(ActionPopoverItem, { disabled: canAddNotes === false, onClick: onDeleteNote(note), icon: "delete" }, resolveTranslationKey(stringLiterals, 'delete'))))));
        }))),
        state.mode === 'adding' && (React__default.createElement(Box, { mt: "var(--spacing100)", display: "flex", flexDirection: "column", gap: "var(--spacing200)", boxSizing: "border-box" },
            React__default.createElement(Textarea, { ref: textareaRef, mt: "var(--spacing100)", value: state.newNoteText, onChange: onEditNewNote }),
            React__default.createElement(Box, { display: "flex", gap: "var(--spacing100)", boxSizing: "border-box" },
                React__default.createElement(Button, { ref: saveButtonRef, buttonType: "secondary", size: "small", disabled: !state.newNoteText, onClick: onAddNote }, resolveTranslationKey(stringLiterals, 'save')),
                React__default.createElement(Button, { buttonType: "tertiary", size: "small", onClick: onCancel }, resolveTranslationKey(stringLiterals, 'cancel'))))),
        state.mode === 'editing' && (React__default.createElement(Box, { mt: "var(--spacing100)", display: "flex", flexDirection: "column", gap: "var(--spacing200)", boxSizing: "border-box" },
            React__default.createElement(Textarea, { ref: textareaRef, mt: "var(--spacing100)", value: state.note.text, onChange: onEditLocalNote }),
            React__default.createElement(Box, { display: "flex", gap: "var(--spacing100)", boxSizing: "border-box" },
                React__default.createElement(Button, { ref: saveButtonRef, buttonType: "secondary", size: "small", disabled: !state.note.text, onClick: onEditNoteSave }, resolveTranslationKey(stringLiterals, 'save')),
                React__default.createElement(Button, { buttonType: "tertiary", size: "small", onClick: onCancel }, resolveTranslationKey(stringLiterals, 'cancel')))))));
};

const ContactCardEmptyState = ({ contactType, onContactAdd, stringLiterals, onAddressAdd }) => {
    const onAddContact = React__default.useCallback(() => {
        if (contactType === 'contact') {
            onContactAdd?.();
        }
        else {
            onAddressAdd?.();
        }
    }, [contactType, onContactAdd, onAddressAdd]);
    return (React__default.createElement(Box, { padding: "25px", display: "flex", flexDirection: "column", alignItems: "center", justifyContent: "center", gap: "var(--spacing200)", boxSizing: "border-box" },
        React__default.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", width: "96", height: "96", viewBox: "0 0 96 96", fill: "none" },
            React__default.createElement("g", { clipPath: "url(#clip0_1007_17508)" },
                React__default.createElement("path", { d: "M91.1999 96H30.5439C29.2709 96 28.05 95.4943 27.1498 94.5941C26.2496 93.6939 25.7439 92.473 25.7439 91.2V4.8C25.7439 3.52696 26.2496 2.30606 27.1498 1.40588C28.05 0.505709 29.2709 0 30.5439 0H91.1999C92.4729 0 93.6938 0.505709 94.594 1.40588C95.4942 2.30606 95.9999 3.52696 95.9999 4.8V91.2C95.9999 92.473 95.4942 93.6939 94.594 94.5941C93.6938 95.4943 92.4729 96 91.1999 96Z", fill: "#8D8D8D" }),
                React__default.createElement("path", { d: "M42.1441 80.0001C35.1357 80.1547 28.3524 77.5223 23.2829 72.6808C18.2134 67.8393 15.2719 61.1841 15.104 54.1761C14.9624 47.1748 17.6032 40.4031 22.4472 35.3461C27.2913 30.2891 33.9432 27.3596 40.9441 27.2001C47.9605 27.0413 54.7528 29.6761 59.8267 34.525C64.9006 39.3738 67.8406 46.0396 68.0001 53.0561C68.133 60.0546 65.4863 66.8204 60.6397 71.8709C55.7931 76.9214 49.1421 79.8447 42.1441 80.0001ZM75.7441 12.8001C75.7441 11.9514 75.4069 11.1375 74.8068 10.5374C74.2067 9.93724 73.3927 9.6001 72.5441 9.6001H34.1441C33.2954 9.6001 32.4814 9.93724 31.8813 10.5374C31.2812 11.1375 30.9441 11.9514 30.9441 12.8001C30.9441 13.6488 31.2812 14.4627 31.8813 15.0628C32.4814 15.663 33.2954 16.0001 34.1441 16.0001H72.5441C73.3927 16.0001 74.2067 15.663 74.8068 15.0628C75.4069 14.4627 75.7441 13.6488 75.7441 12.8001ZM85.3441 28.8001H34.1441C33.2954 28.8001 32.4814 28.463 31.8813 27.8628C31.2812 27.2627 30.9441 26.4488 30.9441 25.6001C30.9441 24.7514 31.2812 23.9375 31.8813 23.3374C32.4814 22.7372 33.2954 22.4001 34.1441 22.4001H85.3441C86.1927 22.4001 87.0067 22.7372 87.6068 23.3374C88.2069 23.9375 88.5441 24.7514 88.5441 25.6001C88.5441 26.4488 88.2069 27.2627 87.6068 27.8628C87.0067 28.463 86.1927 28.8001 85.3441 28.8001ZM85.3441 41.6001H34.1441C33.2954 41.6001 32.4814 41.263 31.8813 40.6628C31.2812 40.0627 30.9441 39.2488 30.9441 38.4001C30.9441 37.5514 31.2812 36.7375 31.8813 36.1374C32.4814 35.5372 33.2954 35.2001 34.1441 35.2001H85.3441C86.1927 35.2001 87.0067 35.5372 87.6068 36.1374C88.2069 36.7375 88.5441 37.5514 88.5441 38.4001C88.5441 39.2488 88.2069 40.0627 87.6068 40.6628C87.0067 41.263 86.1927 41.6001 85.3441 41.6001ZM85.3441 54.4001H34.1441C33.2954 54.4001 32.4814 54.063 31.8813 53.4628C31.2812 52.8627 30.9441 52.0488 30.9441 51.2001C30.9441 50.3514 31.2812 49.5375 31.8813 48.9374C32.4814 48.3372 33.2954 48.0001 34.1441 48.0001H85.3441C86.1927 48.0001 87.0067 48.3372 87.6068 48.9374C88.2069 49.5375 88.5441 50.3514 88.5441 51.2001C88.5441 52.0488 88.2069 52.8627 87.6068 53.4628C87.0067 54.063 86.1927 54.4001 85.3441 54.4001ZM85.3441 80.0001H72.6881C71.8394 80.0001 71.0254 79.663 70.4253 79.0628C69.8252 78.4627 69.4881 77.6488 69.4881 76.8001C69.4881 75.9514 69.8252 75.1375 70.4253 74.5374C71.0254 73.9372 71.8394 73.6001 72.6881 73.6001H85.3441C86.1927 73.6001 87.0067 73.9372 87.6068 74.5374C88.2069 75.1375 88.5441 75.9514 88.5441 76.8001C88.5441 77.6488 88.2069 78.4627 87.6068 79.0628C87.0067 79.663 86.1927 80.0001 85.3441 80.0001Z", fill: "white" }),
                React__default.createElement("g", { clipPath: "url(#clip1_1007_17508)" },
                    React__default.createElement("path", { d: "M52.8787 63.2986C52.2607 63.9214 51.4478 64.2336 50.635 64.2336C49.8283 64.2336 49.0202 63.9261 48.4035 63.3103L42.7742 57.6953L37.1741 63.3381C36.5561 63.9609 35.7432 64.2731 34.9304 64.2731C34.1237 64.2731 33.3156 63.9656 32.699 63.3498C31.4597 62.1136 31.4535 60.1051 32.6866 58.8627L38.287 53.2196L32.6588 47.6057C31.4195 46.3695 31.4133 44.361 32.6464 43.1186C33.878 41.8794 35.8837 41.8708 37.1214 43.107L42.7508 48.722L48.3509 43.0793C49.5824 41.8392 51.5882 41.8322 52.8259 43.0677C54.0652 44.3039 54.0713 46.3124 52.8382 47.5548L47.2378 53.1979L52.8661 58.8118C54.1054 60.048 54.1115 62.0565 52.8784 63.2989L52.8787 63.2986ZM42.1468 24.5332C41.9196 24.5332 41.6925 24.5356 41.4653 24.541C33.634 24.7207 26.3403 27.9476 20.9273 33.6267C15.5158 39.306 12.6355 46.7581 12.8146 54.61C12.967 61.2866 15.3628 67.3798 19.2304 72.2341L0.927095 90.584C-0.309032 91.8226 -0.309032 93.8318 0.927095 95.0703C1.54516 95.6899 2.35484 95.9997 3.16469 95.9997C3.97453 95.9997 4.78405 95.6899 5.40228 95.0703L5.45908 95.0133L5.54912 95.1036L23.4525 77.1573C23.5587 77.0509 23.6435 76.9321 23.7315 76.8152C28.7759 80.9006 35.1853 83.3423 42.127 83.3423C42.3543 83.3423 42.5813 83.3399 42.8085 83.3345C58.9766 82.9635 71.83 69.4742 71.4591 53.2645C71.0942 37.2828 58.0104 24.5332 42.1468 24.5332ZM42.663 76.9909C42.4837 76.9947 42.3076 76.9971 42.1299 76.9971C29.689 76.9971 19.4285 66.9973 19.1411 54.4646C19.0005 48.3075 21.2611 42.4633 25.5045 38.0101C29.7493 33.5563 35.4684 31.0258 41.6107 30.8847C41.79 30.8809 41.9661 30.8786 42.1438 30.8786C54.5847 30.8786 64.8452 40.8776 65.1326 53.4103C65.4215 66.122 55.3434 76.6997 42.663 76.9908V76.9909Z", fill: "#C6C6C6" }))),
            React__default.createElement("defs", null,
                React__default.createElement("clipPath", { id: "clip0_1007_17508" },
                    React__default.createElement("rect", { width: "96", height: "96", fill: "white" })),
                React__default.createElement("clipPath", { id: "clip1_1007_17508" },
                    React__default.createElement("rect", { width: "71.4667", height: "71.4667", fill: "white", transform: "translate(0 24.5332)" })))),
        React__default.createElement(Typography, { variant: "h2" }, resolveTranslationKey(stringLiterals, contactType === 'contact' ? 'contact_card_empty_text' : 'site_card_empty_text')),
        React__default.createElement(Button, { buttonType: "primary", iconType: "add", onClick: onAddContact }, resolveTranslationKey(stringLiterals, contactType === 'contact' ? 'add_contact' : 'add_site'))));
};

const ContactCard = ({ address, addresses, addressFunction, addressRole, canAddNotes, contactEmailAddress, contactImage, contactName, contactPhoneNumber, contactPosition, contactRole, contacts, iconSrc: icon, notes, numberOfAddresses, numberOfContacts, onAddressAdd, onContactAdd = void noop$1, onContactTypeSwitchChanged = void noop$1, onNoteAdded = void noop$1, onNoteDeleted = void noop$1, onNoteEdited = void noop$1, onSelectedAddressChanged: controlledOnSelectedAddressChanged, onSelectedContactChanged: controlledOnSelectedContactChanged, selectedAddressId, selectedContactId, stringLiterals, }) => {
    const [contactType, setContactType] = React__default.useState('contact');
    const isEmptyContact = React__default.useMemo(() => contactType === 'contact' && numberOfContacts === 0, [
        contactType,
        numberOfContacts,
    ]);
    const isEmptySite = React__default.useMemo(() => contactType === 'site' && numberOfAddresses === 0, [
        contactType,
        numberOfAddresses,
    ]);
    const [lastSelectedAddressId, setLastSelectedAddressId] = React__default.useState('');
    const [lastSelectedContactId, setLastSelectedContactId] = React__default.useState('');
    React__default.useEffect(() => {
        if (lastSelectedAddressId === '') {
            return;
        }
        controlledOnSelectedAddressChanged?.(lastSelectedAddressId);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [lastSelectedAddressId]);
    React__default.useEffect(() => {
        if (lastSelectedContactId === '') {
            return;
        }
        controlledOnSelectedContactChanged?.(lastSelectedContactId);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [lastSelectedContactId]);
    const onSelectedAddressChanged = React__default.useCallback(async (selectedAddressId) => {
        await setLastSelectedAddressId(selectedAddressId);
    }, []);
    const onSelectedContactChanged = React__default.useCallback(async (selectedAddressId) => {
        await setLastSelectedContactId(selectedAddressId);
    }, []);
    return (React__default.createElement(Box, { alignItems: "stretch", backgroundColor: "white", borderRadius: "borderRadius200", boxSizing: "border-box", className: "db-contact-card", display: "flex", flexDirection: "column", justifyContent: "center" },
        React__default.createElement(ContactCardHeader, { iconSrc: icon, numberOfAddresses: numberOfAddresses, numberOfContacts: numberOfContacts, stringLiterals: stringLiterals }),
        React__default.createElement(ContactCardSelectionBar, { addresses: addresses, contacts: contacts, contactType: contactType, onContactTypeSwitchChanged: onContactTypeSwitchChanged, onSelectedAddressChanged: onSelectedAddressChanged, onSelectedContactChanged: onSelectedContactChanged, selectedAddressId: selectedAddressId, selectedContactId: selectedContactId, setContactType: setContactType, stringLiterals: stringLiterals }),
        React__default.createElement("div", { className: "db-contact-card-body-wrapper" },
            !isEmptyContact && !isEmptySite && (React__default.createElement(ContactCardBody, { address: address, addressFunction: addressFunction, addressRole: addressRole, canAddNotes: canAddNotes, contactEmailAddress: contactEmailAddress, contactImage: contactImage, contactName: contactName, contactPhoneNumber: contactPhoneNumber, contactPosition: contactPosition, contactRole: contactRole, contacts: contacts, contactType: contactType, notes: notes, onSelectedContactChanged: onSelectedContactChanged, selectedContactId: selectedContactId, stringLiterals: stringLiterals })),
            contactType === 'contact' && !isEmptyContact && (React__default.createElement(ContactCardNotes, { notes: notes, onNoteAdded: onNoteAdded, onNoteDeleted: onNoteDeleted, onNoteEdited: onNoteEdited, stringLiterals: stringLiterals, canAddNotes: canAddNotes }))),
        (isEmptyContact || isEmptySite) && (React__default.createElement(ContactCardEmptyState, { contactType: contactType, onAddressAdd: onAddressAdd, onContactAdd: onContactAdd, stringLiterals: stringLiterals }))));
};

const useScrollButtons = (tilesLength) => {
    const [canScrollLeft, setCanScrollLeft] = useState(false);
    const [canScrollRight, setCanScrollRight] = useState(false);
    const gridRef = useRef(null);
    const scrollTiles = useCallback((direction, event) => {
        event.preventDefault();
        const grid = gridRef.current;
        if (!grid)
            return;
        const containerWidth = grid.clientWidth;
        const scrollLeft = grid.scrollLeft;
        const scrollWidth = grid.scrollWidth;
        let scrollAmount = 200;
        if (direction === 'right') {
            const remainingScroll = scrollWidth - (scrollLeft + containerWidth);
            scrollAmount = Math.min(scrollAmount, remainingScroll);
        }
        else if (direction === 'left') {
            scrollAmount = Math.min(scrollAmount, scrollLeft);
        }
        const targetScroll = direction === 'left' ? scrollLeft - scrollAmount : scrollLeft + scrollAmount;
        grid.scrollLeft = targetScroll;
    }, []);
    const updateScrollButtons = useCallback(() => {
        const grid = gridRef.current;
        if (grid) {
            const containerWidth = grid.clientWidth;
            const scrollLeft = grid.scrollLeft;
            const scrollWidth = grid.scrollWidth;
            setCanScrollLeft(scrollLeft > 0);
            setCanScrollRight(scrollLeft + containerWidth < scrollWidth);
        }
    }, []);
    const addResizeObserver = useCallback((grid) => {
        const resizeObserver = new ResizeObserver(() => {
            updateScrollButtons();
        });
        resizeObserver.observe(grid);
        return resizeObserver;
    }, [updateScrollButtons]);
    const addScrollListener = useCallback((grid) => {
        grid.addEventListener('scroll', updateScrollButtons);
    }, [updateScrollButtons]);
    const removeScrollListener = useCallback((grid) => {
        grid.removeEventListener('scroll', updateScrollButtons);
    }, [updateScrollButtons]);
    useEffect(() => {
        const grid = gridRef.current;
        if (!grid)
            return;
        const resizeObserver = addResizeObserver(grid);
        addScrollListener(grid);
        updateScrollButtons();
        return () => {
            removeScrollListener(grid);
            resizeObserver.unobserve(grid);
        };
    }, [
        tilesLength,
        addResizeObserver,
        addScrollListener,
        removeScrollListener,
        updateScrollButtons,
    ]);
    return {
        canScrollLeft,
        canScrollRight,
        scrollTiles,
        gridRef,
    };
};

const Tile = ({ tile, index }) => {
    const tileBody = (React__default.createElement("div", { className: "tile-content" },
        React__default.createElement(Typography, { fontSize: "14px", mb: 0, as: "span", color: "blackOpacity65", className: "tile-indicator-description" }, tile.title),
        React__default.createElement("div", { className: "tile-icon-value" },
            tile.icon && (React__default.createElement(Icon$1, { className: "tile-icon", type: tile.icon, color: tile.iconColor, fontSize: "small" })),
            React__default.createElement(Typography, { fontWeight: "700", lineHeight: "25px", fontSize: "20px", mb: "0", as: "span", "data-testid": "db-tile-indicator-value", className: "tile-indicator-value", color: tile.valueColor }, tile.value))));
    return tile.onClick ? (React__default.createElement("button", { className: `db-draggable-element-indicator tile-indicator ${tile.hasSeparatorAfter ? 'has-separator' : ''} ${tile.contentAlignment === 'right' ? 'align-right' : 'align-left'}`, onClick: () => tile.onClick?.(index) }, tileBody)) : (React__default.createElement("div", { className: `db-draggable-element-indicator tile-indicator ${tile.hasSeparatorAfter ? 'has-separator' : ''} ${tile.contentAlignment === 'right' ? 'align-right' : 'align-left'}` }, tileBody));
};

const TileIndicatorGroup = (props) => {
    const tileWidth = 233;
    const { canScrollLeft, canScrollRight, scrollTiles, gridRef } = useScrollButtons(props.tiles.length);
    return (React.createElement("div", { className: "widgetTileIndicatorGroup tile-indicator-group-wrapper" },
        React.createElement("div", { className: "tile-indicator-group" },
            canScrollLeft && (React.createElement("button", { className: "scroll-button left", "aria-label": resolveTranslationKey(props.stringLiterals, 'Scroll left'), onClick: (event) => scrollTiles('left', event) },
                React.createElement(Icon$1, { type: "chevron_left", color: "black", fontSize: "small" }))),
            React.createElement("div", { ref: gridRef, className: "tile-indicator-list", style: { gridTemplateColumns: `repeat(auto-fit, minmax(${tileWidth}px, 1fr))` } }, props.tiles.map((tile, index) => (React.createElement(Tile, { key: `${tile.title}-${index}`, tile: tile, index: index })))),
            canScrollRight && (React.createElement("button", { className: "scroll-button right", "aria-label": resolveTranslationKey(props.stringLiterals, 'Scroll right'), onClick: (event) => scrollTiles('right', event) },
                React.createElement(Icon$1, { type: "chevron_right", color: "black", fontSize: "small" }))))));
};

const HtmlContainer = ({ data }) => (React__default.createElement("div", { dangerouslySetInnerHTML: { __html: data?.htmlContent ?? '' } }));
function RenderComponentType({ widget, locale, title, stringLiterals, }) {
    const commonProps = React__default.useMemo(() => ({
        stringLiterals,
        locale,
        isAiGenerated: widget.isAiGenerated,
    }), [locale, stringLiterals, widget.isAiGenerated]);
    switch (widget.type) {
        case 'bar-chart':
            return React__default.createElement(BarChartComponent, { ...widget.data, ...commonProps });
        case 'contact-card':
            return React__default.createElement(ContactCard, { ...widget.data, ...commonProps });
        case 'line-chart':
            return React__default.createElement(LineChartComponent, { ...widget.data, ...commonProps });
        case 'pie-chart':
            return React__default.createElement(PieChartWidget, { ...widget.data, ...commonProps });
        case 'basic':
            return React__default.createElement(HtmlContainer, { data: widget.data, ...commonProps });
        case 'line':
            return React__default.createElement(LineGraph, { data: widget.data, ...commonProps });
        case 'gauge':
            return React__default.createElement(Gauge, { ...widget.data, ...commonProps });
        case 'treemap':
            return React__default.createElement(TableTreeMap, { data: widget.data, ...commonProps });
        case 'preview':
            return React__default.createElement(Preview, null);
        case 'progress-tracker':
            return React__default.createElement(ProgressTracker, { ...widget.data, ...commonProps });
        case 'simple-indicator':
            return React__default.createElement(SimpleIndicator, { ...widget.data, ...commonProps });
        case 'multiple-indicators':
            return React__default.createElement(MultipleIndicators, { ...widget.data, ...commonProps });
        case 'simple-list':
            return React__default.createElement(SimpleList, { ...widget.data, ...commonProps });
        case 'horizontal-bar':
            return React__default.createElement(HorizontalBar, { ...widget.data, ...commonProps });
        case 'status-report':
            return React__default.createElement(StatusReport, { ...widget.data, ...commonProps });
        case 'tile-indicator':
            return React__default.createElement(TileIndicator, { ...widget.data, ...commonProps });
        case 'tile-group-indicator':
            return React__default.createElement(TileIndicatorGroup, { ...widget.data, ...commonProps });
        case 'table':
            return (React__default.createElement(TableComponent, { ...widget.data, ...commonProps, title: title, callToActions: widget.callToActions }));
        case 'visual-process':
            return React__default.createElement(VisualProcess, { ...widget.data, ...commonProps });
        case 'custom-widget':
            return React__default.createElement(CustomWidget, { component: widget.data.component });
        case 'quick-links':
            return React__default.createElement(QuickLinks, { ...widget.data, ...commonProps });
        default:
            return React__default.createElement("div", null,
                "Unknown widget type: ",
                widget.type);
    }
}

const WidgetWrapper = React__default.memo(({ widget, locale, stringLiterals }) => {
    if (widget.showLoader && !widget.data) {
        return React__default.createElement(Preview, null);
    }
    return (React__default.createElement("div", { "data-testid": `${widget.type}-${widget.i}`, className: `db-widget-skeleton-content db-widget-type-${widget.type}`, tabIndex: 0 },
        React__default.createElement(RenderComponentType, { widget: widget, locale: locale, title: widget.title, stringLiterals: stringLiterals })));
});
WidgetWrapper.displayName = 'WidgetWrapper';

const getWidgetTestId = (widget) => {
    if (widget.title) {
        return `db-widget-container-${camelCase(widget.title)}`;
    }
    if (widget.type === 'simple-indicator') {
        const testIdTitle = widget.data.mainIndicator.subtitle ||
            widget.data.mainIndicator.title;
        if (testIdTitle) {
            return `db-widget-container-${camelCase(testIdTitle)}`;
        }
    }
    return undefined;
};
/** ## Widget re-usable skeleton container
 *
 * This is a UI-normalized container to be used as a widget with a header, a title, and some options.
 * This skeleton model is meant to be used inside the drag'n'drop dashboard layout/canvas.
 *
 * Props from **ToggleActionsMenu** are inherited.
 * **props.align** is not used: since the position of the menu is set to right, and then dropdown aligns from right.
 * The content (through **props.children**) is meant to be filled with pure rendered KPI.
 *
 * props.isSeparator enables an alternative simplified version of the widget, to be used as a separator.
 * So, it just has a title.
 * Same stylistic result can be achieved if the regular widget template is used 'under' a container with class name "db-section-separator".
 * In that second case, widget will be built as a regular one, but unnecessary parts will be hidden by CSS.
 *
 * @todo  Build footer area.
 * @todo  Build case when widget content is empty.
 * @todo  Build case when widget is inside of available-widgets list (+ set category).
 */
function WidgetSkeleton({ locale, stringLiterals, widget, isEditMode, onWidgetRemoved, }) {
    const { callToActions, description, dropdown, i, icon, isAiGenerated, isLoading, showLoader, title, type, url, } = widget;
    const [isLoadingTranslations, setLoadingTranslations] = React__default.useState(!!url);
    const [widgetData, setWidgetData] = React__default.useState({});
    // In the BMS product a URL may be provided for each URL to fetch additional strings which can be used within the scope of the widget
    React__default.useEffect(() => {
        if (url) {
            httpService
                .get(url)
                .then(result => {
                const successToJSON = JSON.parse(result.data.value.json);
                setWidgetData({ data: translateObject(successToJSON, stringLiterals) });
            })
                .finally(() => {
                setLoadingTranslations(false);
            });
        }
    }, [stringLiterals, url]);
    // While the translations are getting fetched, we display the placeholder widget kind
    const widgetType = React__default.useMemo(() => (isLoadingTranslations || Boolean(isLoading) ? 'preview' : type), [isLoadingTranslations, isLoading, type]);
    const isSeparator = React__default.useMemo(() => widgetType === 'separator', [widgetType]);
    const testId = React__default.useMemo(() => getWidgetTestId(widget), [widget]);
    const actionDropdownTestId = React__default.useMemo(() => (title ? `db-widget-container-dropdown-actions-${camelCase(title)}` : undefined), [title]);
    const hasCallToActions = React__default.useMemo(() => callToActions &&
        Object.keys(callToActions).filter(a => !callToActions?.[a].isHidden).length > 0, [callToActions]);
    const triggerOnClick = React__default.useCallback((item) => () => {
        if (item.action === 'closeWidget') {
            onWidgetRemoved(i);
        }
        if (item.onClick) {
            item.onClick();
        }
    }, [onWidgetRemoved, i]);
    const classNames = React__default.useMemo(() => {
        const cNames = ['db-widget-skeleton'];
        cNames.push(isSeparator ? 'db-section-separator' : 'sectionWidget');
        if (hasCallToActions) {
            cNames.push('db-widget-skeleton-with-call-to-actions');
        }
        if (isAiGenerated) {
            cNames.push('db-widget-skeleton-ai-generated');
        }
        return cNames.join(' ');
    }, [hasCallToActions, isSeparator, isAiGenerated]);
    const widgetStringLiterals = React__default.useMemo(() => ({
        ...stringLiterals,
    }), [stringLiterals]);
    const renderDropdown = React__default.useCallback(() => (React__default.createElement("div", { className: "db-header-right" }, isSeparator ? (isEditMode && React__default.createElement("a", { href: "#", onClick: () => onWidgetRemoved(i), className: "close" })) : (React__default.createElement(ActionPopover, { "aria-label": stringLiterals.more_options, "data-testid": actionDropdownTestId, renderButton: renderActionPopoverButton(stringLiterals) }, dropdown?.items?.map(a => {
        if ('isDivider' in a) {
            return React__default.createElement(ActionPopoverDivider, { key: a.i });
        }
        return (React__default.createElement(ActionPopoverItem, { key: a.i, icon: a.icon, "data-testid": `db-widget-action-item-${camelCase(resolveTranslationKey(widgetStringLiterals, 'label' in a ? a.label : undefined))}`, onClick: triggerOnClick(a) }, resolveTranslationKey(widgetStringLiterals, 'label' in a ? a.label : undefined)));
    }))))), [
        actionDropdownTestId,
        isEditMode,
        isSeparator,
        onWidgetRemoved,
        stringLiterals,
        triggerOnClick,
        dropdown?.items,
        i,
        widgetStringLiterals,
    ]);
    const renderCallsToAction = React__default.useCallback(() => (React__default.createElement("div", { className: "db-widget-call-to-actions-container" }, Object.keys(callToActions ?? {}).map(a => {
        const action = callToActions?.[a];
        if (action?.isHidden) {
            return null;
        }
        const buttonProps = {
            buttonType: 'tertiary',
            disabled: action?.isDisabled,
        };
        if (typeof action?.onClick === 'string') {
            buttonProps.href = action.onClick;
            buttonProps.target = action.target;
        }
        else {
            buttonProps.onClick = action?.onClick;
        }
        return (React__default.createElement(Button, { key: a, ...buttonProps, "data-testid": `db-widget-call-to-actions-${camelCase(action?.title)}` }, resolveTranslationKey(widgetStringLiterals, action?.title)));
    }))), [callToActions, widgetStringLiterals]);
    return (React__default.createElement("div", { className: classNames, id: i, "data-testid": testId },
        title && type !== 'table' && type !== 'contact-card' && (React__default.createElement("div", { className: "db-widget-skeletonHeader" },
            React__default.createElement("div", { className: "skeletonHeader" },
                React__default.createElement("div", { className: "db-header-left db-draggable-element-indicator" },
                    React__default.createElement(Typography, { variant: isSeparator ? 'h2' : 'h3', fontWeight: isSeparator ? '700' : undefined },
                        React__default.createElement("div", { className: "db-widget-skeletonHeader-title" },
                            icon ? (React__default.createElement("div", { className: "db-widget-skeletonHeader-icon" },
                                React__default.createElement(SvgIcon, { ...icon }))) : (React__default.createElement(React__default.Fragment, null)),
                            resolveTranslationKey(widgetStringLiterals, title),
                            isAiGenerated && React__default.createElement(AiGeneratedWarning, { stringLiterals: widgetStringLiterals }))),
                    description && (React__default.createElement(Typography, { fontSize: "12px", mb: 0, color: "blackOpacity65", className: "db-widget-skeletonHeader-description" }, resolveTranslationKey(widgetStringLiterals, description))))))),
        dropdown && renderDropdown(),
        !isSeparator && (React__default.createElement(React__default.Fragment, null,
            React__default.createElement(WidgetWrapper, { widget: {
                    ...widget,
                    ...widgetData,
                    showLoader: showLoader ?? false,
                    type: widgetType,
                }, locale: locale, stringLiterals: widgetStringLiterals }),
            hasCallToActions && renderCallsToAction()))));
}

const BackgroundPlaceholders$1 = (layout) => {
    const colHorizontalMargin = 24;
    const colVerticalMargin = 24;
    const { width, cols, rowHeight } = layout;
    const dropAreaWidth = width - 48;
    const tileWidth = (dropAreaWidth - colHorizontalMargin * (cols - 1)) / cols;
    const tileHeight = rowHeight;
    const patternWidth = tileWidth + colHorizontalMargin;
    const patternHeight = tileHeight + colVerticalMargin;
    return (React__default.createElement("div", { className: "placeholders" },
        React__default.createElement("svg", { width: "100%", height: "100%" },
            React__default.createElement("pattern", { id: "pattern-placeholder", x: "0", y: "0", width: patternWidth, height: patternHeight, patternUnits: "userSpaceOnUse" },
                React__default.createElement("rect", { className: "placeholderTile", width: tileWidth, x: "0", y: "0", height: tileHeight, fill: tokens.colorsUtilityMajor040 })),
            React__default.createElement("rect", { x: "0", y: "0", width: "100%", height: "100%", fill: "url(#pattern-placeholder)" }))));
};
function DragDrop({ isEditMode, widgets, layout, locale, stringLiterals, onNewItemAdded, onWidgetRemoved, onWidgetLayoutChanged, isContainerPaddingDisabled, }) {
    const [ref, size] = useElementSize();
    return (React__default.createElement("div", { className: "dragDropContainer", ref: ref }, size?.width && (React__default.createElement(React__default.Fragment, null,
        isEditMode ? React__default.createElement(BackgroundPlaceholders$1, { ...layout, ...size }) : null,
        React__default.createElement(ReactGridLayout, { className: "db-draggable-area", onLayoutChange: onWidgetLayoutChanged, layout: widgets, cols: layout.cols, rowHeight: layout.rowHeight, isDraggable: isEditMode, isDroppable: isEditMode, isResizable: isEditMode, droppingItem: { i: PENDING_WIDGET_ID, h: 2, w: 2 }, draggableCancel: "", onDrop: (_widgets, newItem, event) => {
                if (onNewItemAdded === undefined) {
                    return;
                }
                const dataTransfer = event.dataTransfer;
                const eventData = dataTransfer.getData('application/json');
                const parsedData = eventData ? JSON.parse(eventData) : {};
                onNewItemAdded({ ...newItem, i: NEW_WIDGET_ID, ...parsedData });
            }, width: size.width, draggableHandle: ".db-draggable-element-indicator", margin: [24, 24], ...(isContainerPaddingDisabled && { containerPadding: [0, 0] }) }, widgets.map(widget => (React__default.createElement("div", { key: widget.i, className: "db-widget-container" },
            React__default.createElement(WidgetSkeleton, { widget: widget, locale: locale, stringLiterals: stringLiterals, isEditMode: isEditMode, onWidgetRemoved: onWidgetRemoved })))))))));
}

const ActionPopoverWrapper$1 = styled.div `
  float: right;
`;
/** ## Flexible Masonry drag'n'drop canvas
 *
 * This component uses [React Grid Layout](https://github.com/react-grid-layout/react-grid-layout) as an alternative to [Gridstack](https://gridstackjs.com/).
 *
 * This component is meant to be used as full-available width.
 * It uses a responsive/elastic grid of 12 virtual columns.
 *
 * Each child container of the RGL component is initially positionable (x, y), has initial size (w, h), and other get/set properties (see [React Grid Layout Github repo](https://github.com/react-grid-layout/react-grid-layout) for more API details such as drag'n'drop, resize, etc...).
 * Use **WidgetSleketon** inside child containers of the RGL component.
 * Rendered graphs and KPI are the "children" of **WidgetSkeleton**, so we should virtually be able to display anything as a content.
 *
 * A widget can be dragged by its title.
 * Widget that are resizable can be resized by using bottom-right corner (cursor will change on hover).
 *
 * props.layout is the dataset to handle shapes, positions, and other properties.
 *
 * props.layout.i is different from props.layout.id .
 * props.layout.i refers to the "key" attribute used by React Grid Layout to identify each container. This is mandatory, and each key must be unique.
 * props.layout.id refers to the ID attribute of an HTML tag. Here it is used to define a target for the anchors links. This is optional, and each id must be unique.
 *
 * props.editMode is the starter point for the template edit mode for the user. Actually work in progress through CSS, but can be improved with specific state. FIXME PIFAL: to be completed.
 *
 * props.displayAnchors split the screen in 2 columns: main content on the left (12 columns grid), auto-generated anchors sticky on the right.
 *
 * props.highlightAnchorsFullSection defines the behavior of the anchors highlights: false (default) = once at a time | true = multiple anchors can be highlighted if they appear on screen.
 *
 * @todo  Bind dynamic properties, combine layout dataset with building calls of **WidgetSkeleton**.
 * @todo  Check if **WidgetSkeleton** children can be filled through layout dataset, or through the RGL component props.
 * @todo  Build case when widgets are just a simple list of available boxes to drag and drop inside the canvas.
 * @todo  FIXME: overlap of options menu depending of z-index of RGL elements.
 * @todo  Refine styles + check z-index of options menu + resize handler.
 * @todo  Define minimal sizes of widgets.
 * @todo  Check naming rules.
 * @todo  Optimize and split this long component into smaller parts.
 * @todo  Add smooth scroll animations when clicking on anchor link.
 */
function DragDropCanvas(props) {
    const { locale, layout: controlledLayout, widgets: controlledWidgets, isEditMode: controlledIsEditMode, areAnchorsDisplayed, isHeaderHidden, onSavedData, onDataChange, stringLiterals: controlledStringLiterals, onEditAction, newItem: controlledNewItem, isContainerPaddingDisabled, } = props;
    const stringLiterals = merge({}, defaultLiterals, controlledStringLiterals);
    const [displayAnchors, setDisplayAnchors] = React__default.useState(areAnchorsDisplayed || false);
    React__default.useEffect(() => {
        setDisplayAnchors(areAnchorsDisplayed || false);
    }, [areAnchorsDisplayed]);
    const [widgets, setWidgets] = React__default.useState(controlledWidgets || []);
    React__default.useEffect(() => {
        setWidgets(controlledWidgets || []);
    }, [controlledWidgets]);
    const [isEditMode, setEditMode] = React__default.useState(controlledIsEditMode || false);
    React__default.useEffect(() => {
        setEditMode(controlledIsEditMode || false);
    }, [controlledIsEditMode]);
    const availableAnchors = getWidgetAnchors(widgets, displayAnchors);
    const anchorState = useAnchors(availableAnchors, props, stringLiterals);
    const popoverMenuItems = [
        {
            id: '0',
            action: 'editLayout',
            icon: 'edit',
            label: resolveTranslationKey(stringLiterals, 'action_edit'),
            onClick: () => (onEditAction ? onEditAction() : setEditMode(true)),
        },
        {
            id: '1',
            action: 'cloneLayout',
            icon: 'duplicate',
            label: resolveTranslationKey(stringLiterals, 'action_clone'),
        },
        {
            id: '2',
            action: 'shareLayout',
            icon: 'share',
            label: resolveTranslationKey(stringLiterals, 'action_share'),
        },
        {
            id: '3',
            action: 'deleteLayout',
            icon: 'delete',
            label: resolveTranslationKey(stringLiterals, 'action_delete'),
        },
    ];
    async function saveLayout() {
        setEditMode(false);
        const layout = {
            ...controlledLayout,
            isDraggable: false,
            isEditMode: false,
        };
        onSavedData?.({ layout, widgets });
        if (controlledLayout.url) {
            const params = {
                Json: {
                    ...props,
                    layout,
                    widgets,
                },
            };
            await httpService.put(controlledLayout.url, params, {
                headers: {
                    'content-type': 'text/plain',
                },
            });
        }
    }
    const onWidgetRemoved = React__default.useCallback((i) => {
        const updatedWidgets = [...widgets];
        const indexToRemove = widgets.findIndex(w => w.i === i);
        updatedWidgets.splice(indexToRemove, 0);
        setWidgets(updatedWidgets);
    }, [widgets]);
    const onNewItemAdded = React__default.useCallback((newItem) => {
        const newWidgets = [
            ...widgets,
            {
                ...newItem,
                ...controlledNewItem,
                type: newItem.type || 'basic',
                i: newItem.i,
                x: newItem.x,
                y: newItem.y,
                w: newItem.w,
                h: newItem.h,
            },
        ];
        setWidgets(newWidgets);
        if (onDataChange) {
            onDataChange({ layout: controlledLayout, widgets: newWidgets });
        }
    }, [controlledLayout, controlledNewItem, onDataChange, widgets]);
    const onWidgetLayoutChanged = React__default.useCallback((newLayout) => {
        // First we check if the layout was changed with properties we care about
        const hasLayoutChanged = !!newLayout.find(l => {
            const existingWidget = widgets.find(w => w.i === l.i);
            if (!existingWidget) {
                return true;
            }
            return (existingWidget.x !== l.x ||
                existingWidget.y !== l.y ||
                existingWidget.h !== l.h ||
                existingWidget.w !== l.w);
        });
        // If it hasn't we stopped processing
        if (!hasLayoutChanged) {
            return;
        }
        const newWidgets = widgets
            .filter(widget => newLayout.find(l => widget.i === l.i))
            .map(widget => {
            const newLayoutProps = newLayout.find(l => widget.i === l.i);
            return {
                ...widget,
                x: newLayoutProps.x,
                y: newLayoutProps.y,
                w: newLayoutProps.w,
                h: newLayoutProps.h,
            };
        });
        setWidgets(newWidgets);
        if (onDataChange) {
            const filteredWidgets = newWidgets.filter(i => i.i !== PENDING_WIDGET_ID);
            if (!isEqual(widgets, filteredWidgets)) {
                onDataChange({ layout: controlledLayout, widgets: filteredWidgets });
            }
        }
    }, [widgets, onDataChange, controlledLayout]);
    return (React__default.createElement("div", { id: "dashboardWrapper", className: isEditMode ? 'dashboardEditMode' : '' },
        !isHeaderHidden && (React__default.createElement("div", { id: "dashboardHeader" },
            React__default.createElement("div", { className: "actionsBar" }, isEditMode ? (React__default.createElement("button", { onClick: saveLayout, className: "exitEditMode" },
                React__default.createElement("span", null, resolveTranslationKey(stringLiterals, 'general_finish_editing')),
                "\u00A0",
                React__default.createElement("i", { className: "icon bmsIcon-tick" }))) : (React__default.createElement(React__default.Fragment, null,
                React__default.createElement("button", { className: "tabButton" }, resolveTranslationKey(stringLiterals, 'general_welcome_page')),
                React__default.createElement(ActionPopoverWrapper$1, null,
                    React__default.createElement(ActionPopover, { "aria-label": stringLiterals.more_options, renderButton: ({ tabIndex, 'data-element': dataElement, ariaAttributes }) => (React__default.createElement("div", null,
                            React__default.createElement(ActionPopoverMenuButton, { buttonType: "tertiary", ariaAttributes: ariaAttributes, iconPosition: "after", iconType: "dropdown", size: "small", tabIndex: tabIndex, "data-element": dataElement }, resolveTranslationKey(stringLiterals, 'dashboard_settings')))) }, popoverMenuItems.map(a => (React__default.createElement(ActionPopoverItem, { key: a.id, icon: a.icon, onClick: a.onClick }, a.label)))))))))),
        !isEditMode ? (areAnchorsDisplayed ? (React__default.createElement("div", { id: "dashboard", className: (areAnchorsDisplayed ? 'withAnchors ' : '') + (isEditMode ? 'dashboardEditMode ' : '') },
            React__default.createElement("div", { className: "canvasFull" },
                React__default.createElement("div", { className: "canvasLeft" },
                    React__default.createElement(DragDrop, { type: "fixed", layout: controlledLayout, widgets: widgets, isEditMode: isEditMode, locale: locale, stringLiterals: stringLiterals, onWidgetRemoved: onWidgetRemoved, onWidgetLayoutChanged: onWidgetLayoutChanged, isContainerPaddingDisabled: isContainerPaddingDisabled })),
                React__default.createElement("div", { className: "canvasRight" },
                    React__default.createElement("div", null,
                        React__default.createElement(Tabs, { align: "right", position: "left", selectedTabId: anchorState.find(t => t.visible)?.id, onTabChange: selectedId => (window.location.hash = selectedId) }, anchorState.map(a => (React__default.createElement(Tab, { key: a.id, tabId: a.id, title: a.title }))))))))) : (React__default.createElement(DragDrop, { type: "fixed", layout: controlledLayout, widgets: widgets, isEditMode: isEditMode, locale: locale, stringLiterals: stringLiterals, onWidgetRemoved: onWidgetRemoved, onWidgetLayoutChanged: onWidgetLayoutChanged, isContainerPaddingDisabled: isContainerPaddingDisabled }))) : (React__default.createElement(DragDrop, { type: "fixed", isEditMode: isEditMode, layout: controlledLayout, locale: locale, onNewItemAdded: onNewItemAdded, onWidgetLayoutChanged: onWidgetLayoutChanged, onWidgetRemoved: onWidgetRemoved, stringLiterals: stringLiterals, widgets: widgets, isContainerPaddingDisabled: isContainerPaddingDisabled }))));
}

const colHorizontalMargin = 24;
const colVerticalMargin = 24;
const BackgroundPlaceholders = ({ width, cols, rowHeight, }) => {
    const dropAreaWidth = React__default.useMemo(() => width - 48, [width]);
    const tileWidth = React__default.useMemo(() => (dropAreaWidth - colHorizontalMargin * (cols - 1)) / cols, [
        cols,
        dropAreaWidth,
    ]);
    const tileHeight = React__default.useMemo(() => rowHeight, [rowHeight]);
    const patternWidth = React__default.useMemo(() => tileWidth + colHorizontalMargin, [tileWidth]);
    const patternHeight = React__default.useMemo(() => tileHeight + colVerticalMargin, [tileHeight]);
    return (React__default.createElement("div", { className: "placeholders" },
        React__default.createElement("svg", { key: `pattern-placeholder-svg-${String(tileWidth)}`, id: `pattern-placeholder-svg-${String(tileWidth)}`, width: "100%", height: "100%" },
            React__default.createElement("pattern", { key: `pattern-placeholder-${String(tileWidth)}`, id: `pattern-placeholder-${String(tileWidth)}`, x: "0", y: "0", width: patternWidth, height: patternHeight, patternUnits: "userSpaceOnUse" },
                React__default.createElement("rect", { key: `pattern-placeholder-tile-${String(tileWidth)}`, id: `pattern-placeholder-tile-${String(tileWidth)}`, className: "placeholderTile", width: tileWidth, x: "0", y: "0", height: tileHeight, fill: tokens.colorsUtilityMajor040 })),
            React__default.createElement("rect", { key: `pattern-placeholder-bg-${String(tileWidth)}`, id: `pattern-placeholder-bg-${String(tileWidth)}`, x: "0", y: "0", width: "100%", height: "100%", fill: `url(#pattern-placeholder-${String(tileWidth)})` }))));
};

const DEFAULT_BREAKPOINTS = { xxs: 0, xs: 480, sm: 768, md: 996, lg: 1200 };
const DEFAULT_COLS = { xxs: 2, xs: 4, sm: 6, md: 10, lg: 12 };
const ResponsiveGridLayout = WidthProvider(Responsive);
function ResponsiveDragDrop({ isEditMode, widgets, layout, locale, stringLiterals, onNewItemAdded, onWidgetRemoved, onWidgetLayoutChanged, onBreakpointChange: controlledOnBreakpointChange, isContainerPaddingDisabled, }) {
    const [width, setWidth] = React__default.useState(0);
    const onWidthChange = React__default.useCallback(w => {
        setWidth(w);
    }, []);
    const [breakpoint, setBreakpoint] = React__default.useState('md');
    const breakpoints = React__default.useMemo(() => layout.breakpoints ?? DEFAULT_BREAKPOINTS, [
        layout.breakpoints,
    ]);
    const [currentLayout, setCurrentLayout] = React__default.useState([]);
    const [allLayouts, setAllLayouts] = React__default.useState({});
    const previousBreakpoint = usePrevious(breakpoint);
    const previousLayout = usePrevious(currentLayout);
    const previousLayouts = usePrevious(allLayouts);
    const cols = React__default.useMemo(() => layout.cols ?? DEFAULT_COLS, [layout.cols]);
    const columnCount = React__default.useMemo(() => cols[breakpoint], [breakpoint, cols]);
    const onLayoutChange = React__default.useCallback((currentLayout, allLayouts, type) => {
        /**
         * 'react-grid-layout' has been patched to track the type of the layout change
         * This is used to prevent a layout change event from being triggered when the layout is "fixed"
         * by the library. This only ever happens on mount and when some x or y values overflow.
         * When the user makes manual changes by dragging or resizing, the type will be 'drag' or 'resize'
         * and layout changes will be triggered as expected.
         */
        if (type === 'fix' || type === 'update') {
            return;
        }
        setCurrentLayout(currentLayout);
        setAllLayouts(allLayouts);
    }, []);
    React__default.useEffect(() => {
        controlledOnBreakpointChange?.(breakpoint, columnCount);
    }, [controlledOnBreakpointChange, columnCount, breakpoint]);
    React__default.useEffect(() => {
        if (isEqual(previousLayout, currentLayout) &&
            isEqual(previousLayouts, allLayouts) &&
            isEqual(previousBreakpoint, breakpoint)) {
            return;
        }
        onWidgetLayoutChanged(currentLayout, allLayouts, breakpoint);
    }, [
        allLayouts,
        breakpoint,
        currentLayout,
        onWidgetLayoutChanged,
        previousBreakpoint,
        previousLayout,
        previousLayouts,
    ]);
    const onBreakpointChange = React__default.useCallback(newBreakPoint => {
        setBreakpoint(newBreakPoint);
    }, []);
    const layouts = React__default.useMemo(() => ({
        xxs: widgets.map(w => ({ i: w.i, ...w.xxs })),
        xs: widgets.map(w => ({ i: w.i, ...w.xs })),
        sm: widgets.map(w => ({ i: w.i, ...w.sm })),
        md: widgets.map(w => ({ i: w.i, ...w.md })),
        lg: widgets.map(w => ({ i: w.i, ...w.lg })),
    }), [widgets]);
    const onDrop = React__default.useCallback((_widgets, newItem, event) => {
        if (onNewItemAdded === undefined) {
            return;
        }
        const dataTransfer = event.dataTransfer;
        const eventData = dataTransfer.getData('application/json');
        const parsedData = eventData ? JSON.parse(eventData) : {};
        onNewItemAdded({
            newItem: { ...newItem, i: NEW_WIDGET_ID, ...parsedData },
            breakpoint,
        });
    }, [breakpoint, onNewItemAdded]);
    return (React__default.createElement("div", { className: "dragDropContainer" },
        React__default.createElement(React__default.Fragment, null,
            isEditMode && (React__default.createElement(BackgroundPlaceholders, { cols: columnCount, rowHeight: layout.rowHeight, width: width })),
            React__default.createElement(ResponsiveGridLayout, { className: "db-draggable-area", onLayoutChange: onLayoutChange, cols: cols, breakpoints: breakpoints, onBreakpointChange: onBreakpointChange, onWidthChange: onWidthChange, layouts: layouts, compactType: "vertical", rowHeight: layout.rowHeight, isDraggable: isEditMode, isDroppable: isEditMode, isResizable: isEditMode, droppingItem: { i: PENDING_WIDGET_ID, h: 2, w: 2 }, draggableCancel: "", onDrop: onDrop, width: width, draggableHandle: ".db-draggable-element-indicator", margin: [24, 24], ...(isContainerPaddingDisabled && { containerPadding: [0, 0] }) }, widgets.map(widget => (React__default.createElement("div", { key: widget.i, className: "db-widget-container" },
                React__default.createElement(WidgetSkeleton, { widget: { ...widget, ...widget[breakpoint] }, locale: locale, stringLiterals: stringLiterals, isEditMode: isEditMode, onWidgetRemoved: onWidgetRemoved }))))))));
}

const ActionPopoverWrapper = styled.div `
  float: right;
`;
/** ## Flexible Masonry drag'n'drop canvas
 *
 * This component uses [React Grid Layout](https://github.com/react-grid-layout/react-grid-layout) as an alternative to [Gridstack](https://gridstackjs.com/).
 *
 * This component is meant to be used as full-available width.
 * It uses a responsive/elastic grid of 12 virtual columns.
 *
 * Each child container of the RGL component is initially positionable (x, y), has initial size (w, h), and other get/set properties (see [React Grid Layout Github repo](https://github.com/react-grid-layout/react-grid-layout) for more API details such as drag'n'drop, resize, etc...).
 * Use **WidgetSleketon** inside child containers of the RGL component.
 * Rendered graphs and KPI are the "children" of **WidgetSkeleton**, so we should virtually be able to display anything as a content.
 *
 * A widget can be dragged by its title.
 * Widget that are resizable can be resized by using bottom-right corner (cursor will change on hover).
 *
 * props.layout is the dataset to handle shapes, positions, and other properties.
 *
 * props.layout.i is different from props.layout.id .
 * props.layout.i refers to the "key" attribute used by React Grid Layout to identify each container. This is mandatory, and each key must be unique.
 * props.layout.id refers to the ID attribute of an HTML tag. Here it is used to define a target for the anchors links. This is optional, and each id must be unique.
 *
 * props.editMode is the starter point for the template edit mode for the user. Actually work in progress through CSS, but can be improved with specific state. FIXME PIFAL: to be completed.
 *
 * props.displayAnchors split the screen in 2 columns: main content on the left (12 columns grid), auto-generated anchors sticky on the right.
 *
 * props.highlightAnchorsFullSection defines the behavior of the anchors highlights: false (default) = once at a time | true = multiple anchors can be highlighted if they appear on screen.
 *
 * @todo  Bind dynamic properties, combine layout dataset with building calls of **WidgetSkeleton**.
 * @todo  Check if **WidgetSkeleton** children can be filled through layout dataset, or through the RGL component props.
 * @todo  Build case when widgets are just a simple list of available boxes to drag and drop inside the canvas.
 * @todo  FIXME: overlap of options menu depending of z-index of RGL elements.
 * @todo  Refine styles + check z-index of options menu + resize handler.
 * @todo  Define minimal sizes of widgets.
 * @todo  Check naming rules.
 * @todo  Optimize and split this long component into smaller parts.
 * @todo  Add smooth scroll animations when clicking on anchor link.
 */
function ResponsiveDragDropCanvas(props) {
    const { locale, layout: controlledLayout, widgets: controlledWidgets, isEditMode: controlledIsEditMode, areAnchorsDisplayed, isHeaderHidden, onSavedData, onDataChange, stringLiterals: controlledStringLiterals, onEditAction, newItem: controlledNewItem, onBreakpointChange, isContainerPaddingDisabled, } = props;
    const stringLiterals = merge({}, defaultLiterals, controlledStringLiterals);
    const [displayAnchors, setDisplayAnchors] = React__default.useState(areAnchorsDisplayed || false);
    React__default.useEffect(() => {
        setDisplayAnchors(areAnchorsDisplayed || false);
    }, [areAnchorsDisplayed]);
    const [widgets, setWidgets] = React__default.useState(controlledWidgets || []);
    React__default.useEffect(() => {
        setWidgets(controlledWidgets || []);
    }, [controlledWidgets]);
    const [isEditMode, setEditMode] = React__default.useState(controlledIsEditMode || false);
    React__default.useEffect(() => {
        setEditMode(controlledIsEditMode || false);
    }, [controlledIsEditMode]);
    const availableAnchors = React__default.useMemo(() => getWidgetAnchors(widgets, displayAnchors), [
        displayAnchors,
        widgets,
    ]);
    const anchorState = useAnchors(availableAnchors, props, stringLiterals);
    const popoverMenuItems = React__default.useMemo(() => [
        {
            id: '0',
            action: 'editLayout',
            icon: 'edit',
            label: resolveTranslationKey(stringLiterals, 'action_edit'),
            onClick: () => (onEditAction ? onEditAction() : setEditMode(true)),
        },
        {
            id: '1',
            action: 'cloneLayout',
            icon: 'duplicate',
            label: resolveTranslationKey(stringLiterals, 'action_clone'),
        },
        {
            id: '2',
            action: 'shareLayout',
            icon: 'share',
            label: resolveTranslationKey(stringLiterals, 'action_share'),
        },
        {
            id: '3',
            action: 'deleteLayout',
            icon: 'delete',
            label: resolveTranslationKey(stringLiterals, 'action_delete'),
        },
    ], [onEditAction, stringLiterals]);
    const saveLayout = React__default.useCallback(async () => {
        setEditMode(false);
        const layout = {
            ...controlledLayout,
            isDraggable: false,
            isEditMode: false,
        };
        onSavedData?.({ layout, widgets });
        if (controlledLayout.url) {
            const params = {
                Json: {
                    ...props,
                    layout,
                    widgets,
                },
            };
            await httpService.put(controlledLayout.url, params, {
                headers: {
                    'content-type': 'text/plain',
                },
            });
        }
    }, [controlledLayout, onSavedData, props, widgets]);
    const onWidgetRemoved = React__default.useCallback((i) => {
        const updatedWidgets = [...widgets];
        const indexToRemove = widgets.findIndex(w => w.i === i);
        updatedWidgets.splice(indexToRemove, 0);
        setWidgets(updatedWidgets);
    }, [widgets]);
    const onNewItemAdded = React__default.useCallback(({ newItem, breakpoint }) => {
        const newWidgets = [
            ...widgets,
            {
                ...newItem,
                ...controlledNewItem,
                type: newItem.type || 'basic',
                i: newItem.i,
                xxs: { x: newItem.x, y: newItem.y, w: newItem.w, h: newItem.h },
                xs: { x: newItem.x, y: newItem.y, w: newItem.w, h: newItem.h },
                sm: { x: newItem.x, y: newItem.y, w: newItem.w, h: newItem.h },
                md: { x: newItem.x, y: newItem.y, w: newItem.w, h: newItem.h },
                lg: { x: newItem.x, y: newItem.y, w: newItem.w, h: newItem.h },
            },
        ];
        setWidgets(newWidgets);
        if (onDataChange) {
            onDataChange({ layout: controlledLayout, widgets: newWidgets, breakpoint });
        }
    }, [controlledLayout, controlledNewItem, onDataChange, widgets]);
    const onWidgetLayoutChanged = React__default.useCallback((newLayout, _allLayouts, breakpoint) => {
        // First we check if the layout was changed with properties we care about
        const hasLayoutChanged = !!newLayout.find(l => {
            const existingWidget = widgets.find(w => w.i === l.i);
            if (!existingWidget) {
                return true;
            }
            return (existingWidget[breakpoint].x !== l.x ||
                existingWidget[breakpoint].y !== l.y ||
                existingWidget[breakpoint].h !== l.h ||
                existingWidget[breakpoint].w !== l.w);
        });
        // If it hasn't we stopped processing
        if (!hasLayoutChanged) {
            return;
        }
        const newWidgets = widgets
            .filter(widget => newLayout.find(l => widget.i === l.i))
            .map(widget => {
            const newLayoutProps = newLayout.find(l => widget.i === l.i);
            return {
                ...widget,
                [breakpoint]: {
                    x: newLayoutProps.x,
                    y: newLayoutProps.y,
                    w: newLayoutProps.w,
                    h: newLayoutProps.h,
                },
            };
        });
        setWidgets(newWidgets);
        if (onDataChange) {
            const filteredWidgets = newWidgets.filter(i => i.i !== PENDING_WIDGET_ID);
            if (!isEqual(widgets, filteredWidgets)) {
                onDataChange({ layout: controlledLayout, widgets: filteredWidgets, breakpoint });
            }
        }
    }, [widgets, onDataChange, controlledLayout]);
    return (React__default.createElement("div", { id: "dashboardWrapper", className: isEditMode ? 'dashboardEditMode' : '' },
        !isHeaderHidden && (React__default.createElement("div", { id: "dashboardHeader" },
            React__default.createElement("div", { className: "actionsBar" }, isEditMode ? (React__default.createElement("button", { onClick: saveLayout, className: "exitEditMode" },
                React__default.createElement("span", null, resolveTranslationKey(stringLiterals, 'general_finish_editing')),
                "\u00A0",
                React__default.createElement("i", { className: "icon bmsIcon-tick" }))) : (React__default.createElement(React__default.Fragment, null,
                React__default.createElement("button", { className: "tabButton" }, resolveTranslationKey(stringLiterals, 'general_welcome_page')),
                React__default.createElement(ActionPopoverWrapper, null,
                    React__default.createElement(ActionPopover, { renderButton: ({ tabIndex, 'data-element': dataElement, ariaAttributes }) => (React__default.createElement("div", null,
                            React__default.createElement(ActionPopoverMenuButton, { buttonType: "tertiary", ariaAttributes: ariaAttributes, iconPosition: "after", iconType: "dropdown", size: "small", tabIndex: tabIndex, "data-element": dataElement }, resolveTranslationKey(stringLiterals, 'dashboard_settings')))) }, popoverMenuItems.map(a => (React__default.createElement(ActionPopoverItem, { key: a.id, icon: a.icon, onClick: a.onClick }, a.label)))))))))),
        !isEditMode ? (areAnchorsDisplayed ? (React__default.createElement("div", { id: "dashboard", className: (areAnchorsDisplayed ? 'withAnchors ' : '') + (isEditMode ? 'dashboardEditMode ' : '') },
            React__default.createElement("div", { className: "canvasFull" },
                React__default.createElement("div", { className: "canvasLeft" },
                    React__default.createElement(ResponsiveDragDrop, { type: "responsive", layout: controlledLayout, widgets: widgets, isEditMode: isEditMode, locale: locale, stringLiterals: stringLiterals, onWidgetRemoved: onWidgetRemoved, onWidgetLayoutChanged: onWidgetLayoutChanged, onBreakpointChange: onBreakpointChange, isContainerPaddingDisabled: isContainerPaddingDisabled })),
                React__default.createElement("div", { className: "canvasRight" },
                    React__default.createElement("div", null,
                        React__default.createElement(Tabs, { align: "right", position: "left", selectedTabId: anchorState.find(t => t.visible)?.id, onTabChange: selectedId => (window.location.hash = selectedId) }, anchorState.map(a => (React__default.createElement(Tab, { key: a.id, tabId: a.id, title: a.title }))))))))) : (React__default.createElement(ResponsiveDragDrop, { type: "responsive", layout: controlledLayout, widgets: widgets, isEditMode: isEditMode, locale: locale, stringLiterals: stringLiterals, onWidgetRemoved: onWidgetRemoved, onWidgetLayoutChanged: onWidgetLayoutChanged, onBreakpointChange: onBreakpointChange }))) : (React__default.createElement(ResponsiveDragDrop, { type: "responsive", isEditMode: isEditMode, layout: controlledLayout, locale: locale, onNewItemAdded: onNewItemAdded, onWidgetLayoutChanged: onWidgetLayoutChanged, onWidgetRemoved: onWidgetRemoved, stringLiterals: stringLiterals, widgets: widgets, onBreakpointChange: onBreakpointChange }))));
}

function DragDropCanvasWrapper(props) {
    return props.json.type === 'responsive' ? (React__default.createElement(ResponsiveDragDropCanvas, { ...props.json })) : (React__default.createElement(DragDropCanvas, { ...props.json }));
}

export { BarChartComponent as BarChart, ContactCard, DateFilterPeriodType, DragDropCanvas, DragDropCanvasWrapper, Gauge, HorizontalBar, LineChartComponent as LineChart, LineGraph, MultipleIndicators, NEW_WIDGET_ID, PENDING_WIDGET_ID, PieChartWidget as PieChart, ProgressTracker, SimpleIndicator, SimpleList, StatusReport, TableComponent as Table, TableTreeMap, TileIndicator, VisualProcess, WidgetSkeleton, dashboard_http_service as dashboardHttpService, DragDropCanvasWrapper as default };
//# sourceMappingURL=index.js.map
