# Xtrem Testing Methodology

## Overview

Xtrem framework supports comprehensive testing with two distinct approaches:

1. **Traditional Mocha Unit Tests**: For explicit testing of business logic, error scenarios, and edge cases
2. **GraphQL Tests**: For API integration testing with extensive implicit coverage of node lifecycle and operations

## Testing Philosophy

### Explicit vs. Implicit Testing Logic

**Implicit Testing (GraphQL Tests)**:

- Tests that verify proper operation of the full system through successful API operations
- Successful CRUD operations implicitly test property validation, node lifecycle, data persistence
- Business rule enforcement is tested through successful workflows
- GraphQL tests provide extensive coverage with minimal implementation cost

**Explicit Testing (Mocha Unit Tests)**:

- Tests that specifically target error scenarios, edge cases, and failure modes
- Direct testing of business logic methods without full system involvement
- Validation of error handling, boundary conditions, and negative test cases
- Required for scenarios where implicit testing cannot verify specific behaviors

### Test Strategy Hierarchy

1. **GraphQL Tests** (Primary Coverage):
    - CRUD operations for basic node functionality
    - Business workflow validation
    - Integration testing with realistic data flows
    - Comprehensive happy path coverage

2. **Mocha Unit Tests** (Targeted Coverage):
    - Error scenario validation
    - Edge case handling
    - Complex business logic isolation
    - Async operations and messaging patterns

## Test Coverage Categories

### 🟢 High Coverage (80-100%)

- All major operations tested (CRUD + business logic)
- Error scenarios explicitly covered
- Async operations verified
- Edge cases documented and tested

### 🟡 Medium Coverage (60-79%)

- Basic CRUD operations tested
- Some business logic covered
- Limited error scenario testing
- Async patterns partially verified

### 🔴 Low Coverage (40-59%)

- Minimal testing present
- Missing critical business logic tests
- No error scenario coverage
- Async operations untested

### ❌ No Coverage (0-39%)

- No meaningful tests found
- Critical functionality untested
- High risk for production issues

## Testing Best Practices

### Pattern-Based Test Development

1. **Always Search for Existing Patterns**:

    ```bash
    # Search for similar node tests
    grep_search "test.*describe.*[NodeName]" --isRegexp=true

    # Search for GraphQL test patterns
    file_search "**/test/graphql/**"

    ```

2. **Reference Proven Test Approaches**:
    - Use existing test utilities and helpers
    - Follow established async testing patterns
    - Maintain consistency with codebase conventions

3. **Validate Implementation**:
    - Run tests to ensure they work correctly
    - Verify coverage improvements
    - Check for test stability and reliability

### Node Identification in Tests

Always use proper node identification methods in tests:

```typescript
// ✅ Correct - Using node identification
const item = await context.xtremMasterData.item.getByCode('TEST001');

// ❌ Incorrect - Direct database queries in tests
const item = await context.db.query('SELECT * FROM items WHERE code = ?', ['TEST001']);
```

### Async Operations & SQS Messaging Testing

**Critical Requirements**:

- MUST verify both successful processing and error scenarios
- MUST test notification generation and delivery

## Test Implementation Guidelines

### Coverage Analysis Before Test Suggestions

**MANDATORY Process**:

1. **Analyze Existing Tests**:
    - Search for existing GraphQL tests
    - Identify existing Mocha unit tests
    - Document current coverage areas

2. **Calculate Coverage Components**:
    - CRUD operations coverage
    - Business logic coverage
    - Error scenario coverage
    - Async operations coverage

3. **Identify Critical Gaps**:
    - Missing error handling tests
    - Untested business logic flows
    - Missing async operation verification
    - Unverified integration points

4. **Prioritize Test Development**:
    - Critical: Error scenarios and edge cases
    - Important: Complex business logic
    - Supplementary: Additional happy path coverage

### Test File Organization

```text
lib/nodes/[node-name]/
├── test/
│   ├── mocha                         # Mocha unit tests
│   └── graphql/
│       ├── create-[node-name]/       # Parametric GraphQL test
│       ├── update-[node-name]/       # Parametric GraphQL test
│       └── [scenario-name]/          # Parametric GraphQL test
│           ├── parameters.json
│           ├── request.graphql.hbs
│           └── response.json.hbs
```

### Common Testing Utilities

**Test Setup Patterns**:

```typescript
import { expect } from 'chai';
import { Context } from '@sage/xtrem-core';

describe('NodeName Tests', () => {
    let context: Context;

    beforeEach(async () => {
        context = await createTestContext();
    });

    afterEach(async () => {
        await cleanupTestContext(context);
    });
});
```

## Test Execution Commands

### GraphQL Tests

```bash
# Run all GraphQL tests
npm run test:graphql

# Run specific GraphQL test
npm run test:graphql -- --grep "test-name"

# Run GraphQL tests for specific node
npm run test:graphql -- --grep "node-name"
```

### Mocha Unit Tests

```bash
# Run all unit tests
npm run test:unit

# Run specific test file
npm run test:unit -- lib/nodes/customer/test/customer.test.ts

# Run tests with coverage
npm run test:coverage
```

### Combined Test Execution

```bash
# Run all tests
npm test

# Run tests with verbose output
npm test -- --verbose

# Run tests in watch mode
npm run test:watch
```

## Usage Keywords

Use these phrases to trigger testing assistance:

- "analyze test coverage"
- "identify missing tests"
- "suggest unit tests"
- "test gap analysis"
- "generate test cases"
- "xtrem testing patterns"
- "coverage analysis for [filename]"
- "notification testing patterns"
- "graphql test analysis"
- "suggest graphql tests"
- "create test for [node]"
