## Development Support

This section covers tools, patterns, and resources to support Xtrem application development and maintenance.

### Testing Patterns

### Feature File Structure

```gherkin
@xtrem @show-case @functional
Feature: Product Management in ShowCase

    Background:
        Given the user is authenticated as "show-case-user"
        And the user navigates to the "ShowCase - Product" page

    Scenario: Create a new product with provider
        When the user clicks the "New" button
        And the user fills the "Product" field with "Test Electronics Device"
        And the user fills the "Description" field with "High-quality electronic device"
        And the user selects a provider from the "Provider" reference field
        And the user fills the "List price" field with "299.99"
        And the user selects "electronics" from the "Category" dropdown
        And the user clicks the "Save" button
        Then the product should be created successfully
        And the user should see a success notification
        And the product should appear in the navigation panel

    Scenario: Filter products by category
        Given there are products with different categories
        When the user applies a filter for "Category" equals "electronics"
        Then only products with "electronics" category should be displayed
        And the product count should be updated accordingly

    Scenario: Update product pricing
        Given a product exists with ID "PROD-001"
        When the user opens the product details
        And the user changes the "List price" to "349.99"
        And the user clicks the "Save" button
        Then the product price should be updated
        And the changes should be reflected in the product list
```

### Step Definition Patterns

```typescript
import { Given, When, Then } from '@cucumber/cucumber';
import { expect } from 'chai';

Given(/^there are products with different categories$/, async function () {
    // Setup test data with various product categories
    await this.testDataManager.createProducts([
        { product: 'Laptop', category: 'electronics', listPrice: 999.99 },
        { product: 'Desk', category: 'furniture', listPrice: 299.99 },
        { product: 'Smartphone', category: 'electronics', listPrice: 699.99 },
    ]);
});

When(/^the user fills the "([^"]*)" field with "([^"]*)"$/, async function (fieldName: string, value: string) {
    const field = await this.page.getField(fieldName);
    await field.setValue(value);
});

When(/^the user selects a provider from the "([^"]*)" reference field$/, async function (fieldName: string) {
    const referenceField = await this.page.getReferenceField(fieldName);
    await referenceField.openLookup();
    await referenceField.selectFirstOption();
});

Then(/^the product should be created successfully$/, async function () {
    const notification = await this.page.getLastNotification();
    expect(notification.type).to.equal('success');
    expect(notification.message).to.include('created');
});
```

### Unit Test Patterns

```typescript
import { Test } from '@sage/xtrem-core';
import { assert } from 'chai';

describe('ShowCaseProduct Node Tests', () => {
    before(async () => {
        await setup({
            application: await createApplicationWithApi({
                nodes: { ShowCaseProduct, ShowCaseProvider },
            }),
        });
        await initTables([
            { nodeConstructor: ShowCaseProduct, data: [] },
            { nodeConstructor: ShowCaseProvider, data: providerTestData },
        ]);
    });

    it('can create a product with valid data', () =>
        Test.withContext(async context => {
            const product = await context.create(ShowCaseProduct, {
                product: 'Test Product',
                description: 'Test Description',
                listPrice: decimal.parse('99.99'),
                category: 'electronics',
            });

            assert.equal(await product.product, 'Test Product');
            assert.equal(await product.listPrice, decimal.parse('99.99'));
        }));

    it('validates required fields', () =>
        Test.withContext(async context => {
            const product = await context.create(ShowCaseProduct, {
                description: 'Missing product name',
            });

            const isValid = await product.$.control();
            assert.isFalse(isValid);
            assert.isTrue(context.diagnoses.some(d => d.path.includes('product')));
        }));

    it('can filter products by category', () =>
        Test.withReadonlyContext(async context => {
            const electronicsProducts = await context.select(
                ShowCaseProduct,
                { _id: true, product: true, category: true },
                { filter: { category: 'electronics' } },
            );

            assert.isTrue(electronicsProducts.every(p => p.category === 'electronics'));
        }));

    after(() => restoreTables());
});
```

