# Workspace Overview

This workspace is a monorepo that contains a GraphQL application framework and various web applications. It is built using React and Node.js, and uses PostgreSQL for data storage or MSSQL or Oracle for the X3 applications.

## Folder Structure

- `/platform/back-end`: Contains the source code for the Node.js backend.
- `/platform/cli`: Contains the source code for the xtrem CLI tools.
- `/platform/front-end`: Contains the source code for the frontend.
- `/services`: Contains the source code for the SDMO (Sage Distribution and Manufacturing) Application.
- `/services/application`: Contains the source code for the SDMO (Sage Distribution and Manufacturing) functional modules.
- `/services/shared`: Contains the source code for the SDMO (Sage Distribution and Manufacturing) shared modules.
- `/shopfloor`: Contains the source code for the Shopfloor application.
- `tools/glossary`: Contains the source code for the Glossary application.
- `/x3-services`: Contains the source code for the X3 services, including the CLI and sync services.
- `/x3-services/cli`: Contains the CLI tools for interacting with X3 services.
- `/x3-services/platform`: Contains the platform-specific code for X3 services.
- `/x3-services/platform/xtrem-x3-sync`: Contains the sync service for X3 services.
- `/x3-services/application`: Contains the source code for the X3 Services functional modules.
- `/x3-services/shared`: Contains the source code for the X3 Services shared modules.
- `/wh-services`: Contains the source code for the X3 Warehousing (GEODE)services, including the CLI and sync services.
- `/wh-services/application`: Contains the source code for the X3 Warehousing functional modules.
- `/wh-services/shared`: Contains the source code for the X3 Warehousing shared modules.
- `/documentation`: Contains documentation for the project, including API specifications and user guides.
- `pipelines`: Contains CI/CD pipeline configurations.

## Libraries and Frameworks

- React for the frontend.
- Node.js and Express for the backend.
- PostgreSQL for data storage and MSSQL or Oracle for X3 applications.

## Coding Standards

- Use semicolons at the end of each statement.
- Use single quotes for strings except to avoid escape.
- Use arrow functions for callbacks.
- Use `const` for variables that are not reassigned, and `let` for those that are.
- Use `async/await` for asynchronous code.
- Use `camelCase` for variable, constant and function names.
- Use `PascalCase` for class names.
- Use `kebab-case` for file and directory names except for well known files like `Dockerfile`, etc.

### Typescript Configuration

Each package in the monorepo has its own TypeScript configuration. Always follow the TypeScript rules defined for the specific package you're working in:

#### tsconfig.json

```json
{
    "extends": "../../tsconfig-base.json",
    "compilerOptions": {
        "outDir": "build",
        "rootDir": ".",
        "baseUrl": "."
    },
    "include": ["index.ts", "application.ts", "lib/**/*.ts", "test/**/*.ts", "test/**/*.json", "api/api.d.ts"],
    "exclude": [
        ".eslintrc*.cjs",
        "lib/pages/**/*",
        "lib/widgets/**/*",
        "lib/page-extensions/**/*",
        "lib/page-fragments/**/*",
        "lib/stickers/**/*",
        "lib/i18n/**/*",
        "**/*.feature",
        "**/*.png",
        "lib/client-functions/**/*"
    ],
    "references": []
}
```

#### tsconfig-artifacts.json

```json
{
    "extends": "../../tsconfig-base.json",
    "compilerOptions": {
        "outDir": "build",
        "rootDir": ".",
        "baseUrl": "."
    },
    "include": ["index.ts", "application.ts", "lib/**/*.ts", "test/**/*.ts", "test/**/*.json", "api/api.d.ts"],
    "exclude": [
        ".eslintrc*.cjs",
        "lib/pages/**/*",
        "lib/widgets/**/*",
        "lib/page-extensions/**/*",
        "lib/page-fragments/**/*",
        "lib/stickers/**/*",
        "lib/i18n/**/*",
        "**/*.feature",
        "**/*.png",
        "lib/client-functions/**/*"
    ],
    "references": []
}
```

### ESLint Configuration

Each package in the monorepo has its own ESLint configuration. Always follow the ESLint rules defined for the specific package you're working in:

- **Check package-specific rules**: Before generating code, review the `eslint.config.mjs` or `.eslintrc` file in the target package
- **Run ESLint validation**: Use `npm run lint` or `pnpm lint` to validate code against package rules
- **Auto-fix when possible**: Use `npm run lint:fix` or `pnpm lint:fix` to automatically resolve fixable issues
- **Common configurations**:
    - Services packages: Use `@sage/eslint-config-xtrem` with Xtrem-specific rules
    - Platform packages: May have stricter or different rule sets
    - X3 services: May have X3-specific linting requirements

### ESLint Rule Compliance

When generating code:

1. **Follow the target package's ESLint configuration**
2. **Ensure code passes linting before suggesting**
3. **Mention ESLint fixes if code requires manual adjustments**
4. **Use consistent formatting that aligns with the package's ESLint rules**

Example ESLint configurations found in packages:

```javascript
// Standard Xtrem package configuration
// eslint.config.mjs (ESLint v9+)
import { defineConfig, globalIgnores } from 'eslint/config';
import globals from 'globals';
import baseConfig from '../../eslint-truth.config.mjs';
import xtremPlugin from '@sage/eslint-plugin-xtrem';

const artifactFiles = [
    'lib/pages/**/*.{ts,tsx}',
    'lib/widgets/**/*.{ts,tsx}',
    'lib/page-extensions/**/*.{ts,tsx}',
    'lib/page-fragments/**/*.{ts,tsx}',
    'lib/stickers/**/*.{ts,tsx}',
    'api/api.d.ts',
    'lib/menu-items/**/*.{ts,tsx}',
    'lib/client-functions/**/*.{ts,tsx}',
    'lib/shared-functions/**/*.{ts,tsx}',
];

export default defineConfig([
    globalIgnores(['build/**', 'tmp/**', 'node_modules/**', '**/coverage', '**/i18n/*.json', 'data/**']),
    {
        files: artifactFiles,
        languageOptions: {
            parserOptions: {
                tsconfigRootDir: import.meta.dirname,
                project: 'tsconfig-artifacts.json',
            },
        },
    },
    {
        files: ['**/*.{ts,tsx,mts,cts}'],
        ignores: artifactFiles,
        languageOptions: {
            parser: '@typescript-eslint/parser',
            parserOptions: {
                tsconfigRootDir: import.meta.dirname,
                project: 'tsconfig.json',
            },
        },
    },
    ...baseConfig,
    {
        plugins: {
            '@sage/xtrem': xtremPlugin,
        },
    },
    ...xtremPlugin.configs.recommended,
    // config envs
    {
        languageOptions: {
            globals: { ...globals.browser, ...globals.node },
        },
    },
]);
```

```javascript
// eslint-filename.config.mjs (ESLint v9+)
// This file is the filename for ESLint rules
import baseConfig from '../../../eslint-filename.config.mjs';

export default [...baseConfig];
```

## Xtrem Framework Patterns

**IMPORTANT: Always consult the comprehensive Xtrem Framework Knowledge Base at `.github/xtrem-framework-knowledge.md` for detailed patterns, examples, and best practices before implementing any Xtrem-related code.**

### ESLint and Code Quality

- **Use Xtrem ESLint plugin**: `@sage/eslint-plugin-xtrem` enforces framework-specific patterns
- **Follow import conventions**: ESLint rules enforce proper import patterns (`prefer-xtrem-imports`)
- **Avoid hardcoded strings**: Use internationalization patterns (`no-hardcoded-strings`)
- **Validate before commit**: All code must pass ESLint validation

### Import Conventions

```typescript
// Preferred imports
import * as ui from '@sage/xtrem-ui';
import { decorators, Node, Context } from '@sage/xtrem-core';
```

### Framework Development Rules

1. **Always reference the knowledge base**: Before creating nodes, pages, or any Xtrem components, read the relevant sections in `xtrem-framework-knowledge.md`
2. **Follow established patterns**: Use the examples in the knowledge base as templates
3. **Use proper decorators**: All properties must use the correct decorator patterns with Promise<T> types
4. **Package structure**: Follow the standard package layout defined in the knowledge base
5. **Node patterns**: Use the proper node decorator structure with all required options

## Xtrem Development Assistance

GitHub Copilot supports specialized Xtrem agents through prompt files in `.github/prompts/`:

### Available Xtrem Agents

- **xtrem-dev-assistant** - Senior Xtrem Framework Architect for development patterns
- **xtrem-code-review** - Code review focusing on Xtrem patterns and compliance
- **generate-feature-file** - Generate Gherkin feature files for testing

### Using Prompt-Based Agents

#### Method 1: Reference Prompt Files Directly

```
Use the xtrem-dev-assistant prompt to help me create a Customer node
Follow the xtrem-code-review prompt to review this code
Apply the generate-feature-file prompt to create tests for this component
```

#### Method 2: Use @workspace with Xtrem Context

```
@workspace How do I create a new Xtrem node following the SalesRep pattern?
@workspace Review this code for Xtrem framework compliance
@workspace Help me generate tests for this Xtrem component
```

#### Method 3: Use Descriptive Prompts

```
"As an Xtrem framework developer, help me create a Customer node"
"Generate Xtrem tests with Gherkin feature files for this component"
"Review this code for Xtrem decorator patterns and best practices"
"Help migrate this code to newer Xtrem framework patterns"
```

#### Method 4: Use Slash Commands in Messages

```
/xtrem-node Create a Customer node with name, email, and phone properties
/xtrem-page Create a customer management page with CRUD operations
/xtrem-test Generate tests for the SalesOrder node
/xtrem-migrate Update this component to use latest decorator patterns
/xtrem-review Check this code for framework compliance
/xtrem-fix Fix common Xtrem framework issues and patterns
```

### Xtrem-Specific Keywords

When asking for help, use these keywords to trigger the right context:

- **"Xtrem framework"** - For general framework questions
- **"Xtrem node"** - For node creation and management
- **"Xtrem decorators"** - For property and node decorators
- **"Xtrem UI"** - For page and component creation
- **"Xtrem tests"** - For test generation and Gherkin features
- **"Xtrem patterns"** - For best practices and code review

## Async Processing Utilities

The Xtrem framework provides async processing utilities through `@sage/xtrem-async-helper`. For server-side code, these utilities are re-exported from `@sage/xtrem-core`.

### Key Utilities Available

- **asyncArray**: Async array operations with chainable methods and event loop yielding
- **funnel**: Concurrency control to limit parallel executions
- **getAsync**: Safe access to nested async properties
- **AsyncReader**: Advanced async iteration for streaming data

### When to Use Async Utilities

1. **asyncArray**: For processing arrays with async operations while maintaining event loop health
2. **funnel**: To control concurrency and prevent resource exhaustion
3. **getAsync**: For safely accessing nested object properties that might be Promises
4. **AsyncReader**: For streaming large datasets or async iteration

### Import Conventions for Async Utilities

```typescript
// Client-side
import { asyncArray, funnel, getAsync, AsyncReader } from '@sage/xtrem-async-helper';

// Server-side (preferred - re-exported from core)
import { asyncArray, funnel, getAsync, AsyncReader } from '@sage/xtrem-core';
```

### Code Generation Guidelines for Async Operations

When generating code that involves async operations:

1. **Use asyncArray** instead of manual for loops for array processing
2. **Control concurrency** with funnel to prevent overwhelming external services
3. **Use getAsync** for nested property access in complex objects
4. **Process streams** with AsyncReader for large datasets
5. **Monitor funnel metrics** to optimize performance

### Common Async Patterns in Xtrem

- **Node properties**: Use asyncArray for collection processing, funnel for external API calls
- **Page data loading**: Use asyncArray.forEachParallel() for controlled concurrent loading
- **Activity processing**: Use funnel for sequential operations, asyncArray for batch processing
- **Large dataset processing**: Use AsyncReader for streaming, funnel to control concurrency
- **Schema operations**: Use asyncArray.forEach() for sequential table creation
