"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AsyncArrayReader = exports.AsyncArray = void 0;
exports.asyncArray = asyncArray;
exports.asyncArrayReader = asyncArrayReader;
const xtrem_decimal_1 = require("@sage/xtrem-decimal");
const lodash_1 = require("lodash");
const _1 = require(".");
const async_reader_1 = require("./async-reader");
const promise_markers_1 = require("./promise-markers");
/**
 * Array wrapper for chainable async versions of the ES5 Array methods.
 *
 * Only the very basic features of the ES5 API are supported.
 * The goal is to favor speed and not get distracted by rarely used features.
 */
class AsyncArray {
    constructor(getSource) {
        this.getSource = getSource;
    }
    async forEach(body) {
        const array = await this.getSource();
        const len = array.length;
        for (let i = 0; i < len; i += 1) {
            if (_1.asyncHealtyEventLoop.shouldYield) {
                await _1.asyncHealtyEventLoop.yield();
            }
            await body(array[i], i);
        }
    }
    /**
     * Asynchronously creates a shallow copy of a portion of the array, specified by the `start` and `end` parameters,
     * and returns a new `AsyncArray` instance containing the sliced portion. This method does not modify the original array.
     */
    async slice(start, end) {
        const slicedArray = (await this.getSource()).slice(start, end);
        return new AsyncArray(() => slicedArray);
    }
    /**
     * Similar to forEach but parallelizes the execution of `body`.
     * The `maxParallel` parameter controls the number of parallel executions.
     */
    async forEachParallel(maxParallel, body) {
        if (maxParallel < 1)
            throw new Error(`invalid maxParallel value: ${maxParallel}`);
        const array = await this.getSource();
        // See https://maximorlov.com/parallel-tasks-with-pure-javascript/ for a detailed explanation of the trick.
        const iterator = array.entries();
        // Create maxParallel workers which consume the iterator in parallel
        const workers = Array.from({ length: maxParallel })
            .fill(iterator)
            .map(async (iter) => {
            // This is the trick: each worker reads the iterator but a given entry of the iterator is only read by one worker.
            // eslint-disable-next-line no-restricted-syntax
            for (const [i, value] of iter) {
                if (_1.asyncHealtyEventLoop.shouldYield) {
                    await _1.asyncHealtyEventLoop.yield();
                }
                await body(value, i);
            }
        });
        await Promise.all(workers);
    }
    map(body) {
        return new AsyncArrayReader(() => (async () => {
            const array = await this.getSource();
            const results = Array.from({ length: array.length });
            const len = array.length;
            for (let i = 0; i < len; i += 1) {
                if (_1.asyncHealtyEventLoop.shouldYield) {
                    await _1.asyncHealtyEventLoop.yield();
                }
                results[i] = await body(array[i], i);
            }
            return results;
        })());
    }
    filter(body) {
        return new AsyncArrayReader(() => (async () => {
            const array = await this.getSource();
            const results = [];
            const len = array.length;
            for (let i = 0; i < len; i += 1) {
                if (_1.asyncHealtyEventLoop.shouldYield) {
                    await _1.asyncHealtyEventLoop.yield();
                }
                const item = array[i];
                if (await body(item))
                    results.push(item);
            }
            return results;
        })());
    }
    async find(body) {
        const array = await this.getSource();
        const len = array.length;
        for (let i = 0; i < len; i += 1) {
            if (_1.asyncHealtyEventLoop.shouldYield) {
                await _1.asyncHealtyEventLoop.yield();
            }
            const item = array[i];
            if (await body(item, i))
                return item;
        }
        return undefined;
    }
    async findIndex(body) {
        const array = await this.getSource();
        const len = array.length;
        for (let i = 0; i < len; i += 1) {
            if (_1.asyncHealtyEventLoop.shouldYield) {
                await _1.asyncHealtyEventLoop.yield();
            }
            if (await body(array[i], i))
                return i;
        }
        return -1;
    }
    async some(body) {
        return (await this.find(body)) !== undefined;
    }
    async every(body) {
        return ((await this.find(async (item) => {
            if (_1.asyncHealtyEventLoop.shouldYield) {
                await _1.asyncHealtyEventLoop.yield();
            }
            return !(await body(item));
        })) === undefined);
    }
    async reduce(body, initial) {
        const array = await this.getSource();
        const len = array.length;
        if (len === 0 && arguments.length === 1)
            throw new TypeError('Reduce of empty array with no initial value');
        let result = arguments.length === 1 ? array[0] : initial;
        const start = arguments.length === 1 ? 1 : 0;
        for (let i = start; i < len; i += 1) {
            if (_1.asyncHealtyEventLoop.shouldYield) {
                await _1.asyncHealtyEventLoop.yield();
            }
            result = await body(result, array[i], i);
        }
        return result;
    }
    async qsort(array, compare, beg, end) {
        if (beg >= end)
            return promise_markers_1.voidPromise;
        let tmp;
        if (end === beg + 1) {
            if ((await compare(array[beg], array[end])) > 0) {
                tmp = array[beg];
                array[beg] = array[end];
                array[end] = tmp;
            }
            return promise_markers_1.voidPromise;
        }
        const mid = Math.floor((beg + end) / 2);
        const o = array[mid];
        let nbeg = beg;
        let nend = end;
        while (nbeg <= nend) {
            while (nbeg < end && array[nbeg] !== o && (await compare(array[nbeg], o)) < 0)
                nbeg += 1;
            while (beg < nend && array[nend] !== o && (await compare(o, array[nend])) < 0)
                nend -= 1;
            if (nbeg <= nend) {
                tmp = array[nbeg];
                array[nbeg] = array[nend];
                array[nend] = tmp;
                nbeg += 1;
                nend -= 1;
            }
        }
        if (nbeg < end)
            await this.qsort(array, compare, nbeg, end);
        if (beg < nend)
            await this.qsort(array, compare, beg, nend);
        return promise_markers_1.voidPromise;
    }
    sort(compare, begIndex, endIndex) {
        return new AsyncArrayReader(async () => {
            const array = (await this.getSource()).slice();
            await this.qsort(array, compare, begIndex || 0, endIndex == null ? array.length - 1 : endIndex);
            return array;
        });
    }
    async join(sep) {
        const array = await this.getSource();
        return array.join(sep);
    }
    uniq() {
        // lodash uniq is much faster than [...new Set(...)]
        // https://www.measurethat.net/Benchmarks/Show/9647/0/lodashs-uniq-vs-array-from-set
        return new AsyncArrayReader(async () => (0, lodash_1.uniq)(await this.getSource()));
    }
    async elementAt(i) {
        const array = await this.getSource();
        const j = i >= 0 ? i : array.length + i;
        if (j < 0 || j >= array.length)
            throw new Error(`index out of bounds: ${i}`);
        return array[j];
    }
    async at(i) {
        const array = await this.getSource();
        const j = i >= 0 ? i : array.length + i;
        return array[j];
    }
    get length() {
        return (async () => {
            return (await this.getSource()).length;
        })();
    }
    // eslint-disable-next-line require-await
    async toArray() {
        return this.getSource();
    }
    async min(body) {
        return (await this.map(body).toArray()).reduce((prev, item) => {
            if (prev == null)
                return item;
            if (item == null)
                return prev;
            return prev < item ? prev : item;
        });
    }
    async max(body) {
        return (await this.map(body).toArray()).reduce((prev, item) => {
            if (prev == null)
                return item;
            if (item == null)
                return prev;
            return prev > item ? prev : item;
        });
    }
    async sum(body) {
        return (await this.map(body).toArray()).reduce((prev, item) => (0, xtrem_decimal_1.add)(prev, item), 0);
    }
}
exports.AsyncArray = AsyncArray;
/**
 * Wraps an array with an AsyncArray.
 */
function asyncArray(array) {
    return new AsyncArray(() => array);
}
/**
 * AsyncReader wrapper around arrays.
 */
class AsyncArrayReader extends AsyncArray {
    #index = 0;
    get readCount() {
        return this.#index;
    }
    async read() {
        const array = await this.getSource();
        if (_1.asyncHealtyEventLoop.shouldYield) {
            await _1.asyncHealtyEventLoop.yield();
        }
        const item = array[this.#index];
        if (item !== undefined)
            this.#index += 1;
        return item;
    }
    async stop() {
        this.#index = (await this.getSource()).length;
    }
    async readAll() {
        const array = await this.getSource();
        this.#index = array.length;
        return array;
    }
    toAsyncGenerator() {
        return (0, async_reader_1.createAsyncGenerator)(this);
    }
}
exports.AsyncArrayReader = AsyncArrayReader;
/**
 * Wraps an array with an AsyncArray
 */
function asyncArrayReader(array) {
    return new AsyncArrayReader(() => array);
}
//# sourceMappingURL=async-array.js.map