import { AsyncReader } from './async-reader';
import { AnyValue, AsyncResponse } from './promise-markers';
type Widen<T> = T extends '' ? string : T extends number ? number : T extends boolean ? boolean : T;
/**
 * Async reader interface. Implemented by AsyncGenericReader and AsyncArrayReader.
 */
export interface AsyncArrayLike<T extends AnyValue> {
    forEach(body: (item: T, i: number) => AsyncResponse<void>): Promise<void>;
    map<R extends AnyValue>(body: (item: T, i: number) => AsyncResponse<R>): AsyncArrayLike<R>;
    filter(body: (item: T) => AsyncResponse<boolean>): AsyncArrayLike<T>;
    reduce<R extends AnyValue>(body: (prev: Widen<R>, item: T) => AsyncResponse<Widen<R>>, initial: Widen<R>): Promise<Widen<R>>;
    find(body: (item: T, i: number) => AsyncResponse<boolean>): Promise<T | undefined>;
    findIndex(body: (item: T, i: number) => AsyncResponse<boolean>): Promise<number>;
    some(body: (item: T) => AsyncResponse<boolean>): Promise<boolean>;
    every(body: (item: T) => AsyncResponse<boolean>): Promise<boolean>;
    sort(cmp?: (elt1: T, elt2: T) => AsyncResponse<number>): AsyncArrayLike<T>;
}
/**
 * Array wrapper for chainable async versions of the ES5 Array methods.
 *
 * Only the very basic features of the ES5 API are supported.
 * The goal is to favor speed and not get distracted by rarely used features.
 */
export declare class AsyncArray<T extends AnyValue> implements AsyncArrayLike<T> {
    protected getSource: () => Promise<T[]> | T[];
    constructor(getSource: () => Promise<T[]> | T[]);
    forEach(body: (item: T, i: number) => AsyncResponse<void>): Promise<void>;
    /**
     * Asynchronously creates a shallow copy of a portion of the array, specified by the `start` and `end` parameters,
     * and returns a new `AsyncArray` instance containing the sliced portion. This method does not modify the original array.
     */
    slice(start?: number, end?: number): Promise<AsyncArray<T>>;
    /**
     * Similar to forEach but parallelizes the execution of `body`.
     * The `maxParallel` parameter controls the number of parallel executions.
     */
    forEachParallel(maxParallel: number, body: (item: T, i: number) => AsyncResponse<void>): Promise<void>;
    map<R extends AnyValue>(body: (item: T, i: number) => AsyncResponse<R>): AsyncArrayReader<R>;
    filter(body: (item: T) => AsyncResponse<boolean>): AsyncArrayReader<T>;
    find(body: (item: T, i: number) => AsyncResponse<boolean>): Promise<T | undefined>;
    findIndex(body: (item: T, i: number) => AsyncResponse<boolean>): Promise<number>;
    some(body: (item: T) => AsyncResponse<boolean>): Promise<boolean>;
    every(body: (item: T) => AsyncResponse<boolean>): Promise<boolean>;
    reduce(body: (prev: T, item: T, index: number) => AsyncResponse<T>): Promise<T>;
    reduce<R extends AnyValue>(body: (prev: Widen<R>, item: T, index: number) => AsyncResponse<Widen<R>>, initial: Widen<R>): Promise<Widen<R>>;
    private qsort;
    sort(compare: (elt1: T, elt2: T) => AsyncResponse<number>, begIndex?: number, endIndex?: number): AsyncArrayReader<T>;
    join(sep?: string): Promise<string>;
    uniq(): AsyncArrayReader<T>;
    elementAt(i: number): Promise<T>;
    at(i: number): Promise<T | undefined>;
    get length(): Promise<number>;
    toArray(): Promise<T[]>;
    min<R extends AnyValue>(body: (item: T, i: number) => AsyncResponse<R>): Promise<R>;
    max<R extends AnyValue>(body: (item: T, i: number) => AsyncResponse<R>): Promise<R>;
    sum<R extends number>(body: (item: T, i: number) => AsyncResponse<R>): Promise<R>;
}
/**
 * Wraps an array with an AsyncArray.
 */
export declare function asyncArray<T extends AnyValue>(array: T[]): AsyncArray<T>;
/**
 * AsyncReader wrapper around arrays.
 */
export declare class AsyncArrayReader<T extends AnyValue> extends AsyncArray<T> implements AsyncReader<T> {
    #private;
    get readCount(): number;
    read(): Promise<T | undefined>;
    stop(): Promise<void>;
    readAll(): Promise<T[]>;
    toAsyncGenerator(): AsyncGenerator<T>;
}
/**
 * Wraps an array with an AsyncArray
 */
export declare function asyncArrayReader<T extends AnyValue>(array: T[]): AsyncArrayReader<T>;
export {};
//# sourceMappingURL=async-array.d.ts.map