/** @packageDocumentation @module runtime */
import { AnyRecord, AnyValue, UnPromised } from '@sage/xtrem-async-helper';
import { Decimal } from '@sage/xtrem-decimal';
import { AsyncResponse, Dict, integer } from '@sage/xtrem-shared';
import { Collection, datetime, OrderBy } from '.';
import { InternalAttachmentAssociationData } from '../application/attachment-manager';
import { InternalNoteAssociationData } from '../application/note-manager';
import { Property } from '../properties/property';
import { Context, NodeDeleteOptions } from '../runtime/context';
import type { NodeFactory } from '../runtime/node-factory';
import { SyncInfo } from '../synchronization/sync-info';
import { NodePayloadData, NodeUpdateData } from './create-data';
import { Node } from './node';
import { NodeStatus } from './node-status';
export interface NodeSaveOptions {
    controlOnly?: boolean;
    useUpsert?: boolean;
    /**
     * Should the save be deferred at the end of the transaction ?
     */
    deferred?: boolean;
    /**
     * Flushes the deferred actions at the end of the save
     * @see context.flushDeferredActions()
     */
    flushDeferredActions?: boolean;
}
export interface NodePayloadOptions<T> {
    omitDefaultValues?: boolean;
    inputValuesOnly?: boolean;
    withIds?: boolean;
    withoutCustomData?: boolean;
    thunkOnly?: boolean;
    propertyNames?: PayloadSelector<T>;
    returnEnumsAsNumber?: boolean;
    convertScalarValue?: (value: AnyValue) => AnyValue;
    withLayer?: boolean;
    withTimeStamps?: boolean;
    withNaturalKeyWhenThunk?: boolean;
    limitCollections?: number;
}
export type PayloadSelector<T> = {
    [K in keyof T]?: T[K] extends Promise<infer U> ? (U extends Node | null ? PayloadSelector<U> : boolean) : T[K];
};
export type PropertyName<T extends Node> = Exclude<keyof T, symbol | number>;
/** System properties and methods accessible from `node.$` */
export declare class Node$<NodeT extends Node> {
    /** returns the node's status */
    get status(): NodeStatus;
    /** Is the node readonly? */
    get isReadonly(): boolean;
    /**
     * Is the node effectively readonly?
     *
     * Writable states are put in readonly mode during the execution of some rules
     * (defaultValue, updatedValue, control, controlDelete).
     * This accessor returns true while these rules are executed, even if the node is writable.
     */
    get isEffectivelyReadonly(): boolean;
    /** Is the node writable? (opposite of isReadonly) */
    get isWritable(): boolean;
    /** Was the node created (rather than read)  */
    get isNew(): boolean;
    get isOnlyForDefaultValues(): boolean;
    /** return the decrypted value */
    decryptValue<K extends keyof NodeT>(propertyName: K): Promise<string>;
    /** the _context_ in which the node was created. */
    get context(): Context;
    /** The state of the object at the beginning of the transaction, before any changes made to it */
    get old(): Promise<NodeT>;
    /** The factory that provides metadata for the node instance */
    get factory(): NodeFactory;
    /**
     * The node's ID, a unique integer for all nodes of the same node class.
     *
     * Same as `node._id`
     */
    get id(): number;
    /**
     * The node's source ID, an ID that relates the record to the source system.
     *
     * Same as `node._sourceId`
     */
    get sourceId(): Promise<string>;
    /**
     * The node's layer.
     *
     * Same as `node._layer`
     */
    get layer(): Promise<number>;
    /**
     * The node's sort value. Only valid on vital child nodes
     */
    get sortValue(): Promise<integer>;
    setSortValue(value: integer): Promise<void>;
    /** The _createUser of the node */
    get createdBy(): Promise<Node | number>;
    /** The _updateUser of the node */
    get updatedBy(): Promise<Node | number>;
    /** The _createStamp of the node */
    get createStamp(): Promise<datetime>;
    /** The _updateStamp of the node */
    get updateStamp(): Promise<datetime>;
    /** The _syncTick of the node */
    get syncTick(): Promise<Decimal>;
    /** The _syncInfo of the node */
    get syncInfo(): Promise<SyncInfo>;
    /**
     * The vendor. Only valid if the node has a _vendor property
     */
    get vendor(): Promise<Node | null>;
    get etag(): Promise<string>;
    get customData(): Promise<object>;
    get valuesHash(): Promise<string>;
    get attachments(): Collection<InternalAttachmentAssociationData & Node>;
    get notes(): Collection<InternalNoteAssociationData & Node>;
    getCursorValue(orderBy: OrderBy<NodeT>): Promise<string>;
    /** Controls the node before saving it. */
    control(options?: NodeSaveOptions): Promise<boolean>;
    /**
     * Saves a newly created node or an updated node to the database.
     * If the nodes has vital children, they are also saved.
     * This method executes all the validation rules before saving.
     * It throws and does not modify the database if some validation rules raise an error.
     */
    save(options?: NodeSaveOptions): Promise<void>;
    /**
     * `node.$.save` with a boolean return value.
     * Returns true if node has been saved, false if some validation rules failed.
     */
    trySave(options?: NodeSaveOptions): Promise<boolean>;
    update(data: NodeUpdateData<NodeT>): Promise<void>;
    /**
     * Deletes the node.
     *
     * All the nodes that are related via _vital_ references or collections will also be deleted.
     *
     * This call triggers the `delete` rules on the node decorators and throws an error if any
     * rule forbids the deletion.
     * An error will also be thrown if the deletion violates foreign key constraints in the database
     * (for example an attempt to delete a product which is referenced from sales documents).
     */
    delete(options?: NodeDeleteOptions): Promise<void>;
    /**
     * Tries to delete the node.
     *
     * Similar to `delete()` but returns a true value in case of success and a false value otherwise,
     * instead of throwing an exception.
     */
    tryDelete(options?: NodeDeleteOptions): Promise<boolean>;
    /**
     * Sets multiple property values in a single call.
     * @param data the values, as a plain object. `data` may contain values for collection properties, as arrays of plain objects
     *
     * Note: `node.$.set(data)` only assigns the values. It does not save the node to the database.
     * To save the node, call `node.$.save()`.
     *
     * @example
     * ```
     * user.$.set({ lastName: 'Smith', children: [{name: 'Mary' }, {name: 'Tim'}] })
     * ```
     */
    set(data: NodeUpdateData<NodeT>): Promise<void>;
    /**
     * Gets the value of a property, given the property name
     */
    getValue<ValT extends AnyValue = AnyValue, K extends PropertyName<NodeT> | '' = '', ResultT = K extends PropertyName<NodeT> ? UnPromised<NodeT[K]> & ValT : ValT>(propertyName: K | string): Promise<(ResultT & ValT) | undefined>;
    getRawPropertyValue(propertyName: string): AnyValue;
    getRawPropertyValues(): Dict<AnyValue>;
    isValueDeferred(propertyName: string): boolean;
    /**
     * Get the natural key value as a string (concatenated with '|' if composite).
     * Throws if the node factory does not have a natural key.
     */
    getNaturalKeyValue(): Promise<string>;
    /** The key token, a string that uniquely identifies a node within its factory */
    get keyToken(): string;
    /**
     * Does the node have a property with a given name
     */
    hasProperty(propertyName: string): boolean;
    /**
     * Returns whether the property is frozen
     */
    isPropertyFrozen(property: Property): AsyncResponse<boolean>;
    /**
     * The node's property values as a plain TypeScript object which may be serialized
     * with JSON.stringify().
     */
    payload(options?: NodePayloadOptions<NodeT>): Promise<NodePayloadData<NodeT>>;
    /**
     * Untyped variant of this.payload(options?). For compat with exising code.
     */
    payloadAsAny<V extends Dict<any> = Dict<any>>(options?: NodePayloadOptions<NodeT>): Promise<V>;
    isPropertyEnabled<K extends keyof NodeT>(propertyName: K): Promise<boolean>;
    /**
     * Return a duplicate of the node with negative _id values to create a copy
     * Where the node is a vitalCollectionChild, remove sort value and keep parent _id
     */
    duplicate(data?: AnyRecord): Promise<NodeT>;
    /**
     * Return the value of a property from a path
     */
    get(path: string): Promise<AnyValue>;
    /**
     * Special method for x3-gateway to clear the caches
     */
    clearCaches(): void;
    /**
     * Special method used only by ItemSite.
     * TODO: investigate why applicative code really needs this and remove if possible.
     */
    get isInsideCollection(): boolean;
}
//# sourceMappingURL=node-$.d.ts.map