import { Dict } from '@sage/xtrem-shared';
import { Table } from '../schema/table';
export declare class SqlNaturalKeyUtils {
    /**
     * Builds all the parts needed to build a SELECT query where all the natural keys will be returned as 'foo|bar'.
     * This function will take care of base tables (joins and aliases will be created).
     * For reference array properties, the natural keys will be aggregated into an array using ARRAY_AGG.
     * Note: When reference arrays are present, you may need to add a GROUP BY clause to your query.
     * Returns:
     *   - aliasedColumnNames: the name of all the columns, with aliases (t1.a AS a, t2.b AS b, t1.c || '|' || t2.d AS c, ...)
     *     For reference arrays: ARRAY_AGG(t2.name) FILTER (WHERE t2.name IS NOT NULL) AS service_options
     *   - aliases: list of created aliases, indexed by table name
     *   - joins: list of joins required to follow the references / base relations
     *   - args: args to be used by the query (will by used to store the constructor names when dealing with base tables)
     * @param schemaName
     * @param table
     * @param columnNamesToQuery the columns to query from the table (a column can be either from the table or from its base tables)
     * @param options
     */
    static getSqlQueryPartsWithNaturalKeys(schemaName: string, table: Table, columnNamesToQuery: string[], options?: {
        /**
         * The alias already assigned to the table
         */
        tableAlias?: string;
        /**
         * When dealing with base tables, should the name of the constructor be inserted in
         * the LEFT JOIN clause (true) or inserted in the args array (false) ?
         */
        doNotUseArgsForConstructors?: boolean;
        /**
         * When chaining many calls to the function in the same context, you can pass the aliases, joins, args returned by the previous
         * calls.
         */
        aliases?: Dict<string>;
        /**
         * When chaining many calls to the function in the same context, you can pass the aliases, joins, args returned by the previous
         * calls.
         */
        joins?: string[];
        /**
         * When chaining many calls to the function in the same context, you can pass the aliases, joins, args returned by the previous
         * calls.
         */
        args?: any[];
    }): {
        columns: {
            /**
             * The column with the 'AS'. Something like "t2.name || '|' || t1.name || '|' || t1.action AS operation"
             */
            aliased: string;
            /**
             * The 'simple' column name (something like 'operation')
             */
            unaliased: string;
        }[];
        aliases: Dict<string>;
        joins: string[];
        args: any[];
    };
    /**
     * Returns the column name (with its alias) that should be used to in a SQL query.
     * If needed (column belongs to a base table, references, ...), some joins will be generated and some new aliases will be created
     */
    private static _getAliasedColumnForSqlQuery;
}
//# sourceMappingURL=sql-natural-key-utils.d.ts.map