import { AnyRecord } from '@sage/xtrem-async-helper';
import { Logger } from '@sage/xtrem-log';
import { AnyValue, AsyncResponse, Dict, MetaCustomFields, NotificationProgress } from '@sage/xtrem-shared';
import { RequestHandler } from 'express';
import { Application, DataValidationManager, HotUpgradeManager, PrintingManager } from '../application';
import { AttachmentManager } from '../application/attachment-manager';
import { ChatbotManager } from '../application/chatbot-manager';
import { ClientSettingsManager } from '../application/client-settings-manager';
import { CsvChecksumManager } from '../application/csv-checksum-manager';
import { DashboardManager } from '../application/dashboard-manager';
import { NoteManager } from '../application/note-manager';
import { NotificationManager } from '../application/notification-manager';
import { CreateSchemaOptions, PackageManager } from '../application/package-manager';
import { SageAiManager } from '../application/sage-ai-manager';
import { ServiceOptionManager } from '../application/service-option-manager';
import { TagManager } from '../application/tag-manager';
import { PlainOperationDecorator, StaticThis } from '../decorators';
import { AuditManagerInterface } from '../hooks/auditing';
import { InteropAppInfo } from '../interop';
import { Node } from '../ts-api';
import { WorkflowManagerInterface } from '../workflow';
import { Context, UserInfo } from './context';
import { NodeFactory } from './node-factory';
export interface SourceCode {
    content: string;
    file: string;
    contentToLint?: string;
}
export interface WebSocketMessageRequest {
    tenantId: string;
    developerId: string;
    contextId: string;
    attributes: Dict<AnyValue>;
    type: string;
    source: string;
    request: string;
    replyId: string;
    replyTopic: string;
    locale?: string;
}
export interface WebSocketMessageResponse extends WebSocketMessageRequest {
    response: AnyValue;
}
/**
 * A source code resolver is responsible for returning the response to a web socket request
 */
export interface WebSocketMessageHandler {
    getResponse(application: Application, data: WebSocketMessageRequest): Promise<WebSocketMessageResponse>;
}
export interface ExportTemplate {
    id: string;
    name: string;
    isDefault?: boolean;
}
export interface CsvTemplateContent {
    _id: number;
    path: string;
    dataType: string;
    description: string;
    locale: string;
    isCustom?: boolean;
}
export interface HeaderDefinition {
    path: string;
    title: string;
}
export interface NodeExportTemplate {
    name: string;
    exportTemplates: ExportTemplate[];
}
export interface SecretConfig {
    xtremEnv?: string;
    clusterId?: string;
    secretCacheTimeoutInMinutes?: number;
}
export type BatchLogLevel = 'info' | 'warning' | 'error' | 'exception' | 'result' | 'test';
export interface BatchLogEntry {
    level: BatchLogLevel;
    message: string;
}
/**
 * The Manager global object contain methods that other packages (like xtrem-system) will override to inject
 * optional functionality into xtrem-core
 *
 * This API is currently very sketchy and will evolve soon.
 *
 * The other managers which are currently registered on Context (localizationManager, authorizationManager, ...)
 * will be moved here.
 *
 * TODO: improve this API and suggest a better name.
 * Maybe something like `InjectedDependencies`, or just 'Hooks'
 */
export declare abstract class CoreHooks {
    static createPackageManager: (application: Application) => PackageManager;
    static createServiceOptionManager: (application: Application) => ServiceOptionManager;
    static createHotUpgradeManager: (_application: Application) => HotUpgradeManager;
    static createDashboardManager: (_application: Application) => DashboardManager;
    static getChatbotManager: (_application: Application) => ChatbotManager;
    static createPrintingManager: (_context: Context) => PrintingManager | null;
    static createClientSettingsManager: (_application: Application) => ClientSettingsManager;
    static createNotificationManager: (_application: Application) => NotificationManager;
    static createCsvChecksumManager: (_application: Application) => CsvChecksumManager;
    static getDataValidationManager: () => DataValidationManager;
    static createWorkflowManager: (_application: Application) => WorkflowManagerInterface;
    static auditManager: AuditManagerInterface;
    static getAttachmentManager: () => AttachmentManager;
    static getTagManager: () => TagManager;
    static getNoteManager: () => NoteManager;
    static getSageAiManager: () => SageAiManager;
    static testManager: {
        loadTestData(_application: Application, _options: {
            testLayers?: string[];
        }, _tenantId: string): Promise<void>;
        /**
         * Initializes the pack allocation for system factories
         */
        initTestTenantActivePackages(_application: Application): Promise<void>;
    };
    static webSocketHandlers: Dict<WebSocketMessageHandler>;
    /** API injected by xtrem-communication */
    static communicationManager: {
        /**
         * Sends a notification
         */
        notify(_context: Context, topic: string, _payload: AnyRecord): Promise<string>;
        /**
         * Starts an async mutation
         */
        startAsyncMutation(_context: Context, factoryName: string, mutationName: string, _payload: AnyRecord): Promise<string>;
        /**
         * Tracks the progress of an async mutation
         */
        trackAsyncMutation(_context: Context, _factory: NodeFactory, trackingId: string, _op: PlainOperationDecorator): Promise<{
            status: string;
            result?: AnyValue;
            errorMessage?: string;
            logMessages?: BatchLogEntry[];
        }>;
        /**
         * Sets ths status of an async mutation to stopRequested
         */
        requestStop(_context: Context, trackingId: string, _reason: string): Promise<void>;
        /**
         * Sets ths status of an async mutation to stopped
         */
        setStopped(_context: Context, trackingId: string): Promise<void>;
        /**
         * Sets ths status of an async mutation to stopped
         */
        requestUserNotification(_context: Context, trackingId: string): Promise<void>;
        /**
         * Tracks the progress of an async mutation
         */
        isStopRequested(_context: Context): Promise<boolean>;
        extendClassDecorator(_factory: NodeFactory): void;
        updateProgress(_context: Context, _partialProgress: Partial<NotificationProgress>): Promise<void>;
        logErrorBatchMessage(_context: Context, _message: string): Promise<void>;
    };
    static interopManager: {
        getInteropAppInfo(_context: Context, _app: string): Promise<InteropAppInfo>;
        interopProxyMiddleware(_application: Application): RequestHandler;
        getTenantApps(_tenantId: string): Promise<string[] | null>;
        /**
         * Fix the activation of sysNodeTransformations at the end of upgrades
         * @param _context
         */
        fixInteropTransformationsActivation(_context: Context, _options?: {
            logger?: Logger;
        }): AsyncResponse<void>;
    };
    /** API injected by xtrem-metadata */
    static metadataManager: {
        /**
         * Synchronize the metadata to the database
         */
        upgradeMetadata(_application: Application, _options?: {
            fullReload?: boolean;
            onlyCreate?: boolean;
        }): AsyncResponse<void>;
        getMetaNodeFactoryConstructor(): StaticThis<Node> | null;
    };
    /** API injected by xtrem-system */
    static sysManager: {
        /**
         * Gets the User node
         */
        getUserNode(): StaticThis<Node>;
    };
    /** API injected by xtrem-customization */
    static customizationManager: {
        getWizardUrl(): string;
        /**
         * Gets the metadata of the custom fields for a list of node names.
         */
        getCustomFields(_context: Context, _nodeNames: string[], _options?: {
            includeMutableChildren: boolean;
        }): Promise<MetaCustomFields | undefined>;
    };
    /** API injected for import-export data */
    static importExportManager: {
        executeExport(_context: Context, _id: string, _outputFormat: string, _filter: string, _orderBy: string): Promise<string>;
        executeExportByDefinition(_context: Context, _templateDefinition: HeaderDefinition[], _nodeName: string, _outputFormat: string, _filter: string, _orderBy: string): Promise<string>;
        getNodeExportTemplates(_context: Context, _nodeNames: string[]): Promise<Dict<ExportTemplate[]>>;
        getExportPageUrl(): string;
    };
    static remoteAppManager: {
        getRemoteAppToken(context: Context): AsyncResponse<string | null>;
        getLocalUserFromRemoteUser(_context: Context): AsyncResponse<UserInfo | null>;
    };
}
export { CreateSchemaOptions, PackageManager, ServiceOptionManager };
//# sourceMappingURL=core-hooks.d.ts.map