import { AnyRecord, Funnel } from '@sage/xtrem-async-helper';
import type { NodeFactory } from '../system-exports';
/**
 * This file implements prefetching for the framework.
 *
 * Each context has a context.prefetcher field that references a ContextPrefetcher instance.
 * This instance tracks the keys of records that may be fetched later, and of the records that have already been fetched.
 *
 * Prefetching proceeds in two passes:
 *
 * 1. A visit pass.
 *    In this pass, the visit method is called for each record that was just fetched from the database,
 *    or that is being inserted/updated. It scans the references and collections of the record and it collects their keys
 *    in the prefetcher buckets.
 * 2. A fetch pass.
 *    When the frameork needs to read or query records, it calls the prefetcher tryRead/tryQuery method, instead of
 *    fetching directly from the database.
 *    These methods return the record(s) if they can be prefetched, undefined otherwise.
 *    If the prefetcher return undefined, the framework fetches the records.
 *
 * The prefetcher uses buckets to track the keys that have been collected during the visit pass, and the records
 * that have been fetched.
 * There are several buckets for each factory because we have to track references and collections separately
 * (references have complete keys whereas collection have partial keys), and because records that have been read with
 * forUpdate false cannot be used when we read with forUpdate true.
 *
 * When tryRead is called, the prefetcher checks if the requested key is already in the bucket map. If so, it returns it.
 * Otherwise, it fetches all the keys that have been collected, including the requested key.
 * All the records read are put in the bucket's fetched records map, indexed by their key.
 * The prefetcher only returns the record that was requested. The other records are stored in the bucket map.
 * When a subsequent tryRead is called in the same transaction, the prefetcher is likely to find the record in the map.
 *
 * This reduces significantly the number of database calls.
 *
 * Let us see how this works when we read a sales order, its lines and the items of the lines.
 * - First, we read the sales order header (1 SQL query)
 * - Then we read the lines (1 SQL query)
 * - The line records are visited. All their item keys are collected in the 'Item' bucket.
 * - Then the application code processes the first line and reads its item.
 * - The prefetcher checks if the item is already in the bucket. It won't be because it was not fetched yet.
 * - So it fetches the item key and all the other keys that have collected in the 'Item' bucket (1 SQL query).
 * - The prefetcher returns the item that was requested (or null if not found) and puts the fetched records in the bucket's map.
 * - When the application code processes the following lines and reads their items, the prefetcher finds the items in the
 *   bucket map, without having to fetch them from the database.
 *
 * So for a sales order with 50 lines and 50 items, we have only 3 SQL queries instead of 52: 1 to fetch the header,
 * 1 to fetch the lines and 50 to fetch the items individually.
 *
 * Note that we cannot know in advance if the transaction will enventually read the items or not.
 * Some transactions might only need other fields from the lines, like amount, quantity, etc.
 * This is why the visit pass collects the item keys, but does not actually fetch the items.
 * The items are only fetched when the applicative code tries to fetch the first item.
 * The prefetcher assumes that, if we need to fetch an item on one line, it is likely that we will also need to fetch items
 * on the other lines.
 *
 * In the visit pass the prefetcher will usually collect some keys that will never be used by the transaction.
 * This is not a problem because the visit pass is very fast and uses little memory to store the keys.
 *
 * In the fetch pass, the prefetcher may read records that will not be used by the transaction.
 * This is a little more problematic but the bet is that, most of the time, the transaction will execute similar logic
 * on similar records and will thus end up fetching all the records or most of them.
 * Also, reading more records than needed is much less expensive than reading them one by one.
 *
 * The prefetcher also has methods that get called after an insert or an update. These methods adjust the buckets to
 * take into account the changes, so that subsequent reads and queries will return correct results.
 * We do not try to do a perfect job here, we just ensure that the results will always been correct, event if this
 * means invalidating maps that could have been patched with more sophisticated logic.
 *
 * Limitations: prefetching is limited to nodes with 'sql' storage. External storage may be supported later.
 */
/** Type for the records that are fetched from the database and stored in the bucket maps */
export type PrefetchedRecord = AnyRecord;
/** Data type for record or collection key values (including _id) */
export type Key = string;
/** Data type for _id values - string so that we can handle external storage later */
export type SysId = string;
/** Bucket for keys and records that are visited as references with a key other than _id (a reverseReference, or a join) */
export interface ReferenceBucket {
    /** The factory bucket */
    readonly factoryBucket: FactoryBucket;
    /** The bucketId - names of the key properties concatenated with '|'  */
    readonly bucketId: string;
    /** The key values that have collected during the visit pass but not fetched yet */
    readonly pendingRecordKeys: Set<Key>;
    /**
     * The map from key values to _id values (or null if record was not found).
     * The records are not stored in this bucket but in factoryBucket.fetchedRecords.
     */
    readonly fetchedSysIds: Map<Key, SysId | null>;
}
/** Bucket for keys and records that are visited as collections */
export interface CollectionBucket {
    /** The factory bucket */
    readonly factoryBucket: FactoryBucket;
    /** The bucketId - names of the collection join properties concatenated with '|'  */
    readonly bucketId: string;
    /** The key values that have collected during the visit pass but not fetched yet */
    readonly pendingRecordKeys: Set<Key>;
    /**
     * The map from key values to _id values (or null if record was not found).
     * The records are not stored in this bucket but in factoryBucket.fetchedRecords.
     */
    readonly fetchSysIdArrays: Map<Key, SysId[]>;
}
/**
 * Factory-level bucket
 * A factory may have 2 such buckets, for update true and update false.
 */
export interface FactoryBucket {
    /** The factory of the records that will be fetched */
    readonly factory: NodeFactory;
    /** Are the records fetched _for update_ or not? */
    readonly forUpdate: boolean;
    /** The _id values that have collected during the visit pass but not fetched yet */
    readonly pendingSysIds: Set<SysId>;
    /** The map from _id values to fetched records */
    readonly fetchedRecords: Map<SysId, PrefetchedRecord>;
    /** The reference buckets for keys other than _id */
    readonly referenceBuckets: Map<string, ReferenceBucket>;
    /** The collection buckets */
    readonly collectionBuckets: Map<string, CollectionBucket>;
    /** The funnel to prevent concurrent fetching of the same key */
    readonly funnel: Funnel;
}
/** Camel case that preserves leading _ */
export declare function camelCase(str: string): string;
//# sourceMappingURL=context-prefetcher.d.ts.map