import * as stream from 'stream';
import { Application } from '../application';
import { RestoreEnvelop } from './restore-envelop';
import { S3ObjectInfo } from './s3-helper';
type S3Configuration = {
    bucket: string;
    folder: string;
    s3Key?: string;
    /**
     * Returns the full folder (S3 prefix) to be used for this configuration
     * When not set, the folder will be used
     */
    getFullFolder?: (app: Application) => string;
};
declare const s3Configurations: {
    /**
     * The S3 bucket used to store the reference database backups (the backups used for the generation of SQL files)
     */
    forSqlFiles: S3Configuration;
    /**
     * The S3 bucket used to store cluster-cu backups
     * Used to restore a backup of cluster-cu, from the begining of the month
     * It's an exact copy of sdmo_cu but we have to keep it for some releases for backward compatibility
     */
    clusterCuBackup: S3Configuration;
    /**
     * The S3 bucket used to store sdmo-cu backups
     * Used to restore a backup of sdmo-cu, from the begining of the month
     */
    sdmo_cu: S3Configuration;
    /**
     * The S3 bucket used to store cluster-ci/sdmo backups
     * Used to restore a backup of cluster-ci/sdmo, after the last successful upgrade
     * It's an exact copy of sdmo but we have to keep it for some releases for backward compatibility
     */
    clusterCiBackup: S3Configuration;
    /**
     * The S3 bucket used to store sdmo-ci/sdmo backups
     * Used to restore a backup of sdmo-ci/sdmo, after the last successful upgrade
     */
    sdmo: S3Configuration;
    /**
     * Cluster release (dev)
     */
    clusterDevRelease: S3Configuration;
    /**
     * Cluster release (qa)
     */
    clusterQaRelease: S3Configuration;
    /**
     * Glossary
     * To regenerate the S3 backup : Use https://sage-liveservices.visualstudio.com/X3%20XTREM/_build?definitionId=2311
     *    - Name of cluster to dump : glossary
     *    - Env of cluster to dump : dev-eu
     *    - withDate : unchecked
     *    - suffix : latestWorking
     */
    glossary: S3Configuration;
    /**
     * Showcase
     * To regenerate the S3 backup : Use https://sage-liveservices.visualstudio.com/X3%20XTREM/_build?definitionId=2311
     *    - Name of cluster to dump : showcase
     *    - Env of cluster to dump : dev-eu
     *    - withDate : unchecked
     *    - suffix : latestWorking
     */
    showcase: S3Configuration;
    /**
     * Shopfloor (ci)
     */
    shopfloor: S3Configuration;
    /**
     * Shopfloor (cu)
     */
    shopfloor_cu: S3Configuration;
    /**
     * X3 Connector
     */
    x3_connector: S3Configuration;
};
export type S3ConfigurationType = keyof typeof s3Configurations;
export declare class S3Manager {
    readonly application: Application;
    private static _postgreSqlTools;
    readonly tempFolder: string;
    constructor(application: Application);
    /**
     * Dump the schema of an application to a S3 bucket (and returns the S3 URI of the written object)
     * @param app the application to dump the schema from
     */
    dumpSchemaToS3Bucket(): Promise<string>;
    /**
     * Returns the full folder (S3 prefix) to be used for a given S3 configuration
     */
    private _getFullS3Folder;
    /**
     * Resolve the version of the backup to be used to generate a new SQL file
     * @param app
     * @param version
     * @returns
     */
    private _resolveVersionToRestoreForSqlFiles;
    /**
     * Raise an error when a specific version is not part of the available backups used to generate SQL files
     * @param app
     * @param version
     * @param versions
     */
    private _raiseErrorVersionNotFoundForSqlFile;
    /**
     * Restore a schema version from a S3 bucket.
     * If the version is 'latest', the latest available dump for the application will be restored
     * @param app the application to restore the schema for
     * @param versionOrS3Uri either a version to restore, or a full S3 URI of the backup to restore
     * @param s3ConfigurationType the type of the S3 configuration to restore from
     * @param options.checkSingleSchema if true, will raise an error if there are more than 2 schemas in the database
     * @param options.skipValuesHash if true, the re-compute of _valuesHash will be skipped
     */
    restoreSchemaFromS3Bucket(versionOrS3Uri: string, s3ConfigurationType?: S3ConfigurationType, options?: {
        checkSingleSchema: boolean;
        skipValuesHash: boolean;
    }): Promise<void>;
    private _listSchemas;
    /**
     * Upload a local db dump to a S3 bucket and returns the S3 URL of the written object
     * @param localFolder the local folder that contains the file to copy
     * @param filename the name of the file to copy
     * @param dumpKey the key to use to store the dump into S3
     */
    private uploadDumpToS3;
    /**
     * Upload a local file to a S3 bucket and returns the S3 URL of the written object
     * @param s3Configuration see s3Configurations
     * @param localFilename the (full) filename of the local file to copy
     * @param objectKey the key to use to store the file into S3
     */
    private _uploadLocalFileToS3;
    /**
     * Returns whether an objectKey matches a db backup that can be used to generate SQL files for a given application
     */
    private _isBackupForSqlFile;
    /**
     * List the versions hosted in the S3 bucket for a given application.
     * Will only return versions for backups that can be used to generate SQL files.
     * Will only return the versions with the same major version as the application.
     * @param app
     */
    getAvailableVersions(): Promise<string[]>;
    /**
     * Extracts the version from a S3 key.
     * @sage/xtrem-services-main@6.0.13.zip -> 6.0.13
     *
     * @param s3Key
     * @returns
     */
    private static _getVersionFromS3Key;
    /**
     * Returns the list of backups that can be used to generate SQL files for a given application
     * @param app
     * @param bucketName
     * @returns
     */
    private _listS3BackupsForSqlFile;
    getS3Info(s3ConfigurationType: S3ConfigurationType, versionOrS3Uri: string): Promise<S3ObjectInfo>;
    private static _tailLogLine;
    /**
     * Pipes a readable stream to a writable stream.
     * Transformations are applied according to the provided mappings.
     */
    private static _pipeStreamWithTransformations;
    private static _pgClientToolCommand;
    /**
     * Dumps the current schema (described by the current configuration) to a local dump file.
     * Note: if the filename is a zip file, the dump will be a valid zip archive.
     *
     * Note: the created file will be anonymized: it will contain no schema/user name, they will be replaced with tags:
     * - schema -> [SCHEMA_NAME]
     * - user -> [USER_NAME]
     *
     * @param folder the folder that contains the dump file
     * @param filename the name of the dump file
     */
    static dumpSchemaToFile(folder: string, filename: string, schemaName: string): Promise<void>;
    /**
     * Returns the mappings that should be applied to a sql file before it get restored
     * @param schemaName
     * @param dumpFullFilename
     * @param sqlCfg
     */
    private static _getMappingsForDumpFiles;
    static runRestoreProcess(restoreEnvelop: RestoreEnvelop, filenameOrStream: string | stream.Readable): Promise<void>;
    /**
     * Run the restore process for a schema from a mapped file (i.e. a dump file where the mappings have been applied)
     * @param filenameOrStream the FULL filename of the '.mapped' file
     * @param schemaName
     */
    static runRestoreFromMappedFile(restoreEnvelop: RestoreEnvelop, filenameOrStream: string | stream.Readable): Promise<void>;
    /**
     * Restore a schema from a mapped file (i.e. a dump file where the mappings have been applied)
     * @param dumpFullFilename the FULL filename of the 'sql.mapped' file
     * @param schemaName
     */
    private static _restoreFromMappedFile;
    /**
     * Restore a schema from a dump file
     * @param tempFolder the 'temp' folder where to create the 'sql.mapped.dump' file
     * @param dumpFullFilename the FULL filename of the 'sql.dump' file
     * @param schemaName
     */
    static restoreFromDumpFile(folder: string, dumpFullFilename: string, schemaName: string): Promise<void>;
    /**
     * Restore from a gz dump stream
     * @param dumpFullFilename the FULL filename of the dump to restore
     * @param sourceStream
     * @param hackForSdmo
     */
    restoreSchemaFromBackupStream(dumpFullFilename: string, sourceStream: stream.Readable, hackForSdmo?: boolean): Promise<void>;
    /**
     * Process the table of content of a postgres dump.
     * Mainly : some entries related to extensions will be filtered and the content will be parsed to retrieve the name of schemas
     */
    private _processTableOfContentsForRestoration;
    dropApplicationSchema(): Promise<void>;
    /**
     * Drops a bunch of schemas (if they exist) before restoring a dump
     */
    private _dropSchemasBeforeRestoration;
    /** Generic asynchronous handler for child process stdout, stderr and close event listeners */
    private static _childProcessHandler;
    static normalizeApplicationName(appName: string): string;
}
export {};
//# sourceMappingURL=s3-manager.d.ts.map