let silentOutput = false;
let logToString = false;
const correlationHeader = "x-correlation";
const correlationIdPrefix = correlationHeader + "-id";
const getErrorMsgFromMaybeObject = (notError) => notError instanceof Object && "toString" in notError ? notError.toString() : "no msg";
export const getErrorMsg = (error) => error instanceof Error ? error.message : getErrorMsgFromMaybeObject(error);
const ContextTool = {
    clearAll: () => global.CONTEXT = undefined,
    replaceAllWith: (ctx) => global.CONTEXT = ctx,
    set: (key, value) => {
        //This test is required because SDX is using this header
        if (key === correlationHeader) {
            key = correlationIdPrefix;
        }
        if (!key.startsWith(correlationIdPrefix + "-")) {
            key = `${correlationIdPrefix}-${key}`;
        }
        const globalTyped = global;
        if (!globalTyped.CONTEXT) {
            globalTyped.CONTEXT = {};
        }
        globalTyped.CONTEXT[key] = value;
    },
    get: () => global.CONTEXT || {},
    responseWithContextHeaders: (request, headersObj) => {
        if (!headersObj) {
            headersObj = {};
        }
        if (!request) {
            request = {};
        }
        const headers = Object.assign({}, ContextTool.get(), headersObj);
        const headerNames = Object.keys(headers);
        if (!request.headers) {
            request.headers = {};
        }
        for (const name of headerNames) {
            request.headers[name] = headers[name];
        }
        return request;
    },
    setRequestContext: (event, context) => {
        if (!context) {
            context = {};
        }
        if (!event.headers) {
            LogTool.warn(`Request ${context.awsRequestId} is missing headers`);
            return;
        }
        if (!context.awsRequestId) {
            LogTool.warn(`Request ${context.awsRequestId} is missing awsRequestId`);
            return;
        }
        const ctx = { awsRequestId: context.awsRequestId };
        for (const header in event.headers) {
            if (header.toLowerCase() === correlationHeader) {
                event.headers[header.toLowerCase()] = event.headers[header];
            }
            if (header.toLowerCase().startsWith("x-correlation-")) {
                ctx[header.toLowerCase()] = event.headers[header];
            }
        }
        if (!ctx[correlationIdPrefix]) {
            ctx[correlationIdPrefix] = ctx.awsRequestId;
        }
        //fallback due to SWD using another token for context
        if (event.headers[correlationHeader]) {
            ctx["x-correlation-id"] = event.headers[correlationHeader];
        }
        if (event.headers["User-Agent"]) {
            ctx["User-Agent"] = event.headers["User-Agent"];
        }
        if (event.headers["Debug-Log-Enabled"] === "true") {
            ctx["Debug-Log-Enabled"] = "true";
        }
        else {
            // enable debug logging on 5% of cases
            ctx["Debug-Log-Enabled"] = Math.random() < 0.05 ? "true" : "false";
        }
        if (event.headers["True-Client-IP"]) {
            ctx["True-Client-IP"] = event.headers["True-Client-IP"];
        }
        ContextTool.replaceAllWith(ctx);
    }
};
function getContext() {
    // note: this is a shallow copy
    return Object.assign({}, ContextTool.get());
}
function isDebugEnabled() {
    // disable debug logging by default, but allow override via env variables
    // or if enabled via forwarded request context
    return process.env.DEBUG_LOG === "true" || ContextTool.get()["Debug-Log-Enabled"] === "true";
}
function isSecretNotObfuscated() {
    // secret obfuscation is enabled by default by default, but allow override via env variables for debug
    return process.env.OBFUSCATE_SECRET_IN_LOGS === "false";
}
function autoConvertError(objOrErr) {
    if (objOrErr instanceof Error) {
        return {
            name: objOrErr.name,
            message: objOrErr.message,
            stack: isDebugEnabled() ? objOrErr.stack : ""
        };
    }
    else {
        return objOrErr;
    }
}
function log(level, msg, params) {
    if (level === "DEBUG" && !isDebugEnabled() || silentOutput) {
        return undefined;
    }
    const logMsg = getContext();
    logMsg.level = level;
    if (level === "SECURITY") {
        msg = "[SECURITY_ALERT] " + msg;
    }
    logMsg.message = msg;
    if (params) {
        logMsg.params = autoConvertError(params);
    }
    const res = JSON.stringify(logMsg);
    if (logToString) {
        return res;
    }
    else {
        console.log(res);
    }
    return undefined;
}
const LogTool = {
    isDebugLoggingEnabled: isDebugEnabled,
    debug: (msg, params) => log("DEBUG", msg, params),
    info: (msg, params) => log("INFO", msg, params),
    warn: (msg, params) => log("WARN", msg, params),
    error: (msg, params) => log("ERROR", msg, params),
    security: (msg, params) => log("SECURITY", msg, params),
    audit: (msg, params) => log("AUDIT", msg, params),
    _setConsoleSilent: (value) => {
        silentOutput = value;
    },
    _setLogToString: (value) => {
        logToString = value;
    }
};
const Tools = {
    Context: ContextTool,
    Log: LogTool,
    obfuscate: (txt, lastCharNbPercentToObfuscate) => {
        if (!txt) {
            return txt;
        }
        if (isSecretNotObfuscated()) {
            return txt;
        }
        if (lastCharNbPercentToObfuscate !== 0 && !lastCharNbPercentToObfuscate) {
            lastCharNbPercentToObfuscate = 50;
        }
        const nbChar = Math.ceil(txt.length / (100 / (100 - lastCharNbPercentToObfuscate)));
        let newTxt = "";
        for (let x = 0; x < txt.length; x++) {
            if (x <= (nbChar - 1)) {
                newTxt += txt[x];
            }
            else {
                newTxt += "-";
            }
        }
        if (txt !== "" && newTxt === txt) {
            newTxt = newTxt.substring(0, newTxt.length - 1) + "-";
        }
        return newTxt;
    },
    deepObfuscate: (obj, lastCharNbPercentToObfuscate) => {
        if (obj instanceof Array) {
            return obj.map((data) => {
                switch (true) {
                    case typeof (data) === "string":
                        return Tools.obfuscate(data, lastCharNbPercentToObfuscate);
                    case data instanceof Object:
                        return Tools.deepObfuscate(data, lastCharNbPercentToObfuscate);
                    default:
                        return data;
                }
            });
        }
        else if (obj instanceof Object) {
            const objType = obj;
            return Object.keys(obj).reduce(function (result, key) {
                switch (true) {
                    case typeof (objType[key]) === "string":
                        result[key] = Tools.obfuscate(objType[key], lastCharNbPercentToObfuscate);
                        break;
                    case objType[key] instanceof Object:
                        result[key] = Tools.deepObfuscate(objType[key], lastCharNbPercentToObfuscate);
                        break;
                    default:
                        result[key] = objType[key];
                }
                return result;
            }, {});
        }
        else if (typeof (obj) === "string") {
            return Tools.obfuscate(obj, lastCharNbPercentToObfuscate);
        }
        else {
            return obj;
        }
    }
};
export default Tools;
//# sourceMappingURL=data:application/json;base64,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