import { DateValue } from '@sage/xtrem-date-time';

export enum ElementTypeEnum {
    string = 'string',
    number = 'number',
    date = 'date',
}
export type ElementType = keyof typeof ElementTypeEnum;
export type DataTypeSupported = string | DateValue | number;
export type UnitTypeSupported = string | undefined;

/*
 * GetElementType returns the ElementTypeEnum for a given DataTypeSupported
 * @param _currentType: DataTypeSupported - the data type
 * @returns ElementTypeEnum - the element type
 */
export function getElementType(_currentType: DataTypeSupported): ElementTypeEnum {
    switch (_currentType) {
        case 'string':
            return ElementTypeEnum.string;
        case 'number':
            return ElementTypeEnum.number;
        case 'DateValue':
            return ElementTypeEnum.date;
        default:
            return ElementTypeEnum.string;
    }
}

/*
 * ParseGenericdBarcode represents the parsed barcode data
 * @param codeName: string - the name of the barcode
 * @param parsedCodeItems: ParsedGenericElement[] - the parsed elements of the barcode
 */

export interface ParsedGenericBarcode {
    codeName: string;
    parsedCodeItems: ParsedGenericElement[];
}

/**
 * ParsedGenericElement class represents a single element of the barcode data
 * It is common for compatibility reasons with its GS1 counterpart except that the elementDataTitle is a string.
 */
export class ParsedGenericElement {
    private _data: DataTypeSupported;

    private _unit: UnitTypeSupported;

    constructor(
        public readonly ai: string,
        public readonly elementDataTitle: string,
        public readonly elementType: ElementType,
    ) {
        this._defineData(elementType);
    }

    protected _defineData(elementType: ElementType): void {
        /* defines the object which represents a single element
         */
        switch (elementType) {
            case ElementTypeEnum.string:
                this._data = '';
                break;
            case ElementTypeEnum.number:
                this._data = 0;
                break;
            case ElementTypeEnum.date:
                this._data = '';
                break;
            default:
                this._data = '';
                break;
        }
        this.unit = undefined; // some elements are accompanied by an unit of
    }

    get data(): DataTypeSupported {
        return this._data;
    }

    set data(data: DataTypeSupported) {
        this._data = data;
    }

    get unit(): UnitTypeSupported {
        return this._unit;
    }

    set unit(unit: UnitTypeSupported) {
        this._unit = unit;
    }
}
