/**
 * P. Brockfeld, 2014-02-05
 *
 * JavaScript for parsing GS1 barCodes, see
 *
 * https://github.com/hprange/BarcodeParser (active fork)
 *
 * https://github.com/PeterBrockfeld/BarcodeParser (original repo)
 *
 * for details.
 */

import { date as dateXtrem } from '@sage/xtrem-date-time';
import { SystemError } from '@sage/xtrem-shared';
import { AbstractBarcodeParser } from './barcode-parser-abstract';
import { DataTitle, ParsedBarcode, ParsedElement, UnitOfMeasure } from './parsed-element';
import { ElementTypeEnum } from './parsed-generic-element';

/**
 * This section store internal errors
 */
enum ErrorMessages {
    invalidAIAfter0 = "invalid AI after '0'",
    invalidAIAfter1 = "invalid AI after '1'",
    invalidAIAfter24 = "invalid AI after '24'",
    invalidAIAfter25 = "invalid AI after '25'",
    invalidAIAfter2 = "invalid AI after '2'",
    invalidAIAfter31 = "invalid AI after '31'",
    invalidAIAfter32 = "invalid AI after '32'",
    invalidAIAfter33 = "invalid AI after '33'",
    invalidAIAfter34 = "invalid AI after '34'",
    invalidAIAfter35 = "invalid AI after '35'",
    invalidAIAfter36 = "invalid AI after '36'",
    invalidAIAfter39 = "invalid AI after '39'",
    invalidAIAfter3 = "invalid AI after '3'",
    invalidAIAfter40 = "invalid AI after '40'",
    invalidAIAfter41 = "invalid AI after '41'",
    invalidAIAfter42 = "invalid AI after '42'",
    invalidAIAfter4 = "invalid AI after '4'",
    invalidAIAfter700 = "invalid AI after '700'",
    invalidAIAfter70 = "invalid AI after '70'",
    invalidAIAfter71 = "invalid AI after '71'",
    invalidAIAfter7 = "invalid AI after '7'",
    invalidAIAfter800 = "invalid AI after '800'",
    invalidAIAfter801 = "invalid AI after '801'",
    invalidAIAfter802 = "invalid AI after '802'",
    invalidAIAfter80 = "invalid AI after '80'",
    invalidAIAfter810 = "invalid AI after '810'",
    invalidAIAfter811 = "invalid AI after '811'",
    invalidAIAfter81 = "invalid AI after '81'",
    invalidAIAfter82 = "invalid AI after '82'",
    invalidAIAfter8 = "invalid AI after '8'",
    invalidAIAfter9 = "invalid AI after '9'",
    noValidAI = 'no valid AI',
    invalidNumber = 'invalid number',
    invalidFixedSizeInParse = 'invalid fixed size in parse',
    invalidVariableSizeInParse = 'invalid variable size in parse',
    invalidFixedSizeMeasureInParse = 'invalid fixed size measure in parse',
    invalidVariableSizeMeasureInParse = 'invalid variable size measure in parse',
    invalidVariableSizeWithISONumbersInParse = 'invalid variable size with ISO numbers in parse',
    invalidVariableSizeWithISOCharsInParse = 'invalid variable size with ISO chars in parse',
    invalidDateSizeInParse = 'invalid date size in parse',
}

/**
 * This is the main routine provided by the parse library. It takes a string,
 * splices it from left to right into its elements and tries to parse it as an
 * GS1 - element. If it succeeds, the result is returned as an object composed of
 * an identifier and an array.It accepts
 * @param   {String}   barcode is the contents of the barcode you'd like to get parsed
 * @returns {Array}    an array with elements which are objects of type "ParsedElement"
 */

/** @internal */
type ParsedElementAndCode = {
    element: ParsedElement;
    codeString: string;
};

export class BarcodeParserGs1 extends AbstractBarcodeParser {
    // Main function

    /* @internal */
    private readonly _fncChar: string;

    /* @internal */
    protected _codeString = '';

    /* @internal */
    private _codeStringLength = 0;

    constructor() {
        super();
        this._fncChar = String.fromCharCode(29); // the ASCII "group separator"
    }

    /**
     * String before change
     */
    private get codeString(): string {
        return this._codeString;
    }

    private set codeString(codeString: string) {
        this._codeString = codeString;
        this._codeStringLength = codeString.length;
    }

    /**
     * String length before change
     */
    private get codeStringLength(): number {
        return this._codeStringLength;
    }

    /**
     * Special separator of variable record
     */
    private get fncChar(): string {
        return this._fncChar;
    }

    /** Evaluate barcode received */
    public parse(barcode: string): ParsedBarcode {
        const barCodeLength = barcode.length;
        const answer: ParsedBarcode = {} as ParsedBarcode; // the object to return
        let restOfBarcode = ''; // the rest of the barcode, when first
        // elements are spliced away
        const symbologyIdentifier = barcode.slice(0, 3);
        let firstElement: ParsedElementAndCode = {} as ParsedElementAndCode;

        /**
         * =========== BEGIN of main routine ===================
         */

        /**
         *
         * ==== First step: ====
         *
         * IF there is any symbology identifier
         *   chop it off;
         *   put as "codeName" into the answer;
         *   fill restOfBarcode with the rest
         *   after the symbology identifier;
         * ELSE
         *   leave "codeName" empty;
         *   put the whole barcode into restOfBarcode;
         */

        if (barcode.startsWith(this.fncChar)) {
            answer.codeName = 'GS1-128';
            restOfBarcode = barcode.slice(1, barCodeLength);
        } else {
            switch (symbologyIdentifier) {
                case ']C1':
                    answer.codeName = 'GS1-128';
                    restOfBarcode = barcode.slice(3, barCodeLength);
                    break;
                case ']e0':
                    answer.codeName = 'GS1 DataBar';
                    restOfBarcode = barcode.slice(3, barCodeLength);
                    break;
                case ']e1':
                case ']e2':
                    answer.codeName = 'GS1 Composite';
                    restOfBarcode = barcode.slice(3, barCodeLength);
                    break;
                case ']d2':
                    answer.codeName = 'GS1 DataMatrix';
                    restOfBarcode = barcode.slice(3, barCodeLength);
                    break;
                case ']Q3':
                    answer.codeName = 'GS1 QR Code';
                    restOfBarcode = barcode.slice(3, barCodeLength);
                    break;
                default:
                    answer.codeName = '';
                    restOfBarcode = barcode;
                    break;
            }
        }

        /**
         * we have chopped off any symbology identifier. Now we can
         * try to parse the rest. It should give us an array of
         * ParsedElements.
         */

        /**
         * ===== Second step: ====
         *
         * Parse "barcode" data element by data element using
         * identifyAI.
         *
         */

        answer.parsedCodeItems = [];

        /**
         * The following part calls "identifyAI" in a loop, until
         * the whole barcode is parsed (or an error occurs).
         *
         * It uses the following strategy:
         *
         *   try to parse the part after the symbology identifier:
         *   - identify the first AI;
         *   - make a parsed element from the part after the AI;
         *   - append the parsed element to answer;
         *   - chop off the parsed part;
         *  do so while there is left something to parse;
         */

        while (restOfBarcode.length > 0) {
            try {
                firstElement = this._identifyAI(restOfBarcode);
                restOfBarcode = firstElement.codeString;
                answer.parsedCodeItems.push(firstElement.element);
            } catch (e) {
                throw new SystemError(e.message, e);
            }
        }
        /**
         * ==== Third and last step: =====
         *
         */
        return answer;
    }

    // auxiliary functions

    /**
     * "ParsedElement" is the
     *
     * @constructor for ParsedElements, the components of the array returned by parse
     * @param {String} elementAI        the AI of the recognized element
     * @param {String} elementDataTitle the title of the element, i.e. its short description
     * @param {String} elementType      a one-letter string describing the type of the element.
     *                                  allowed values are
     *                                  "string" for strings,
     *                                  "numeric" for numbers and
     *                                  "date" for dates
     */

    /**
     *
     * ================== BEGIN of identifyAI =======================
     *
     * does the main work:
     *   what AI is in the beginning of the restOfBarcode?
     *     If identified:
     *       which function to call with
     *       which parameters to parse the element?[Description]]
     * @param   {String} codeString a string; the function tries to
     *                   identify an AI in the beginning of this string
     * @returns {Object} if it succeeds in identifying an AI the
     *                   ParsedElement is returned with, together with the
     *                   still unparsed rest of codeString.
     */

    private _identifyAI(codeString: string): ParsedElementAndCode {
        // find first identifier. AIs have a minimum length of 2
        // digits, some have 3, some even 4.
        const firstNumber = codeString.slice(0, 1);
        const secondNumber = codeString.slice(1, 2);
        let parsedElementAndCode: ParsedElementAndCode;
        this.codeString = codeString;

        /**
         *
         * ======= BEGIN of the big switch =======================
         *
         * and now a very big "switch", which tries to find a valid
         * AI within the first digits of the codeString.
         *
         * See the documentation for an explanation why it is made
         * this way (and not by some configuration file).
         */
        switch (firstNumber) {
            case '0':
                parsedElementAndCode = this._evaluateCodeStartWith0(secondNumber);
                break;
            case '1':
                parsedElementAndCode = this._evaluateCodeStartWith1(secondNumber);
                break;
            case '2':
                parsedElementAndCode = this._evaluateCodeStartWith2(secondNumber);
                break;
            case '3':
                parsedElementAndCode = this._evaluateCodeStartWith3(secondNumber);
                break;
            case '4':
                parsedElementAndCode = this._evaluateCodeStartWith4(secondNumber);
                break;
            // first digits 5 and 6 are not used
            case '7':
                parsedElementAndCode = this._evaluateCodeStartWith7(secondNumber);
                break;
            case '8':
                parsedElementAndCode = this._evaluateCodeStartWith8(secondNumber);
                break;
            case '9':
                parsedElementAndCode = this._evaluateCodeStartWith9(secondNumber);
                break;
            default:
                throw new SystemError(ErrorMessages.noValidAI);
        }

        /**
         *
         * ======= END of the big switch =======================
         *
         * now identifyAI has just to return the new
         * ParsedElement (create by one of the parsing
         * functions) and the (cleaned) rest of codeString.
         */

        return {
            element: parsedElementAndCode.element,
            codeString: this._cleanCodeString(parsedElementAndCode.codeString),
        } as ParsedElementAndCode;
    }

    private _evaluateCodeStartWith0(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;

        switch (secondNumber) {
            case '0':
                // SSCC (Serial Shipping Container Code)
                parsedElementAndCode = this._parseFixedLength('00', DataTitle.sscc, 18);
                break;
            case '1':
                // Global Trade Item Number (GTIN)
                parsedElementAndCode = this._parseFixedLength('01', DataTitle.gtin, 14);
                break;
            case '2':
                // GTIN of Contained Trade Items
                parsedElementAndCode = this._parseFixedLength('02', DataTitle.content, 14);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter0);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith1(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;

        switch (secondNumber) {
            case '0':
                // Batch or Lot Number
                parsedElementAndCode = this._parseVariableLength('10', DataTitle.batchLot, 20);
                break;
            case '1':
                // Production Date (YYMMDD)
                parsedElementAndCode = this._parseDate('11', DataTitle.prodDate);
                break;
            case '2':
                // Due Date (YYMMDD)
                parsedElementAndCode = this._parseDate('12', DataTitle.dueDate);
                break;
            case '3':
                // Packaging Date (YYMMDD)
                parsedElementAndCode = this._parseDate('13', DataTitle.packDate);
                break;
            // AI "14" isn't defined
            case '5':
                // Best Before Date (YYMMDD)
                parsedElementAndCode = this._parseDate('15', DataTitle.bestBefore);
                break;
            case '6':
                // Sell By Date (YYMMDD)
                parsedElementAndCode = this._parseDate('16', DataTitle.sellBy);
                break;
            case '7':
                // Expiration Date (YYMMDD)
                parsedElementAndCode = this._parseDate('17', DataTitle.expiryDate);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter1);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith2(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;

        switch (secondNumber) {
            case '0':
                // Variant Number
                parsedElementAndCode = this._parseFixedLength('20', DataTitle.variant, 2);
                break;
            case '1':
                // Serial Number
                parsedElementAndCode = this._parseVariableLength('21', DataTitle.serial, 20);
                break;
            case '2':
                // Consumer product variant
                parsedElementAndCode = this._parseVariableLength('22', DataTitle.cpv, 20);
                break;
            // AI 23 is not defined
            case '4':
                // from now, the third number matters:
                parsedElementAndCode = this._evaluateCodeStartWith24(this.codeString.slice(2, 3));
                break;
            case '5':
                // from now, the third number matters:
                parsedElementAndCode = this._evaluateCodeStartWith25(this.codeString.slice(2, 3));
                break;
            // AI "26" to "29" aren't defined
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter2);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith24(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Additional Item Identification
                parsedElementAndCode = this._parseVariableLength('240', DataTitle.additionalId, 30);
                break;
            case '1':
                // Customer Part Number
                parsedElementAndCode = this._parseVariableLength('241', DataTitle.customPartNumber, 30);
                break;
            case '2':
                // Made-to-Order Variation Number
                parsedElementAndCode = this._parseVariableLength('242', DataTitle.mtoVariant, 6);
                break;
            case '3':
                // Packaging Component Number
                parsedElementAndCode = this._parseVariableLength('243', DataTitle.pcn, 20);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter24);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith25(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Secondary Serial Number
                parsedElementAndCode = this._parseVariableLength('250', 'SECONDARY SERIAL', 30);
                break;
            case '1':
                // Reference to Source Entity
                parsedElementAndCode = this._parseVariableLength('251', 'REF. TO SOURCE', 30);
                break;
            // AI "252" isn't defined
            case '3':
                // Global Document Type Identifier (GDTI)
                parsedElementAndCode = this._parseVariableLength('253', 'GDTI', 13 + 17);
                break;
            case '4':
                // GLN Extension Component
                parsedElementAndCode = this._parseVariableLength('254', 'GLN EXTENSION COMPONENT', 20);
                break;
            case '5':
                // Global Coupon Number (GCN)
                parsedElementAndCode = this._parseVariableLength('255', 'GCN', 13 + 12);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter25);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith3(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;

        switch (secondNumber) {
            case '0':
                // Count of Items (Variable Measure Trade Item)
                parsedElementAndCode = this._parseVariableLength('30', DataTitle.varCount, 8);
                break;
            case '1':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith31(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '2':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith32(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '3':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith33(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '4':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith34(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );

                break;
            case '5':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith35(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '6':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith36(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '7':
                // Count of Trade Items
                parsedElementAndCode = this._parseVariableLength('37', 'COUNT', 8);
                break;
            // AI "38" isn't defined
            case '9':
                // third and fourth numbers matter:
                parsedElementAndCode = this._evaluateCodeStartWith39(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter3);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith31(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Net weight, kilograms (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '310',
                    fourthNumber,
                    DataTitle.netWeight,
                    UnitOfMeasure.KILOGRAM,
                );
                break;
            case '1':
            // 1 - Length or first dimension, metres (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '2':
            // 2 - Width, diameter, or second dimension, metres (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '3':
                // 3 - Depth, thickness, height, or third dimension, metres (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `31${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.METER,
                );
                break;

            case '4':
                // Area, square metres (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '314',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_METER,
                );
                break;
            case '5':
                // Net volume, litres (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '315',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.LITER,
                );
                break;
            case '6':
                // Net volume, cubic metres (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '316',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.CUBIC_METER,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter31);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith32(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Net weight, pounds (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '320',
                    fourthNumber,
                    DataTitle.netWeight,
                    UnitOfMeasure.POUND,
                );
                break;
            case '1':
            // 1 - Length or first dimension, inches (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '4':
            // 4 - Width, diameter, or second dimension, inches (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '7':
                // 7 - Depth, thickness, height, or third dimension, inches (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `32${thirdNumber}`,
                    fourthNumber,
                    DataTitle.width,
                    UnitOfMeasure.INCH,
                );
                break;
            case '2':
            // 2 - Length or first dimension, feet (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '5':
            // 5 - Width, diameter, or second dimension, feet (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '8':
                // 8 - Depth, thickness, height, or third dimension, feet (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `32${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.FOOT,
                );
                break;
            case '3':
            // 3 - Length or first dimension, yards (Variable Measure Trade Item)
            // eslint-disable-next-line no-fallthrough
            case '6':
            // 6 - Width, diameter, or second dimension, yards (Variable Measure Trade Item
            // eslint-disable-next-line no-fallthrough
            case '9':
                // 9 - Depth, thickness, height, or third dimension, yards (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `32${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.YARD,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter32);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith33(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Logistic weight, kilograms
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '330',
                    fourthNumber,
                    DataTitle.grossWeight,
                    UnitOfMeasure.KILOGRAM,
                );
                break;
            case '1':
                // Length or first dimension, metres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '331',
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.METER,
                );
                break;
            case '2':
                // Width, diameter, or second dimension, metres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '332',
                    fourthNumber,
                    DataTitle.width,
                    UnitOfMeasure.METER,
                );
                break;
            case '3':
                // Depth, thickness, height, or third dimension, metres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '333',
                    fourthNumber,
                    DataTitle.height,
                    UnitOfMeasure.METER,
                );
                break;
            case '4':
                // Area, square metres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '334',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_METER,
                );
                break;
            case '5':
                // Logistic volume, litres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '335',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.LITER,
                );
                break;
            case '6':
                // Logistic volume, cubic metres
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '336',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_METER,
                );
                break;
            case '7':
                // Kilograms per square metre, yes, the ISO code for this _is_ "28".
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '337',
                    fourthNumber,
                    DataTitle.kgPerSquareMeter,
                    UnitOfMeasure._28,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter33);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith34(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // 340 - Logistic weight, pounds
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '340',
                    fourthNumber,
                    DataTitle.grossWeight,
                    UnitOfMeasure.POUND,
                );
                break;
            case '1':
            // 341 - Length or first dimension, inches
            // eslint-disable-next-line no-fallthrough
            case '4':
            // 344 - Width, diameter, or second dimension, inches
            // eslint-disable-next-line no-fallthrough
            case '7':
                // 347 - Depth, thickness, height, or third dimension, inches

                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `34${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.INCH,
                );
                break;
            case '2':
            // 342 - Length or first dimension, feet
            // eslint-disable-next-line no-fallthrough
            case '5':
            // 345 - Width, diameter, or second dimension, feet
            // eslint-disable-next-line no-fallthrough
            case '8':
                // 348 - Depth, thickness, height, or third dimension, feet
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `34${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.FOOT,
                );
                break;
            case '3':
            // 343 - Length or first dimension, yards
            // eslint-disable-next-line no-fallthrough
            case '6':
            // 346 - Width, diameter, or second dimension, yard
            // eslint-disable-next-line no-fallthrough
            case '9':
                // 349 - Depth, thickness, height, or third dimension, yards
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    `34${thirdNumber}`,
                    fourthNumber,
                    DataTitle.length,
                    UnitOfMeasure.YARD,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter34);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith35(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Area, square inches (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '350',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_INCH,
                );
                break;
            case '1':
                // Area, square feet (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '351',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_FOOT,
                );
                break;
            case '2':
                // Area, square yards (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '352',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_YARD,
                );
                break;
            case '3':
                // Area, square inches
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '353',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_INCH,
                );
                break;
            case '4':
                // Area, square feet
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '354',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_FOOT,
                );
                break;
            case '5':
                // Area, square yards
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '355',
                    fourthNumber,
                    DataTitle.area,
                    UnitOfMeasure.SQUARE_YARD,
                );
                break;
            case '6':
                // Net weight, troy ounces (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '356',
                    fourthNumber,
                    DataTitle.netWeight,
                    UnitOfMeasure.TROY_OUNCES,
                );
                break;
            case '7':
                // Net weight (or volume), ounces (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '357',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.OUNCE,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter35);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith36(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Net volume, quarts (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '360',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.QUART,
                );
                break;
            case '1':
                // Net volume, gallons U.S. (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '361',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.GALLON,
                );
                break;
            case '2':
                // Logistic volume, quarts
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '362',
                    fourthNumber,
                    DataTitle.netVolume,
                    UnitOfMeasure.QUART,
                );
                break;
            case '3':
                // Logistic volume, gallons U.S.
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '363',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.GALLON,
                );
                break;
            case '4':
                // Net volume, cubic inches (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '364',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_INCH,
                );
                break;
            case '5':
                // Net volume, cubic feet (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '365',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_FOOT,
                );
                break;
            case '6':
                // Net volume, cubic yards (Variable Measure Trade Item)
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '366',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_YARD,
                );
                break;
            case '7':
                // Logistic volume, cubic inches
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '367',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_INCH,
                );
                break;
            case '8':
                // Logistic volume, cubic feet
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '368',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_FOOT,
                );
                break;
            case '9':
                // Logistic volume, cubic yards
                parsedElementAndCode = this._parseFixedLengthMeasure(
                    '369',
                    fourthNumber,
                    DataTitle.volume,
                    UnitOfMeasure.CUBIC_YARD,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter36);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith39(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Applicable Amount Payable, local currency
                parsedElementAndCode = this._parseVariableLengthMeasure('390', fourthNumber, DataTitle.amount, 15);
                break;
            case '1':
                // Applicable Amount Payable with ISO Currency Code
                parsedElementAndCode = this._parseVariableLengthWithISONumbers(
                    '391',
                    fourthNumber,
                    DataTitle.amount,
                    15,
                );
                break;
            case '2':
                // Applicable Amount Payable, single monetary area (Variable Measure Trade Item)
                parsedElementAndCode = this._parseVariableLengthMeasure('392', fourthNumber, DataTitle.price, 15);
                break;
            case '3':
                // Applicable Amount Payable with ISO Currency Code (Variable Measure Trade Item)
                parsedElementAndCode = this._parseVariableLengthWithISONumbers(
                    '393',
                    fourthNumber,
                    DataTitle.price,
                    15,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter39);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith4(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (secondNumber) {
            case '0':
                // third number matters:
                parsedElementAndCode = this._evaluateCodeStartWith40(this.codeString.slice(2, 3));
                break;
            case '1':
                // third number matters:
                parsedElementAndCode = this._evaluateCodeStartWith41(this.codeString.slice(2, 3));
                break;
            case '2':
                // third number matters:
                parsedElementAndCode = this._evaluateCodeStartWith42(this.codeString.slice(2, 3));
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter4);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith40(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Customer's Purchase Order Number
                parsedElementAndCode = this._parseVariableLength('400', DataTitle.orderNumber, 30);
                break;
            case '1':
                // Global Identification Number for Consignment (GINC)
                parsedElementAndCode = this._parseVariableLength('401', DataTitle.ginc, 30);
                break;
            case '2':
                // Global Shipment Identification Number (GSIN)
                parsedElementAndCode = this._parseVariableLength('402', DataTitle.gsin, 17); // should be 17 digits long
                break;
            case '3':
                // Routing Code
                parsedElementAndCode = this._parseVariableLength('403', DataTitle.route, 30);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter40);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith41(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Ship to - Deliver to Global Location Number
                parsedElementAndCode = this._parseFixedLength('410', DataTitle.shipToLoc, 13);
                break;
            case '1':
                // Bill to - Invoice to Global Location Number
                parsedElementAndCode = this._parseFixedLength('411', DataTitle.billTo, 13);
                break;
            case '2':
                // Purchased from Global Location Number
                parsedElementAndCode = this._parseFixedLength('412', DataTitle.purchaseFrom, 13);
                break;
            case '3':
                // Ship for - Deliver for - Forward to Global Location Number
                parsedElementAndCode = this._parseFixedLength('413', DataTitle.shipForLoc, 13);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter41);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith42(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // Ship to - Deliver to Postal Code Within a Single Postal Authority
                parsedElementAndCode = this._parseVariableLength('420', DataTitle.shipToPost, 20);
                break;
            case '1':
                // Ship to - Deliver to Postal Code with ISO Country Code
                parsedElementAndCode = this._parseVariableLengthWithISOChars('421', DataTitle.shipToPost, 9);
                break;
            case '2':
                // Country of Origin of a Trade Item
                parsedElementAndCode = this._parseFixedLength('422', DataTitle.origin, 3);
                break;
            case '3':
                // Country of Initial Processing
                // Up to 5 3-digit ISO-countryCodes
                parsedElementAndCode = this._parseVariableLength('423', DataTitle.countryInitialProcess, 3 + 12);
                break;
            case '4':
                // Country of Processing
                parsedElementAndCode = this._parseFixedLength('424', DataTitle.countryProcess, 3);
                break;
            case '5':
                // Country of Disassembly
                parsedElementAndCode = this._parseFixedLength('425', DataTitle.countryDisassembly, 3);
                break;
            case '6':
                // Country Covering full Process Chain
                parsedElementAndCode = this._parseFixedLength('426', DataTitle.countryFullProcess, 3);
                break;
            case '7':
                // Country Subdivision of Origin
                parsedElementAndCode = this._parseVariableLength('427', DataTitle.countrySubdivision, 3);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter42);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith7(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (secondNumber) {
            case '0':
                // third and fourth number matter:
                parsedElementAndCode = this._evaluateCodeStartWith70(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '1':
                parsedElementAndCode = this._evaluateCodeStartWith71(this.codeString.slice(2, 3));
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter7);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith70(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                parsedElementAndCode = this._evaluateCodeStartWith700(fourthNumber);
                break;
            // 1 and 2 are not used
            case '3':
                // Approval Number of Processor with ISO Country Code

                // Title and stem for parsing are build from 4th number:

                parsedElementAndCode = this._parseVariableLengthWithISOChars(
                    `703${fourthNumber}`,
                    `PROCESSOR # ${fourthNumber}`,
                    27,
                );
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter70);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith700(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (fourthNumber) {
            case '1':
                // NATO Stock Number (NSN)
                parsedElementAndCode = this._parseVariableLength('7001', DataTitle.nsn, 13); // should be 13 digits long
                break;
            case '2':
                // UN/ECE Meat Carcasses and Cuts Classification
                parsedElementAndCode = this._parseVariableLength('7002', DataTitle.meatCut, 30);
                break;
            case '3':
                // Expiration Date and Time
                parsedElementAndCode = this._parseVariableLength('7003', DataTitle.expiryTime, 10); // should be 10 digits long
                break;
            case '4':
                // Active Potency
                parsedElementAndCode = this._parseVariableLength('7004', DataTitle.activePotency, 4);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter700);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith71(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                // National Healthcare Reimbursement Number (NHRN) – Germany PZN
                parsedElementAndCode = this._parseVariableLength('710', 'NHRN PZN', 20);
                break;
            case '1':
                // National Healthcare Reimbursement Number (NHRN) – France CIP
                parsedElementAndCode = this._parseVariableLength('711', 'NHRN CIP', 20);
                break;
            case '2':
                // National Healthcare Reimbursement Number (NHRN) – Spain CN
                parsedElementAndCode = this._parseVariableLength('712', 'NHRN CN', 20);
                break;
            case '3':
                // National Healthcare Reimbursement Number (NHRN) – Brasil DRN
                parsedElementAndCode = this._parseVariableLength('713', 'NHRN DRN', 20);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter71);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith8(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (secondNumber) {
            case '0':
                parsedElementAndCode = this._evaluateCodeStartWith80(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '1':
                parsedElementAndCode = this._evaluateCodeStartWith81(
                    this.codeString.slice(2, 3),
                    this.codeString.slice(3, 4),
                );
                break;
            case '2':
                parsedElementAndCode = this._evaluateCodeStartWith82(this.codeString.slice(2, 3));
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter8);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith80(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                parsedElementAndCode = this._evaluateCodeStartWith800(fourthNumber);
                break;
            case '1':
                parsedElementAndCode = this._evaluateCodeStartWith801(fourthNumber);
                break;
            case '2':
                parsedElementAndCode = this._evaluateCodeStartWith802(fourthNumber);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter80);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith800(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (fourthNumber) {
            case '1':
                // Roll Products (Width, Length, Core Diameter, Direction, Splices)
                parsedElementAndCode = this._parseVariableLength('8001', 'DIMENSIONS', 14); // should be 14 digits long
                break;
            case '2':
                // Cellular Mobile Telephone Identifier
                parsedElementAndCode = this._parseVariableLength('8002', 'CMT No', 20);
                break;
            case '3':
                // Global Returnable Asset Identifier (GRAI)
                parsedElementAndCode = this._parseVariableLength('8003', 'GRAI', 14 + 16); // should contain at least 14 digits
                break;
            case '4':
                // Global Individual Asset Identifier (GIAI)
                parsedElementAndCode = this._parseVariableLength('8004', 'GIAI', 30);
                break;
            case '5':
                // Price Per Unit of Measure
                parsedElementAndCode = this._parseVariableLength('8005', 'PRICE PER UNIT', 6); // should be 6 digits long
                break;
            case '6':
                // Identification of the Components of a Trade Item
                parsedElementAndCode = this._parseVariableLength('8006', 'GCTIN', 14 + 2 + 2); // should be exactly 18 digits long
                break;
            case '7':
                // International Bank Account Number (IBAN)
                parsedElementAndCode = this._parseVariableLength('8007', 'IBAN', 34);
                break;
            case '8':
                // Date and Time of Production
                parsedElementAndCode = this._parseVariableLength('8008', 'PROD TIME', 8 + 4); // should be exactly 12 digits long
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter800);
        }

        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith801(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (fourthNumber) {
            case '0':
                // Component / Part Identifier (CPID)
                parsedElementAndCode = this._parseVariableLength('8010', 'CPID', 30);
                break;
            case '1':
                // Component / Part Identifier Serial Number (CPID SERIAL)
                parsedElementAndCode = this._parseVariableLength('8011', 'CPID SERIAL', 12);
                break;
            case '7':
                // Global Service Relation Number to identify the relationship between an organisation offering services and the provider of services
                parsedElementAndCode = this._parseVariableLength('8017', 'GSRN - PROVIDER', 18); // should be 18 digits long
                break;
            case '8':
                // Global Service Relation Number to identify the relationship between an organization offering services and the recipient of services
                parsedElementAndCode = this._parseVariableLength('8018', 'GSRN - RECIPIENT', 18); // should be 18 digits long
                break;
            case '9':
                // Service Relation Instance Number (SRIN)
                parsedElementAndCode = this._parseVariableLength('8019', 'SRIN', 10);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter801);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith802(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        if (fourthNumber === '0') {
            // Payment Slip Reference Number
            parsedElementAndCode = this._parseVariableLength('8020', 'REF No', 25);
        } else {
            throw new SystemError(ErrorMessages.invalidAIAfter802);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith81(thirdNumber: string, fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (thirdNumber) {
            case '0':
                parsedElementAndCode = this._evaluateCodeStartWith810(fourthNumber);
                break;
            case '1':
                parsedElementAndCode = this._evaluateCodeStartWith811(fourthNumber);
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter81);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith810(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        switch (fourthNumber) {
            case '0':
                // GS1-128 Coupon Extended Code
                parsedElementAndCode = this._parseVariableLength('8100', '-', 6); // should be 6 digits long
                break;
            case '1':
                // GS1-128 Coupon Extended Code
                parsedElementAndCode = this._parseVariableLength('8101', '-', 10); // should be 10 digits long
                break;
            case '2':
                // GS1-128 Coupon Extended Code
                parsedElementAndCode = this._parseVariableLength('8102', '-', 2); // should be 2 digits long
                break;
            default:
                throw new SystemError(ErrorMessages.invalidAIAfter810);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith811(fourthNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        if (fourthNumber === '0') {
            // Coupon Code Identification for Use in North America
            parsedElementAndCode = this._parseVariableLength('8110', '-', 70);
        } else {
            throw new SystemError(ErrorMessages.invalidAIAfter811);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith82(thirdNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        if (thirdNumber === '0') {
            // Extended Packaging URL
            parsedElementAndCode = this._parseVariableLength('8200', 'PRODUCT URL', 70);
        } else {
            throw new SystemError(ErrorMessages.invalidAIAfter82);
        }
        return parsedElementAndCode;
    }

    private _evaluateCodeStartWith9(secondNumber: string): ParsedElementAndCode {
        let parsedElementAndCode: ParsedElementAndCode;
        if (secondNumber === '0') {
            // Information Mutually Agreed Between Trading Partners
            parsedElementAndCode = this._parseVariableLength('90', 'INTERNAL', 30);
        } else if (secondNumber >= '1' && secondNumber <= '9') {
            // Company Internal Information
            parsedElementAndCode = this._parseVariableLength(`9${secondNumber}`, 'INTERNAL', 90);
        } else {
            throw new SystemError(ErrorMessages.invalidAIAfter9);
        }

        return parsedElementAndCode;
    }

    /**
     *
     * =========== END of identifyAI =======================
     *
     */

    /**
     * ============ auxiliary functions for identifyAI =============
     */

    /**
     * some data items are followed by an ParsedElementAndCode  even in case of
     * fixed length, so the codeStringToReturn may have
     * leading FNCs.
     *
     * This function eliminates these leading FNCs.
     *
     * @param   {String} stringToClean string which has to be cleaned
     * @returns {String} the cleaned string
     */
    private _cleanCodeString(stringToClean: string): string {
        //
        let firstChar = stringToClean.slice(0, 1);
        let currentStringToClean = stringToClean;
        while (firstChar === this.fncChar) {
            currentStringToClean = currentStringToClean.slice(1, currentStringToClean.length);
            firstChar = currentStringToClean.slice(0, 1);
        }
        return currentStringToClean;
    }

    /**
     * Used for calculating numbers which are given as string
     * with a given number of fractional decimals.
     *
     * To avoid conversion errors binary decimal I _don't_
     * just divide by 10 numberOfFractionals times.
     *
     * Parse a string with given fractional to float
     * @param {string} stringToParse
     * @param {number} numberOfFractionals
     * @returns {number} value
     */
    private static _parseFloatingPoint(stringToParse: string, numberOfFractionals: number): number {
        let auxFloat = 0.0;

        try {
            const offset = stringToParse.length - numberOfFractionals;
            const auxString = `${stringToParse.slice(0, offset)}.${stringToParse.slice(offset, stringToParse.length)}`;
            auxFloat = parseFloat(auxString);
        } catch (_e) {
            throw new SystemError(ErrorMessages.invalidNumber);
        }

        return auxFloat;
    }

    /**
     * ======== END of auxiliary function for identifyAI =======
     */

    /**
     *
     * ======== BEGIN of parsing privates in identifyAI =======
     *
     * Some privates to parse the various GS1 formats. They
     * create a new ParsedElement and set its properties.
     *
     * They all modify the variables "elementToReturn" and
     * "codeStringToReturn".
     */

    /**
     * dates in GS1-elements have the format "YYMMDD".
     * This private generates a new ParsedElement and tries to fill a
     * JS-date into the "data"-part.
     * @param {String} ai    the AI to use for the ParsedElement
     * @param {String} title the title to use for the ParsedElement
     */
    private _parseDate(ai: string, title: DataTitle): ParsedElementAndCode {
        const elementParsed = new ParsedElement(ai, title, ElementTypeEnum.date);
        const offSet = ai.length;
        const dateYYMMDD = this.codeString.slice(offSet, offSet + 6);

        if (dateYYMMDD.length !== 6) {
            throw new SystemError(ErrorMessages.invalidDateSizeInParse);
        }

        elementParsed.data = dateXtrem.parse(dateYYMMDD, undefined, 'YYMMDD');

        const codeStringToReturn = this.codeString.slice(offSet + 6, this.codeStringLength);
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * simple: the element has a fixed length AND is not followed by an FNC1.
     * @param {String} ai     the AI to use
     * @param {String} title  its title, i.e. its short description
     * @param {Number} length the fixed length
     */
    private _parseFixedLength(ai: string, title: DataTitle, length: number): ParsedElementAndCode {
        const elementParsed = new ParsedElement(ai, title, ElementTypeEnum.string);
        const offSet = ai.length;
        elementParsed.data = this.codeString.slice(offSet, length + offSet);
        const codeStringToReturn = this.codeString.slice(length + offSet, this.codeStringLength);
        if (elementParsed.data.length !== length) {
            throw new SystemError(ErrorMessages.invalidFixedSizeInParse);
        }
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * tries to parse an element of variable length
     * some fixed length AIs are terminated by FNC1, so this private
     * is used even for fixed length items
     * @param {String} ai           the AI to use
     * @param {String} title        its title, i.e. its short description
     * @param {number} maxLength    maximum data size allowed or 0
     * @return parsed element
     */
    private _parseVariableLength(ai: string, title: any, maxLength = 0): ParsedElementAndCode {
        //
        const elementParsed = new ParsedElement(ai, title, ElementTypeEnum.string);
        const offSet = ai.length;
        const posOfFNC = this.codeString.indexOf(this.fncChar);
        const codeStringLength = this.codeString.length;
        let codeStringToReturn = '';

        if (posOfFNC === -1) {
            // we've got the last element of the barcode
            elementParsed.data = this.codeString.slice(offSet, codeStringLength);
        } else {
            elementParsed.data = this.codeString.slice(offSet, posOfFNC);
            codeStringToReturn = this.codeString.slice(posOfFNC + 1, codeStringLength);
        }
        if (maxLength && elementParsed.data.length > maxLength) {
            throw new SystemError(ErrorMessages.invalidVariableSizeInParse);
        }
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * the place of the decimal fraction is given by the fourth number, that's
     * the first after the identifier itself.
     *
     * All of theses elements have a length of 6 characters.
     * @param {String} aiStem       the first digits of the AI, _not_ the fourth digit
     * @param {Number} currentFourthNumber the 4th number indicating the count of valid fractionals
     * @param {String} title        the title of the AI
     * @param {String} unit         often these elements have an implicit unit of measurement
     */
    private _parseFixedLengthMeasure(
        aiStem: string,
        currentFourthNumber: string,
        title: DataTitle,
        unit: UnitOfMeasure,
    ): ParsedElementAndCode {
        //
        const elementParsed = new ParsedElement(aiStem + currentFourthNumber, title, ElementTypeEnum.number);
        const offSet = aiStem.length + 1;
        const numberOfDecimals = parseInt(currentFourthNumber, 10);
        const numberPart = this.codeString.slice(offSet, offSet + 6);

        elementParsed.data = BarcodeParserGs1._parseFloatingPoint(numberPart, numberOfDecimals);

        elementParsed.unit = unit;
        const codeStringToReturn = this.codeString.slice(offSet + 6, this.codeString.length);

        if (elementParsed.data !== 6) {
            throw new SystemError(ErrorMessages.invalidFixedSizeMeasureInParse);
        }
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * parses data elements of variable length, which additionally have
     *
     * - an indicator for the number of valid decimals
     * - an implicit unit of measurement
     *
     * These data elements contain e.g. a weight or length.
     *
     */
    private _parseVariableLengthMeasure(
        aiStem: string,
        currentFourthNumber: string,
        title: DataTitle,
        maxLength = 0,
        unit?: UnitOfMeasure,
    ): ParsedElementAndCode {
        // the place of the decimal fraction is given by the fourth number, that's
        // the first after the identifier itself.
        const elementParsed = new ParsedElement(aiStem + currentFourthNumber, title, ElementTypeEnum.number);
        const offSet = aiStem.length + 1;
        const posOfFNC = this.codeString.indexOf(this.fncChar);
        const numberOfDecimals = parseInt(currentFourthNumber, 10);
        let codeStringToReturn = '';
        let numberPart = '';

        if (posOfFNC === -1) {
            numberPart = this.codeString.slice(offSet, this.codeStringLength);
        } else {
            numberPart = this.codeString.slice(offSet, posOfFNC);
            codeStringToReturn = this.codeString.slice(posOfFNC + 1, this.codeStringLength);
        }

        if (maxLength && numberPart.length > maxLength) {
            throw new SystemError(ErrorMessages.invalidVariableSizeMeasureInParse);
        }

        // adjust decimals according to fourthNumber:

        elementParsed.data = BarcodeParserGs1._parseFloatingPoint(numberPart, numberOfDecimals);
        elementParsed.unit = unit;

        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * parses data elements of variable length, which additionally have
     *
     * - an indicator for the number of valid decimals
     * - an explicit unit of measurement
     *
     * These data element contain amounts to pay or prices.
     *
     */
    private _parseVariableLengthWithISONumbers(
        aiStem: string,
        currentFourthNumber: string,
        title: DataTitle,
        maxLength = 0,
    ): ParsedElementAndCode {
        // an element of variable length, representing a number, followed by
        // some ISO-code.
        const elementParsed = new ParsedElement(aiStem + currentFourthNumber, title, ElementTypeEnum.number);
        const offSet = aiStem.length + 1;
        const posOfFNC = this.codeString.indexOf(this.fncChar);
        const numberOfDecimals = parseInt(currentFourthNumber, 10);
        let codeStringToReturn = '';
        let isoPlusNumbers = '';
        let numberPart = '';

        if (posOfFNC === -1) {
            isoPlusNumbers = this.codeString.slice(offSet, this.codeStringLength);
        } else {
            isoPlusNumbers = this.codeString.slice(offSet, posOfFNC);
            codeStringToReturn = this.codeString.slice(posOfFNC + 1, this.codeStringLength);
        }
        // cut off ISO-Code
        numberPart = isoPlusNumbers.slice(3, isoPlusNumbers.length);
        elementParsed.data = BarcodeParserGs1._parseFloatingPoint(numberPart, numberOfDecimals);

        elementParsed.unit = isoPlusNumbers.slice(0, 3);
        if (maxLength && isoPlusNumbers.length > maxLength) {
            throw new SystemError(ErrorMessages.invalidVariableSizeWithISONumbersInParse);
        }
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }

    /**
     * parses data elements of variable length, which additionally have
     *
     * - an explicit unit of measurement or reference
     *
     * These data element contain countries, authorities within countries.
     *
     */
    private _parseVariableLengthWithISOChars(aiStem: string, title: any, maxLength = 0): ParsedElementAndCode {
        // an element of variable length, representing a sequence of chars, followed by
        // some ISO-code.
        const elementParsed = new ParsedElement(aiStem, title, ElementTypeEnum.string);
        const offSet = aiStem.length;
        const posOfFNC = this.codeString.indexOf(this.fncChar);
        let codeStringToReturn = '';
        let isoPlusNumbers = '';

        if (posOfFNC === -1) {
            isoPlusNumbers = this.codeString.slice(offSet, this.codeStringLength);
        } else {
            isoPlusNumbers = this.codeString.slice(offSet, posOfFNC);
            codeStringToReturn = this.codeString.slice(posOfFNC + 1, this.codeStringLength);
        }
        // cut off ISO-Code
        elementParsed.data = isoPlusNumbers.slice(3, isoPlusNumbers.length);
        elementParsed.unit = isoPlusNumbers.slice(0, 3);
        if (maxLength && elementParsed.data.length > maxLength) {
            throw new SystemError(ErrorMessages.invalidVariableSizeWithISOCharsInParse);
        }
        return { element: elementParsed, codeString: codeStringToReturn } as ParsedElementAndCode;
    }
    /**
     *
     * ======== END of parsing privates in identifyAI =======
     *
     */
}
