import { DataTypeSupported, ElementType, ElementTypeEnum, UnitTypeSupported } from './parsed-generic-element';

export type TypeOfBarcodeParser = 'gs1' | 'composite';

export enum CompositeDateFormat {
    ddmmyy = 'ddmmyy',
    ddmmyyyy = 'ddmmyyyy',
    yymmdd = 'yymmdd',
    yyyymmdd = 'yyyymmdd',
    yyyy_mm_dd = 'yyyy_mm_dd',
}

export enum CompositeFormat {
    delimited = 'delimited',
    fixedLength = 'fixed',
}

export interface ParserConfigurationLine {
    fieldKey: string;
    fieldAlias?: string;
    fieldDataType: DataTypeSupported;
    description?: string;
    startPosition?: number;
    length?: number;
}

export interface ParserConfiguration {
    typeOfBarcodeParser: TypeOfBarcodeParser;
    compositeCode: string;
    localizedDescription?: string;
    isBlankDeletion: boolean;
    compositeFormat: CompositeFormat;
    dateFormat: CompositeDateFormat;
    dateLabel: string;
    fieldSeparator: string;
    decimalSeparator: string;
    recordSize: number;
    numberOfAlias: number;
    lines: ParserConfigurationLine[];
}

/* getDateFormatSize returns the size of the date format
 * @param dateFormat: CompositeDateFormat - the date format
 * @returns number - the size of the date format
 */
export function getDateFormatSize(dateFormat: CompositeDateFormat): number {
    switch (dateFormat) {
        case CompositeDateFormat.ddmmyy:
        case CompositeDateFormat.yymmdd:
            return 6;
        case CompositeDateFormat.ddmmyyyy:
        case CompositeDateFormat.yyyymmdd:
            return 8;
        case CompositeDateFormat.yyyy_mm_dd:
            return 10;
        default:
            return 0;
    }
}

/* getDateFormat returns the date format
 * @param dateFormat: CompositeDateFormat - the date format
 * @returns string - the date format
 */
export function getDateFormat(dateFormat: CompositeDateFormat): string {
    switch (dateFormat) {
        case CompositeDateFormat.ddmmyy:
            return 'DDMMYY';
        case CompositeDateFormat.ddmmyyyy:
            return 'DDMMYYYY';
        case CompositeDateFormat.yymmdd:
            return 'YYMMDD';
        case CompositeDateFormat.yyyymmdd:
            return 'YYYYMMDD';
        case CompositeDateFormat.yyyy_mm_dd:
            return 'YYYY-MM-DD';
        default:
            return 'YYYYMMDD';
    }
}

/*
 * ParseGenericBarcode represents the parsed barcode data
 * @param codeName: string - the name of the barcode
 * @param parsedCodeItems: ParsedGenericElement[] - the parsed elements of the barcode
 */

export interface ParsedGenericBarcode {
    codeName: string;
    parsedCodeItems: ParsedGenericElement[];
}

/**
 * ParsedGenericElement class represents a single element of the barcode data
 * It is common for compatibility reasons with its GS1 counterpart except that the elementDataTitle is a string.
 */
export class ParsedGenericElement {
    private _data: DataTypeSupported;

    private _unit: UnitTypeSupported;

    constructor(
        public readonly ai: string,
        public readonly elementDataTitle: string,
        public readonly elementType: ElementType,
    ) {
        this._defineData(elementType);
    }

    protected _defineData(elementType: ElementType): void {
        /* defines the object which represents a single element
         */
        switch (elementType) {
            case ElementTypeEnum.string:
                this._data = '';
                break;
            case ElementTypeEnum.number:
                this._data = 0;
                break;
            case ElementTypeEnum.date:
                this._data = '';
                break;
            default:
                this._data = '';
                break;
        }
        this.unit = undefined; // some elements are accompanied by an unit of
    }

    set unit(unit: UnitTypeSupported) {
        this._unit = unit;
    }

    get unit(): UnitTypeSupported {
        return this._unit;
    }

    set data(data: DataTypeSupported) {
        this._data = data;
    }

    get data(): DataTypeSupported {
        return this._data;
    }
}
