import { DateValue, date as dateXtrem } from '@sage/xtrem-date-time';
import { SystemError } from '@sage/xtrem-shared';
import { AbstractBarcodeParser } from './barcode-parser-abstract';
import { getDateFormat, getDateFormatSize, ParserConfigurationLine } from './barcode-parser-configuration';
import { ParsedBarcode } from './parsed-element';
import { DataTypeSupported, ElementTypeEnum, getElementType, ParsedGenericElement } from './parsed-generic-element';

export class BarcodeParserComposite extends AbstractBarcodeParser {
    parse(barCode: string): ParsedBarcode {
        try {
            const _parserConfiguration = this.parserConfiguration;

            if (!_parserConfiguration) {
                throw new Error('Parser configuration is missing');
            }

            const _recordSize = _parserConfiguration.recordSize;

            if (!_recordSize) {
                throw new Error('Record size is missing');
            }

            if (barCode.length < _recordSize) {
                throw new Error(
                    `Barcode length is not valid, expected minimum ${_recordSize} but got ${barCode.length}`,
                );
            }

            const _sizeDate = getDateFormatSize(_parserConfiguration.dateFormat);
            const _dateFormat = getDateFormat(_parserConfiguration.dateFormat);

            if (_dateFormat.length !== _sizeDate) {
                throw new Error(`Date format size  ${_dateFormat} is not valid`);
            }

            let _parsedCodeItems = <ParsedGenericElement[]>[];

            // Use string comparison to avoid enum comparison issues
            const compositeFormat = String(_parserConfiguration.compositeFormat);
            if (compositeFormat === 'fixed') {
                _parsedCodeItems = this._parseFixed(barCode, _dateFormat, _sizeDate);
            } else if (compositeFormat === 'delimited') {
                _parsedCodeItems = this._parseDelimited(barCode, _dateFormat, _sizeDate);
            } else {
                throw new Error(`Composite format ${_parserConfiguration.compositeFormat} is not supported`);
            }

            return <ParsedBarcode>{
                codeName: _parserConfiguration.compositeCode ?? _parserConfiguration.localizedDescription ?? '',
                parsedCodeItems: _parsedCodeItems,
            };
        } catch (error) {
            throw new SystemError(`Parser error :\n${error}`, error);
        }
    }

    /*
     * Parse the fixed length barcode
     * @param barCode: string - the barcode to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns ParsedGenericElement[] - the parsed elements of the barcode
     */
    private _parseFixed(barCode: string, dateFormat: string, sizeDate: number): ParsedGenericElement[] {
        const _parsedCodeItems = <ParsedGenericElement[]>[];
        const _parserConfiguration = this.parserConfiguration;
        let _startPosition = 0;

        if (!_parserConfiguration) {
            return [];
        }

        const { isBlankDeletion } = _parserConfiguration;

        _parserConfiguration?.lines.forEach(line => {
            const _length = line.length || 0;
            const _fieldKey = line.fieldKey;
            if (_length === 0) {
                throw new Error(`Length is missing for field ${_fieldKey}`);
            }

            const _isDateField = line.fieldDataType === <DataTypeSupported>'date';
            if (_isDateField && _length < sizeDate) {
                throw new Error(
                    `Field size ${_fieldKey} is not valid, expected minimum ${sizeDate} but got ${_length}`,
                );
            }

            const _fieldSize = _isDateField ? Math.min(sizeDate, _length) : _length;

            const _endPosition = _startPosition + _length;
            const _value = barCode.substring(_startPosition, _startPosition + _fieldSize) || '';
            _startPosition = _endPosition;

            _parsedCodeItems.push(
                ...this._getParsedElement(line, isBlankDeletion ? _value.trim() : _value, dateFormat, sizeDate),
            );
        });

        return _parsedCodeItems;
    }

    /*
     * Parse the delimited barcode
     * @param barCode: string - the barcode to parse
     * @param _sizeDate: number - the size of the date
     *   @returns ParsedGenericElement[] - the parsed elements of the barcode
     */
    private _parseDelimited(barCode: string, dateFormat: string, sizeDate: number): ParsedGenericElement[] {
        const _parsedCodeItems = <ParsedGenericElement[]>[];
        const _parserConfiguration = this.parserConfiguration;
        if (!_parserConfiguration) {
            return [];
        }

        const { isBlankDeletion } = _parserConfiguration;
        const _fields = barCode.split(_parserConfiguration.fieldSeparator);
        if (_fields.length < _parserConfiguration.lines.length) {
            throw new Error(
                `Barcode fields count is not valid, expected ${_parserConfiguration.lines.length} but got ${_fields.length}`,
            );
        }

        for (let index = 0; index < _parserConfiguration.lines.length; index += 1) {
            const _line = _parserConfiguration.lines[index];
            const _isDateField = _line.fieldDataType === <DataTypeSupported>'date';
            const _value = (isBlankDeletion || _isDateField ? _fields[index]?.trim() : _fields[index]) || '';
            if (_isDateField && _value.length < sizeDate) {
                if (_value.length) {
                    throw new Error(
                        `Field size ${_line.fieldKey} is not valid, expected minimum ${sizeDate} but got ${_value.length}`,
                    );
                }

                // When the date is empty, we don't throw an error but we return an empty date
                const _elementType = getElementType(_line.fieldDataType);
                _parsedCodeItems.push(new ParsedGenericElement(_line.fieldKey, _line.fieldKey, _elementType));
                if (_line.fieldAlias) {
                    _parsedCodeItems.push(new ParsedGenericElement(_line.fieldAlias, _line.fieldAlias, _elementType));
                }
            } else {
                _parsedCodeItems.push(...this._getParsedElement(_line, _value, dateFormat, sizeDate));
            }
        }

        return _parsedCodeItems;
    }

    /*
     * Get the element type
     * @param line: ParserConfigurationLine - the line to parse
     * @param value: string - the value to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns ParsedGenericElement - the parsed element
     */
    private _getParsedElement(
        line: ParserConfigurationLine,
        value: string,
        dateFormat: string,
        sizeDate: number,
    ): ParsedGenericElement[] {
        const _elementType = getElementType(line.fieldDataType);
        const _parsedElement = new ParsedGenericElement(line.fieldKey, line.fieldKey, _elementType);
        let _parsedAliasElement: ParsedGenericElement | undefined;

        switch (_elementType) {
            case ElementTypeEnum.string:
                _parsedElement.data = value;
                break;
            case ElementTypeEnum.number:
                _parsedElement.data = BarcodeParserComposite._parseNumber(
                    line.fieldKey,
                    value,
                    this.parserConfiguration?.decimalSeparator,
                );
                break;
            case ElementTypeEnum.date:
                _parsedElement.data = BarcodeParserComposite._parseDate(line.fieldKey, value, dateFormat, sizeDate);
                break;
            default:
                throw new Error(`Element type ${_elementType} is not supported`);
        }

        if (line.fieldAlias) {
            _parsedAliasElement = new ParsedGenericElement(line.fieldAlias, line.fieldAlias, _elementType);
            _parsedAliasElement.data = _parsedElement.data;
        }

        return _parsedAliasElement ? [_parsedElement, _parsedAliasElement] : [_parsedElement];
    }

    /**
     * Parse the date
     * @param fieldKey: string - the field key
     * @param value: string - the value to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns string - the parsed date
     */
    private static _parseDate(fieldKey: string, value: string, dateFormat: string, sizeDate: number): DateValue {
        const dateToFormat = value.slice(0, sizeDate);

        if (!dateToFormat.length) {
            throw new Error(`Date  ${fieldKey} is empty`);
        }

        if (dateToFormat.length !== sizeDate) {
            throw new Error(`Date size ${fieldKey} is not valid, expected ${sizeDate} but got ${dateToFormat.length}`);
        }

        return dateXtrem.parse(dateToFormat, undefined, dateFormat);
    }

    /**
     * Parse the number
     * @param fieldKey: string - the field key
     * @param value: string - the value to parse
     * @param decimalSeparator: string - the decimal separator
     * @returns number - the parsed number
     */
    private static _parseNumber(fieldKey: string, value: string, decimalSeparator?: string): number {
        let _value = value;
        if (decimalSeparator && decimalSeparator !== '.') {
            _value = value.replace(decimalSeparator, '.');
        }
        const parsedNumber = parseFloat(_value);
        if (Number.isNaN(parsedNumber)) {
            throw new Error(`Number format ${fieldKey} is not valid, expected number but got ${value}`);
        }
        return parsedNumber;
    }
}
