import * as ui from '@sage/xtrem-ui';
import { BarcodeManagementServiceFactory } from './barcode-management-service-factory';
import { ManagementServiceGs1 } from './management-service-gs1';
import {
    AsyncCompositeAllowed,
    BarcodeManagerSupportedFields,
    DictionaryDataGs1,
    DictionaryFieldSupported,
} from './screen-management';
import { BarcodeManagementServiceInterface, TypeOfServiceBarcode } from './type-of-barcode-management';

const defaultManagementServiceKey = 'DEFAULT_MOBILE_COMPOSITE_DATA_KEY';
export class SupportManagementService<GraphqlApi = any, NodeType extends ui.ClientNode = any> extends ui.Page<
    GraphqlApi,
    NodeType
> {
    /**
     * Internal only
     */

    private _factory = new BarcodeManagementServiceFactory();

    /**
     *  Constructor of control manager GS 1
     * @param typeOfServiceBarCode type of service barcode
     */
    constructor() {
        super();
        this._factory = new BarcodeManagementServiceFactory();
        this._controlManager = new ManagementServiceGs1(defaultManagementServiceKey);
    }

    /**
     * Manager services for General Specifications One (GS 1).
     */

    /**
     * Current instance of management service
     * A temporary instance is created here to prevent any preliminary function calls
     * before the real initialization.
     *
     */
    /** @internal */
    private _controlManager: BarcodeManagementServiceInterface;

    /**
     * Create a new instance of service barcode
     * @param codeFlow optional flow code
     */
    /** @internal */
    private _createService(codeFlow?: string): Promise<BarcodeManagementServiceInterface> {
        return this._factory.createService(this as any, defaultManagementServiceKey, codeFlow);
    }

    /**
     * Used only for disable scan operation (service must stay active)
     */
    /** @internal */
    private _disableBarcodeService = false;

    /**
     * Initialize ControlManager :
     * Depending on the type of manager used (not Gs1), the dictionary and checkCompositeDataAllowed parameters
     * may be ignored because the factory retrieves the configuration to know whether or not to use them.
     * The flow code is mandatory in order to load the configuration, except in the case of Gs1.
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param codeFlow optional flow code (mandatory for non-Gs1)
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @returns true when ControlManager is usable
     */
    /** @internal */
    private async _createAndInitControlManager(
        site: string,
        compositePageKey: string,
        codeFlow: string | undefined,
        dictionaryFieldSupported: DictionaryFieldSupported,
        dictionaryDataComposites?: DictionaryDataGs1,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        /**
         * Create and initialize manager for a given service :
         *
         */
        this._controlManager = await this._createService(codeFlow);
        // Including erroneous field misspelled dateTestFake instead dateTest
        if (site && compositePageKey) {
            if (
                await this._controlManager
                    .initialize(
                        this as any,
                        dictionaryFieldSupported,
                        dictionaryDataComposites,
                        checkCompositeDataAllowed,
                    )
                    .catch(() => false)
            ) {
                return true;
            }
            // Error detected, abort controller
            ui.console.error(`Service initialization failure for site ${site}`);
        }
        // Initialization failure : reset to default instance (gs1)
        this._controlManager = await this._createService();
        return false;
    }

    /**
     * Remove composite data
     */
    /** @protected */
    protected clearCompositeData(): void {
        this._controlManager.clearCompositeData();
    }

    /**
     * Remove composite data from storage
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    protected clearCompositeDataStorage(storageKey?: string): void {
        this._controlManager.clearCompositeDataStorage(this as ui.Page, storageKey);
    }

    /**
     * Remove all composite and storage data
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    protected clearAllCompositeDataAndStorage(storageKey?: string): void {
        this._controlManager.clearAllCompositeDataAndStorage(this as ui.Page, storageKey);
    }

    /**
     * Get current control manager instance
     * @returns current instance of control manager
     */
    /** @protected */
    protected get controlManager(): BarcodeManagementServiceInterface {
        return this._controlManager;
    }

    /**
     * Create and initialize manager for a given service :
     * Depending on the type of manager used (not Gs1), the dictionary and checkCompositeDataAllowed parameters
     * may be ignored because the factory retrieves the configuration to know whether or not to use them.
     * The flow code is mandatory in order to load the configuration, except in the case of Gs1.
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param codeFlow optional flow code (mandatory for non-Gs1)
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @return false when service has not properly initialized (unavailable)
     */
    /** @protected */
    protected async createAndInitServiceManagement(
        site: string,
        compositePageKey: string,
        codeFlow: string | undefined,
        dictionaryFieldSupported: DictionaryFieldSupported,
        dictionaryDataComposites?: DictionaryDataGs1,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        if (
            !site ||
            !(await this._createAndInitControlManager(
                site,
                compositePageKey,
                codeFlow,
                dictionaryFieldSupported,
                dictionaryDataComposites,
                checkCompositeDataAllowed,
            ))
        ) {
            ui.console.error(`Unable to initialize service control manager for site ${site}`);
            return false;
        }
        return true;
    }

    /**
     * Disable or enable service (default : enable) :
     * Disabling clearing all composite data
     */
    /** @protected */
    protected set disableBarcodeService(disableServiceBarcode: boolean) {
        this._disableBarcodeService = disableServiceBarcode;
        if (this._disableBarcodeService) {
            this._controlManager.clearCompositeData();
        }
    }

    /**
     * Protected only
     */
    /**
     * Return current state of service
     */
    /** @protected */
    protected get isBarcodeServiceDisabled(): boolean {
        return this._disableBarcodeService;
    }

    /**
     * Load composite data from storage and erase them
     * @returns true when some data has been loaded
     */
    /** @protected */
    protected loadCompositeData(): boolean {
        return this._controlManager.loadCompositeData(this as ui.Page);
    }

    /**
     * Save composite data, zero elements allowed
     * @returns true when operation has performed
     */
    /** @protected */
    protected saveCompositeData(): boolean {
        return this._controlManager.saveCompositeData(this as ui.Page);
    }

    /**
     * Update screen fields supported (only when exists Gs1 parameters).
     * mapping field has been performed after
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    protected setScreenFieldSupported(dictionaryFieldSupported: DictionaryFieldSupported): Promise<boolean> {
        return Promise.resolve(this._controlManager.setScreenFieldSupported(this as ui.Page, dictionaryFieldSupported));
    }

    /**
     * call scan evaluator only when manager available and rawData has minimum requirement
     * @param mainField current screen field reference
     * @param rawData data to parse
     * @return false when is not a bar code composite
     */
    /** @protected */
    protected scan(mainField: BarcodeManagerSupportedFields, rawData: string): Promise<boolean> {
        // class must be created, initialized and not busy, data must be started by a numeric value,
        // minimum size 2 to 4 numerics or prefix ]<alphabetic><numeric>
        if (!(this._disableBarcodeService ?? false) && this._controlManager.isCompositeData(rawData)) {
            return this._controlManager.scan(this as ui.Page, mainField, rawData).catch(() => false);
        }
        return Promise.resolve(false);
    }

    /**
     * Return type of service barcode
     * @returns type of service barcode
     *
     */
    protected get typeOfServiceBarcode(): TypeOfServiceBarcode {
        return this._controlManager.typeOfServiceBarcode ?? <TypeOfServiceBarcode>'gs1';
    }
}
