/** Interfaces and types for management code GS 1 */

import { DateValue } from '@sage/xtrem-date-time';
import { integer } from '@sage/xtrem-shared';
import { fields } from '@sage/xtrem-ui';
import { DataTitle } from '../shared-functions/index';

// Incoming page fields definition

export type BarcodeManagerSupportedFields =
    | fields.Date
    | fields.DropdownList
    | fields.FilterSelect
    | fields.Numeric
    | fields.Reference
    | fields.Select
    | fields.Text;

/**
 * Check if the supported field support sound
 * @param field
 */
export function isSoundSupportedField(field: any): boolean {
    return [fields.Reference, fields.FilterSelect, fields.Select, fields.DropdownList].some(
        supportedFieldType => field instanceof supportedFieldType,
    );
}

/**
 * Check if the field is an instance of any type in BarcodeManagerSupportedFields
 * @param field
 */
export function isSupportedField(field: any): boolean {
    return [
        fields.Reference,
        fields.Date,
        fields.Text,
        fields.Numeric,
        fields.FilterSelect,
        fields.Select,
        fields.DropdownList,
    ].some(supportedFieldType => field instanceof supportedFieldType);
}

export type ElementTypeSupported = string | DateValue | number | undefined;

export type AsyncVoidFunction = () => Promise<void>;

export type VoidFunction = () => void;

export type AsyncAndVoidFunction = AsyncVoidFunction | VoidFunction;

export interface FieldSupported {
    mainField: BarcodeManagerSupportedFields;
    unitField?: BarcodeManagerSupportedFields;
}

// For applicative side
export type DictionaryFieldSupported = Record<DataTitle, FieldSupported>;
export type DictionaryGenericFieldSupported = Record<string, FieldSupported>;

// Internal page fields definition

/** @internal */
export interface FieldSegment {
    currentField: BarcodeManagerSupportedFields;
}

/**
 * This attribute is used for very order of segment into page...
 * when the value has not changed, this segment has unused
 */
/** @internal */
export interface SegmentForDictionary extends FieldSegment {
    sequence: integer;
}

/** @internal */
export type DictionarySegment = Record<string, SegmentForDictionary>;

// This expose only data during mapping operations
/** @internal */
export interface DataSegmentMapped extends FieldSegment {
    data: any;
}

// This expose only data during mapping operations

/** @internal */
export interface FieldDataType {
    DataTitle: string;
}

/**
 * This attribute is used for very order of field into page...
 * when the value has not changed, this field has unused
 * This interface store only minimum data..
 */
/** @internal */
export interface DataFieldMapped extends FieldDataType, FieldSupported {
    sequence: number;
    data?: ElementTypeSupported;
    unit?: ElementTypeSupported;
}

// Supported page fields definition without data

/**
 * This attribute is used for very order of field into page...
 * when the value has not changed, this field has unused
 */
/** @internal */
export interface ScreenFieldSupported extends FieldDataType {
    sequence: number;
    segments: FieldSegment[];
}

// Data composite

export interface DataGeneric {
    dataKey: string;
    data: ElementTypeSupported;
    unit?: ElementTypeSupported;
}

/**
 * This composite item is used in storage and service
 */
export interface DataGs1 extends DataGeneric {
    dataKey: DataTitle;
}

/**
 * All composite data have stored in dictionary.
 */
export type DictionaryDataGs1 = Record<DataTitle, DataGs1>;
export type DictionaryDataGeneric = Record<string, DataGeneric>;

/**
 * This asynchronous function signature is used only for specifying
 * a call back to check composite data before continuous
 * @param dictionaryDataComposite : current dictionary  (may be empty)
 * @param scannedFieldId : the field id that has been scanned
 * @return true when dispatching is allowing to continue action
 *         false when dispatching must clear all composites and exiting
 */
export type AsyncCompositeAllowed = (
    dictionaryDataComposite: DictionaryDataGs1 | DictionaryDataGeneric,
    scannedFieldId?: string,
) => Promise<boolean>;
