/** Service of management code GS 1 */
import type { GraphApi } from '@sage/wh-system-api';
import * as ui from '@sage/xtrem-ui';
import { BarcodeManagementService } from './barcode-management-service';
import {
    AsyncCompositeAllowed,
    BarcodeManagerSupportedFields,
    DictionaryDataGs1,
    DictionaryFieldSupported,
} from './screen-management';

export class ManagementServiceGs1 extends BarcodeManagementService {
    /**
     * This Method build instance with retrieving all data
     * @param pageInstance : current page
     * @param dictionaryFieldSupported : page fields supported
     * @param dictionaryDataComposites : extracted composite data, may be send to storage
     * @param checkCompositeDataAllowed? : optional client callback to check composite data code before to dispatch them
     */

    // eslint-disable-next-line require-await
    public override async initialize(
        pageInstance: ui.Page<GraphApi>,
        dictionaryFieldSupported: DictionaryFieldSupported = {} as DictionaryFieldSupported,
        dictionaryDataComposites: DictionaryDataGs1 = {} as DictionaryDataGs1,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        return super.initialize(
            pageInstance,
            ManagementServiceGs1._getDictionaryFieldSupported(dictionaryFieldSupported),
            dictionaryDataComposites,
            checkCompositeDataAllowed,
        );
    }

    /**
     * Check if raw data is a composite data
     * @param rawData
     * @returns
     */
    // eslint-disable-next-line class-methods-use-this
    public override isCompositeData(rawData: any): boolean {
        const groupSeparator = String.fromCharCode(0x1d); // GS1 Group Separator
        const compositeCodePrefix = new RegExp(
            `^(\\d{2,4}|\\][A-Za-z]\\d{3,5}|${groupSeparator.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\d{2,4})`,
        );
        return (
            rawData &&
            typeof rawData === 'string' &&
            rawData.length >= 4 &&
            !!Number(rawData.match(compositeCodePrefix)?.length ?? 0)
        );
    }

    /**
     * Update screen fields supported.
     * field mapping was done after
     * @param pageInstance current page
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    // eslint-disable-next-line require-await
    public override async setScreenFieldSupported(
        pageInstance: ui.Page<GraphApi>,
        dictionaryFieldSupported: DictionaryFieldSupported,
    ): Promise<boolean> {
        return super.setScreenFieldSupported(
            pageInstance,
            ManagementServiceGs1._getDictionaryFieldSupported(dictionaryFieldSupported),
        );
    }

    /**
     * Called when field update is in progress
     * @param pageInstance current page
     * @param currentField reference of field to updating
     * @param rawData received raw data
     * @param appendCompositeData true when is necessary to appending current date to previous set
     * @return true when is not necessary to continue and affect a value : composite value has been processed
     */
    public override scan(
        pageInstance: ui.Page<GraphApi>,
        currentField: BarcodeManagerSupportedFields,
        rawData: any,
        appendCompositeData: boolean = false,
    ): Promise<boolean> {
        return super.scan(pageInstance, currentField, rawData, appendCompositeData);
    }

    /**
     * Load composite data from storage and erase them
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when some data has been loaded
     */
    private static _getDictionaryFieldSupported(
        dictionaryFieldSupported: DictionaryFieldSupported,
    ): DictionaryFieldSupported {
        return dictionaryFieldSupported ?? ({} as DictionaryFieldSupported);
    }
}
